/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:element-boxscale
 *
 * <refsect2>
 * <para>
 * Performs video resizing (using <link linkend="GstVideoScale">videoscale</link>)
 * while possibly adding top and bottom (or left and right) bars
 * (using <link linkend="GstVideoBox">videobox</link>
 * to preserve the aspect-ratio of the input.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * It is akin to transcode's --keep_asr option [Thomas Oestreich]
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>

#include <gst/video/video.h>

#define GST_TYPE_BOX_SCALE \
  gst_box_scale_get_type ()
#define GST_BOX_SCALE(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), GST_TYPE_BOX_SCALE, GstBoxScale))
#define GST_BOX_SCALE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), GST_TYPE_BOX_SCALE, GstBoxScaleClass))
#define GST_IS_BOX_SCALE(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), GST_TYPE_BOX_SCALE))
#define GST_IS_BOX_SCALE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), GST_TYPE_BOX_SCALE))

typedef struct _GstBoxScale GstBoxScale;
typedef struct _GstBoxScaleClass GstBoxScaleClass;

struct _GstBoxScale
{
  GstBin parent;

  GstElement *scale, *box;

  GstPadSetCapsFunction setcaps;
};

struct _GstBoxScaleClass
{
  GstBinClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (box_scale_debug);
#define GST_CAT_DEFAULT box_scale_debug

/* properties */
enum
{
  PROP_0
};

static gboolean gst_box_scale_setcaps (GstPad * pad, GstCaps * caps);

static GstStateChangeReturn gst_box_scale_change_state (GstElement * element,
    GstStateChange transition);

/* properties */
static void gst_box_scale_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_box_scale_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstBoxScale, gst_box_scale, GstBin, GST_TYPE_BIN);

static void
gst_box_scale_base_init (gpointer klass)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_set_details_simple (element_class, "BoxScaler",
      "Filter/Effect/Video",
      "Resizes video and/or boxes if needed to preserve aspect ratio",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");
}

static void
gst_box_scale_class_init (GstBoxScaleClass * klass)
{
  GstElementClass *element_class;
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);
  element_class = GST_ELEMENT_CLASS (klass);

  GST_DEBUG_CATEGORY_INIT (box_scale_debug, "y4mtrans", 0, "y4m wrapper");

  gobject_class->set_property = gst_box_scale_set_property;
  gobject_class->get_property = gst_box_scale_get_property;

  element_class->change_state = GST_DEBUG_FUNCPTR (gst_box_scale_change_state);

#if 0
  /* loading a plugin while building a class during plugin loading
   * seems to be problematic */
  /* some proxy'ing for containing elements */
  el = gst_element_factory_make ("videoscale", NULL);
  el_class = GST_ELEMENT_GET_CLASS (el);
  gst_element_class_add_pad_template (element_class,
      gst_element_class_get_pad_template (el_class, "sink"));
  gst_object_unref (el);
#endif
}

static void
gst_box_scale_init (GstBoxScale * bin, GstBoxScaleClass * klass)
{
  GstPad *pad, *gpad;

  bin->scale = gst_element_factory_make ("videoscale", "videoscale");
  if (!bin->scale) {
    g_warning ("Can't find videoscale element, boxscale will not work");
    return;
  }
  if (!gst_bin_add (GST_BIN (bin), bin->scale)) {
    g_warning ("Could not add videoscale element, boxscale will not work");
    gst_object_unref (bin->scale);
    bin->scale = NULL;
    return;
  }

  bin->box = gst_element_factory_make ("videobox", "videobox");
  if (!bin->scale) {
    g_warning ("Can't find videobox element, boxscale will not work");
    return;
  }
  if (!gst_bin_add (GST_BIN (bin), bin->box)) {
    g_warning ("Could not add videobox element, boxscale will not work");
    gst_object_unref (bin->box);
    bin->box = NULL;
    return;
  }

  /* get the sinkpad */
  pad = gst_element_get_static_pad (bin->scale, "sink");

  /* ghost the sink pad to ourself */
  gpad = gst_ghost_pad_new ("sink", pad);
  gst_element_add_pad (GST_ELEMENT (bin), gpad);

  gst_object_unref (pad);

  bin->setcaps = GST_PAD_SETCAPSFUNC (gpad);
  gst_pad_set_setcaps_function (gpad,
      GST_DEBUG_FUNCPTR (gst_box_scale_setcaps));

  /* get the srcpad */
  pad = gst_element_get_static_pad (bin->box, "src");

  /* ghost the sink pad to ourself */
  gpad = gst_ghost_pad_new ("src", pad);
  gst_element_add_pad (GST_ELEMENT (bin), gpad);

  gst_object_unref (pad);

  if (!gst_element_link (bin->scale, bin->box))
    g_warning ("Could not link internal elements, boxscale will not work");
}

static gboolean
gst_box_scale_setcaps (GstPad * pad, GstCaps * caps)
{
  GstBoxScale *bin;
  GstPad *src;
  GstCaps *outcaps;
  GstStructure *ins, *outs;
  gint in_w, in_h, out_w, out_h;
  gint count;
  gint top = 0, bottom = 0, left = 0, right = 0;
  gboolean ret = TRUE;

  bin = GST_BOX_SCALE (GST_PAD_PARENT (pad));

  ins = gst_caps_get_structure (caps, 0);
  g_return_val_if_fail (ins != NULL, FALSE);

  src = gst_element_get_static_pad (GST_ELEMENT (bin), "src");
  g_return_val_if_fail (src != NULL, FALSE);
  outcaps = gst_pad_get_allowed_caps (src);
  gst_object_unref (src);
  g_return_val_if_fail (outcaps != NULL, FALSE);
  outs = gst_caps_get_structure (outcaps, 0);
  g_return_val_if_fail (outs != NULL, FALSE);

  count = 0;
  if (gst_structure_get_int (outs, "width", &out_w))
    ++count;
  if (gst_structure_get_int (outs, "height", &out_h))
    ++count;

  /* if no restriction in one dimension, no boxing needed, just scaling */
  if (count == 2) {
    gint new, par_num = 1, par_denom = 1;
    const GValue *par;

    ret &= gst_structure_get_int (ins, "width", &in_w);
    ret &= gst_structure_get_int (ins, "height", &in_h);
    if (!ret)
      goto done;

    par = gst_structure_get_value (ins, "pixel-aspect-ratio");
    if (par != NULL && GST_VALUE_HOLDS_FRACTION (par)) {
      par_num = gst_value_get_fraction_numerator (par);
      par_denom = gst_value_get_fraction_denominator (par);
    }

    in_w *= par_num;
    in_h *= par_denom;

    // TODO somehow consider outgoing par ??
    new = (gint) gst_util_uint64_scale_int (out_w, in_h, in_w);
    if (new <= out_h) {         /* need top and bottom bar */
      top = (out_h - new) / 2;
      bottom = (out_h - new) - top;
    } else {
      new = (gint) gst_util_uint64_scale_int (out_h, in_w, in_h);
      left = (out_w - new) / 2; /* this must be positive in this case */
      right = (out_w - new) - left;
    }
  }

  g_object_set (bin->box, "left", -left, NULL);
  g_object_set (bin->box, "right", -right, NULL);
  g_object_set (bin->box, "top", -top, NULL);
  g_object_set (bin->box, "bottom", -bottom, NULL);

done:
  gst_caps_unref (outcaps);

  return (ret && bin->setcaps (pad, caps));
}

static void
gst_box_scale_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstBoxScale *trans;

  g_return_if_fail (GST_IS_BOX_SCALE (object));

  trans = GST_BOX_SCALE (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_box_scale_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstBoxScale *trans;

  g_return_if_fail (GST_IS_BOX_SCALE (object));

  trans = GST_BOX_SCALE (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstStateChangeReturn
gst_box_scale_change_state (GstElement * element, GstStateChange transition)
{
  GstStateChangeReturn ret;

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
    goto done;

  switch (transition) {
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      break;
    default:
      break;
  }

done:
  return ret;
}
