//*****************************************************************************
//                              PnlGnuCapFO.cpp                               *
//                             -----------------                              *
// Started     : 26/04/2004                                                   *
// Last Update : 21/08/2009                                                   *
// Copyright   : (C) 2004 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/panels/PnlGnuCapFO.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlGnuCapFO, PnlAnaBase )

  EVT_BUTTON( ID_BTN_SETUP, PnlGnuCapFO::OnBtnSetup )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlGnuCapFO::PnlGnuCapFO( wxWindow * poWin )
                        : PnlAnaBase( poWin ), m_oDlgCfgGEN( poWin )
{
  bSetSimrEng( eSIMR_GNUCAP );
  bSetAnaType( eCMD_FO );

  Create( );  // Create the analysis panel
  bClear( );  // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlGnuCapFO::~PnlGnuCapFO( )
{
}

//*****************************************************************************
// Create the display objects.

void  PnlGnuCapFO::Create( void )
{
  wxStaticBox * poSbxSigSrc;
  int           ix, iy;

  // Set the frequency sweep parameter labels
  m_oSbxSwpPars.GetSize( &ix, &iy );
  m_oSbxSwpPars.SetSize(  ix, 167 );
  m_oSbxSwpPars.SetLabel( wxT(" Frequency Sweep ") );
  m_oPnlStart  .bSetName( wxT("Start Frequency") );
  m_oPnlStop   .bSetName( wxT("Stop Frequency") );
  m_oPnlStep   .bSetName( wxT("Step Increment") );

  // Set sweep parameter units
  m_oPnlStart.bSetUnitsType( eUNITS_FREQ );
  m_oPnlStop .bSetUnitsType( eUNITS_FREQ );
  m_oPnlStep .bSetUnitsType( eUNITS_FREQ );

  // Create and add generator controls
  poSbxSigSrc = new wxStaticBox( this, ID_UNUSED, wxT(" Signal Source "),
                           wxPoint( 6, 175 ),  wxSize( 241, 55 ) );
  m_oChoSrcName.Create( this, ID_CHO_SRCNAME,
                        wxPoint( 18, 195 ),  wxSize( 107, 29 ) );
  m_oBtnSetup  .Create( this, ID_BTN_SETUP, wxT("Setup ..."),
                        wxPoint( 130, 195 ), wxSize( 105, 29 ) );
  poSbxSigSrc->SetOwnFont( FONT_SLANT );

  CreateOutputs( );           // Create and add output parameters
  PnlAnaBase::CreateInitC( ); // Create and add the initial conditions controls
  PnlAnaBase::CreateTemp( );  // Create and add the analysis temp. controls
}

//*****************************************************************************

void  PnlGnuCapFO::GetConfig( void )
{
  wxConfig * poConfig;
  long       li1;

  // Set the output parameters to be extracted
  poConfig = (wxConfig *) wxConfig::Get( );
  if( poConfig == NULL ) return;

  poConfig->SetPath( wxT("/GNU-Cap") );

  poConfig->Read( wxT("FO_Mag") , &li1, (long) 1 );
  m_oCbxMag   .SetValue( li1 != 0 ? TRUE : FALSE );

  poConfig->Read( wxT("FO_Phase") , &li1, (long) 0 );
  m_oCbxPhase .SetValue( li1 != 0 ? TRUE : FALSE );

  poConfig->Read( wxT("FO_MagRel"), &li1, (long) 1 );
  m_oCbxMagRel.SetValue( li1 != 0 ? TRUE : FALSE );

  poConfig->Read( wxT("FO_MagDB") , &li1, (long) 0 );
  m_oCbxMagDb .SetValue( li1 != 0 ? TRUE : FALSE );
}

//*****************************************************************************

void  PnlGnuCapFO::SetConfig( void )
{
  wxConfig * poConfig;

  poConfig = (wxConfig *) wxConfig::Get( );
  if( poConfig == NULL ) return;

  poConfig->SetPath( wxT("/GNU-Cap") );

  poConfig->Write( wxT("FO_Mag")   , m_oCbxMag   .GetValue( ) );
  poConfig->Write( wxT("FO_Phase") , m_oCbxPhase .GetValue( ) );
  poConfig->Write( wxT("FO_MagRel"), m_oCbxMagRel.GetValue( ) );
  poConfig->Write( wxT("FO_MagDB") , m_oCbxMagDb .GetValue( ) );
}

//*****************************************************************************
// Create the parameters to included in the results file output.

void  PnlGnuCapFO::CreateOutputs( void )
{
  if( bIsCreatedOutputs( ) ) return;

  // Create and add output parameters
  m_oSbxOutputs.Create( this, ID_UNUSED,     wxT(" Outputs "),
                        wxPoint( 455,  5 ), wxSize( 110, 122 ) );
  m_oCbxMag    .Create( this, ID_CBX_MAG,    wxT("Magnitude"),
                        wxPoint( 461, 24 ) );
  m_oCbxPhase  .Create( this, ID_CBX_PHASE,  wxT("Phase"),
                        wxPoint( 461, 49 ) );
  m_oCbxMagDb  .Create( this, ID_CBX_MAGDB,  wxT("Mag. in dB"),
                        wxPoint( 461, 74 ) );
  m_oCbxMagRel .Create( this, ID_CBX_MAGREL, wxT("Mag. is Rel."),
                        wxPoint( 461, 99 ) );

  m_oSbxOutputs.SetOwnFont( FONT_SLANT );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapFO::bClear( void )
{
  bool  bRtn=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )    bRtn = FALSE;

  // Set default sweep values
  m_oPnlStart.bSetValue( (float)   1.0 );
  m_oPnlStop .bSetValue( (float) 100.0 );
  m_oPnlStep .bSetValue( (float)   1.0 );
  m_oPnlStart.bSetUnits( wxT("kHz") );
  m_oPnlStop .bSetUnits( wxT("kHz") );
  m_oPnlStep .bSetUnits( wxT("kHz") );

  // Set default initial condition values
  if( ! bSetInitC( eINITC_WARM ) ) bRtn = FALSE;

  // Clear the signal generator configuration dialog
  m_oDlgCfgGEN.bClear( );

  // Set the output parameter check box default values
  m_oCbxMag   .SetValue( TRUE  );
  m_oCbxPhase .SetValue( FALSE );
  m_oCbxMagDb .SetValue( TRUE  );
  m_oCbxMagRel.SetValue( FALSE );

  return( bRtn );
}

//*****************************************************************************
// Load information from a simulation object.
//
// Argument List :
//   roSimn - A SimnGnuCap object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapFO::bLoad( SimnGnuCap & roSimn )
{
  bool  bRtn=TRUE;

  // Load the components into the signal source choice box
  PnlAnaBase::LoadSrcNames( roSimn.m_oaCpnts, wxT("VI") );

  // Set the sweep source
  if( ! PnlAnaBase::bSetSrcCpnt( roSimn.m_oCpntSwpSrc ) )    bRtn = FALSE;

  // Set the sweep values
  if( ! m_oPnlStart.bSetValue( roSimn.m_oCmdFO.m_osStart ) ) bRtn = FALSE;
  if( ! m_oPnlStop .bSetValue( roSimn.m_oCmdFO.m_osStop  ) ) bRtn = FALSE;
  if( ! m_oPnlStep .bSetValue( roSimn.m_oCmdFO.m_osStep  ) ) bRtn = FALSE;
  if( m_oPnlStart.dfGetValue( ) == 0.0 )
        m_oPnlStart.bSetUnits( m_oPnlStop.rosGetUnits( ) );

  // Set the initial conditions
  if( roSimn.m_oCmdFO.m_eInitC != eINITC_NONE )
    m_oRbxInitC.SetSelection( roSimn.m_oCmdFO.m_eInitC );

  // Load the generator setup dialog
  if( ! m_oDlgCfgGEN.bSetValues( roSimn.m_oCmdGEN ) )        bRtn = FALSE;

  if( roSimn.eGetAnaType( ) == eCMD_FO )
  {
    // Set the parameters to derive
    m_oCbxVoltage.SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] );
    m_oCbxCurrent.SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] );
    m_oCbxPower  .SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] );
    m_oCbxResist .SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] );

    // Set the output parameters to be extracted
    GetConfig( );
  }

  // Set the analysis temperature
  if( ! m_oPnlTemp.bSetValue( roSimn.m_oCmdFO.m_osTempC ) )  bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Save information to a simulation object.
//
// Argument List :
//   roSimn - A SimnGnuCap object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapFO::bSave( SimnGnuCap & roSimn )
{
  wxString  os1;

  m_osErrMsg.Empty( );

  // Set the sweep values
  roSimn.m_oCmdFO.m_osStart = m_oPnlStart.rosGetValue( );
  roSimn.m_oCmdFO.m_osStop  = m_oPnlStop .rosGetValue( );
  roSimn.m_oCmdFO.m_osStep  = m_oPnlStep .rosGetValue( );

  // Set the initial conditions
  roSimn.m_oCmdFO.m_eInitC = (eInitCType) m_oRbxInitC.GetSelection( );

  // Set the component to be used as a signal source
  if( m_oChoSrcName.GetStringSelection( ) != wxT("None") )
  {
    // Get the unmodified signal source component, set it as the signal source
    os1 = m_oChoSrcName.GetStringSelection( );
    (Component &) roSimn.m_oCpntSwpSrc = roSimn.roGetCpnt( os1 );
    roSimn.m_oCpntSwpSrc.m_osValue = wxT("GENERATOR(1)");
    // Get the GENERATOR command from the Generator Setup dialog
    m_oDlgCfgGEN.bGetValues( roSimn.m_oCmdGEN );
    if( ! roSimn.m_oCmdGEN.bIsValid( ) )
    {
      os1 = wxT("The signal source ")
          + roSimn.m_oCmdGEN.rosGetErrMsg( ) + wxT(".   ");
      SetErrMsg( os1 );
    }
  }
  else
  {
    roSimn.m_oCpntSwpSrc.bClear( );
    roSimn.m_oCmdGEN    .bClear( );
  }

  // Set the analysis type
  roSimn.m_oCmdPR.m_eAnaType = eCMD_FO;

  // Store the parameters to derive
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] = m_oCbxVoltage.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] = m_oCbxCurrent.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] = m_oCbxPower  .GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] = m_oCbxResist .GetValue( );

  // Store the output parameters to be extracted
  if( m_oCbxMag.IsChecked( ) || m_oCbxPhase.IsChecked( ) ) SetConfig( );
  else SetErrMsg( wxT("An Outputs value hasn't been specified.") );

  // Set the analysis temperature
  roSimn.m_oCmdFO.m_osTempC = m_oPnlTemp.rosGetValue( );

  // Create the command strings
  roSimn.m_oCmdFO.bFormat( );
  roSimn.m_oCmdPR.bFormat( );

  // Check for errors
  if( ! roSimn.m_oCmdFO.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdFO.rosGetErrMsg( ) );
  if( ! roSimn.m_oCmdPR.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdPR.rosGetErrMsg( ) );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Generator component setup button control event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapFO::OnBtnSetup( wxCommandEvent & roEvtCmd )
{
  m_oDlgCfgGEN.CenterOnParent( );
  m_oDlgCfgGEN.ShowModal( );
}

//*****************************************************************************
