//*****************************************************************************
//                                FrmMain.cpp                                 *
//                               -------------                                *
//  Started     : 18/08/2003                                                  *
//  Last Update : 25/10/2007                                                  *
//  Copyright   : (C) 2003 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "main/FrmMain.hpp"

// The application icon
#if defined( __WXGTK__ ) || defined( __WXMOTIF__ ) || defined( __WXMAC__ ) || \
    defined( __WXMGL__ ) || defined( __WXX11__ )
  #include "icons/gspiceui-32x32.xpm"
  #include "icons/file-open.xpm"
  #include "icons/file-import.xpm"
  #include "icons/file-reload.xpm"
  #include "icons/file-close.xpm"
  #include "icons/sim-create.xpm"
  #include "icons/sim-run.xpm"
  #include "icons/sim-stop.xpm"
  #include "icons/sim-plot.xpm"
  #include "icons/help.xpm"
#endif

//*****************************************************************************
// Implement an event table in which the events are routed to their respective
// handler functions in the class. If -1 is given as the ID, the given handler
// will be invoked for any event of the specified type.

BEGIN_EVENT_TABLE( FrmMain, wxFrame )

  EVT_MENU( ID_MNU_OPEN,    FrmMain::OnOpen      )
  EVT_MENU( ID_MNU_IMPORT,  FrmMain::OnImport    )
  EVT_MENU( ID_MNU_RELOAD,  FrmMain::OnReload    )
  EVT_MENU( ID_MNU_CLOSE,   FrmMain::OnClose     )
  EVT_MENU( ID_MNU_EXIT,    FrmMain::OnAppExit   )
  EVT_MENU( ID_MNU_CREATE,  FrmMain::OnSimCreate )
  EVT_MENU( ID_MNU_RUN,     FrmMain::OnSimRun    )
  EVT_MENU( ID_MNU_STOP,    FrmMain::OnSimStop   )
  EVT_MENU( ID_MNU_PLOT,    FrmMain::OnPlot      )
  EVT_MENU( ID_MNU_GNUCAP,  FrmMain::OnSelSimr   )
  EVT_MENU( ID_MNU_NGSPICE, FrmMain::OnSelSimr   )
  EVT_MENU( ID_MNU_PREFS,   FrmMain::OnPrefs     )
  EVT_MENU( ID_MNU_MANUAL,  FrmMain::OnManual    )
  EVT_MENU( ID_MNU_ABOUT,   FrmMain::OnAbout     )

  EVT_TOOL( ID_TBR_OPEN,    FrmMain::OnOpen      )
  EVT_TOOL( ID_TBR_IMPORT,  FrmMain::OnImport    )
  EVT_TOOL( ID_TBR_RELOAD,  FrmMain::OnReload    )
  EVT_TOOL( ID_TBR_CLOSE,   FrmMain::OnClose     )
  EVT_TOOL( ID_TBR_CREATE,  FrmMain::OnSimCreate )
  EVT_TOOL( ID_TBR_RUN,     FrmMain::OnSimRun    )
  EVT_TOOL( ID_TBR_STOP,    FrmMain::OnSimStop   )
  EVT_TOOL( ID_TBR_PLOT,    FrmMain::OnPlot      )
  EVT_TOOL( ID_TBR_HELP,    FrmMain::OnManual    )

  EVT_TOOL_ENTER( -1, FrmMain::OnTbrEnter )

  EVT_CLOSE( FrmMain::OnSysExit )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Argument List:
//   poApp - A pointer to the class that created this object

FrmMain::FrmMain( const wxApp * poApp ) :
               wxFrame( (wxFrame *) NULL, -1, wxT(""), wxDefaultPosition,
                        wxDefaultSize, wxDEFAULT_FRAME_STYLE&~wxMAXIMIZE_BOX ),
               m_oFileTsks( this ), m_oHelpTsks( this ), m_oDlgPrefs( this )
{
  // Indicate that the main frame is open
  m_bIsOpen = TRUE;

  // Set pointer to parent application
  m_poApp = poApp;

  // Get the global configuration object
  m_poConfig = (wxConfig *) wxConfig::Get( );

  // Automatically update the frame layout when it is resized
  SetAutoLayout( TRUE );

  // Initialize pointers
  m_poPrcSimr = NULL;
  m_poNbkSimr = NULL;
}

//*****************************************************************************
// Destructor.

FrmMain::~FrmMain( )
{
  if( m_poNbkSimr != NULL ) { delete m_poNbkSimr; m_poNbkSimr = NULL; }
  if( m_poPrcSimr != NULL ) { delete m_poPrcSimr; m_poPrcSimr = NULL; }
}

//*****************************************************************************
// Initialize the main frame.

void  FrmMain::Initialize( void )
{
  // Set the frame icon and font
  SetIcon( wxICON( gspiceui ) );
  SetFont( FONT_NORM );

  // Set the frame title
  m_oFileTsks.bSetTitle( );

  // Call all the initialization functions
  InitMenuBar ( );
  InitToolBar ( );
  InitLstBoxs ( );
  InitNbkTCtls( );
  InitStatBar ( );
  InitToolTips( );
  InitLogFiles( );
  InitSimrEng ( );

  // Layout the frame's display objects and set it's size and position
  DoLayout( );
  InitPosnSize( );

  // Select the analysis page ??? 14/11/2005
  m_poConfig->SetPath( wxT("/Spice") );
  m_poNbkSimr->bSetPage( m_poConfig->Read( wxT("Analysis"), wxT("DC") ) );
}

//*****************************************************************************
// Initialize the menu bar.

void  FrmMain::InitMenuBar( void )
{
  // Create the menu bar and menus
  wxMenuBar * poMenuBar  = new wxMenuBar;
  wxMenu    * poMenuFile = new wxMenu;
  wxMenu    * poMenuSimu = new wxMenu;
  wxMenu    * poMenuSets = new wxMenu;
  wxMenu    * poMenuHelp = new wxMenu;

  poMenuBar ->SetFont( FONT_NORM );

  // Load the menus with items
  poMenuFile->Append( ID_MNU_OPEN,             wxT(" &Open ... ") );
  poMenuFile->Append( ID_MNU_IMPORT,           wxT(" &Import ... ") );
  poMenuFile->Append( ID_MNU_RELOAD,           wxT(" &Reload ") );
  poMenuFile->Append( ID_MNU_CLOSE,            wxT(" &Close ") );
  poMenuFile->AppendSeparator( );
  poMenuFile->Append( ID_MNU_EXIT,             wxT(" E&xit ") );

  poMenuSimu->Append( ID_MNU_CREATE,           wxT(" &Create    ") );
  poMenuSimu->Append( ID_MNU_RUN,              wxT(" &Run ") );
  poMenuSimu->Append( ID_MNU_STOP,             wxT(" &Stop ") );
  poMenuSimu->Append( ID_MNU_PLOT,             wxT(" &Plot ... ") );

  poMenuSets->AppendRadioItem( ID_MNU_GNUCAP,  wxT("  &GNU-Cap ") );
  poMenuSets->AppendRadioItem( ID_MNU_NGSPICE, wxT("  &NG-Spice ") );
  poMenuSets->AppendSeparator( );
  poMenuSets->Append( ID_MNU_PREFS,            wxT("&Preferences ... ") );

  poMenuHelp->Append( ID_MNU_MANUAL,           wxT(" &Manual ... ") );
  poMenuHelp->AppendSeparator( );
  poMenuHelp->Append( ID_MNU_ABOUT,            wxT(" &About ... ") );

  // Load the menubar with menus
  poMenuBar ->Append( poMenuFile,              wxT(" &File ") );
  poMenuBar ->Append( poMenuSimu,              wxT(" &Simulate ") );
  poMenuBar ->Append( poMenuSets,              wxT(" &Settings ") );
  poMenuBar ->Append( poMenuHelp,              wxT(" &Help ") );

  SetMenuBar( poMenuBar );

  // Select the appropriate simulation engine menu item
  m_poConfig->SetPath( wxT("/Spice") );
  if( m_poConfig->Read( wxT("Engine"), wxT("GNUCAP") ) == wxT("NGSPICE") )
       poMenuBar->FindItem( FrmMain::ID_MNU_NGSPICE )->Check( );
  else poMenuBar->FindItem( FrmMain::ID_MNU_GNUCAP  )->Check( );
}

//*****************************************************************************
// Initialize the tool bar.

void  FrmMain::InitToolBar( void )
{
  wxBitmap  * poPixMap[ 9 ];
  wxToolBar * poToolBar;

  // Create the tool bar
  poToolBar = CreateToolBar( wxHORIZONTAL | wxTB_FLAT );

  // Create the bitmaps for the tools
  poPixMap[ 0 ] = new wxBitmap( file_open_xpm );
  poPixMap[ 1 ] = new wxBitmap( file_import_xpm );
  poPixMap[ 2 ] = new wxBitmap( file_reload_xpm );
  poPixMap[ 3 ] = new wxBitmap( file_close_xpm );
  poPixMap[ 4 ] = new wxBitmap( sim_create_xpm );
  poPixMap[ 5 ] = new wxBitmap( sim_run_xpm );
  poPixMap[ 6 ] = new wxBitmap( sim_stop_xpm );
  poPixMap[ 7 ] = new wxBitmap( sim_plot_xpm );
  poPixMap[ 8 ] = new wxBitmap( help_xpm );

  // Add the tools to the toolbar
  poToolBar->AddTool( ID_TBR_OPEN,   wxT(""), *(poPixMap[ 0 ]),
                      wxT("Open Schem/Netlist") );
  poToolBar->AddTool( ID_TBR_IMPORT, wxT(""), *(poPixMap[ 1 ]),
                      wxT("Import Schematic") );
  poToolBar->AddTool( ID_TBR_RELOAD, wxT(""), *(poPixMap[ 2 ]),
                      wxT("Reload Schem/Netlist") );
  poToolBar->AddTool( ID_TBR_CLOSE,  wxT(""), *(poPixMap[ 3 ]),
                      wxT("Close Schem/Netlist") );
  poToolBar->AddSeparator( );
  poToolBar->AddTool( ID_TBR_CREATE, wxT(""), *(poPixMap[ 4 ]),
                      wxT("Create Simulation") );
  poToolBar->AddTool( ID_TBR_RUN,    wxT(""), *(poPixMap[ 5 ]),
                      wxT("Run Simulation") );
  poToolBar->AddTool( ID_TBR_STOP,   wxT(""), *(poPixMap[ 6 ]),
                      wxT("Stop Simulation") );
  poToolBar->AddSeparator( );
  poToolBar->AddTool( ID_TBR_PLOT,   wxT(""), *(poPixMap[ 7 ]),
                      wxT("Plot Simulation Results") );
  poToolBar->AddSeparator( );
  poToolBar->AddTool( ID_TBR_HELP,   wxT(""), *(poPixMap[ 8 ]),
                      wxT("View Manual") );

  // Realize the toolbar
  poToolBar->Realize( );

  // Delete the bitmaps
  for( int i1=0; i1<9; i1++ ) delete poPixMap[ i1 ];
}

//*****************************************************************************
// Initialize the test node and component list boxes.

void  FrmMain::InitLstBoxs( void )
{
  long  lStyle;

  // Create the node and component list labels
  lStyle = wxALIGN_CENTRE;
  m_oLblNodes.Create( this, ID_UNUSED, wxT("Nodes"),      wxDefaultPosition,
                      wxDefaultSize, lStyle );
  m_oLblCpnts.Create( this, ID_UNUSED, wxT("Components"), wxDefaultPosition,
                      wxDefaultSize, lStyle );

  // Create the node and component list boxes
  lStyle = wxLB_MULTIPLE | wxLB_ALWAYS_SB;
  m_oLbxNodes.Create( this, ID_LBX_NODES, wxDefaultPosition,
                      wxSize(  90, -1 ), 0, NULL, lStyle );
  m_oLbxCpnts.Create( this, ID_LBX_CPNTS, wxDefaultPosition,
                      wxSize( 110, -1 ), 0, NULL, lStyle );
}

//*****************************************************************************
// Set the electronic circuit simulator engine to use.

void  FrmMain::InitSimrEng( void )
{
  // Does anything need to be done
  if( m_poNbkSimr != NULL )
  {
    if( GetMenuBar( )->FindItem( FrmMain::ID_MNU_GNUCAP  )->IsChecked( ) &&
        m_poNbkSimr->eGetSimrType( )==eSIMR_GNUCAP  ) return;
    if( GetMenuBar( )->FindItem( FrmMain::ID_MNU_NGSPICE )->IsChecked( ) &&
        m_poNbkSimr->eGetSimrType( )==eSIMR_NGSPICE ) return;
  }

  // Delete any previous simulator objects
  if( m_poNbkSimr != NULL )
  { // Delete the old simulator notebook object
    GetSizer( )->Detach( m_poNbkSimr );
    delete m_poNbkSimr;
    m_poNbkSimr = NULL;
  }
  if( m_poPrcSimr != NULL )
  { // Delete the old simulator process object
    delete m_poPrcSimr;
    m_poPrcSimr = NULL;
  }

  // Create the new simulator objects
  if( GetMenuBar( )->FindItem( FrmMain::ID_MNU_GNUCAP )->IsChecked( ) )
  {
    // Create the simulator process object and the analysis notebook control
    m_poPrcSimr = new PrcGnuCap( );
    m_poNbkSimr = new NbkGnuCap( this, ID_NBK_ANALYS );
    m_oSim.bSetSimrType( eSIMR_GNUCAP );
    // Check the simulator item in the Options menu and the status bar message
    GetMenuBar( )->FindItem( FrmMain::ID_MNU_GNUCAP )->Check( );
    SetStatusText( wxT(" Simulation Engine : GNU-Cap"), 2 );
  }
  else
  {
    // Create the simulator process object and the analysis notebook control
    m_poNbkSimr = new NbkNgSpice( this, ID_NBK_ANALYS );
    m_poPrcSimr = new PrcNgSpice( );
    m_oSim.bSetSimrType( eSIMR_NGSPICE );
    // Check the simulator item in the Options menu and the status bar message
    GetMenuBar( )->FindItem( FrmMain::ID_MNU_NGSPICE )->Check( );
    SetStatusText( wxT(" Simulation Engine : NG-Spice"), 2 );
  }

  InitLogFiles( ); // Set the simulator process log file name
}

//*****************************************************************************
// Initialize the console notebook.

void  FrmMain::InitNbkTCtls( void )
{
  // Create the text control notebook
  m_oNbkTxtCtls.bCreate( this, ID_NBK_TXTCTLS );

  // Set the text control maximum lines
  m_oNbkTxtCtls.bSetLinesMax( m_oDlgPrefs.iGetMaxLines( ) );
}

//*****************************************************************************
// Initialize the status bar.
// Note : The first field in the status bar has a width of zero and is not used
//        since the frame has the bad habit of writing text into it as the user
//        moves the mouse over frame controls.

void  FrmMain::InitStatBar( void )
{
  int  iaWidths[3];

  // Create the status bar
  CreateStatusBar( 3 );

  // Determine the status bar field widths
  iaWidths[ 0 ] = 3;   // Vanishingly thin
  iaWidths[ 1 ] = -1;  // Variable width
  iaWidths[ 2 ] = 215; // Fixed width
  SetStatusWidths( 3, iaWidths );

  // Set status bar text
  SetStatusText( wxT(" Welcome to GNU Spice GUI"), 1 );
  SetStatusText( wxT(" Simulation Engine : None"), 2 );
}

//*****************************************************************************
// Initialize the tool tips.

void  FrmMain::InitToolTips( void )
{
  long  li1;

  // Define tool tips for each control
  m_oLbxNodes.SetToolTip( wxT("Test Nodes") );
  m_oLbxCpnts.SetToolTip( wxT("Test Components") );

  // Set global tool tip attributes
  m_poConfig->SetPath( wxT("/Main") );
  m_poConfig->Read( wxT("ToolTips"), &li1, (long) 0 );
  wxToolTip::Enable( li1 != 0 ? TRUE : FALSE );
  wxToolTip::SetDelay( 700 );
}

//*****************************************************************************
// Initialize the log file names in the process objects.

void  FrmMain::InitLogFiles( void )
{
  wxFileName  ofn1;
  wxString    os1;

  // Check that a schematic or netlist file has been defined
  ofn1 = m_oFileTsks.rosGetNetLstFile( );
  if( ! ofn1.IsOk( )       ) return;
  if( ! ofn1.FileExists( ) ) return;

  // Get the path to the schematic or netlist file
  os1 = ofn1.GetPath( ) + wxT("/gspiceui.log");

  // Set the log file path for the gNetList process object
  m_oFileTsks.bSetLogFile( os1 );

  // Set the log file path for the simulation process object
  if( m_poPrcSimr != NULL ) m_poPrcSimr->bSetLogFile( os1 );

  // Set the log file path for the gWave process object
  m_oPrcGWave.bSetLogFile( os1 );
}

//*****************************************************************************
// Initialize the frames position amd size.

void  FrmMain::InitPosnSize( void )
{
  int  ix, iy, iw, ih;

  // Get the default position and size
  GetPosition( &ix, &iy );
  GetClientSize( &iw, &ih );

  // Get the position and size from the configuration object
  ix = m_poConfig->Read( wxT("/Main/PosnX"), ix );
  iy = m_poConfig->Read( wxT("/Main/PosnY"), iy );
  iw = m_poConfig->Read( wxT("/Main/SizeW"), iw );
  ih = m_poConfig->Read( wxT("/Main/SizeH"), ih );

  // Set the position and size
  Move( ix, iy );
  SetClientSize( iw, ih );
}

//*****************************************************************************
// Layout the main frame display objects.

void  FrmMain::DoLayout( void )
{
  wxGridBagSizer * poSizer;
  wxGBPosition  oGBPosn;
  wxGBSpan      oGBSpan;
  int  iFlags = wxALL | wxALIGN_CENTER | wxEXPAND;
  int  iBorder = 3;

  if( GetSizer( ) == NULL )
  { // Create the sizer
    poSizer = new wxGridBagSizer( 1, 1 );
  }
  else
  { // Detach the display objects from the sizer
    poSizer = (wxGridBagSizer *) GetSizer( );
    poSizer->Detach( &m_oLblNodes );
    poSizer->Detach( &m_oLbxNodes );
    poSizer->Detach( &m_oLblCpnts );
    poSizer->Detach( &m_oLbxCpnts );
    poSizer->Detach( m_poNbkSimr );
    poSizer->Detach( &m_oNbkTxtCtls );
  }

  // Add the nodes label
  oGBPosn.SetCol( 0 );          oGBPosn.SetRow( 0 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 1 );
  poSizer->Add( &m_oLblNodes,   oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the nodes list box
  oGBPosn.SetCol( 0 );          oGBPosn.SetRow( 1 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 19 );
  poSizer->Add( &m_oLbxNodes,   oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the components label
  oGBPosn.SetCol( 1 );          oGBPosn.SetRow( 0 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 1 );
  poSizer->Add( &m_oLblCpnts,   oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the components list box
  oGBPosn.SetCol( 1 );          oGBPosn.SetRow( 1 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 19 );
  poSizer->Add( &m_oLbxCpnts,   oGBPosn, oGBSpan, iFlags, iBorder );

  // Add the simulator notebook
  oGBPosn.SetCol( 2 );          oGBPosn.SetRow( 0 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 13 );
  poSizer->Add( m_poNbkSimr,    oGBPosn, oGBSpan, iFlags, iBorder );
  poSizer->SetItemMinSize( m_poNbkSimr, 575, 266 );

  // Add the console notebook
  oGBPosn.SetCol( 2 );          oGBPosn.SetRow( 13 );
  oGBSpan.SetColspan( 1 );      oGBSpan.SetRowspan( 7 );
  poSizer->Add( &m_oNbkTxtCtls, oGBPosn, oGBSpan, iFlags, iBorder );

  // Specify how the sizer will grow when resized
  poSizer->SetNonFlexibleGrowMode( wxFLEX_GROWMODE_SPECIFIED );
  poSizer->AddGrowableCol( 0 );
  poSizer->AddGrowableCol( 1 );
  for( size_t szt1=13; szt1<20; szt1++ )
    poSizer->AddGrowableRow( szt1 );

  if( GetSizer( ) == NULL ) SetSizer( poSizer ); // Set the main frame sizer
  poSizer->SetSizeHints( this ); // Set min. and init. sizes as calc'd by sizer
}

//*****************************************************************************
// Load information from the Simulation object.
//
// Argument List:
//   bReload - Flag indicating if it is a load or reload operation
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bLoadSim( bool bReload )
{
  TextCtrl * poTxtCtl;
  uint       ui1;
  size_t     sz1;

  // Load the nodes into the list box
  const wxArrayString & roasNodeLbls = m_oSim.roasGetNodeLbls( );
  for( ui1=0; ui1<roasNodeLbls.GetCount( ); ui1++ )
    m_oLbxNodes.Append( roasNodeLbls.Item( ui1 ) );
  // Select the test nodes
  const wxArrayString & roasTstNodes = m_oSim.roasGetTstNodes( );
  for( ui1=0; ui1<roasTstNodes.GetCount( ); ui1++ )
    m_oLbxNodes.SetStringSelection( roasTstNodes.Item( ui1 ) );

  // Load the components into the list box
  const wxArrayString & roasCpntLbls = m_oSim.roasGetCpntLbls( );
  for( ui1=0; ui1<roasCpntLbls.GetCount( ); ui1++ )
    m_oLbxCpnts.Append( roasCpntLbls.Item( ui1 ) );
  // Select the test components
  const wxArrayString & roasTstCpnts = m_oSim.roasGetTstCpnts( );
  for( ui1=0; ui1<roasTstCpnts.GetCount( ); ui1++ )
    m_oLbxCpnts.SetStringSelection( roasTstCpnts.Item( ui1 ) );

  // Load the raw circuit description into the text control
  m_oNbkTxtCtls.bClear( NbkTxtCtls::ePAGE_NETLIST );
  poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_NETLIST );
  poTxtCtl->bClear( );
  for( sz1=0; sz1<m_oSim.GetCount( ); sz1++ )
    poTxtCtl->bAppendLine( m_oSim.Item( sz1 ) );
  poTxtCtl->SetInsertionPoint( 0 ); // Go to top of the netlist text control

  // Load the analysis notebook with simulation information
  if( ! bReload ) m_poNbkSimr->bLoad( m_oSim );

  return( TRUE );
}

//*****************************************************************************
// Reload information from the Simulation object.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bReloadSim( void )
{
  wxArrayString  oasTstNodes;
  wxArrayString  oasTstCpnts;
  size_t         sz1;
  int            i1;

  // Parse the simulator commands
  InitSimrEng( );
  m_poPrcSimr->bParseSim( m_oSim );

  // Record the nodes that are currently selected
  for( i1=0; i1<(int) m_oLbxNodes.GetCount( ); i1++ )
    if( m_oLbxNodes.IsSelected( i1 ) )
      oasTstNodes.Add( m_oLbxNodes.GetString( i1 ) );

  // Record the components that are currently selected
  for( i1=0; i1<(int) m_oLbxCpnts.GetCount( ); i1++ )
    if( m_oLbxCpnts.IsSelected( i1 ) )
      oasTstCpnts.Add( m_oLbxCpnts.GetString( i1 ) );

  // Clear the lists of nodes and components
  m_oLbxNodes.Clear( );
  m_oLbxCpnts.Clear( );

  bLoadSim( TRUE );  // Load the Simulation object

  // Select nodes that where previously selected
  if( ! oasTstNodes.IsEmpty( ) )
  {
    for( sz1=0; sz1<oasTstNodes.GetCount( ); sz1++ )
    {
      i1 = m_oLbxNodes.FindString( oasTstNodes.Item( sz1 ) );
      if( i1 != wxNOT_FOUND ) m_oLbxNodes.Select( i1 );
    }
  }

  // Select components that where previously selected
  if( ! oasTstCpnts.IsEmpty( ) )
  {
    for( sz1=0; sz1<oasTstCpnts.GetCount( ); sz1++ )
    {
      i1 = m_oLbxCpnts.FindString( oasTstCpnts.Item( sz1 ) );
      if( i1 != wxNOT_FOUND ) m_oLbxCpnts.Select( i1 );
    }
  }

  // Load the analysis notebook with simulation information
  m_poNbkSimr->bLoad( m_oSim );

  return( TRUE );
}

//*****************************************************************************
// Save information to the Simulation object.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bSaveSim( void )
{
  wxArrayInt  oaiSelNodes, oaiSelCpnts;
  size_t      sz1;
  int         iSelNo;
  wxString    osSel;

  // Clear all previously selected test points
  m_oSim.ClrTstNodes( );
  m_oSim.ClrTstCpnts( );

  // Get an array of indexes to the selected test points
  m_oLbxNodes.GetSelections( oaiSelNodes );
  m_oLbxCpnts.GetSelections( oaiSelCpnts );

  // Have any test points (ie. nodes or components) been selected?
  if( oaiSelNodes.GetCount( )>0 || oaiSelCpnts.GetCount( )>0 )
  {
    // Load the test nodes into the simulation object
    for( sz1=0; sz1<oaiSelNodes.GetCount( ); sz1++ )
    {
      iSelNo = oaiSelNodes.Item( sz1 );
      osSel = m_oLbxNodes.GetString( iSelNo );
      m_oSim.bAddTstNode( osSel );
    }

    // Load the test components into the simulation object
    for( sz1=0; sz1<oaiSelCpnts.GetCount( ); sz1++ )
    {
      iSelNo = oaiSelCpnts.Item( sz1 );
      osSel = m_oLbxCpnts.GetString( iSelNo );
      m_oSim.bAddTstCpnt( (const wxString &) osSel );
    }

    // Setup the simulation parameters
    if( m_poNbkSimr->bSave( m_oSim ) )
    {
      // Create the simulation file
      if( ! m_poPrcSimr->bMakeSim( m_oSim ) )
        m_osErrMsg = wxT("Couldn't create simulation.");
    }
    else m_osErrMsg = m_poNbkSimr->rosGetErrMsg( );
  }
  else m_osErrMsg = wxT("No nodes or components have been selected.");

  return( m_osErrMsg.IsEmpty( ) );
}

//*****************************************************************************
// Display a dialog box containing an error message.
//
// Argument List:
//   rosTitle  - The dialogue box title
//   rosMsg    - The error message

void  FrmMain::DlgErrMsg( const wxString & rosTitle, const wxString & rosMsg )
{
  wxStringTokenizer  ostk1;
  wxString  os1, os2, os3;

  // Tokenize the message into lines
  ostk1.SetString( rosMsg, wxT("\n\r"), wxTOKEN_STRTOK );

  // Use the first complete sentence as the status bar message
  os1 << wxT(" Error :");
  while( ostk1.HasMoreTokens( ) )
  {
    // Get the next line
    os2 = ostk1.GetNextToken( ).Strip( wxString::both );

    // Remove the path from any file names
    if( os2.Freq( wxT('/') ) > 1 )
    {
      os3 << os2.AfterLast( wxT('/') ) << wxT(',');
      os2 = os3;
    }

    // Remove duplicate white spaces
    while( os2.Replace( wxT("  "), wxT(" ") ) ) continue;

    // Append the line to the message
    os1 << wxT(' ') << os2;

    // Is this the end of the sentence?
    if( os1.Last( ) == wxT('.') )
    {
      os1.Truncate( os1.Length( )-1 );
      break;
    }
  }

  // Set the status line message
  SetStatusText( os1, 1 );

  // Display the error message dialog
  wxMessageDialog  oMsgDlg( this, rosMsg, rosTitle, wxOK | wxICON_ERROR );
  oMsgDlg.ShowModal( );
}

//*****************************************************************************
// Check that the electronic circuit simulator engine is present and accounted
// for, if not display an error message.
//
// Argument List:
//   poPrcSimr - The utility object to be tested
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bIsOkSimEng( PrcSimrBase * poPrcSimr )
{
  wxString  osErrMsg;

  // Check that gnetlist exists and is accessible
  if( ! poPrcSimr->bBinExists( ) )
  {
    osErrMsg << wxT("\nCan't find ") << poPrcSimr->rofnGetBinary( ).GetFullName( )
             << wxT(" which is required to run electronic simulation.\n")
             << wxT("There is no path to it or it has not been installed.\n\n");
    DlgErrMsg( wxT("Configuration Fault"), osErrMsg );
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Check that the results plotter utility is present and accounted for, if not
// display an error message.
//
// Argument List:
//   poPrcGWave - The utility object to be tested
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  FrmMain::bIsOkGWave( PrcGWave * poPrcGWave )
{
  wxString  os1;

  // Check that the plotter utility exists and is accessible
  if( ! poPrcGWave->bBinExists( ) )
  {
    os1 << wxT("\nCan't find ") << poPrcGWave->rofnGetBinary( ).GetFullName( )
        << wxT(" which is required to plot simulation results.\n")
        << wxT("There is no path to it or it has not been installed.\n\n");
    DlgErrMsg( wxT("Configuration Fault"), os1 );
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Clear the object attributes.

bool  FrmMain::bClear( void )
{
  bool  bRtn=TRUE;

  // Clear all previously selected test points
  m_oLbxNodes.Clear( );
  m_oLbxCpnts.Clear( );

  // Clear simulation object attributes
  m_oSim.bClear( );

  // Clear the simulation object and the analysis panels
  if( ! m_poNbkSimr->bClear( ) )  bRtn = FALSE;

  // Clear the text controls
  if( ! m_oNbkTxtCtls.bClear( ) ) bRtn = FALSE;
  m_oNbkTxtCtls.Initialize( );

  // Terminate any simulation processes
  if( ! m_poPrcSimr->bKill( ) )   bRtn = FALSE;

  // Terminate any viewer processes
  if( ! m_oPrcGWave.bKill( ) )    bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
//                                                                            *
//                               Event Handlers                               *
//                                                                            *
//*****************************************************************************
// Open a circuit description file.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnOpen( wxCommandEvent & roEvtCmd )
{
  // Get a net list file name from the user and attempt to open it
  if( m_oFileTsks.bOpen( m_oSim ) )
  {
    // Check the appropriate simulation engine menu bar item
    if( m_oSim.eGetSimrType( ) != eSIMR_NONE )
    {
      if( m_oSim.eGetSimrType( ) == eSIMR_NGSPICE )
           GetMenuBar( )->FindItem( FrmMain::ID_MNU_NGSPICE )->Check( );
      else GetMenuBar( )->FindItem( FrmMain::ID_MNU_GNUCAP  )->Check( );
      InitSimrEng( );
    }

    // Parse the simulator commands
    m_poPrcSimr->bParseSim( m_oSim );

    // Attempt to load the simulation information
    if( bLoadSim( ) )
         SetStatusText( wxT(" Net list file opened successfully"), 1 );
    else SetStatusText( wxT(" Error/s encountered opening a net list file"), 1 );

    // Display the netlist text control and go to the top
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
    m_oNbkTxtCtls.bSetPosn( 0 );
  }
}

//*****************************************************************************
// Import a schematic file using gnetlist to convert the schematic to a circuit
// description.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnImport( wxCommandEvent & roEvtCmd )
{
  // Get schematic file name/s from the user and attempt to import them
  if( m_oFileTsks.bImport( m_oSim ) )
  {
    // Attempt to load the simulation information
    if( bLoadSim( ) )
         SetStatusText( wxT(" Schematic file/s imported successfully"), 1 );
    else SetStatusText( wxT(" Error/s encountered importing schematic file/s"), 1 );

    // Set the simulator type in the Simulation object
    m_oSim.bSetSimrType( m_poNbkSimr->eGetSimrType( ) );

    // Display the netlist text control and go to the top
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
    m_oNbkTxtCtls.bSetPosn( 0 );
  }
}

//*****************************************************************************
// Reload the schematic or net list file.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnReload( wxCommandEvent & roEvtCmd )
{
  // Reload schematic file/s or a net list file into the Simulation object
  if( m_oFileTsks.bReload( m_oSim ) )
  {
    // Attempt to load the simulation information
    if( bReloadSim( ) )
         SetStatusText( wxT(" Netlist file reloaded successfully"), 1 );
    else SetStatusText( wxT(" Error/s encountered reloading the netlist file"), 1 );

    // Display the netlist text control and go to the top
    m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_NETLIST );
    m_oNbkTxtCtls.bSetPosn( 0 );
  }
}

//*****************************************************************************
// Close the circuit description file.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnClose( wxCommandEvent & roEvtCmd )
{
  m_oFileTsks.bClose( );             // Delete temporary files
  bClear( );                         // Clear the object attributes
  m_oFileTsks.bSetTitle( );          // Set the frame title
  m_poNbkSimr->bSetPage( eANA_OP );  // Display first analysis panel
}

//*****************************************************************************
// Exit the application.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnAppExit( wxCommandEvent & roEvtCmd )
{
  Close( TRUE ); // Generates a wxCloseEvent which is handled by OnSysExit( )
}

//*****************************************************************************
// Create the simulation.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimCreate( wxCommandEvent & roEvtCmd )
{
  TextCtrl * poTxtCtl;

  // Change the cursor to the wait symbol
  ::wxBeginBusyCursor( );

  m_osErrMsg.Empty( );

  if( ! m_oSim.IsEmpty( ) )
  {
    if( bSaveSim( ) )
    {
      // Print the simulation to the simulation text control
      m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );
      poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_SIMULTN );
      poTxtCtl->bLoadFile( m_oSim.rofnGetSaveFile( ).GetFullPath( ) );
      poTxtCtl->SetEditable( TRUE );
    }
    else SetStatusText( wxT(" Error/s encountered creating simulation"), 1 );
  }
  else m_osErrMsg = wxT("There is no net list loaded.");

  // Change the cursor to the default
  ::wxEndBusyCursor( );

  if( m_osErrMsg.IsEmpty( ) )
    SetStatusText( wxT(" Simulation created successfully"), 1 );
  else
  { // Display an error message dialogue
    SetStatusText( wxT(" Error/s encountered creating simulation"), 1 );
    DlgErrMsg( wxT("Create Simulation Error"), m_osErrMsg );
  }
}

//*****************************************************************************
// Run the simulation.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimRun( wxCommandEvent & roEvtCmd )
{
  TextCtrl * poTxtCtl;
  NbkTxtCtls::ePageType  ePage;
  wxFileName  ofn1;
  wxString  os1;

  m_osErrMsg.Empty( );

  // Check if a schematic or net list file is currently open
  ofn1 = m_oFileTsks.rosGetNetLstFile( );
  if( ofn1.GetFullPath( ).IsEmpty( ) )
  {
    m_osErrMsg = wxT("There is no schematic or net list file currently loaded.");
    DlgErrMsg( wxT("Run Simulation Error"), m_osErrMsg );
    return;
  }

  // Change the cursor to the wait symbol
  ::wxBeginBusyCursor( );

  // Load the simulation text control
  poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_SIMULTN );
  if( poTxtCtl->IsModified( ) )
  { // Save the simulation in the text control to file
    poTxtCtl->SaveFile( m_oSim.rofnGetSaveFile( ).GetFullPath( ) );
  }
  else if( bSaveSim( ) )
  { // Load the simulation on file into the simulation text control
    poTxtCtl->bLoadFile( m_oSim.rofnGetSaveFile( ).GetFullPath( ) );
    // poTxcSimn is saved immediately after loading to clear IsModified flag
    poTxtCtl->SaveFile( m_oSim.rofnGetSaveFile( ).GetFullPath( ) );
    poTxtCtl->SetEditable( TRUE );
  }
  else
  {
    DlgErrMsg( wxT("Run Simulation Error"), m_osErrMsg );
    ::wxEndBusyCursor( );             // Change the cursor to the default
    return;
  }
  m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_SIMULTN );

  // Run the simulation
  SetStatusText( wxT(" Running the simulation ..."), 1 );
  m_oNbkTxtCtls.bSetPage( NbkTxtCtls::ePAGE_CONSOLE );
  poTxtCtl = m_oNbkTxtCtls.poGetPage( );
  poTxtCtl->bClear( );                // Clear the text control
  m_poPrcSimr->PrintCmd( *poTxtCtl ); // Print the simulator command
  if( ! m_poPrcSimr->bExec( ) )
  {
    m_osErrMsg = m_poPrcSimr->psGetErrMsg( );
    DlgErrMsg( wxT( "Run Simulation Error" ), m_osErrMsg );
    ::wxEndBusyCursor( );             // Change the cursor to the default
    return;
  }
  if( GetStatusBar()->GetStatusText( 1 ).Contains( wxT("Simulation aborted") ) )
  { // Just exit ammediately if the user aborts the simulation
    ::wxEndBusyCursor( );             // Change the cursor to the default
    return;
  }
  m_poPrcSimr->PrintRsp( *poTxtCtl ); // Print the simulator output

  // Format the simulation results
  SetStatusText( wxT(" Formatting the simulation results ..."), 1 );
  wxYield( );
  if( ! m_poPrcSimr->bFmtResults( ) )
  {
    m_osErrMsg = m_poPrcSimr->psGetErrMsg( );
    DlgErrMsg( wxT( "Run Simulation Error" ), m_osErrMsg );
    ::wxEndBusyCursor( );             // Change the cursor to the default
    return;
  }

  // Load the simulation output into the results text controls
  if( m_poNbkSimr->eGetSimrType( ) == eSIMR_GNUCAP )
       ePage = NbkTxtCtls::ePAGE_GNUCAP;
  else ePage = NbkTxtCtls::ePAGE_NGSPICE;
  poTxtCtl = m_oNbkTxtCtls.poGetPage( ePage );
  os1 = m_poPrcSimr->roGetResultsFile( ).GetFullPath( );
  if( ! poTxtCtl->bLoadFile( os1 ) )  // Print results
  {
    m_osErrMsg = wxT("Couldn't load the results file : \n\n") + os1;
    DlgErrMsg( wxT( "Run Simulation Error" ), m_osErrMsg );
    ::wxEndBusyCursor( );             // Change the cursor to the default
    return;
  }
  m_oNbkTxtCtls.bSetPage( ePage );    // Change to the appropriate results page

  // Change the cursor to the default
  ::wxEndBusyCursor( );

  SetStatusText( wxT(" Simulation ran successfully"), 1 );
}

//*****************************************************************************
// Stop the simulation.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnSimStop( wxCommandEvent & roEvtCmd )
{
  if( m_poPrcSimr->bIsExec( ) )
  {
    m_poPrcSimr->bKill( );

    SetStatusText( wxT(" Simulation aborted by user"), 1 );

    wxMessageDialog  oMsgDlg( this, wxT("The simulation has been aborted."),
                              wxT("Run Simulation Event"), wxICON_INFORMATION );
    oMsgDlg.ShowModal( );
  }
}

//*****************************************************************************
// Plot the simulation results.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnPlot( wxCommandEvent & roEvtCmd )
{
  TextCtrl * poTxtCtl;
  wxFileName  ofn1;
  wxString  os1;

  // Check if the plotter process is already running
  if( m_oPrcGWave.bIsExec( ) )
  {
    os1 = wxT("The gWave process is already running.");
    DlgErrMsg( wxT( "Plot Results Error" ), os1 );
    return;
  }

  // Check if a schematic or net list file is currently open
  ofn1 = m_oFileTsks.rosGetNetLstFile( );
  if( ofn1.GetFullPath( ).IsEmpty( ) )
  {
    os1 = wxT("There is no schematic or net list file currently loaded.");
    DlgErrMsg( wxT( "Plot Results Error" ), os1 );
    return;
  }

  // Create the appropriate name for the results file
  os1 = ofn1.GetName( );
  switch( m_poNbkSimr->eGetSimrType( ) )
  {
    case eSIMR_GNUCAP :
      if( m_oNbkTxtCtls.eGetPage( ) == NbkTxtCtls::ePAGE_NGSPICE )
           os1 << wxT(".ngspice");
      else os1 << wxT(".gnucap");
      break;
    case eSIMR_NGSPICE :
      if( m_oNbkTxtCtls.eGetPage( ) == NbkTxtCtls::ePAGE_GNUCAP )
           os1 << wxT(".gnucap");
      else os1 << wxT(".ngspice");
      break;
    default: return;
  }
  ofn1.SetName( os1 );
  ofn1.SetExt( m_poNbkSimr->rosGetPage( ) );

  // Set the results file in the plotter process
  if( ! ofn1.FileExists( ) )
  {
    os1.Empty( );
    os1 << wxT("The simulation results file:\n\n") << ofn1.GetFullPath( )
        << wxT("\n\nhasn't yet been generated.");
    DlgErrMsg( wxT( "Plot Results Error" ), os1 );
    return;
  }
  m_oPrcGWave.bSetResults( ofn1.GetFullPath( ) );

  // Execute the plotter process
  if( m_oPrcGWave.bExec( ) )
  {
    // Load the plotter output into the console text control
    poTxtCtl = m_oNbkTxtCtls.poGetPage( NbkTxtCtls::ePAGE_CONSOLE );
    m_oPrcGWave.Print( *poTxtCtl ); // Print the results
  }
  else
  {
    os1 = m_oPrcGWave.psGetErrMsg( );
    DlgErrMsg( wxT( "Plot Results Error" ), os1 );
  }
}

//*****************************************************************************
// Select which simulator to use.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnSelSimr( wxCommandEvent & roEvtCmd )
{
  int  ix, iy, iw, ih;

  if( ! roEvtCmd.IsChecked( ) ) return;

  // Change the cursor to the wait symbol and display a progress dialog
  ::wxBeginBusyCursor( );
  Show( FALSE );             // Hide the GUI while reconstructing it
  wxProgressDialog  oDlgProgress( wxT(" GNU Spice GUI"),
                             wxT("Changing simulation engines ..."), 100, this,
                             wxPD_AUTO_HIDE | wxPD_APP_MODAL | wxPD_SMOOTH );
  oDlgProgress.CenterOnParent( );
  oDlgProgress.Update( 0 );
  ::wxMilliSleep( 200 );

  // Change the simulator
  GetPosition( &ix, &iy );   // Set the position
  GetClientSize( &iw, &ih ); // Set the size
  oDlgProgress.Update( 10 );
  InitSimrEng( );            // Create sim. engine & associated display objects
  oDlgProgress.Update( 60 );
  ::wxMilliSleep( 200 );
  DoLayout( );               // Layout the GUI components
  oDlgProgress.Update( 80 );
  ::wxMilliSleep( 200 );
  Move( ix, iy );            // Set the position
  SetClientSize( iw, ih );   // Set the size
  oDlgProgress.Update( 99 );
  ::wxMilliSleep( 200 );

  // Change the cursor to the default
  oDlgProgress.Update( 100 ); // Kill the progress dialog
  Show( TRUE );               // Show the GUI again
  ::wxEndBusyCursor( );

  // Select the analysis page ??? 14/11/2005
  m_poConfig->SetPath( wxT("/Spice") );
  m_poNbkSimr->bSetPage( m_poConfig->Read( wxT("Analysis"), wxT("DC") ) );

  // Load the GUI with the simulation settings
  m_poNbkSimr->bLoad( m_oSim );
}

//*****************************************************************************
// Show the application preferences dialog.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnPrefs( wxCommandEvent & roEvtCmd )
{
  m_oDlgPrefs.CenterOnParent( );
  m_oDlgPrefs.ShowModal( );

  if( m_oDlgPrefs.GetReturnCode( ) == wxID_OK )
  {
    // Set the text control maximum lines
    m_oNbkTxtCtls.bSetLinesMax( m_oDlgPrefs.iGetMaxLines( ) );

    // Set default temporary file management strategy
    m_oFileTsks.bSetTmpFileMgt( (FileTasks::eTmpFMgtType) m_oDlgPrefs.iGetTmpFileMgt( ) );
  }
}

//*****************************************************************************
// Display the gSpiceUI manual.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnManual( wxCommandEvent & roEvtCmd )
{
  m_oHelpTsks.Manual( );
}

//*****************************************************************************
// Display about message dialog.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnAbout( wxCommandEvent & roEvtCmd )
{
  m_oHelpTsks.About( );
}

//*****************************************************************************
// Change the cursor when a simulation is running and the mouse is over the
// stop button.
//
// Argument List:
//   roEvtCmd - The event to be processed

void  FrmMain::OnTbrEnter( wxCommandEvent & roEvtCmd )
{
  if( roEvtCmd.GetSelection( ) == ID_TBR_STOP )
  {
    if( m_poPrcSimr->bIsExec( ) &&  ::wxIsBusy( ) )
    {
      ::wxEndBusyCursor( );    // Change the cursor to the default
      ::wxYield( );            // Allow the display to update
    }
  }
  else
  {
    if( m_poPrcSimr->bIsExec( ) && !::wxIsBusy( ) )
    {
      ::wxBeginBusyCursor( );  // Change the cursor to the hour glass
      ::wxYield( );            // Allow the display to update
    }
  }
}

//*****************************************************************************
// Event handler for system close.
//
// Argument List:
//   roEvtClose - The event to be processed

void  FrmMain::OnSysExit( wxCloseEvent & roEvtClose )
{
  wxString  os1;
  int  ix, iy, iw, ih;

  // Save the frame size and position
  GetClientSize( &iw, &ih );
  GetPosition( &ix, &iy );
  m_poConfig->Write( wxT("/Main/PosnX"), (long) ix );
  m_poConfig->Write( wxT("/Main/PosnY"), (long) iy );
  m_poConfig->Write( wxT("/Main/SizeW"), (long) iw );
  m_poConfig->Write( wxT("/Main/SizeH"), (long) ih );
  m_poConfig->Flush( );

  // Record the simulation engine
  m_poConfig->SetPath( wxT("/Spice") );
  if( m_poPrcSimr->eGetSimrType( ) == eSIMR_GNUCAP ) os1 = wxT("GNUCAP");
  else                                               os1 = wxT("NGSPICE");
  m_poConfig->Write( wxT("Engine"), os1 );

  // Record the analysis type
  os1 = m_poNbkSimr->rosGetPage( );
  m_poConfig->Write( wxT("Analysis"), os1.Upper( ) );

  bClear( );            // Kill any processes currently running

  m_oFileTsks.bExit( ); // Delete temporary files

  m_bIsOpen = FALSE;    // Indicate that the main frame has been closed

  // Destroys the window safely (used instead on the delete operator)
  Destroy( );
}

//*****************************************************************************
