/*
 * Copyright 2003-2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.codehaus.groovy.binding;

import java.util.LinkedHashSet;

/**
 * An aggregation of multiple bindings
 * @author Danno Ferrin
 * @since Groovy 1.6
 */
public class AggregateBinding implements BindingUpdatable {

    protected  boolean bound;

    // use linked hash set so order is preserved
    protected LinkedHashSet<BindingUpdatable> bindings = new LinkedHashSet();

    public void addBinding(BindingUpdatable binding) {
        if (bound) binding.bind(); // bind is idempotent, so no state checking
        bindings.add(binding);
    }

    public void removeBinding(BindingUpdatable binding) {
        bindings.remove(binding);
    }

    public void bind() {
        if (!bound) {
            bound = true;
            for (BindingUpdatable binding : bindings) {
                binding.bind();
            }
        }
    }

    public void unbind() {
        if (bound) {
            for (BindingUpdatable binding : bindings) {
                binding.unbind();
            }
            bound = false;
        }
    }

    public void rebind() {
        if (bound) {
            unbind();
            bind();
        }
    }

    public void update() {
        for (BindingUpdatable binding : bindings) {
            binding.update();
        }
    }

    public void reverseUpdate() {
        for (BindingUpdatable binding : bindings) {
            binding.reverseUpdate();
        }
    }
}
