/*
 *  save.c 
 *
 *  Copyright (c) 2005 Giansalvo Gusinu <giansalvo at gusinu.net>
 *  Copyright (C) 1999 Cory Lueninghoener (cluenin1@bigred.unl.edu)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <pthread.h>
#include <gtk/gtk.h>
#include <linux/types.h> 
#include <linux/videodev.h>
#include <stdio.h>
#include <jpeglib.h>
#include <png.h>

#include "gqcam.h"
#include "frontend.h"
#include "save.h"
#include "time.h"

/* common file extensions added to filename */
static struct symbolic_list fname_postfix_plist[] = {
	{ SAVE_PNG, ".png" },
	{ SAVE_PPM, ".ppm" },
	{ SAVE_JPEG,".jpg" },
	{ SAVE_GIF, ".gif" },
	{ SAVE_RAW, ".raw" },
	{ -1,NULL }
};


void select_ok(GtkWidget *widget, struct Camera *camera)
{
  gtk_entry_set_text(GTK_ENTRY(camera->save_struct.textbox), gtk_file_selection_get_filename(GTK_FILE_SELECTION (camera->save_struct.fileselect)));
  gtk_widget_destroy(camera->save_struct.fileselect);
  return;
}

void select_cancel(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  gtk_widget_destroy(GTK_WIDGET(widget));
  return;
}

void file_select(GtkWidget *widget, struct Camera *camera)
{

  //  camera->saving = 1;
  camera->save_struct.fileselect = gtk_file_selection_new("Save Image");
  gtk_file_selection_set_filename (GTK_FILE_SELECTION(camera->save_struct.fileselect), camera->savefileclean);
  gtk_widget_show(camera->save_struct.fileselect);

  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (camera->save_struct.fileselect)->cancel_button), "clicked", (GtkSignalFunc) select_cancel, GTK_OBJECT (camera->save_struct.fileselect));

  gtk_signal_connect(GTK_OBJECT (GTK_FILE_SELECTION (camera->save_struct.fileselect)->ok_button), "clicked", (GtkSignalFunc) select_ok, camera);

  return;
}


void save_ok(GtkWidget *widget, struct Camera *camera)
{
  int ls=0, x, y;
  int i;
  int lines, pixels_per_line;
  FILE *outfile;
  unsigned char buff[3];

  struct jpeg_compress_struct cinfo;
  struct jpeg_error_mgr jerr;
  JSAMPROW row_pointer[1];     // pointer to JSAMPLE row(s)
  int row_stride;              // physical row width in image buffer

  png_byte interlace;

  if(strlen(gtk_entry_get_text(GTK_ENTRY(camera->save_struct.textbox))) > 255){
    errordialog("File name too long...\n");
    return;
  }

  strcpy(camera->savefile, gtk_entry_get_text(GTK_ENTRY(camera->save_struct.textbox)));
  strcpy(camera->savefileclean, gtk_entry_get_text(GTK_ENTRY(camera->save_struct.textbox)));
  camera->save_struct.isinfo = 1;

  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(camera->save_struct.timebutton)))
    savefile_append_time(camera);

  strcat(camera->savefile,symbolic(fname_postfix_plist, camera->savetype)); // TODO add buffer overrun checking
  fprintf(stderr, "save.c: saving to file %s\n", camera->savefile);

  switch (camera->savetype) {

  case SAVE_PPM:
    if (GTK_TOGGLE_BUTTON(camera->page_ppm.radio_raw_ppm)->active){
      camera->save_struct.format = PPM_RAW;
    }
    else if (GTK_TOGGLE_BUTTON(camera->page_ppm.radio_ascii_ppm)->active){
      camera->save_struct.format = PPM_ASCII;
    }
    if (camera->saving)
      ppm_save(camera);

    break; 
    
  case SAVE_JPEG:
    camera->save_struct.quality = GTK_ADJUSTMENT(camera->page_jpeg.quality_adj_jpeg)->value;
    camera->save_struct.smoothness = GTK_ADJUSTMENT(camera->page_jpeg.smooth_adj_jpeg)->value;
    camera->save_struct.optimize = (GTK_TOGGLE_BUTTON(camera->page_jpeg.optimize_jpeg)->active)?1:0;
    if (camera->saving)
      jpeg_save(camera);

    break;
    
  case SAVE_RAW:
 	raw_save(camera);
     break;

  case SAVE_GIF:
    // UNSUPPORTED
    break;

  case SAVE_PNG:
    if(GTK_TOGGLE_BUTTON(camera->page_png.interlace_png)->active)
      camera->save_struct.interlace = PNG_INTERLACE_ADAM7;
    else
      camera->save_struct.interlace = PNG_INTERLACE_NONE;

    camera->save_struct.compression = GTK_ADJUSTMENT(camera->page_png.compression_adj_png)->value;
    if (camera->saving)
      png_save(camera);
    break;

  default:
    // TODO Report an internal error. Either I forgot to code this case or the type is incorrect.
    break;
    
  }

  gtk_widget_destroy(widget->parent->parent->parent);
  pthread_mutex_lock( &camera->pref_mutex );
  if( !camera->frozen )
    pthread_mutex_unlock( &camera->freeze_mutex );
  camera->saving = 0;
  pthread_mutex_unlock( &camera->pref_mutex );
  return;
}

void png_save(struct Camera *camera)
{
  FILE *fp;
  png_structp png_ptr;
  png_infop info_ptr;
  png_uint_32 k, height, width;
  png_bytep row_pointers[camera->vid_win.height];

  /* open the file */
  if(!strcmp(camera->savefile, ""))
    fp = stdout;
  else if(camera->savefile != NULL)
    fp = fopen(camera->savefile, "w");

  if (fp == NULL){
    errordialog("Error - Could not open file");
    return;
  }
  
  /* Create and initialize the png_struct with the desired error handler
   * functions.  If you want to use the default stderr and longjump method,
   * you can supply NULL for the last three parameters.  We also check that
   * the library version is compatible with the one used at compile time,
   * in case we are using dynamically linked libraries.  REQUIRED.
   */
  png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
  
  if (png_ptr == NULL)
    {
      errordialog("Error - could not create PNG");
      fclose(fp);
      return;
    }
  
  /* Allocate/initialize the image information data.  REQUIRED */
  info_ptr = png_create_info_struct(png_ptr);
  if (info_ptr == NULL)
    {
      fclose(fp);
      png_destroy_write_struct(&png_ptr,  (png_infopp)NULL);
      return;
    }
  
  /* Set error handling.  REQUIRED if you aren't supplying your own
   * error hadnling functions in the png_create_write_struct() call.
   */
  if (setjmp(png_ptr->jmpbuf))
    {
      /* If we get here, we had a problem reading the file */
      fclose(fp);
      png_destroy_write_struct(&png_ptr,  (png_infopp)NULL);
      return;
    }
  
  /* One of the following I/O initialization functions is REQUIRED */
  /* set up the output control if you are using standard C streams */
  png_init_io(png_ptr, fp);
  
  /* Set the image information here.  Width and height are up to 2^31,
   * bit_depth is one of 1, 2, 4, 8, or 16, but valid values also depend on
   * the color_type selected. color_type is one of PNG_COLOR_TYPE_GRAY,
   * PNG_COLOR_TYPE_GRAY_ALPHA, PNG_COLOR_TYPE_PALETTE, PNG_COLOR_TYPE_RGB,
   * or PNG_COLOR_TYPE_RGB_ALPHA.  interlace is either PNG_INTERLACE_NONE or
   * PNG_INTERLACE_ADAM7, and the compression_type and filter_type MUST
   * currently be PNG_COMPRESSION_TYPE_BASE and PNG_FILTER_TYPE_BASE. REQUIRED
   */
  if(camera->greyscale)
    png_set_IHDR(png_ptr, info_ptr, camera->vid_win.width, camera->vid_win.height, 8, PNG_COLOR_TYPE_GRAY, camera->save_struct.interlace, PNG_COMPRESSION_TYPE_BASE, PNG_FILTER_TYPE_BASE);
  else
    png_set_IHDR(png_ptr, info_ptr, camera->vid_win.width, camera->vid_win.height, 8, PNG_COLOR_TYPE_RGB, camera->save_struct.interlace, PNG_COMPRESSION_TYPE_BASE, PNG_FILTER_TYPE_BASE);
  png_set_compression_level(png_ptr, camera->save_struct.compression);
  
  /* set the palette if there is one.  REQUIRED for indexed-color images */
  //palette = (png_colorp)png_malloc(png_ptr, 256 * sizeof (png_color));
  /* ... set palette colors ... */
  //png_set_PLTE(png_ptr, info_ptr, palette, 256);
  
  /* optional significant bit chunk */
  /* if we are dealing with a grayscale image then */
  //sig_bit.gray = true_bit_depth;
  /* otherwise, if we are dealing with a color image then */
  //sig_bit.red = true_red_bit_depth;
  //sig_bit.green = true_green_bit_depth;
  //sig_bit.blue = true_blue_bit_depth;
  /* if the image has an alpha channel then */
  //sig_bit.alpha = true_alpha_bit_depth;
  //png_set_sBIT(png_ptr, info_ptr, sig_bit);
  
  
  /* Optional gamma chunk is strongly suggested if you have any guess
   * as to the correct gamma of the image.
   */
  //png_set_gAMA(png_ptr, info_ptr, gamma);
  
  /* Optionally write comments into the image */
/*
  text_ptr[0].key = "Title";
  text_ptr[0].text = "Mona Lisa";
  text_ptr[0].compression = PNG_TEXT_COMPRESSION_NONE;
  text_ptr[1].key = "Author";
  text_ptr[1].text = "Leonardo DaVinci";
  text_ptr[1].compression = PNG_TEXT_COMPRESSION_NONE;
  text_ptr[2].key = "Description";
  text_ptr[2].text = "<long text>";
  text_ptr[2].compression = PNG_TEXT_COMPRESSION_zTXt;
  png_set_text(png_ptr, info_ptr, text_ptr, 3);
*/  
  /* other optional chunks like cHRM, bKGD, tRNS, tIME, oFFs, pHYs, */
  /* note that if sRGB is present the cHRM chunk must be ignored
   * on read and must be written in accordance with the sRGB profile */
  
  /* Write the file header information.  REQUIRED */
  png_write_info(png_ptr, info_ptr);
  
  /* Once we write out the header, the compression type on the text
   * chunks gets changed to PNG_TEXT_COMPRESSION_NONE_WR or
   * PNG_TEXT_COMPRESSION_zTXt_WR, so it doesn't get written out again
   * at the end.
   */
  
  /* set up the transformations you want.  Note that these are
   * all optional.  Only call them if you want them.
   */
  
  /* invert monocrome pixels */
  //png_set_invert_mono(png_ptr);
  
  /* Shift the pixels up to a legal bit depth and fill in
   * as appropriate to correctly scale the image.
   */
  //png_set_shift(png_ptr, &sig_bit);
  
  /* pack pixels into bytes */
  //png_set_packing(png_ptr);
  
  /* swap location of alpha bytes from ARGB to RGBA */
  //png_set_swap_alpha(png_ptr);
  
  /* Get rid of filler (OR ALPHA) bytes, pack XRGB/RGBX/ARGB/RGBA into
   * RGB (4 channels -> 3 channels). The second parameter is not used.
   */
  //png_set_filler(png_ptr, 0, PNG_FILLER_BEFORE);
  
  /* flip BGR pixels to RGB */
  //png_set_bgr(png_ptr);
  
  /* swap bytes of 16-bit files to most significant byte first */
  //png_set_swap(png_ptr);
  
  /* swap bits of 1, 2, 4 bit packed pixel formats */
  //png_set_packswap(png_ptr);
  
  /* turn on interlace handling if you are not using png_write_image() */
/*
  if (interlacing)
    number_passes = png_set_interlace_handling(png_ptr);
  else
    number_passes = 1;
*/  
  /* The easiest way to write the image (you may have a different memory
   * layout, however, so choose what fits your needs best).  You need to
   * use the first method if you aren't handling interlacing yourself.
   */
/*
  png_uint_32 k, height, width;
  png_byte image[height][width];
  png_bytep row_pointers[height];
*/
  if (camera->greyscale) {
    for (k = 0; k < camera->vid_win.height; k++)
      row_pointers[k] = camera->pic + k*camera->vid_win.width;
  }
  else {
    for (k = 0; k < camera->vid_win.height; k++)
      row_pointers[k] = camera->pic + k*camera->vid_win.width*3;
  }
  
  
  /* One of the following output methods is REQUIRED */
  png_write_image(png_ptr, row_pointers);
  
  /* You can write optional chunks like tEXt, zTXt, and tIME at the end
   * as well.
   */
  
  /* It is REQUIRED to call this to finish writing the rest of the file */
  png_write_end(png_ptr, info_ptr);
  
  /* if you malloced the palette, free it here */
  free(info_ptr->palette);
  
  /* if you allocated any text comments, free them here */
  
  /* clean up after the write, and free any memory allocated */
  png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
  
  /* close the file */
  fclose(fp);
  
  /* that's it */
  
  return;
}

void ppm_save(struct Camera *camera)
{
  FILE *outfile;
  unsigned char buff[3];
  int i;

  if(!strcmp(camera->savefile, ""))
    outfile = stdout;
  else if(camera->savefile != NULL)
    outfile = fopen(camera->savefile, "w");
  
  if (outfile == NULL){
    errordialog("Could not open file");
    return;
  }
 
  /*  Raw  */
  if (camera->save_struct.format == PPM_RAW){
    fprintf (outfile, "P6\n%d %d\n%d\n", camera->vid_win.width, camera->vid_win.height, 255);
    if (camera->greyscale){
      for (i = 0; i < camera->vid_win.width*camera->vid_win.height; i++){
	  buff[0] = camera->pic[i];
	  buff[1] = camera->pic[i];
	  buff[2] = camera->pic[i]; 
	  fwrite (buff, 1, 3, outfile); 
      }
    }
    else {
      for (i = 0; i < camera->vid_win.width * camera->vid_win.height * 3; i++) {
	fputc(camera->pic[i], outfile);
      }
    }
  }

  /*  Ascii  */
  else if (camera->save_struct.format == PPM_ASCII){
    fprintf (outfile, "P3\n%d %d\n%d\n", camera->vid_win.width, camera->vid_win.height, 255);
    if (camera->greyscale) {
      for (i = 0; i < camera->vid_win.width*camera->vid_win.height; i++){ 
	fprintf (outfile, "%03d %03d %03d\n", camera->pic[i], camera->pic[i], camera->pic[i]);
      }
    }
    else
      for (i = 0; i < camera->vid_win.width*camera->vid_win.height*3; i=i+3){ 
	fprintf (outfile, "%03d %03d %03d\n", camera->pic[i], camera->pic[i+1], camera->pic[i+2]);
      }
  }

  fclose(outfile);
}


void jpeg_save(struct Camera *camera)
{
  FILE *outfile;
  int ls=0, x, y;
  int i;
  int lines, pixels_per_line;
  unsigned char buff[3];
  struct jpeg_compress_struct cinfo;
  struct jpeg_error_mgr jerr;
  JSAMPROW row_pointer[1];     // pointer to JSAMPLE row(s)
  int row_stride;              // physical row width in image buffer

  if(!strcmp(camera->savefile, ""))
    outfile = stdout;
  else if(camera->savefile != NULL)
    outfile = fopen(camera->savefile, "w");

  if (outfile == NULL){
    errordialog("Could not open file");
    return;
  }

  cinfo.err = jpeg_std_error(&jerr);
  jpeg_create_compress(&cinfo);
  
  jpeg_stdio_dest(&cinfo, outfile);
  cinfo.image_width = camera->vid_win.width;
  cinfo.image_height = camera->vid_win.height;
  
  if (camera->greyscale) {
    cinfo.input_components = 1;
    cinfo.in_color_space = JCS_GRAYSCALE;     
  }
  else {
    cinfo.input_components = 3;
    cinfo.in_color_space = JCS_RGB;
  }
  
  cinfo.smoothing_factor = camera->save_struct.smoothness;
  if (camera->save_struct.smoothness) {
    cinfo.optimize_coding = TRUE;
  }
  
  jpeg_set_defaults(&cinfo);
  jpeg_set_quality(&cinfo, camera->save_struct.quality, TRUE);
  jpeg_start_compress(&cinfo, TRUE);
  row_stride = cinfo.input_components*camera->vid_win.width;
  while (cinfo.next_scanline < cinfo.image_height) {
    row_pointer[0] = &camera->pic[cinfo.next_scanline * row_stride];
    jpeg_write_scanlines(&cinfo, row_pointer, 1);
  }
  jpeg_finish_compress(&cinfo);
  jpeg_destroy_compress(&cinfo);
  
  fclose(outfile);
  return;
}

void raw_save(struct Camera *camera)
{
  FILE *outfile;
  int ls=0, x, y;
  int i;
  int lines, pixels_per_line;
  unsigned char buff[3];
  struct jpeg_compress_struct cinfo;
  struct jpeg_error_mgr jerr;
  JSAMPROW row_pointer[1];     // pointer to JSAMPLE row(s)
  int row_stride;              // physical row width in image buffer

  if(!strcmp(camera->savefile, ""))
    outfile = stdout;
  else if(camera->savefile != NULL)
    outfile = fopen(camera->savefile, "w");

  if (outfile == NULL){
    errordialog("Could not open file");
    return;
  }

  if (fwrite(camera->picbuff, sizeof(unsigned char), (size_t)camera->img_size, outfile) == camera->img_size)
	fprintf(stderr, "raw_save %d bytes... success\n", (int)camera->img_size);
  else
  	fprintf(stderr, "raw_save error\n");

  fclose(outfile);
  
  return;
}

void save_cancel(GtkWidget *widget, struct Camera *camera)
{
  gtk_widget_destroy(widget->parent->parent->parent);
  camera->saving = 0;
  if( !camera->frozen )
    pthread_mutex_unlock(&camera->freeze_mutex);
  return;
}

void destroy( GtkWidget *widget,
	      gpointer   data )
{
  gtk_main_quit();
}

void switch_page_png(GtkWidget *widget, struct Camera *camera)
{
  gtk_widget_hide(camera->currentsavepage);
  gtk_widget_show(camera->page_png.frame_png);
  camera->currentsavepage = camera->page_png.frame_png;
  camera->savetype = SAVE_PNG;
}

void switch_page_jpeg(GtkWidget *widget, struct Camera *camera)
{
  gtk_widget_hide(camera->currentsavepage);
  gtk_widget_show(camera->page_jpeg.frame_jpeg);
  camera->currentsavepage = camera->page_jpeg.frame_jpeg;
  camera->savetype = SAVE_JPEG;
}

void switch_page_raw(GtkWidget *widget, struct Camera *camera)
{
  gtk_widget_hide(camera->currentsavepage);
  camera->savetype = SAVE_RAW;
}


void switch_page_ppm(GtkWidget *widget, struct Camera *camera)
{
  gtk_widget_hide(camera->currentsavepage);
  gtk_widget_show(camera->page_ppm.frame_ppm);
  camera->currentsavepage = camera->page_ppm.frame_ppm;
  camera->savetype = SAVE_PPM;
}

void save_dialog(GtkWidget *widget, struct Camera *camera)
{
  GtkWidget *notebook;
  GtkWidget *window;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *vbox, *hbox;
  GtkWidget *typemenu;
  GtkWidget *typemenubutton;
  GtkWidget *ppmitem;
  GtkWidget *jpegitem;
  GtkWidget *rawitem;
  GtkWidget *pngitem;
  //int openpage = camera->savetype;

  pthread_mutex_lock( &camera->pref_mutex );
  if( !camera->frozen )
    pthread_mutex_lock(&camera->freeze_mutex);
  camera->saving = 1;
  pthread_mutex_unlock( &camera->pref_mutex );

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, TRUE, TRUE);
  gtk_window_set_title (GTK_WINDOW (window), "Gqcam: Save Image");

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_widget_show(vbox);
  
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);
  label = gtk_label_new("File:");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
  gtk_widget_show(label);
  camera->save_struct.textbox = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(hbox), camera->save_struct.textbox, FALSE, FALSE, 2);
  if(camera->save_struct.isinfo)
    gtk_entry_set_text(GTK_ENTRY(camera->save_struct.textbox), camera->savefileclean);
  gtk_widget_show(camera->save_struct.textbox);
  button = gtk_button_new_with_label("Browse...");
  gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (file_select), camera);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 2);
  gtk_widget_show(button);
  gtk_widget_show(hbox);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);

  typemenubutton = gtk_option_menu_new();
  gtk_box_pack_start(GTK_BOX(hbox), typemenubutton, FALSE, FALSE, 2);
  gtk_widget_show(hbox);
  
  typemenu = gtk_menu_new();

  pngitem = gtk_menu_item_new_with_label("png");
  gtk_menu_append(GTK_MENU(typemenu), pngitem);
  gtk_signal_connect(GTK_OBJECT (pngitem), "activate",
                     GTK_SIGNAL_FUNC (switch_page_png), camera);
  gtk_widget_show(pngitem);

  ppmitem = gtk_menu_item_new_with_label("ppm");
  gtk_menu_append(GTK_MENU(typemenu), ppmitem);
  gtk_signal_connect(GTK_OBJECT (ppmitem), "activate",
                     GTK_SIGNAL_FUNC (switch_page_ppm), camera);
  gtk_widget_show(ppmitem);
  
  jpegitem = gtk_menu_item_new_with_label("jpeg");
  gtk_menu_append(GTK_MENU(typemenu), jpegitem);
  gtk_signal_connect(GTK_OBJECT (jpegitem), "activate",
                     GTK_SIGNAL_FUNC (switch_page_jpeg), camera);
  gtk_widget_show(jpegitem);

  rawitem = gtk_menu_item_new_with_label("raw");
  gtk_menu_append(GTK_MENU(typemenu), rawitem);
  gtk_signal_connect(GTK_OBJECT (rawitem), "activate",
                    GTK_SIGNAL_FUNC (switch_page_raw), camera);
  gtk_widget_show(rawitem);
  
  
  gtk_option_menu_set_menu(GTK_OPTION_MENU(typemenubutton), typemenu);
  gtk_option_menu_set_history(GTK_OPTION_MENU(typemenubutton), camera->savetype);

  gtk_widget_show(typemenubutton);

  camera->save_struct.timebutton = gtk_check_button_new_with_label("Append time to filename");
  gtk_box_pack_start(GTK_BOX(hbox), camera->save_struct.timebutton, FALSE, FALSE, 2);
  gtk_widget_show(camera->save_struct.timebutton);
  

  /* PPM PAGE */

  camera->page_ppm.frame_ppm = gtk_frame_new("Save As");
  gtk_frame_set_shadow_type(GTK_FRAME(camera->page_ppm.frame_ppm), GTK_SHADOW_ETCHED_IN);

  camera->page_ppm.vbox_ppm = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(camera->page_ppm.frame_ppm), camera->page_ppm.vbox_ppm);
  gtk_widget_show(camera->page_ppm.vbox_ppm);

  camera->page_ppm.radio_raw_ppm = gtk_radio_button_new_with_label(NULL, "Raw");
  gtk_box_pack_start(GTK_BOX(camera->page_ppm.vbox_ppm), camera->page_ppm.radio_raw_ppm, FALSE, FALSE, 2);
  gtk_widget_show(camera->page_ppm.radio_raw_ppm);
  
  camera->page_ppm.group_ppm = gtk_radio_button_group(GTK_RADIO_BUTTON(camera->page_ppm.radio_raw_ppm));
  camera->page_ppm.radio_ascii_ppm = gtk_radio_button_new_with_label(camera->page_ppm.group_ppm, "ASCII");
  gtk_box_pack_start(GTK_BOX(camera->page_ppm.vbox_ppm), camera->page_ppm.radio_ascii_ppm, FALSE, FALSE, 2);
  gtk_widget_show(camera->page_ppm.radio_ascii_ppm);

  if(camera->savetype == SAVE_PPM){
    gtk_widget_show(camera->page_ppm.frame_ppm);
    camera->currentsavepage = camera->page_ppm.frame_ppm;
  }
  
  gtk_box_pack_start(GTK_BOX(vbox), camera->page_ppm.frame_ppm, FALSE, FALSE, 2);


  /* JPEG PAGE */

  camera->page_jpeg.frame_jpeg = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(camera->page_jpeg.frame_jpeg), GTK_SHADOW_ETCHED_IN);

  camera->page_jpeg.vbox_jpeg = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(camera->page_jpeg.frame_jpeg), camera->page_jpeg.vbox_jpeg);
  gtk_widget_show(camera->page_jpeg.vbox_jpeg);

  camera->page_jpeg.table_jpeg = gtk_table_new(3, 3, TRUE);
  gtk_box_pack_start(GTK_BOX(camera->page_jpeg.vbox_jpeg), camera->page_jpeg.table_jpeg, FALSE, FALSE, 2);
  gtk_widget_show(camera->page_jpeg.table_jpeg);

  camera->page_jpeg.label_jpeg = gtk_label_new("Quality");
  gtk_table_attach_defaults(GTK_TABLE(camera->page_jpeg.table_jpeg), camera->page_jpeg.label_jpeg, 0, 1, 0, 1);
  gtk_widget_show(camera->page_jpeg.label_jpeg);

  camera->page_jpeg.quality_adj_jpeg = gtk_adjustment_new(75, 0, 100, 1, 10, 1);
  camera->page_jpeg.quality_jpeg = gtk_hscale_new(GTK_ADJUSTMENT(camera->page_jpeg.quality_adj_jpeg));
  gtk_scale_set_digits (GTK_SCALE(camera->page_jpeg.quality_jpeg), 0);
  gtk_table_attach_defaults(GTK_TABLE(camera->page_jpeg.table_jpeg), camera->page_jpeg.quality_jpeg, 1, 3, 0, 1);
  gtk_widget_show(camera->page_jpeg.quality_jpeg);

  camera->page_jpeg.label_jpeg = gtk_label_new("Smoothness");
  gtk_table_attach_defaults(GTK_TABLE(camera->page_jpeg.table_jpeg), camera->page_jpeg.label_jpeg, 0, 1, 1, 2);
  gtk_widget_show(camera->page_jpeg.label_jpeg);

  camera->page_jpeg.smooth_adj_jpeg = gtk_adjustment_new( 0, 0, 100, 1, 10, 1 );
  camera->page_jpeg.smooth_jpeg = gtk_hscale_new(GTK_ADJUSTMENT(camera->page_jpeg.smooth_adj_jpeg));
  gtk_range_set_update_policy (GTK_RANGE(camera->page_jpeg.smooth_jpeg), GTK_UPDATE_CONTINUOUS);
  gtk_scale_set_digits (GTK_SCALE(camera->page_jpeg.smooth_jpeg), 0);
  gtk_scale_set_value_pos (GTK_SCALE(camera->page_jpeg.smooth_jpeg), GTK_POS_TOP);
  gtk_scale_set_draw_value (GTK_SCALE(camera->page_jpeg.smooth_jpeg), TRUE);
  gtk_table_attach_defaults(GTK_TABLE(camera->page_jpeg.table_jpeg), camera->page_jpeg.smooth_jpeg, 1, 3, 1, 2);
  gtk_widget_show(camera->page_jpeg.smooth_jpeg);

  camera->page_jpeg.optimize_jpeg = gtk_check_button_new_with_label("Optimize");
  gtk_table_attach_defaults(GTK_TABLE(camera->page_jpeg.table_jpeg), camera->page_jpeg.optimize_jpeg, 0, 1, 2, 3);
  gtk_widget_show(camera->page_jpeg.optimize_jpeg);
  gtk_box_pack_start(GTK_BOX(vbox), camera->page_jpeg.frame_jpeg, FALSE, FALSE, 2);

  if(camera->savetype == SAVE_JPEG){
    gtk_widget_show(camera->page_jpeg.frame_jpeg);
    camera->currentsavepage = camera->page_jpeg.frame_jpeg;
  }

/* PNG PAGE */

  camera->page_png.frame_png = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(camera->page_png.frame_png), GTK_SHADOW_ETCHED_IN);

  camera->page_png.vbox_png = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(camera->page_png.frame_png), camera->page_png.vbox_png);
  gtk_widget_show(camera->page_png.vbox_png);

  camera->page_png.interlace_png = gtk_check_button_new_with_label("Interlace");
  gtk_box_pack_start(GTK_BOX(camera->page_png.vbox_png), camera->page_png.interlace_png, FALSE, FALSE, 2);
  gtk_widget_show(camera->page_png.interlace_png);

  camera->page_png.hbox_png = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(camera->page_png.vbox_png), camera->page_png.hbox_png, FALSE, FALSE, 2);
  gtk_widget_show(camera->page_png.hbox_png);

  camera->page_png.label_png = gtk_label_new("Compression");
  gtk_box_pack_start(GTK_BOX(camera->page_png.hbox_png), camera->page_png.label_png, FALSE, FALSE, 2);
  gtk_widget_show(camera->page_png.label_png);
  
  camera->page_png.compression_adj_png = gtk_adjustment_new(6, 0, 10, 1, 1, 1);
  camera->page_png.compression_png = gtk_hscale_new(GTK_ADJUSTMENT(camera->page_png.compression_adj_png));
  gtk_range_set_update_policy (GTK_RANGE(camera->page_png.compression_png), GTK_UPDATE_CONTINUOUS);
  gtk_scale_set_digits (GTK_SCALE(camera->page_png.compression_png), 0);
  gtk_scale_set_value_pos (GTK_SCALE(camera->page_png.compression_png), GTK_POS_TOP);
  gtk_scale_set_draw_value(GTK_SCALE(camera->page_png.compression_png), TRUE);
  //gtk_table_attach_defaults(GTK_TABLE(camera->page_jpeg.table_jpeg), camera->page_jpeg.smooth_jpeg, 1, 3, 1, 2);
  gtk_box_pack_start(GTK_BOX(camera->page_png.hbox_png), camera->page_png.compression_png, TRUE, TRUE, 2);
  gtk_widget_show(camera->page_png.compression_png);
  
  gtk_box_pack_start(GTK_BOX(vbox), camera->page_png.frame_png, FALSE, FALSE, 2);

  if(camera->savetype == SAVE_PNG){
    gtk_widget_show(camera->page_png.frame_png);
    camera->currentsavepage = camera->page_png.frame_png;
  }
  
/* GIF PAGE */
/*
  page_gif.frame_gif = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(page_gif.frame_gif), GTK_SHADOW_NONE);
  gtk_widget_show(page_gif.frame_gif);

  page_gif.vbox_gif = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page_gif.frame_gif), page_gif.vbox_gif);
  gtk_widget_show(page_gif.vbox_gif);

  page_gif.interlace_gif = gtk_check_button_new_with_label("Interlace");
  gtk_box_pack_start(GTK_BOX(page_gif.vbox_gif), page_gif.interlace_gif, FALSE, FALSE, 2);
  gtk_widget_show(page_gif.interlace_gif);

  page_gif.comment_gif = gtk_check_button_new_with_label("Comment:");
  gtk_box_pack_start(GTK_BOX(page_gif.vbox_gif), page_gif.comment_gif, FALSE, FALSE, 2);
  gtk_widget_show(page_gif.comment_gif);

  page_gif.comment_text_gif = gtk_text_new( NULL, NULL);
  gtk_box_pack_start(GTK_BOX(page_gif.vbox_gif), page_gif.comment_text_gif, FALSE, FALSE, 2);
  gtk_text_set_editable(GTK_TEXT(page_gif.comment_text_gif), TRUE);
  gtk_widget_set_usize(GTK_WIDGET(page_gif.comment_text_gif), 100, 50);
  gtk_widget_show(page_gif.comment_text_gif);
  
  page_gif.label_gif = gtk_label_new("Gif");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page_gif.frame_gif, page_gif.label_gif);
*/

/* END PAGES */
/*
  gtk_notebook_set_page(GTK_NOTEBOOK(notebook), camera->savetype);
*/
  hbox = gtk_hbox_new(TRUE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);
  gtk_widget_show(hbox);
  button = gtk_button_new_with_label("Ok");
  gtk_signal_connect(GTK_OBJECT(button), "clicked", save_ok, camera);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 5);
  gtk_widget_show(button);
  button = gtk_button_new_with_label("Cancel");
  gtk_signal_connect(GTK_OBJECT(button), "clicked", save_cancel, camera);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 5);
  gtk_widget_show(button);
  
  gtk_widget_show(window);
  return;
}

/* 
 * Append timestamp to filename (before the first occurence 
 * of '.' if there is any) 
 */
void savefile_append_time(struct Camera *camera)
{
  char *dotlocation;
  char prefix[255+11], curtime[10];
  int length;

  dotlocation = strrchr(camera->savefileclean, '.');
  if(dotlocation == NULL)
    length = strlen(camera->savefileclean);
  else
    length = (dotlocation - camera->savefileclean)/(sizeof(char));
  strncpy(prefix, camera->savefileclean, length);
  sprintf(curtime, "-%d", time(NULL));
  strcpy(prefix+length, curtime);
  strcpy(prefix+length+strlen(curtime), camera->savefileclean+length);
  //  printf("%s\n", prefix);
  
  strcpy(camera->savefile, prefix);
  
}
