#include "viewer.h"
#include "tabchild.h"
#include "analysisViewer.h"
#include "problemAnalyzer.h"
#include "checkmateViewer.h"
#ifdef ENABLE_NTESUKI
#include "ntesukiViewer.h"
#endif
#include "kifuViewer.h"
#include "quiescenceViewer.h"
#include "networkViewer.h"
#include "piecePairDialog.h"
#include "moveGeneratorDialog.h"
#include "ui_kisendialog4.h"
#include "ui_networkdialog4.h"
#include "kifuFile.h"
#include "evalDebug.h"

#if QT_VERSION < 0x040000
#include "boardEditor.h"
#else
#include "httpwindow.h"
#endif
#include "boardEditor2.h"

#include "osl/search/simpleHashTable.h"
#include "osl/eval/progressEval.h"
#include "osl/eval/ml/openMidEndingEval.h"
#include "osl/state/numEffectState.h"
#include "osl/progress/effect5x3.h"
#include "osl/record/kisen.h"
#include "osl/record/kakinoki.h"
#include "osl/record/csa.h"
#include "osl/search/simpleHashRecord.h"
#include "osl/threatmate/mlPredictor.h"
#include "osl/threatmate/treePredictor.h"
#include "osl/apply_move/applyMove.h"
#include <QFileDialog>
#include <qlayout.h>
#include <qtabwidget.h>
#include <qinputdialog.h>
#include <qmessagebox.h>
#include <qtextcodec.h>
#include <qlistview.h>
#include <qpushbutton.h>

#include <boost/scoped_ptr.hpp>

#include <stdexcept>
#include <iostream>

Viewer::Viewer(QWidget *parent, const char *name)
  : QTabWidget(parent, name), chatIcon(false)
{
  osl::eval::ProgressEval::setUp();
  osl::eval::ml::OpenMidEndingEval::setUp();
  osl::eval::ml::NewProgress::setUp();
  kifuViewer = new KifuViewer(this);
  addTab(kifuViewer, "Kifu");
  analysisViewer = 0;
  checkmateViewer = new CheckmateViewer(this);
  addTab(checkmateViewer, "Check Search");
#ifdef ENABLE_NTESUKI
  ntesukiViewer = new NtesukiViewer(this);
  addTab(ntesukiViewer, "Ntesuki Search");
#endif
  quiescenceViewer = new QuiescenceViewer(this);
  addTab(quiescenceViewer, "Quiescence Search");
  networkViewer = new NetworkViewer(this);
  addTab(networkViewer, "Network");
#if QT_VERSION < 0x040000
  boardEditor = new BoardEditor(this);
  addTab(boardEditor, "Edit");
#endif
  editor = new BoardEditor2(this);
  addTab(editor, "Edit Board");

#if QT_VERSION >= 0x040000
  QPixmap close = QPixmap(":/images/close.png");
#else
  QPixmap close = QPixmap::fromMimeSource("close.png");
#endif

  QPushButton *closeButton = new QPushButton(close, "", this);
  setCornerWidget(closeButton);

  connect(this, SIGNAL(currentChanged(QWidget *)),
	  this, SIGNAL(statusChanged()));
  connect(kifuViewer, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));
  connect(kifuViewer, SIGNAL(analyzeOnlineDisabled()),
	  this, SIGNAL(analyzeOnlineDisabled()));
  connect(checkmateViewer, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));
#ifdef ENABLE_NTESUKI
  connect(ntesukiViewer, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));
#endif
  connect(quiescenceViewer, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));
  connect(networkViewer, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));
#if QT_VERSION < 0x040000
  connect(boardEditor, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));
#endif

  connect(closeButton, SIGNAL(clicked()),
	  this, SLOT(closeTab()));
  connect(this, SIGNAL(currentChanged(QWidget *)),
	  this, SLOT(notifyOrientation(QWidget *)));
  connect(this, SIGNAL(currentChanged(QWidget *)),
	  this, SLOT(notifyEffect(QWidget *)));

  connect(networkViewer, SIGNAL(chatReceived()),
	  this, SLOT(chatReceived()));
  connect(networkViewer, SIGNAL(painted()),
	  this, SLOT(chatDisplayed()));

  resetIcon();
}

const viewer::EvalInfo Viewer::evalState()
{
  viewer::EvalInfo result;
  TabChild *child = (TabChild *)currentPage();
  osl::state::NumEffectState nState(child->getState());
  if (nState.getKingPiece(osl::BLACK).isOnBoard() &&
      nState.getKingPiece(osl::WHITE).isOnBoard())
  {
    osl::eval::ml::OpenMidEndingEval eval(nState);
    result.total =
      eval.value() / osl::progress::ml::NewProgress::maxProgress();;
    result.opening = eval.openingValue();
    result.mid = eval.midgameValue();
    result.ending  = eval.endgameValue();
  }
  return result;
}

void Viewer::progressOfState(osl::stl::vector<int>& out)
{
  TabChild *child = (TabChild *)currentPage();
  // TODO: better abstraction needed
  if (currentPage() == editor)
  {
    for (int i = 0; i < 3; i++)
      out.push_back(0);
    return;
  }
  osl::state::NumEffectState nState(child->getState());
  if (nState.getKingPiece(osl::BLACK).isOnBoard() &&
      nState.getKingPiece(osl::WHITE).isOnBoard())
  {
    osl::progress::ml::NewProgress progress(nState);
    out.push_back(progress.progress16().value());
    out.push_back(progress.progress16(osl::BLACK).value());
    out.push_back(progress.progress16(osl::WHITE).value());
    out.push_back(progress.progressAttack(osl::BLACK).value()-8);
    out.push_back(progress.progressDefense(osl::BLACK).value()-8);
    out.push_back(progress.progressAttack(osl::WHITE).value()-8);
    out.push_back(progress.progressDefense(osl::WHITE).value()-8);
  }
  else
  {
    out.push_back(0);
    out.push_back(0);
    out.push_back(0);
    out.push_back(0);
    out.push_back(0);
    out.push_back(0);
    out.push_back(0);
  }
}

std::pair<double,double> Viewer::checkmateProbability()
{
  TabChild *child = (TabChild *)currentPage();
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);

  if ((moves.size() == 0) || (! moves.back().isNormal()))
    return std::make_pair(0.0, 0.0);

  if (!state.getKingPiece(osl::BLACK).isOnBoard() ||
      !state.getKingPiece(osl::WHITE).isOnBoard())
    return std::make_pair(0.0, 0.0);

  osl::state::NumEffectState nState(state);
  for (size_t i=0; i<moves.size(); ++i)
    osl::ApplyMoveOfTurn::doMove(nState, moves[i]);
  osl::threatmate::MlPredictor predictor;
  osl::threatmate::TreePredictor tree;
  return std::make_pair(predictor.probability(nState, moves.back()),
			tree.probability(nState, moves.back()));
}


void Viewer::forward()
{
  setChatIcon();
  TabChild *child = (TabChild *)currentPage();
  child->forward();
}

void Viewer::backward()
{
  resetIcon();
  TabChild *child = (TabChild *)currentPage();
  child->backward();
}

void Viewer::toInitialState()
{
  TabChild *child = (TabChild *)currentPage();
  child->toInitialState();
}

void Viewer::toLastState()
{
  TabChild *child = (TabChild *)currentPage();
  child->toLastState();
}

void Viewer::open(const std::string &fileName)
{
  kifuViewer->open(new CsaFile(fileName.c_str()));
  showPage(kifuViewer);
}

void Viewer::openKakinoki(const std::string &fileName)
{
  kifuViewer->open(new KakinokiFile(fileName.c_str()));
  showPage(kifuViewer);
}

void Viewer::openKi2(const std::string &fileName)
{
  kifuViewer->open(new Ki2File(fileName.c_str()));
  showPage(kifuViewer);
}

void Viewer::openUsi(const std::string &fileName)
{
  kifuViewer->open(new UsiFile(fileName.c_str()));
  showPage(kifuViewer);
}

void Viewer::open(const std::string &fileName, int index)
{
  kifuViewer->open(new KisenFile(fileName.c_str(), index));
  showPage(kifuViewer);
}

void Viewer::openEvalGraph()
{
  kifuViewer->openEvalGraph(true);
  showPage(kifuViewer);
}

class KisenItem : Q3ListViewItem
{
public:
  KisenItem(Q3ListView *parent, int index, QString str1, QString str2,
	    QString str3, QString str4, QString str5)
    : Q3ListViewItem(parent, str1, str2, str3, str4, str5), index(index) {
  }
  int getIndex() const {
    return index;
  }
  int compare(Q3ListViewItem *i, int col, bool ascending) const;
private:
  int index;
};

int KisenItem::compare(Q3ListViewItem *i, int col, bool ascending) const
{
  KisenItem *item = (KisenItem *)i;
  if (col == 0)
  {
    if (getIndex() < item->getIndex())
      return -1;
    else if (getIndex() == item->getIndex())
      return 0;
    else
      return 1;
  }
  else
    return key(col, ascending).compare(i->key(col, ascending));
}

void Viewer::open()
{
  QFileDialog *dialog = new QFileDialog(this);
  QString filename = kifuViewer->getFilename();
  if (!filename.isNull())
  {
    QFileInfo fileInfo(filename);
    dialog->setDir(fileInfo.dir());
  }
    
  QStringList filters = dialog->filters();
  filters << "Kisen (*.kpf *.kif)";
  filters << "CSA (*.csa)";
  filters << "Kakinoki (*.kif)";
  filters << "Usi (*.usi)";
  filters << "Shogi files (*.csa *.kpf *.kif *.ki2 *.usi)";
  dialog->setFilters(filters);
  if (dialog->exec() == QDialog::Accepted)
  {
    const QString& selectedFile = dialog->selectedFile();
    std::string fileName(selectedFile.ascii());
    if (selectedFile.endsWith(".csa"))
    {
      open(fileName);
    }
    else if (selectedFile.endsWith(".usi"))
    {
      openUsi(fileName);
    }
    else if (selectedFile.endsWith(".ki2"))
    {
      openKi2(fileName);
    }
    else if (selectedFile.endsWith(".kif")
	     && osl::KakinokiFile::isKakinokiFile(fileName))
    {
      openKakinoki(fileName);
    }
    else if (selectedFile.endsWith(".kif") || selectedFile.endsWith(".kpf"))
    {
      QString ipxFile(selectedFile);
      ipxFile.replace(ipxFile.length() - 3, 3, "ipx");
      osl::record::KisenIpxFile ipx(ipxFile.ascii());
      if (ipx.size() > 1000)
      {
	bool ok;
	int index = QInputDialog::getInteger("Kisen Index",
					     QString("Choose Kisen Index %1").arg(ipx.size()),
					     0, 0, ipx.size(), 1, &ok, this);
	if (ok)
	{
	  open(fileName, index);
	}
      }
      else
      {
	boost::scoped_ptr<QDialog> qdialog(new QDialog(this));
	boost::scoped_ptr<Ui::KisenDialog> dialog(new Ui::KisenDialog());
        dialog->setupUi(qdialog.get());
	QTextCodec *codec = QTextCodec::codecForName("Shift-JIS");
	for (size_t i = 0; i < ipx.size(); i++)
	{
	  std::string black = ipx.getPlayer(i, osl::BLACK);
	  std::string white = ipx.getPlayer(i, osl::WHITE);
	  std::string black_title = ipx.getTitle(i, osl::BLACK);
	  std::string white_title = ipx.getTitle(i, osl::WHITE);
	  new KisenItem(dialog->player,
			i,
			QString("%1").arg(i),
			codec->toUnicode(black_title.c_str(),
					 black_title.length()),
			codec->toUnicode(black.c_str(), black.length()),
			codec->toUnicode(white_title.c_str(),
					 white_title.length()),
			codec->toUnicode(white.c_str(), white.length()));
	}
	int ret = qdialog->exec();
	if (ret == QDialog::Accepted)
	{
	  KisenItem *item = (KisenItem *)dialog->player->selectedItem();
	  if (item)
	  {
	    open(fileName, item->getIndex());
	  }
	}
      }
    }
    else
    {
      QMessageBox::warning(this, "Open File",
			   "Unknown file type " + selectedFile);
      return;
    }
    emit statusChanged();
  }
}

void Viewer::nextFile()
{
  const KifuFile *file = kifuViewer->getKifuFile();
  if (file == NULL)
    return;
  KifuFile *next_file = file->nextFile();
  if (next_file)
  {
    kifuViewer->open(next_file);
    showPage(kifuViewer);
    emit statusChanged();
  }
}

void Viewer::prevFile()
{
  const KifuFile *file = kifuViewer->getKifuFile();
  if (file == NULL)
    return;
  KifuFile *prev_file = file->prevFile();
  if (prev_file)
  {
    kifuViewer->open(prev_file);
    showPage(kifuViewer);
    emit statusChanged();
  }
}

void Viewer::watchFile(bool enable)
{
  // FIXME allow each kifuviewer to watch its own file
  kifuViewer->watchFile(enable);
}

void Viewer::openUrl(bool reload)
{
#if QT_VERSION >= 0x040000
  HttpWindow *http = new HttpWindow(this, reload);
  http->show();
  if (http->exec() == QDialog::Accepted)
  {
    std::string filename = http->getFilename().toStdString();
    if (http->getFilename().endsWith(".kif"))
      openKakinoki(filename);
    else if (http->getFilename().endsWith(".ki2"))
      openKi2(filename);
    else if (http->getFilename().endsWith(".usi"))
      openUsi(filename);
    else
      open(filename);
    kifuViewer->toLastState();
  }
#endif
}

void Viewer::openUrl()
{
  openUrl(false);
}

void Viewer::reloadUrl()
{
  openUrl(true);
}

void Viewer::hirate()
{
  TabChild *child = (TabChild *)currentPage();
  if (!child->inherits("KifuViewer"))
    return;
  KifuViewer *kifu = (KifuViewer *)child;

  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state(osl::HIRATE);
  kifu->open(state, moves);
  showPage(kifu);
}

void Viewer::network()
{
  boost::scoped_ptr<QDialog> qdialog(new QDialog(this));
  boost::scoped_ptr<Ui::NetworkDialog> dialog(new Ui::NetworkDialog());
  dialog->setupUi(qdialog.get());

  qdialog->exec();

  networkViewer->setHostname(dialog->servername->text());
  networkViewer->setUsername(dialog->username->text());
  networkViewer->setPassword(dialog->password->text());
  networkViewer->connect();

  showPage(networkViewer);
}

void Viewer::view()
{
  TabChild *child = (TabChild *)currentPage();
  if (child == kifuViewer)
    return;

  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  kifuViewer->open(state, moves);
  kifuViewer->toLastState();
  showPage(kifuViewer);
}

void Viewer::viewInNewTab()
{
  TabChild *child = (TabChild *)currentPage();

  KifuViewer *viewer = new KifuViewer(this);
  connect(viewer, SIGNAL(statusChanged()), this, SIGNAL(statusChanged()));
  connect(viewer, SIGNAL(analyzeOnlineDisabled()),
	  this, SIGNAL(analyzeOnlineDisabled()));
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  viewer->open(state, moves);
  viewer->toLastState();
  addTab(viewer, "Kifu");
  showPage(viewer);
}

int Viewer::moveCount() const
{
  TabChild *child = (TabChild *)currentPage();
  return child->moveCount();
}

osl::Player Viewer::turn() const
{
  TabChild *child = (TabChild *)currentPage();
  return child->turn();
}

const osl::state::SimpleState& Viewer::getState()
{
  TabChild *child = (TabChild *)currentPage();
  return child->getState();
}

void Viewer::analyze()
{
  analyze(false);
}

void Viewer::analyzeInNewTab()
{
  analyze(true);
}

void Viewer::analyze(bool newTab)
{
  bool tabCreated = false;
  TabChild *child = (TabChild *)currentPage();
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  AnalysisViewer *av;

  if (!analysisViewer || newTab)
  {
    tabCreated = true;
    av = new AnalysisViewer(this);
    connect(av, SIGNAL(statusChanged()),
	    this, SIGNAL(statusChanged()));
  }
  else if (child->inherits("AnalysisViewer"))
  {
    av = (AnalysisViewer *)child;
  }
  else
    av = analysisViewer;

  bool success = av->analyze(state, moves);

  if (success)
  {
    if (tabCreated)
    {
      if (!analysisViewer)
	analysisViewer = av;
      addTab(av, "Analyze position");
    }
    showPage(av);
  }
  else
  {
    if (tabCreated)
    {
      delete av;
    }
  }
}

void Viewer::analyzeProblems()
{
  ProblemAnalyzer *analyzer;

  analyzer = new ProblemAnalyzer(this);
  connect(analyzer, SIGNAL(statusChanged()),
	  this, SIGNAL(statusChanged()));

  bool success = analyzer->analyze();

  if (success)
  {
    addTab(analyzer, "Problem Analyze");
    showPage(analyzer);
  }
  else
  {
    delete analyzer;
  }
}

void Viewer::checkmateSearch()
{
  TabChild *child = (TabChild *)currentPage();
  const int node_limit = 100000;
  bool success = checkmateViewer->analyze(child->getState(), node_limit);
  if (success)
    showPage(checkmateViewer);
}

void Viewer::altCheckmateSearch()
{
  TabChild *child = (TabChild *)currentPage();
  const int node_limit = 100000;
  bool success = checkmateViewer->analyze(child->getState(), node_limit, true);
  if (success)
    showPage(checkmateViewer);
}

void Viewer::checkmateSimulation()
{
  TabChild *child = (TabChild *)currentPage();
  bool success = checkmateViewer->analyze(child->getState(), 0);
  if (success)
    showPage(checkmateViewer);
}

void Viewer::ntesukiSearch()
{
#ifdef ENABLE_NTESUKI
  TabChild *child = (TabChild *)currentPage();
  bool success = ntesukiViewer->analyze(child->getState());
  if (success)
    showPage(ntesukiViewer);
#endif
}

void Viewer::quiescenceSearch()
{
  TabChild *child = (TabChild *)currentPage();
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  bool success = quiescenceViewer->analyze(state, moves);
  if (success)
    showPage(quiescenceViewer);
}

void Viewer::quiescenceSearchHalf()
{
  TabChild *child = (TabChild *)currentPage();
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  bool success = quiescenceViewer->analyzeHalfDepth(state, moves);
  if (success)
    showPage(quiescenceViewer);
}

void Viewer::piecePairDialog()
{
  TabChild *child = (TabChild *)currentPage();
  PiecePairDialog *dialog = new PiecePairDialog(child->getState(), this);
  dialog->show();
  dialog->raise();
}

void Viewer::moveGenerateDialog()
{
  if (currentPage()->inherits("BoardAndListTabChild"))
  {
    ((BoardAndListTabChild *)currentPage())->moveGenerateDialog();
    return;
  }

  TabChild *child = (TabChild *)currentPage();
  child->moveGenerateDialog();
}

void Viewer::toggleOrientation()
{
  TabChild *child = (TabChild *)currentPage();
  child->toggleOrientation();
}

QString Viewer::getFilenameToSave()
{
  QString s = QFileDialog::getSaveFileName(0,
					   "CSA (*.csa)",
					   this,
					   "save file",
					   "Save moves up to current state");
  return s;
}

void Viewer::saveMovesToCurrent()
{
  QString s = getFilenameToSave();
  if (s.isNull())
    return;
  TabChild *child = (TabChild *)currentPage();
  osl::stl::vector<osl::Move> moves;
  osl::state::SimpleState state = child->getStateAndMovesToCurrent(moves);
  std::ofstream os(s.ascii());
  os << state;
  for (int i = 0; i < (int)moves.size(); i++)
  {
    os << osl::record::csa::show(moves[i]) << std::endl;
  }
}

void Viewer::exportCurrent()
{
  QString s = getFilenameToSave();
  if (s.isNull())
    return;
  TabChild *child = (TabChild *)currentPage();
  std::ofstream os(s.ascii());
  os << child->getState();
}

void Viewer::moveTo(int index)
{
  TabChild *child = (TabChild *)currentPage();
  if (child == kifuViewer)
  {
    kifuViewer->updateIndex(index);
  }
}

void Viewer::copy()
{
  TabChild *child = (TabChild *)currentPage();
  child->copy();
}

void Viewer::copyBoardAndMoves()
{
  TabChild *child = (TabChild *)currentPage();
  child->copyBoardAndMoves();
}

void Viewer::copyUsi()
{
  TabChild *child = (TabChild *)currentPage();
  child->copyUsi();
}

void Viewer::copyBoardAndMovesUsi()
{
  TabChild *child = (TabChild *)currentPage();
  child->copyBoardAndMovesUsi();
}

QString Viewer::getFilename()
{
  TabChild *child = (TabChild *)currentPage();
  QString filename = child->getFilename();
  if (filename.isNull())
    return "";

  QFileInfo fi(filename);
  return fi.fileName();
}

void Viewer::closeTab()
{
  QWidget *widget = currentPage();
  if (widget->inherits("AnalysisViewer"))
  {
    removePage(widget);
    if (widget == analysisViewer)
    {
      analysisViewer = 0;
      for (int i = 0; i < count(); i++)
      {
	QWidget *child = page(i);
	if (child->inherits("AnalysisViewer"))
	{
	  analysisViewer = (AnalysisViewer *)child;
	  break;
	}
      }
    }
    delete widget;
  }
  else if ((widget != kifuViewer && widget->inherits("KifuViewer"))
	   || widget->inherits("ProblemAnalyzer"))
  {
    removePage(widget);
    delete widget;
  }
}

bool Viewer::isSenteView() const
{
  if (currentPage()->inherits("BoardTabChild"))
  {
    return ((BoardTabChild *)currentPage())->isSenteView();
  }
  return true;
}

void Viewer::setOrientation(bool gote)
{
  if (currentPage()->inherits("BoardTabChild"))
  {
    return ((BoardTabChild *)currentPage())->setOrientation(!gote);
  }
}

void Viewer::setAnalyzeOnline(bool enable)
{
  KifuViewer::setAnalyzeOnline(enable);
  if (enable)
  {
    if (currentPage()->inherits("KifuViewer")) {
      osl::state::SimpleState s;
      osl::stl::vector<osl::Move> m;
      s = ((KifuViewer *)currentPage())->getStateAndMovesToCurrent(m);
      ((KifuViewer *)currentPage())->prepareUpdateForAnalysis(s, m, 0, osl::Move());
    }
    else
      showPage(kifuViewer);
  }
}

void Viewer::notifyOrientation(QWidget *widget)
{
  bool sente;
  if (widget->inherits("BoardTabChild"))
    sente = ((BoardTabChild *)widget)->isSenteView();
  else
    sente = true;

  emit orientationChanged(sente);
}

void Viewer::enableEffect(bool on)
{
  ((TabChild *)currentPage())->enableEffect(on);
}

void Viewer::highlightLastMove(bool on)
{
  ((TabChild *)currentPage())->highlightLastMove(on);
}

void Viewer::notifyEffect(QWidget *widget)
{
  if (widget->inherits("BoardTabChild"))
    emit effectChanged(((BoardTabChild *)widget)->effectEnabled());
}

void Viewer::kifuAnalyze()
{
  if (((TabChild *)currentPage())->inherits("KifuViewer"))
  {
    ((KifuViewer *)currentPage())->analyze();
  }
}

void Viewer::resetIcon()
{
#if QT_VERSION >= 0x040000 && (defined(__APPLE__) || defined(__FreeBSD__))
  qApp->setWindowIcon(QIcon(":/images/icon.png"));
#endif
}

void Viewer::setChatIcon()
{
#if QT_VERSION >= 0x040000 && (defined(__APPLE__) || defined(__FreeBSD__))
  qApp->setWindowIcon(QIcon(":/images/icon-chat.png"));
#endif
}

void Viewer::chatReceived()
{
  if (!chatIcon)
  {
    chatIcon = true;
    setChatIcon();
  }
}

void Viewer::chatDisplayed()
{
  if (chatIcon)
  {
    chatIcon = false;
    resetIcon();
  }
}

void Viewer::editState()
{
  TabChild *child = (TabChild *)currentPage();
  editor->setState(child->getState());
  showPage(editor);
}

void Viewer::evalDebug()
{
  TabChild *child = (TabChild *)currentPage();
  EvalDebugDialog *dialog = new EvalDebugDialog(child->getState(),
						this);
  if (child->inherits("BoardTabChild"))
  {
    BoardTabChild *board_tab_child = (BoardTabChild *)child;
    connect(board_tab_child,
	    SIGNAL(statusChanged(const osl::state::SimpleState &,
				 const osl::stl::vector<osl::Move> &,
				 int, osl::Move)),
	    dialog, SLOT(setStatus(const osl::state::SimpleState &,
				   const osl::stl::vector<osl::Move> &,
				   int, osl::Move)));
  }
  dialog->show();
  dialog->raise();
}

void Viewer::testEvalDebug()
{
  TabChild *child = (TabChild *)currentPage();
  OpenMidEndingEvalDebugDialog *dialog =
    new OpenMidEndingEvalDebugDialog(child->getState(), this);
  if (child->inherits("BoardTabChild"))
  {
    BoardTabChild *board_tab_child = (BoardTabChild *)child;
    connect(board_tab_child,
	    SIGNAL(statusChanged(const osl::state::SimpleState &,
				 const osl::stl::vector<osl::Move> &,
				 int, osl::Move)),
	    dialog, SLOT(setStatus(const osl::state::SimpleState &,
				   const osl::stl::vector<osl::Move> &,
				   int, osl::Move)));
  }
  dialog->show();
  dialog->raise();
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
