#include "eval/pin.h"
#include "eval/indexCache.h"
#include "osl/effect_util/pin.h"
#include "osl/record/csa.h"

template <int DIM, class IndexFunc>
osl::MultiInt gpsshogi::PinBase<DIM, IndexFunc>::
eval(const osl::state::NumEffectState &state, const MultiWeights& weights,
     CArray<MultiInt,2>& /*saved_state*/) const
{
  MultiInt result;
  osl::container::PieceMask black_mask = state.pin(BLACK);
  while (black_mask.any())
  {
    const osl::Piece piece = state.getPieceOf(black_mask.takeOneBit());
    result += weights.value(f(state.getKingPosition<BLACK>(), piece));
  }
  osl::container::PieceMask white_mask = state.pin(WHITE);
  while (white_mask.any())
  {
    const osl::Piece piece = state.getPieceOf(white_mask.takeOneBit());
    result -= weights.value(f(state.getKingPosition<WHITE>(), piece));
  }
  return result;
}

template <int DIM, class IndexFunc>
void gpsshogi::PinBase<DIM, IndexFunc>::
featuresNonUniq(const osl::state::NumEffectState &state, 
	 index_list_t&values,
	 int offset) const
{
  osl::container::PieceMask black_mask = state.pin(BLACK);
  while (black_mask.any())
  {
    const osl::Piece piece = state.getPieceOf(black_mask.takeOneBit());
    const int idx = f(state.getKingPosition<BLACK>(), piece);
    values.add(offset + idx, 1);
  }
  osl::container::PieceMask white_mask = state.pin(WHITE);
  while (white_mask.any())
  {
    const osl::Piece piece = state.getPieceOf(white_mask.takeOneBit());
    const int idx = f(state.getKingPosition<WHITE>(), piece);
    values.add(offset + idx, -1);
  }
}

template <int DIM, class IndexFunc>
void gpsshogi::
PinBase<DIM, IndexFunc>::showSummary(std::ostream& os, const MultiWeights&) const
{
  os << name() << " done" << std::endl;
}

void gpsshogi::Pin::showAll(std::ostream &os,
			    const MultiWeights& weights) const
{
  // TODO: quad?
  for (int stage = 0; stage < 3; ++stage)
  {
    os << name() << " " << stage << std::endl;
    for (int y_diff = 0; y_diff < 17; ++y_diff)
    {
      os << "Y: " << y_diff - 8 << std::endl;
      for (int ptype = PTYPE_MIN; ptype <= PTYPE_MAX; ++ptype)
      {
	if (!isPiece(static_cast<Ptype>(ptype)))
	  continue;
	os << static_cast<Ptype>(ptype);
	for (int x_diff = 0; x_diff < 9; ++x_diff)
	{
	  os << " "
	     << weights.value((ptype - PTYPE_PIECE_MIN) * 17 * 9 +
			      x_diff * 17 + y_diff)[stage];
	}
	os << std::endl;
      }
    }
  }
}

template <osl::Player Defense>
void gpsshogi::
PinPtype::featuresOneKing(
  const NumEffectState &state,
  IndexCacheI<MaxActiveWithDuplication> &features) const
{
  const Position king = state.getKingPosition<Defense>();
  const int weight = (Defense == BLACK ? 1 : -1);
  osl::container::PieceMask pin_mask = state.pin(Defense);
  while (pin_mask.any())
  {
    const Piece piece = state.getPieceOf(pin_mask.takeOneBit());
    if (!state.hasEffectBy<Defense>(piece.position()))
      continue;
    if (king.y() == piece.position().y()) // rook h
    {
      features.add(piece.ptype() + PTYPE_SIZE * 1, weight);
    }
    else if (king.x() == piece.position().x())
    {
      if (state.hasEffectByPtypeStrict<LANCE>(PlayerTraits<Defense>::opponent,
					      piece.position())) // lance
      {
	features.add(piece.ptype() + PTYPE_SIZE * 4, weight);
      }
      else // rook v
      {
	features.add(piece.ptype() + PTYPE_SIZE * 0, weight);
      }
    }
    else // bishop
    {
      if ((Defense == BLACK && piece.position().y() < king.y()) ||
	  (Defense == WHITE && piece.position().y() > king.y())) // u
      {
	features.add(piece.ptype() + PTYPE_SIZE * 2, weight);
      }
      else
      {
	features.add(piece.ptype() + PTYPE_SIZE * 3, weight);
      }
    }
  }
}
void gpsshogi::
PinPtype::featuresOneNonUniq(
  const NumEffectState &state,
  IndexCacheI<MaxActiveWithDuplication> &features) const
{
  featuresOneKing<BLACK>(state, features);
  featuresOneKing<WHITE>(state, features);
}

template <osl::Player Defense>
void gpsshogi::
PinPtypeDistance::featuresOneKing(
  const NumEffectState &state,
  IndexCacheI<MaxActiveWithDuplication> &features) const
{
  const Position king = state.getKingPosition<Defense>();
  const int weight = (Defense == BLACK ? 1 : -1);
  osl::container::PieceMask pin_mask = state.pin(Defense);
  while (pin_mask.any())
  {
    const Piece piece = state.getPieceOf(pin_mask.takeOneBit());
    if (!state.hasEffectBy<Defense>(piece.position()))
      continue;
    if (king.y() == piece.position().y()) // rook h
    {
      features.add((piece.ptype() + PTYPE_SIZE * 1) * 7 +
		   std::abs(king.x() - piece.position().x()) - 1, weight);
    }
    else if (king.x() == piece.position().x())
    {
      if (state.hasEffectByPtypeStrict<LANCE>(PlayerTraits<Defense>::opponent,
					      piece.position())) // lance
      {
	features.add((piece.ptype() + PTYPE_SIZE * 4) * 7 +
		     std::abs(king.y() - piece.position().y()) - 1, weight);
      }
      else // rook v
      {
	features.add((piece.ptype() + PTYPE_SIZE * 0) * 7 +
		     std::abs(king.y() - piece.position().y()) - 1, weight);
      }
    }
    else // bishop
    {
      if ((Defense == BLACK && piece.position().y() < king.y()) ||
	  (Defense == WHITE && piece.position().y() > king.y())) // u
      {
	features.add((piece.ptype() + PTYPE_SIZE * 2) * 7 +
		     std::abs(king.x() - piece.position().x()) - 1, weight);
      }
      else
      {
	features.add((piece.ptype() + PTYPE_SIZE * 3) * 7 +
		     std::abs(king.x() - piece.position().x()) - 1, weight);
      }
    }
  }
}
void gpsshogi::
PinPtypeDistance::featuresOneNonUniq(
  const NumEffectState &state,
  IndexCacheI<MaxActiveWithDuplication> &features) const
{
  featuresOneKing<BLACK>(state, features);
  featuresOneKing<WHITE>(state, features);
}

template <osl::Player Defense>
void gpsshogi::
PinPtypePawnAttack::featuresOneKing(
  const NumEffectState &state,
  IndexCacheI<MaxActiveWithDuplication> &features) const
{
  const Position king = state.getKingPosition<Defense>();
  const int weight = (Defense == BLACK ? 1 : -1);
  osl::container::PieceMask pin_mask = state.pin(Defense);
  while (pin_mask.any())
  {
    const Piece piece = state.getPieceOf(pin_mask.takeOneBit());
    if (!state.hasEffectBy<Defense>(piece.position()))
      continue;
    const Position up =
      piece.position() + DirectionPlayerTraits<U, Defense>::offset();
    if (!up.isOnBoard() ||
	(!state.hasEffectByPtypeStrict<PAWN>(PlayerTraits<Defense>::opponent,
					     up) &&
	 (state.isPawnMaskSet(PlayerTraits<Defense>::opponent,
			      piece.position().x()) ||
	  !state.getPieceAt(up).isEmpty())))
    {
      continue;
    }

    if (king.y() == piece.position().y()) // rook h
    {
      features.add(piece.ptype() + PTYPE_SIZE * 0, weight);
    }
    else if (king.x() == piece.position().x())
    {
      // do nothing
    }
    else // bishop
    {
      if ((Defense == BLACK && piece.position().y() < king.y()) ||
	  (Defense == WHITE && piece.position().y() > king.y())) // u
      {
	features.add(piece.ptype() + PTYPE_SIZE * 1, weight);
      }
      else
      {
	features.add(piece.ptype() + PTYPE_SIZE * 2, weight);
      }
    }
  }
}

void gpsshogi::
PinPtypePawnAttack::featuresOneNonUniq(
  const NumEffectState &state,
  IndexCacheI<MaxActiveWithDuplication> &features) const
{
  featuresOneKing<BLACK>(state, features);
  featuresOneKing<WHITE>(state, features);
}

namespace gpsshogi
{
  template class PinBase<2142, PinF>;
  template class PinBase<10206, PinYF>;
}
