/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*---------------------------------------------------------------------------
   imgproc: processes image

   Copyright (C) 2003, 2004 Gerber van der Graaf

   imgproc is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR 'PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



   RCS version control
   $Log: imgproc.c,v $
   Revision 2.5  2006/01/31 14:18:03  gerber
   version 0.3.2

   Revision 2.4  2005/06/15 09:32:20  gerber
   check on memory leacks: change of internal structure; gpiv_alloc_pivdata and gpiv_free_pivdata in main optimizing by printing progress only if percentage has been changed

   Revision 2.3  2005/02/26 16:29:19  gerber
   parameter flags (parameter_logic) defined as gboolean, added scale_type 
   to struct __GpivPostPar for inverse scaling

   Revision 2.2  2004/10/15 10:11:41  gerber
   GPIV_ and Gpiv prefix to defines and structure names of libgpiv

   Revision 2.1  2004/06/14 14:39:26  gerber
   addied image filtering and processing program




------------------------------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <gpiv.h>

#define PARFILE "gpivrc"	/* Parameter file name */

#define USAGE "\
Usage: imgproc | smooth | hilo | clip | fft | invfft \n\
              [-b int] [-c int] [-f file] [-h | --help] [-o | --operation] \n\
              [-p | --print] [--filter] [-r int] [-t | --threshold int] \n\
              [-w int] [-v] < stdin > stdout \n\
\n\
keys: \n\
\n\
-b bit:           for getbit: bit number [0, .., 7] \n\
-c  ncols:        number of columns  \n\
-f file:          filename (without .r extension) instead of stdin \n\
                  and stdout \n\
--filter:         only for imgproc: image filter to be performed: \n\
                  make test image (0), smoothing (1), high-low filtering (2), \n\
                  clipping (3), fft (4), inverse fft (5), lowpass filter (8), \n\
                  highpass filter (9), getbit (10) \n\
-h | --help:      this on-line help \n\
-p | --print:     print parameters to stdout \n\
-o | --operation: for smoothing: set pixel equal to the mean of the \n\
                  window value (0), subtract mean from from pixel (1) \n\
                  add (2), multiply (3) or divide (4)\n\
-r nrows:         number of rows  \n\
-t | --threshold: only for clip: threshold value to be set to zero \n\
-w size:          window size (default: 15) \n\
-v | --version:   version number \n\
"


#define HELP  "\
Image processing program for PIV images."

#define RCSID "$Id: imgproc.c,v 2.5 2006/01/31 14:18:03 gerber Exp $"

/*
 * Global variables
 */
gboolean fname_logic = FALSE;
gboolean print_par = FALSE;


/*
 * Function prototypes
 */

static void 
command_args(int argc, 
             char *argv[], 
             char fname[GPIV_MAX_CHARS],
             GpivImagePar * image_par, 
             GpivImageProcPar * image_proc_par
             )
/* ----------------------------------------------------------------------------
 * Command line argument handling */
{
    char c;
    int argc_next;
    while (--argc > 0 && (*++argv)[0] == '-') {
	argc_next = 0;
/*
 * argc_next is set to 1 if the next cmd line argument has to be searched for; 
 * in case that the command line argument concerns more than one char or cmd 
 * line argument needs a parameter
 */
	while (argc_next == 0 && (c = *++argv[0])) {
	    switch (c) {
	    case 'v':
/*
 * Use Revision Control System (RCS) for version
 */
		printf("\n%s\n", RCSID);
		exit(0);
		break;
	    case 'h':
		printf("\n%s", RCSID);
		printf("\n%s", HELP);
		printf("\n%s", USAGE);
		exit(0);
		break;
/*
 * bitnumber for getbit
 */
	    case 'b':
		 image_proc_par->bit = atoi(*++argv);
		 image_proc_par->bit_logic = TRUE;
		 argc_next = 1;
		 --argc;
		break;
/*
 * number of image columns
 */
	    case 'c':
		 image_par->ncolumns = atoi(*++argv);
		 image_par->ncolumns_logic = TRUE;
		 argc_next = 1;
		 --argc;
		break;
/*
 * file name
 */
	    case 'f':
		strcpy(fname, *++argv);
		fname_logic = TRUE;
		argc_next = 1;
		--argc;
		break;


/*
 * whar to do with the mean value of a window: operation
 */
	    case 'o':
		 image_proc_par->operator = atoi(*++argv);
		 image_proc_par->operator_logic = TRUE;
		 --argc;
		 argc_next = 1;
		 break;

/*
 * print pararameters
 */
	    case 'p':
		    print_par = TRUE;
		break;
/*
 * number of image rows
 */
	    case 'r':
		 image_par->nrows = atoi(*++argv);
		 image_par->nrows_logic = TRUE;
		 --argc;
		 argc_next = 1;
		 break;

/*
 * threshold
 */
	    case 't':
                image_proc_par->threshold =  atoi(*++argv);
                image_proc_par->threshold_logic = TRUE;
                --argc;
                argc_next = 1;
		break;

/*
 * window size
 */
	    case 'w':
                image_proc_par->window =  atoi(*++argv);
                image_proc_par->window_logic = TRUE;
                --argc;
                argc_next = 1;
		break;

/*
 * long option keys
 */
	    case '-':
		if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);
                    
                } else if (strcmp("-print", *argv) == 0) {
		    print_par = TRUE;
                    
                } else if (strcmp("-filter", *argv) == 0) {
		    image_proc_par->filter = atoi(*++argv);
		    image_proc_par->filter_logic = TRUE;
                    --argc;
                    
                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);
                    
                } else if (strcmp("-operation", *argv) == 0) {
                    image_proc_par->operator = atoi(*++argv);
		 image_proc_par->operator_logic = TRUE;
		 --argc;

                } else if (strcmp("-threshold", *argv) == 0) {
		    image_proc_par->threshold =  atoi(*++argv);
                    image_proc_par->threshold_logic = TRUE;
                     --argc;
                    argc_next = 1;

                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
		break;

	    default:
		gpiv_error("%s: unknown option: %s", RCSID, *argv);
		break;
	    }
	}
    }

    if (argc != 0) {
	gpiv_error("%s: unknown argument: %s", RCSID, *argv);
    }

}



static void
make_fname(char *fname_base, 
           char *fname_header, 
           char *fname_parameter, 
           char *fname_in,
           char *fname_out
           )
/*-----------------------------------------------------------------------------
 * generates filenames for output
 */
{

    if (fname_logic == FALSE) gpiv_error("%s: Filename has to be set", RCSID);

    gpiv_io_make_fname(fname_base, GPIV_EXT_HEADER, fname_header);
    if (print_par) printf("# Image header file: %s\n", fname_header);

    gpiv_io_make_fname(fname_base, GPIV_EXT_PAR, fname_parameter);
    if (print_par) printf("# Parameter file: %s\n", fname_parameter);

    gpiv_io_make_fname(fname_base, GPIV_EXT_RAW_IMAGE, fname_in);
    if (print_par) printf("# Input image file: %s\n", fname_in);

    gpiv_io_make_fname(fname_base, GPIV_EXT_RAW_IMAGE_PROC, fname_out);
    if (print_par) printf("# Output imagecat  file: %s\n", fname_out);

}



/*
 * Image processing functions
 */

/* static void */
/* imgproc_invfft(GpivImagePar image_par, */
/*                GpivImageProcPar image_proc_par, */
/*                guint16 **img_in,  */
/*                guint16 **img_out); */




int 
main(int argc, 
     char *argv[]
     )
/* ----------------------------------------------------------------------------
 *  main routine to process images for PIV 
 */
{
    FILE *fp_par;
    char *err_msg = NULL;
    char fname_base[GPIV_MAX_CHARS], 
        fname_header[GPIV_MAX_CHARS],
        fname_in[GPIV_MAX_CHARS],
        fname_out[GPIV_MAX_CHARS],
	fname_parameter[GPIV_MAX_CHARS];
    guint16 **img1_in = NULL;
    guint16 **img2_in = NULL;
    guint16 **img1_out = NULL;
    guint16 **img2_out = NULL;
    GpivImageProcPar image_proc_par;
    GpivImagePar image_par;

    char *c = NULL;

/*
 * Image header info initialization
 */
    image_par.ncolumns_logic = FALSE;
    image_par.nrows_logic = FALSE;
    image_par.depth_logic = FALSE;
    image_par.x_corr_logic = FALSE;

/*
 * Image processing parameter initialization
 * Define operation type from program name, which is a symbolic link to imgproc
 */
    if ((c = strstr(argv[0], "imgproc")) != NULL) {
        image_proc_par.filter = 0;
        image_proc_par.filter_logic = FALSE;
    } else if ((c = strstr(argv[0], "mktestimg")) != NULL) {
        image_proc_par.filter = GPIV_MK_TSTIMG;
        image_proc_par.filter_logic = TRUE;
    } else if ((c = strstr(argv[0], "smooth")) != NULL) {
        image_proc_par.filter = GPIV_FI_SMOOTH;
        image_proc_par.filter_logic = TRUE;
    } else if ((c = strstr(argv[0], "hilo")) != NULL) {
        image_proc_par.filter = GPIV_FI_HILO;
        image_proc_par.filter_logic = TRUE;
    } else if ((c = strstr(argv[0], "clip")) != NULL) {
        image_proc_par.filter = GPIV_FI_CLIP;
        image_proc_par.filter_logic = TRUE;
    } else if ((c = strstr(argv[0], "fft")) != NULL) {
        image_proc_par.filter = GPIV_FI_FFT;
        image_proc_par.filter_logic = TRUE;
/*     } else if ((c = strstr(argv[0], "invfft")) != NULL) { */
/*         image_proc_par.filter = GPIV_FI_INVFFT; */
/*         image_proc_par.filter_logic = TRUE; */
    } else if ((c = strstr(argv[0], "lowpass")) != NULL) {
        image_proc_par.filter = GPIV_FI_LOWPASS;
        image_proc_par.filter_logic = TRUE;
    } else if ((c = strstr(argv[0], "highpass")) != NULL) {
        image_proc_par.filter = GPIV_FI_HIGHPASS;
        image_proc_par.filter_logic = TRUE;
    } else if ((c = strstr(argv[0], "getbit")) != NULL) {
        image_proc_par.filter = GPIV_PT_GETBIT;
        image_proc_par.filter_logic = TRUE;
    } else {
        gpiv_error("imgproc: unvalid program name or symlink");
    }

    image_proc_par.bit = 0;
    image_proc_par.first_dir = 0;
    image_proc_par.last_dir = 0;
    image_proc_par.dir_prefix = 0;    
    image_proc_par.first_file = 0;
    image_proc_par.last_file = 0;
    image_proc_par.file_prefix = 0;
    image_proc_par.operator = 0;
    image_proc_par.window = 0;
    image_proc_par.threshold = 0;

    image_proc_par.bit_logic = FALSE;
    image_proc_par.first_dir_logic = TRUE;
    image_proc_par.last_dir_logic = TRUE;
    image_proc_par.dir_prefix_logic = TRUE;     
    image_proc_par.first_file_logic = TRUE;
    image_proc_par.last_file_logic = TRUE;
    image_proc_par.file_prefix_logic = TRUE;
    image_proc_par.operator_logic = FALSE; 
    image_proc_par.window_logic = FALSE;
    image_proc_par.threshold_logic = FALSE;


    command_args(argc, argv, fname_base, &image_par, &image_proc_par);

    if (fname_logic == TRUE) {
/* 
 * Generating filenames 
 */
	make_fname(fname_base, fname_header, fname_parameter, 
                   fname_in, fname_out);
	if (print_par) 
            printf("\n# Parameters written to: %s", fname_parameter);


/*
 * Reads image header data from file.h and/or resource files 
 * and image processing parametes
 * if not overridden by the commandline options
 */
        gpiv_scan_parameter("", fname_header, &image_par, print_par);
        gpiv_scan_resourcefiles(GPIV_IMAGE_PAR_KEY, &image_par, print_par);
        gpiv_scan_parameter(GPIV_IMAGEPROC_PAR_KEY, PARFILE, &image_proc_par, 
			    print_par);
        gpiv_scan_resourcefiles(GPIV_IMAGEPROC_PAR_KEY, &image_proc_par, 
                                print_par);
/*         gpiv_imgproc_fprint_parameters(fp_par, image_proc_par); */


    } else {
/*
 * use stdin, stdout
 */
        gpiv_scan_resourcefiles("", &image_par, print_par);
        gpiv_scan_parameter(GPIV_IMAGEPROC_PAR_KEY, PARFILE, &image_proc_par, 
			    print_par);
        gpiv_scan_resourcefiles(GPIV_IMAGEPROC_PAR_KEY, &image_proc_par, 
                                print_par);
    }



/*
 * Set default values
 */
    if (image_proc_par.window_logic == FALSE) {
        image_proc_par.window = GPIV_IMGPROCPAR_DEFAULT__WINDOW;
        image_proc_par.window_logic = TRUE;
    }

    if (image_proc_par.operator_logic == FALSE) {
        image_proc_par.operator = GPIV_SUBTRACT /* GPIV_EQUAL */;
        image_proc_par.operator_logic = TRUE;
    }

    if (image_proc_par.threshold_logic == FALSE) {
        image_proc_par.threshold = GPIV_IMGPROCPAR_DEFAULT__THRESHOLD;
        image_proc_par.threshold_logic = TRUE;
    }

/*
 * Check if all parameters have been read and prints out
 */
    if ((err_msg = 
         gpiv_imgproc_check_parameters_read(image_proc_par))
        != NULL) gpiv_error ("%s: %s", RCSID, err_msg);

    if (fname_logic == TRUE) {
	if ((fp_par = fopen(fname_parameter, "a")) == NULL) {
	    gpiv_error("%s: failure opening %s for input",
                       RCSID, fname_parameter);
	}
        gpiv_img_fprint_header(fp_par, image_par);
	gpiv_imgproc_fprint_parameters(fp_par, image_proc_par);
        fclose(fp_par);
    }

    if (print_par) {
        gpiv_img_print_header(image_par);
        gpiv_imgproc_print_parameters(image_proc_par);
    }
/*
 * Check eventually parameters on correct values and adjust belonging 
 * variables.
 */



/*
 * Now the relevant parameters are known, data memory can be allocated
 */
    if (image_proc_par.filter != GPIV_MK_TSTIMG){
        img1_in = gpiv_alloc_img(image_par);
    }
    img1_out = gpiv_alloc_img(image_par);

    if (image_par.x_corr) {
        if (image_proc_par.filter != GPIV_MK_TSTIMG){
            img2_in = gpiv_alloc_img(image_par);
        }
        img2_out = gpiv_alloc_img(image_par);
    }


/*
 * reads image data
 */
    if (image_proc_par.filter != GPIV_MK_TSTIMG){
        if (fname_logic) {
            if ((err_msg = 
                 gpiv_fread_image(fname_in, img1_in, img2_in, image_par))
                != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
            
        } else {
            gpiv_read_image(img1_in, img2_in, image_par);
        }
    }


/* 
 * Here the function calls of the image processing
 */

    if (image_proc_par.filter == GPIV_MK_TSTIMG) {
        gpiv_imgproc_mktstimg(image_par, image_proc_par, img1_out);
        if (image_par.x_corr) {
            gpiv_imgproc_mktstimg(image_par, image_proc_par, img2_out);
        }
        
    } else if (image_proc_par.filter == GPIV_FI_HILO) {
        gpiv_imgproc_highlow(image_par, image_proc_par, img1_in, img1_out);
        if (image_par.x_corr) {
            gpiv_imgproc_highlow(image_par, image_proc_par, img2_in, img2_out);
        }
        
    } else if (image_proc_par.filter == GPIV_FI_CLIP) {
        gpiv_imgproc_clip(image_par, image_proc_par, img1_in, img1_out);
        if (image_par.x_corr) {
            gpiv_imgproc_clip(image_par, image_proc_par, img2_in, img2_out);
        }
        
    } else if (image_proc_par.filter == GPIV_FI_FFT) {
        gpiv_imgproc_fft(image_par, image_proc_par, img1_in, img1_out);
        if (image_par.x_corr) {
            gpiv_imgproc_fft(image_par, image_proc_par, img2_in, img2_out);
        }
        
/*     } else if (image_proc_par.filter == GPIV_FI_INVFFT) { */
/*         gpiv_imgproc_invfft(image_par, image_proc_par, img1_in, img1_out); */
/*         if (image_par.x_corr) { */
/*             gpiv_imgproc_invfft(image_par, image_proc_par, img2_in, img2_out); */
            
    } else  if (image_proc_par.filter == GPIV_FI_CORR) {
        gpiv_imgproc_correlate(image_par, image_proc_par, img1_in, img2_in, 
                               img1_out);
        
    } else if (image_proc_par.filter == GPIV_FI_CONV) {
        gpiv_imgproc_convolve(image_par, image_proc_par, img1_in, img2_in, 
                              img1_out);
        
    } else if (image_proc_par.filter == GPIV_FI_LOWPASS) {
        gpiv_imgproc_lowpass(image_par, image_proc_par, img1_in, img1_out);
        if (image_par.x_corr) {
            gpiv_imgproc_lowpass(image_par, image_proc_par, img2_in, img2_out);
        }
        
    } else if (image_proc_par.filter == GPIV_FI_HIGHPASS) {
        gpiv_imgproc_highpass(image_par, image_proc_par, img1_in, img1_out);
        if (image_par.x_corr) {
            gpiv_imgproc_highpass(image_par, image_proc_par, img2_in, img2_out);
        }
        
    } else if (image_proc_par.filter == GPIV_PT_GETBIT) {
        gpiv_imgproc_getbit(image_par, image_proc_par, img1_in, img1_out);
        if (image_par.x_corr) {
            gpiv_imgproc_getbit(image_par, image_proc_par, img2_in, img2_out);
        }
        
    } else {
        gpiv_error("img_proc: non-existing filter %d", image_proc_par.filter);
    }


/*
 * And writing to output
 */
    if (fname_logic == TRUE) {
	 if ((err_msg = 
              gpiv_fwrite_image(fname_out, img1_out, img2_out, image_par))
              != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
    } else {
	 gpiv_write_image(img1_out, img2_out, image_par);
    }


/*
 * Freeing allocated memory of matrices
 */
    if (image_proc_par.filter != GPIV_MK_TSTIMG){
        gpiv_free_img(img1_in, image_par);
    }
    gpiv_free_img(img1_out, image_par);

    if (image_par.x_corr) {
        if (image_proc_par.filter != GPIV_MK_TSTIMG){
            gpiv_free_img(img2_in, image_par);
        }
        gpiv_free_img(img2_out, image_par);
    }


    return 0;
}
