# gozerbot/plugins.py
#
#

""" provide plugin infrastructure """

__copyright__ = 'this file is in the public domain'

from gozerbot.datadir import datadir
from gozerbot.eventhandler import commandhandler
from gozerbot.users import users
from gozerbot.monitor import outmonitor, saymonitor, jabbermonitor
from gozerbot.generic import rlog, handle_exception, checkchan, lockdec, \
plugnames, waitforqueue
from gozerbot.myimport import my_import
from gozerbot.persist import Persist
from gozerbot.persistconfig import PersistConfig
from gozerbot.config import config
from gozerbot.commands import cmnds
from gozerbot.callbacks import callbacks, jcallbacks
from gozerbot.redispatcher import rebefore, reafter
from gozerbot.ircevent import makeargrest, Ircevent
from gozerbot.aliases import aliascheck
from gozerbot.limiter import getlastalertmin, setlastalertmin, \
getlastalerthour, setlastalerthour, shouldblockmin, shouldblockhour, addievent
from gozerbot.ignore import shouldignore
from gozerbot.thr import start_new_thread
if config['jabberenable']:
    from gozerbot.jabbermsg import Jabbermsg
import os.path, thread, time, Queue

loadlock = thread.allocate_lock()
locked = lockdec(loadlock)

class Plugins(object):

    """ hold all the plugins """

    def __init__(self):
        self.plugs = {}
        self.reloadlock = thread.allocate_lock()
        self.plugdeny = Persist(datadir + os.sep + 'plugdeny')
        if not self.plugdeny.data:
            self.plugdeny.data = []

    def __getitem__(self, item):
        """ return plugin """
        if self.plugs.has_key(item):
            return self.plugs[item]

    def whatperms(self):
        """ return what permissions are possible """
        result = []
        for i in rebefore.whatperms():
            if not i in result:
                result.append(i)
        for i in cmnds.whatperms():
            if not i in result:
                result.append(i)
        for i in reafter.whatperms():
            if not i in result:
                result.append(i)
        result.sort()
        return result

    def exist(self, name):
        """ see if plugin <name> exists """
        if self.plugs.has_key(name):
            return 1 

    def disable(self, name):
        """ prevent plugins <name> to be loaded """
        if name not in self.plugdeny.data:
            self.plugdeny.data.append(name)
            self.plugdeny.save()

    def enable(self, name):
        """ enable plugin <name> """
        if name in self.plugdeny.data:
            self.plugdeny.data.remove(name)
            self.plugdeny.save()

    def plugsizes(self):
        """ call the size() function in all plugins """
        reslist = []
        for i, j in self.plugs.iteritems():
            try:
                reslist.append("%s: %s" % (i, j.size()))
            except AttributeError:
                pass
        return reslist

    def list(self):
        """ list of registered plugins """
        return self.plugs.keys()

    def regplugin(self, mod, name):
        """ register plugin """
        name = name.lower()
        mod = mod.lower()
        modname = mod + '.' + name
        # see if plugin is in deny
        if name in self.plugdeny.data:
            rlog(10, 'plugins', '%s (%s) in deny .. not loading' % \
(name, mod))
            return 0
        # if plugin is already registered unload it
        if self.plugs.has_key(name):
            rlog(1, 'plugins', 'overloading %s plugin with %s version' \
% (name, mod))
            self.unloadnosave(name)
        # import the plugin
        try:
            plug = my_import(modname)
            self.plugs[name] = plug
        except Exception, ex:
            self.unload(name)
            handle_exception()
            return
        # call init function in plugin
        try:
            rlog(0, 'plugins', 'calling %s init()' % name)
            result = plug.init()
            if not result:
                return 
        except AttributeError:
            pass
        except:
            self.unload(name)
            raise
        rlog(0, 'plugins', "%s (%s) registered" % (name, mod))
        return 1

    def regdir(self, dirname, exclude=[]):
        threads = []
        plugs = []
        for plug in plugnames(dirname):
            if plug in exclude:
                continue
            try:
                thr = start_new_thread(self.regplugin, (dirname, plug))
                threads.append(thr)
                plugs.append(plug)
            except:
                handle_exception()
        for i in threads:
            i.join(5)
        return plugs

    def regcore(self):  
        plugs = my_import('gozerplugs.plugs')
        for i in plugs.__all__:
            self.regplugin('gozerplugs.plugs', i)
        if config['dbenable']:
            dbplugs = my_import('gozerplugs.dbplugs')
            for i in dbplugs.__all__:
                self.regplugin('gozerplugs.dbplugs', i)

    def regplugins(self):
        """ register plugins """
        self.regcore()
        if os.path.isdir('myplugs'):
            self.regdir('myplugs')
        # log loaded plugs
        pluglist = self.list()
        pluglist.sort()
        rlog(10, 'plugins', 'loaded %s' % ' .. '.join(pluglist))
        self.overload()

    def overload(self):
        # see if there is a permoverload file and if so use it to overload
        # permissions based on function name
        try:
            overloadfile = open('permoverload','r')
        except IOError:
            return
        try:
            for i in overloadfile:
                i = i.strip()
                splitted = i.split(',')
                try:
                    funcname = splitted[0].strip()
                    perms = []
                    for j in splitted[1:]:
                        perms.append(j.strip())
                except IndexError:
                    rlog(10, 'plugins', "permoverload: can't set perms of %s" \
% i)
                    continue
                if not funcname:
                    rlog(10, 'plugins', "permoverload: no function provided")
                    continue
                if not perms:
                    rlog(10, 'plugins', "permoverload: no permissions \
provided for %s" % funcname)
                    continue
                if self.permoverload(funcname, perms):
                    rlog(9, 'plugins', '%s permission set to %s' % \
(funcname, perms))
                else:
                    rlog(10, 'plugins', 'overload of %s failed' % funcname)
        except:
            handle_exception()
        
    def available(self):
        """ available pluginsregister """
        resultlist = self.plugs.keys()
        resultlist.sort()
        return resultlist

    def saveplug(self, plugname):
        """ call save on items in plugins savelist """
        try:
            self.save_cfgname(plugname)
            savelist = self.plugs[plugname].savelist
            for saveitem in savelist:
                try:
                    saveitem.save()
                except:
                    handle_exception()
        except AttributeError:
            pass
        except KeyError:
            pass

    def save(self):
        """ call registered plugins save """
        for plug in self.plugs.values():
            try:
                savelist = getattr(plug, 'savelist')
            except AttributeError:
                continue
            for item in savelist:
                try:
                    item.save()
                except:
                    handle_exception()

    def save_cfg(self):
        """ call registered plugins configuration save """
        for plug in self.plugs.values():
            try:
                cfg = getattr(plug, 'cfg')
                if isinstance(cfg, PersistConfig):
                    try:
                        cfg.save()
                    except:
                        handle_exception()
            except AttributeError:
                continue

    def save_cfgname(self, name):
        try:
            plug = self.plugs[name]
            cfg = getattr(plug, 'cfg')
            if isinstance(cfg, PersistConfig):
                try:
                    cfg.save()
                except:
                    handle_exception()
        except (AttributeError, KeyError):
            pass
	
    def exit(self):
        """ call registered plugins save """
        self.save()
        #self.save_cfg()
        threadlist = []
        for name, plug in self.plugs.iteritems():
            try:
                shutdown = getattr(plug, 'shutdown')
                thread = start_new_thread(shutdown, ())
                threadlist.append((name, thread))
            except AttributeError:
                continue
        try:
            for name, thread in threadlist:
                thread.join()
                rlog(1, 'plugins', '%s shutdown' % name)
        except:
            handle_exception()
            return    

    def reload(self, mod, name):
        """ reload plugin """
        ok = 0
        # save data in plugin's savelist before we try to reload
        modname = mod + '.' + name
        if mod == 'gozerplugs.dbplugs' and not config['dbenable']:
            rlog(10, 'plugins', 'db not enabled .. not reloading %s' % \
modname) 
            return 
        # unload plugin
        self.unload(name)
        # see if module is already loaded if not load it
        if not self.plugs.has_key(name):
            self.plugs[name] = my_import(modname)
        # reload the thing
        self.plugs[name] = reload(self.plugs[name])
        # call init now the module has been reloaded
        try:
            rlog(0, 'plugins', 'calling %s init()' % modname)
            ok = self.plugs[name].init()
        except (AttributeError, KeyError):
            ok = 1
            pass
        # enable plugin
        self.enable(name)
        self.overload()
        if not ok:
            rlog(10, 'plugins', '%s module init failed' % name)
            return 0
        rlog(10, 'plugins', 'reloaded plugin %s' % modname)
        return 1

    def unload(self, plugname):
        """ unload plugin """
        # call plugins shutdown function if available
        try:
            self.plugs[plugname].shutdown()
            rlog(0, 'plugins', '%s shutdown called' % plugname)
        except (AttributeError, KeyError):
            pass
        self.saveplug(plugname)
        return self.unloadnosave(plugname)

    def unloadnosave(self, plugname):
        """ unload plugin without saving """
        try:
            cmnds.unload(plugname)
            callbacks.unload(plugname)
            jcallbacks.unload(plugname)
            rebefore.unload(plugname)
            reafter.unload(plugname)
            saymonitor.unload(plugname)
            outmonitor.unload(plugname)
            jabbermonitor.unload(plugname)
            if self.plugs.has_key(plugname):
                del self.plugs[plugname]
        except Exception, ex:
            handle_exception()
            return 0
        rlog(1, 'plugins', '%s unloaded' % plugname)
        return 1

    def whereis(self, what):
        """ locate what in plugins """
        return cmnds.whereis(what)

    def permoverload(self, funcname, perms):
        """ overload permission of a function """
        if not rebefore.permoverload(funcname, perms):
            if not cmnds.permoverload(funcname, perms):
                if not reafter.permoverload(funcname, perms):
                    return 0
        return 1

    def woulddispatch(self, bot, ievent):
        (what, command) = self.dispatchtest(bot, ievent)
        if what and command:
            return 1

    @locked
    def dispatchtest(self, bot, ievent, direct=False):
        """ see if ievent would dispatch """
        chan = checkchan(bot, ievent.txt)
        if chan != None:
            (channel, txt) = chan
            ievent.txt = txt.strip()
            ievent.printto = ievent.channel
            ievent.channel = channel
        if ievent.txt.find(' | ') != -1:
            target = ievent.txt.split(' | ')[0]
        elif ievent.txt.find(' && ') != -1:
            target = ievent.txt.split(' && ')[0]
        else:
            target = ievent.txt
        result = []
        # first check for RE before commands dispatcher
        com = rebefore.getcallback(target)
        if com and not target.startswith('!'):
            com.re = True
            result = [rebefore, com]
        else:
            # try commands 
            if ievent.txt.startswith('!'):
                ievent.txt = ievent.txt[1:]
            aliascheck(ievent)
            com = cmnds.getcommand(ievent.txt)
            if com:
                com.re = False
                result = [cmnds, com]
                ievent.txt = ievent.txt.strip()
            else:
                # try RE after commands
                com = reafter.getcallback(target)
                if com:
                    com.re = True
                    result = [reafter, com]
        time.sleep(0.001)
        if result:
            try:
                chanperms = bot.channels[ievent.channel]['perms']
                for i in result[1].perms:
                    if i in chanperms and not ievent.msg:
                        ievent.speed = 1
                        return result
            except (KeyError, TypeError):
                pass
            if direct:
                return result
            if bot.jabber and ievent.jabber:
                if not ievent.groupchat or ievent.jidchange:
                    if users.allowed(ievent.stripped, result[1].perms):
                        return result
            if users.allowed(ievent.userhost, result[1].perms):
                return result
        return (None, None)

    def cmnd(self, bot, ievent, timeout=15):
        ii = Ircevent(ievent)
        q = Queue.Queue()
        ii.queues.append(q)
        self.trydispatch(bot, ii)
        return waitforqueue(q, timeout)

    def trydispatch(self, bot, ievent, direct=False):
        """ try to dispatch ievent """
        # test for ignore
        if shouldignore(ievent.userhost):
            return
        # limiter test
        lastalertmin = getlastalertmin(ievent.userhost)
        lastalerthour = getlastalerthour(ievent.userhost)
        nr = shouldblockhour(ievent.userhost)
        if nr:
            if time.time() - 3600 > lastalerthour:
                ievent.reply('max %s commands per hour' % nr)
                setlastalerthour(ievent.userhost, time.time())
            return 0
        nr = shouldblockmin(ievent.userhost)
        if nr:
            if time.time() - 60 > lastalertmin:
                ievent.reply('max %s commands per minute' % nr)
                setlastalertmin(ievent.userhost, time.time())
            return 0
        # check if channel is provided
        chan = checkchan(bot, ievent.txt)
        if chan != None:
            (channel, txt) = chan
            ievent.txt = txt.strip()
            ievent.printto = ievent.channel
            ievent.channel = channel
        elif ievent.msg:
            ievent.printto = ievent.nick
        else:
            ievent.printto = ievent.channel
        # see if ievent would dispatch
        # what is rebefore, cmnds of reafter, com is the command object
        # check if redispatcher or commands object needs to be used
        (what, com) = self.dispatchtest(bot, ievent, direct)
        if what:
            if com.allowqueue:
                ievent.txt = ievent.txt.replace(' || ', ' | ')
                if ievent.txt.find(' | ') != -1:
                    if ievent.txt[0] == '!':
                        ievent.txt = ievent.txt[1:]
                    else:
                        self.splitpipe(bot, ievent)
                        return
                elif ievent.txt.find(' && ') != -1:
                    self.multiple(bot, ievent)
                    return
            return self.dispatch(what, com, bot, ievent)

    def dispatch(self, what, com, bot, ievent):
        """ dispatch ievent """
        if bot.stopped:
            return 0
        # check for user provided channel
        makeargrest(ievent)
        ievent.usercmnd = 1
        commandhandler.put(ievent.speed, what, com, bot, ievent)
        addievent(ievent.userhost, ievent)
        return 1

    def multiple(self, bot, ievent):
        for i in ievent.txt.split(' && '):
            if not bot.jabber:
                ie = Ircevent()
                ie.copyin(ievent)
            else:
                ie = Jabbermsg(ievent)
                ie.copyin(ievent)
            ie.txt = i.strip()
            self.trydispatch(bot, ie)

    def splitpipe(self, bot, ievent):
        origqueues = ievent.queues
        ievent.queues = []
        events = []
        txt = ievent.txt.replace(' || ', ' ##')
        for i in txt.split(' | '):
            item = i.replace(' ##', ' | ')
            if not bot.jabber:
                ie = Ircevent()
                ie.copyin(ievent)
            else:
                ie = Jabbermsg(ievent)
                ie.copyin(ievent)
            ie.txt = item.strip()
            events.append(ie)
        prevq = None
        for i in events[:-1]:
            q = Queue.Queue()
            i.queues.append(q)
            if prevq:
                i.inqueue = prevq
            prevq = q
        events[-1].inqueue = prevq
        if origqueues:
            events[-1].queues = origqueues
        for i in events:
            if not self.woulddispatch(bot, i):
                ievent.reply("can't execute %s" % str(i.txt))
                return
        for i in events:
            (what, com) = self.dispatchtest(bot, i)
            if what:
                self.dispatch(what, com, bot, i)

    def listreload(self, pluglist):
        """ reload list of plugins """
        failed = []
        for what in pluglist:
            splitted = what[:-3].split(os.sep)
            mod = '.'.join(splitted[:-1])
            plug  = splitted[-1]
            # test for plugdeny
            if plug in self.plugdeny.data:
                rlog(10, 'upgrade', '%s in plugdeny .. not reloading' % plug) 
                continue
            # reload the plugin
            try:
                self.reload(mod, plug)
            except Exception, ex:
                failed.append(what)
        return failed

plugins = Plugins()
