/*
     This file is part of GNUnet
     (C) 2006 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * @file src/common/iterators.c
 * @brief This file contains GTK helper functions related to iterations
 * @author Igor Wronsky
 * @author Christian Grothoff
 */
#include "platform.h"
#include "gnunetgtk_common.h"
#include <GNUnet/gnunet_util_crypto.h>
#include <glib.h>

/**
 * Identical to "gtk_tree_selection_selected_foreach",
 * except that modifications of the underlying model
 * during the iteration are tolerated.
 */
void
GNUNET_GTK_tree_selection_selected_foreach (GtkTreeSelection * selection,
                                            GtkTreeSelectionForeachFunc func,
                                            gpointer data)
{
  unsigned int i;
  unsigned int size;
  GList *selected;
  GtkTreeRowReference **refs;
  GtkTreeModel *model;
  GtkTreePath *path;
  GtkTreeIter iter;

  selected = gtk_tree_selection_get_selected_rows (selection, &model);

  i = g_list_length (selected);
  size = 0;
  refs = NULL;
  GNUNET_array_grow (refs, size, i);
  for (i = 0; i < size; i++)
    refs[i] = gtk_tree_row_reference_new (model,
                                          g_list_nth_data (selected, i));
  g_list_foreach (selected, (GFunc) & gtk_tree_path_free, NULL);
  g_list_free (selected);
  for (i = 0; i < size; i++)
    {
      path = gtk_tree_row_reference_get_path (refs[i]);
      gtk_tree_row_reference_free (refs[i]);
      if (TRUE == gtk_tree_model_get_iter (model, &iter, path))
        func (model, path, &iter, data);
      gtk_tree_path_free (path);
    }
  GNUNET_array_grow (refs, size, 0);
}

typedef struct
{
  GtkTreeRowReference **refs;
  unsigned int pos;
  unsigned int size;
} CollectData;

static gboolean
collectAllRows (GtkTreeModel * model,
                GtkTreePath * path, GtkTreeIter * iter, gpointer cls)
{
  CollectData *cd = cls;

  if (cd->size == cd->pos)
    GNUNET_array_grow (cd->refs, cd->size, cd->size * 2 + 4);
  cd->refs[cd->pos++] = gtk_tree_row_reference_new (model, path);
  return FALSE;
}


/**
 * Identical to "gtk_tree_model_foreach",
 * except that modifications of the underlying model
 * during the iteration are tolerated.
 */
void
GNUNET_GTK_tree_model_foreach (GtkTreeModel * model,
                               GtkTreeSelectionForeachFunc func,
                               gpointer data)
{
  int i;
  GtkTreePath *path;
  GtkTreeIter iter;
  CollectData cd;

  memset (&cd, 0, sizeof (CollectData));
  gtk_tree_model_foreach (model, &collectAllRows, &cd);
  for (i = cd.pos - 1; i >= 0; i--)
    {
      path = gtk_tree_row_reference_get_path (cd.refs[i]);
      gtk_tree_row_reference_free (cd.refs[i]);
      if (TRUE == gtk_tree_model_get_iter (model, &iter, path))
        func (model, path, &iter, data);
      gtk_tree_path_free (path);
    }
  GNUNET_array_grow (cd.refs, cd.size, 0);
}
