
#include <config.h>
#include "esd-audio.h"

#include <glib/gmem.h>
#include <glib/gstrfuncs.h>
#include <pthread.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <ao/ao.h>

#define BUF_SIZE 8192

static ao_device *device = 0;
static gpointer buffer;
static gboolean going = FALSE, prebuffer, paused = FALSE, remove_prebuffer = FALSE;
static gint buffer_size, prebuffer_size, blk_size = 4096;
static gint rd_index = 0, wr_index = 0;
static gint output_time_offset = 0;
static guint64 written = 0, output_bytes = 0;
static gint bps, ebps;
static gint flush;
static gint channels, frequency, latency;
static AFormat format;
static gint input_bps, input_format, input_frequency, input_channels;
static pthread_t buffer_thread;
static void *(*esd_translate)(void *, gint);

static ao_sample_format output_format;

static void 
esdout_init (void)
{
}

static gint 
get_latency(void)
{
	int amount = 0;

	if (output_format.channels == 2)
		amount = (2 * 44100 * (BUF_SIZE + 128)) / output_format.rate;
	else
		amount = (2 * 44100 * (BUF_SIZE + 256)) / output_format.rate;
	amount += BUF_SIZE * 2;

	return amount;
}

static void *
esd_stou8(void *data, gint length)
{
	int len = length;
	unsigned char *dat = (unsigned char *)data;
	while (len-- > 0)
		*dat++ ^= 0x80;
	return data;
}

static void *esd_utos16sw(void *data, gint length)
{
	int len = length;
	short *dat = data;
	while ( len > 0 ) {
		*dat = GUINT16_SWAP_LE_BE( *dat ) ^ 0x8000;
		dat++;
		len-=2;
	}
	return data;
}

static void *esd_utos16(void *data, gint length)
{
	int len = length;
	short *dat = data;
	while ( len > 0 ) {
		*dat ^= 0x8000;
		dat++;
		len-=2;
	}
	return data;
}

static void *esd_16sw(void *data, gint length)
{
	int len = length;
	short *dat = data;
	while ( len > 0 ) {
		*dat = GUINT16_SWAP_LE_BE( *dat );
		dat++;
		len-=2;
	}
	return data;
}

static void 
esdout_setup_format (AFormat fmt, gint rate, gint nch)
{
	gboolean swap_sign = FALSE;

	format = fmt;
	frequency = rate;
	channels = nch;
	switch (fmt)
	{
		case FMT_S8:
			swap_sign = TRUE;
		case FMT_U8:
			output_format.bits = 8;
			break;
		case FMT_U16_LE:
		case FMT_U16_BE:
		case FMT_U16_NE:
			swap_sign = TRUE;
		case FMT_S16_LE:
		case FMT_S16_BE:
		case FMT_S16_NE:
			output_format.bits = 16;
			break;
	}

	esd_translate = (void*(*)())NULL;
	if (output_format.bits == 8) {
		if (swap_sign == TRUE)
			esd_translate = esd_stou8;
	} else {
		if (swap_sign == TRUE) {
			esd_translate = esd_utos16;
		}
	}

	output_format.rate = rate;

	output_format.byte_format = AO_FMT_NATIVE;

	bps = rate * nch;
	if (output_format.bits == 16)
		bps *= 2;
	if(nch == 1)
		output_format.channels = 1;
	else
		output_format.channels = 2;
	
	latency = ((get_latency() * frequency) / 44100);
	if (format != FMT_U8 && format != FMT_S8)
		latency *= 2;
}
	

static void open_device ()
{
	if (device)
		return;
	device = ao_open_live(ao_default_driver_id(), &output_format, 0);
}

static void close_device ()
{
	if (!device)
		return;

	ao_close(device);
	device = 0;
}

gint 
esdout_get_written_time (void)
{
	if (!going)
		return 0;
	return (gint) ((written * 1000) / input_bps);
}

gint 
esdout_get_output_time (void)
{
	guint64 bytes;

	if (!device || !going) {
		return 0;
	}

	bytes = output_bytes;
	if (!paused) {
		bytes -= (bytes < (guint64)latency ? bytes : (guint64)latency);
	}
	
	return output_time_offset + (gint) ((bytes * 1000) / ebps);
}

/* Returns the number of bytes in our buffer still to be sent to esd. */
gint 
esdout_used (void)
{
	if (wr_index >= rd_index) {
		return wr_index - rd_index;
	}
	return buffer_size - (rd_index - wr_index);
}

/* Returns TRUE if we can play a track, i.e. we can connect to esd. */
int
esdout_can_play (void)
{
	ao_sample_format output_fmt;
	output_fmt.bits = 16;
	output_fmt.rate = 44100;
	output_fmt.channels = 2;
	output_fmt.byte_format = AO_FMT_LITTLE;
	
	if (ao_open_live(ao_default_driver_id(), &output_fmt, 0))
		return TRUE;
	return FALSE;
}

/* Returns TRUE if we are playing a track, i.e. the thread is still streaming
   data to esd. */
int
esdout_playing (void)
{
	/* If our thread is not streaming data to esd, return FALSE. */
	if (!going)
		return FALSE;

	/* If we've run out of data to send, we assume the track is done. */
	if (esdout_used () == 0)
		return FALSE;

	return TRUE;
}

gint 
esdout_free (void)
{
	if (remove_prebuffer && prebuffer) {
		prebuffer = FALSE;
		remove_prebuffer = FALSE;
	}
	
	if (prebuffer) {
		remove_prebuffer = TRUE;
	}

	if (rd_index > wr_index) {
		return (rd_index - wr_index) - 1;
	}
	
	return (buffer_size - (wr_index - rd_index)) - 1;
}

static void 
esdout_write_audio (gpointer data, gint length)
{
	AFormat new_format;
	gint new_frequency,new_channels;
	
	new_format = input_format;
	new_frequency = input_frequency;
	new_channels = input_channels;
		
	if (new_format != format || new_frequency != frequency || new_channels != channels) {
		output_time_offset += (gint) ((output_bytes * 1000) / ebps);
		output_bytes = 0;
		esdout_setup_format(new_format, new_frequency, new_channels);
		frequency = new_frequency;
		channels = new_channels;
		close_device();
		esdout_set_audio_params();
	}
	
        if (esd_translate) {
                ao_play(device,esd_translate(data,length),length);
	} else {
		ao_play(device,data,length);
	}
	output_bytes += length;
}


void 
esdout_write (gpointer ptr, gint length)
{
	gint cnt, off = 0;

	remove_prebuffer = FALSE;

	written += length;
	while (length > 0) {
		cnt = MIN(length, buffer_size - wr_index);
		memcpy((gchar *)buffer + wr_index, (gchar *)ptr + off, cnt);
		wr_index = (wr_index + cnt) % buffer_size;
		length -= cnt;
		off += cnt;

	}
}

void 
esdout_close (void)
{
	if (!going) {
		return;
	}
	
	wr_index = 0;
	rd_index = 0;
	going = 0;
	pthread_join (buffer_thread, NULL);
}

void 
esdout_flush (gint time)
{
	flush = time;
	while (flush != -1) {
		usleep(10000);
	}
}

void 
esdout_pause (gboolean p)
{
	paused = p;
}

static void *
esdout_loop (void *arg)
{
	gint length, cnt;
	
	while (going) {
		if (esdout_used () > prebuffer_size) {
			prebuffer = FALSE;
		}
		
		if (esdout_used () > 0 && !paused && !prebuffer) {
			open_device();
			if (device) {
				length = MIN (blk_size, esdout_used ());
				while (length > 0) {
					cnt = MIN(length,buffer_size-rd_index);
					esdout_write_audio ((gchar *)buffer + rd_index, cnt);
					rd_index=(rd_index+cnt)%buffer_size;
					length-=cnt;				
				}
			} else
				going = 0;
		} else {
			if (paused)
				close_device();
			usleep (10000);
		}

		if (flush != -1) {
			output_time_offset = flush;
			written = (guint64)(flush / 10) * (guint64)(input_bps / 100);
			rd_index = wr_index = output_bytes = 0;
			flush = -1;
			prebuffer = TRUE;
		}
	}

	close_device();
	g_free (buffer);
	pthread_exit (NULL);
	return NULL;  /* make gcc happy */
}

void 
esdout_set_audio_params(void)
{
	/* frequency = GUINT16_SWAP_LE_BE( frequency ); */
	open_device();
	ebps = frequency * channels;
	if (format == FMT_U16_BE || format == FMT_U16_LE || format == FMT_S16_BE || format == FMT_S16_LE || format == FMT_S16_NE || format == FMT_U16_NE)
		ebps *= 2;
}

gint
esdout_open (AFormat fmt, gint rate, gint nch)
{	
	esdout_init ();
	
	esdout_setup_format (fmt,rate,nch);
	
	input_format = format;
	input_channels = channels;
	input_frequency = frequency;
	input_bps = bps;

	buffer_size = 3 * input_bps;
	if (buffer_size < 8192)
		buffer_size = 8192;
	prebuffer_size = buffer_size - 4096;

	buffer = g_malloc0(buffer_size);
	
	flush = -1;
	prebuffer = 1;
	wr_index = rd_index = output_time_offset = written = output_bytes = 0;
	paused = FALSE;
	remove_prebuffer = FALSE;

	esdout_set_audio_params ();
	if (!device) {
		g_free(buffer);
		return 0;
	}
	going = 1;

	pthread_create (&buffer_thread, NULL, esdout_loop, NULL);
	
	return 1;
}
