/* GnomeScanUI - Widgets for scan dialogs
 *
 * gnomescanliststore.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnomescanui.h"
#include "gnomescanui-intl.h"

#define GNOME_SCAN_LIST_STORE_ERROR			(g_type_qname (GNOME_TYPE_SCAN_LIST_STORE))
#define	GNOME_SCAN_LIST_STORE_PARENT_CLASS(klass)	(GTK_WIDGET_CLASS (g_type_class_peek_parent ((klass))))
#define GET_PRIVATE(obj)			(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCAN_LIST_STORE, GnomeScanListStorePrivate))

typedef struct _GnomeScanListStorePrivate		GnomeScanListStorePrivate;

struct _GnomeScanListStorePrivate {
  gboolean		dispose_has_run;

  /* PROPERTIES */
};

enum {
  COLUMN_ICON,
  COLUMN_VENDOR,
  COLUMN_PRODUCT,
  /* state ? */
  COLUMN_SCANNER,
  N_COLUMNS
};


/********************************
 * 	      CALLBACKS		*
 ********************************/

void				gsls_probe_done					(GnomeScanContext *context,
										 GnomeScanListStore *gsls);

void				gsls_tree_probe_done				(GnomeScanContext *context,
										 GtkTreeSelection *tree_selection);

void				gsls_tree_changed				(GtkTreeSelection *tree_selection,
										 GnomeScanContext *context);


/********************************
 * 	       OTHERS		*
 ********************************/

void				gsls_fill_func					(GnomeScanner *scanner,
										 GtkListStore *store);


/********************************
 * 	      GOBJECT		*
 ********************************/

void				gnome_scan_list_store_set_property 		(GObject *obj,
										 guint property_id,
										 const GValue *value,
										 GParamSpec *pspec);

void				gnome_scan_list_store_get_property 		(GObject *obj,
										 guint property_id,
										 GValue *value,
										 GParamSpec *pspec);

void				gnome_scan_list_store_finalize			(GObject *obj);

void				gnome_scan_list_store_dispose			(GObject *obj);


enum {
  PROP_0,
  PROP_CONTEXT
};

enum {
  N_SIGNALS
};

static guint signals[N_SIGNALS];


G_DEFINE_TYPE (GnomeScanListStore, gnome_scan_list_store, GTK_TYPE_LIST_STORE);


void
gnome_scan_list_store_class_init (GnomeScanListStoreClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property	= gnome_scan_list_store_set_property;
  gobject_class->get_property	= gnome_scan_list_store_get_property;
  gobject_class->dispose 	= gnome_scan_list_store_dispose;
  /*   gobject_class->finalize 	= gnome_scan_list_store_finalize; */

  g_type_class_add_private (gobject_class,
			    sizeof (GnomeScanListStorePrivate));

  /* Properties */
  g_object_class_install_property (gobject_class,
				   PROP_CONTEXT,
				   g_param_spec_object ("context",
							"Context",
							"The GnomeScanContext the liststore is connected to.",
							GNOME_TYPE_SCAN_CONTEXT,
							G_PARAM_READWRITE));

  /* Signals */
}

void
gnome_scan_list_store_init (GnomeScanListStore *gsls)
{
  GnomeScanListStorePrivate *priv = GET_PRIVATE (gsls);

  gsls->context = NULL;

  priv->dispose_has_run = FALSE;
}

void
gnome_scan_list_store_dispose (GObject *obj)
{
  GnomeScanListStore *gsls = GNOME_SCAN_LIST_STORE (obj);
  GnomeScanListStorePrivate *priv = GET_PRIVATE (gsls);
  GnomeScanListStoreClass *b_klass = GNOME_SCAN_LIST_STORE_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref devices */
  g_object_unref (gsls->context);
  priv->dispose_has_run = TRUE;

  /* chain */
  /*   GNOME_SCAN_LIST_STORE_PARENT_CLASS (klass)->dispose (obj); */
}

void
gnome_scan_list_store_finalize (GObject *obj)
{
  GnomeScanListStore *gsls = GNOME_SCAN_LIST_STORE (obj);
  GnomeScanListStoreClass *klass = GNOME_SCAN_LIST_STORE_GET_CLASS (gsls);

  /*   GNOME_SCAN_LIST_STORE_PARENT_CLASS (klass)->finalize (obj); */
}

void
gnome_scan_list_store_set_property (GObject *obj,
				    guint property_id,
				    const GValue *value,
				    GParamSpec *pspec)
{
  GnomeScanListStore *gsls = GNOME_SCAN_LIST_STORE (obj);
  GnomeScanListStorePrivate *priv = GET_PRIVATE (gsls);

  switch (property_id) {
  case PROP_CONTEXT:
    gsls->context = GNOME_SCAN_CONTEXT (g_value_dup_object (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}

void
gnome_scan_list_store_get_property (GObject *obj,
				    guint property_id,
				    GValue *value,
				    GParamSpec *pspec)
{
  GnomeScanListStore *gsls = GNOME_SCAN_LIST_STORE (obj);
  GnomeScanListStorePrivate *priv = GET_PRIVATE (gsls);

  switch (property_id) {
  case PROP_CONTEXT:
    g_value_set_object (value, gsls->context);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}




/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_list_store_new:
 * @context: a #GnomeScanContext
 * 
 * Create a new #GnomeScanListStore connected to @context.
 * 
 * Return value: a new #GnomeScanListStore
 **/
GtkListStore*
gnome_scan_list_store_new (GnomeScanContext *context)
{
  GnomeScanListStore *gsls;
  GnomeScanListStorePrivate *priv;
  GtkListStore *list_store;
  GType types[N_COLUMNS] = { GDK_TYPE_PIXBUF,
			     G_TYPE_STRING,
			     G_TYPE_STRING,
			     G_TYPE_POINTER };

  list_store = GTK_LIST_STORE (g_object_new (GNOME_TYPE_SCAN_LIST_STORE,
					     "context", context,
					     NULL));

  gsls = GNOME_SCAN_LIST_STORE (list_store);
  priv = GET_PRIVATE (gsls);


  gtk_list_store_set_column_types (list_store, 
				   N_COLUMNS,
				   types);

  g_signal_connect (context,
		    "probe-done",
		    (GCallback) gsls_probe_done,
		    gsls);

  return list_store;
}

/**
 * gnome_scan_list_store_new_tree_view:
 * @context: 
 * 
 * 
 * 
 * Returns: 
 **/
GtkWidget*
gnome_scan_list_store_new_tree_view (GnomeScanContext *context)
{
  GtkWidget *scrolled, *tree_view;
  GtkListStore *list_store;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  GtkTreeSelection *tree_selection;

  list_store = gnome_scan_list_store_new (context);
  tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list_store));
  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));
  g_signal_connect (tree_selection,
		    "changed",
		    (GCallback) gsls_tree_changed,
		    context);

  g_signal_connect (context,
		    "probe-done",
		    (GCallback) gsls_tree_probe_done,
		    tree_selection);

  /* icon */
  renderer = gtk_cell_renderer_pixbuf_new ();
  gtk_cell_renderer_set_fixed_size (GTK_CELL_RENDERER (renderer),
				    32, 32);

  column = gtk_tree_view_column_new_with_attributes (NULL,
						     renderer,
						     "pixbuf", COLUMN_ICON,
						     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       column);

  /* vendor */
  renderer = gtk_cell_renderer_text_new ();
  /* translator: vendor column in device selector */
  column = gtk_tree_view_column_new_with_attributes (_("Vendor"),
						     renderer,
						     "text", COLUMN_VENDOR,
						     NULL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       column);

  /* product */
  renderer = gtk_cell_renderer_text_new ();
  /* translator: product column in device selector */
  column = gtk_tree_view_column_new_with_attributes (_("Product"),
						     renderer,
						     "text", COLUMN_PRODUCT,
						     NULL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       column);

  scrolled = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled),
				       GTK_SHADOW_IN);

  gtk_container_add (GTK_CONTAINER (scrolled),
		     tree_view);

  return scrolled; 
}

GtkWidget*
gnome_scan_list_store_new_combo_box (GnomeScanContext *context)
{
  GtkCellLayout *cell_layout;
  GtkCellRenderer *cell_renderer;

  /* layout column ICON */
/*   renderer = gtk_cell_renderer_pixbuf_new (); */

/*   gtk_cell_renderer_set_fixed_size (GTK_CELL_RENDERER (renderer), */
/* 				    32, 32); */

/*   gtk_cell_layout_pack_start (cell_layout, */
/* 			      renderer, */
/* 			      FALSE); */

/*   gtk_cell_layout_add_attribute (cell_layout, */
/* 				 renderer, */
/* 				 "pixbuf", */
/* 				 COLUMN_ICON); */

  /* layout column LABEL */
/*   renderer = gtk_cell_renderer_text_new (); */

/*   gtk_cell_layout_pack_start (cell_layout, */
/* 			      renderer, */
/* 			      TRUE); */

/*   gtk_cell_layout_add_attribute (cell_layout, */
/* 				 renderer, */
/* 				 "markup", */
/* 				 COLUMN_LABEL); */

  return NULL;
}

/********************************
 * 	      CALLBACKS		*
 ********************************/

void
gsls_probe_done	(GnomeScanContext *context,
		 GnomeScanListStore *gsls)
{
  GSList *devices;

  devices = gnome_scan_context_get_scanners (gsls->context);

  if (!g_slist_length (devices)) {
    return;
  }

  gtk_list_store_clear (GTK_LIST_STORE (gsls));

  /* Fill the store */
  /* TODO: handle store refill */
  g_slist_foreach (devices,
		   (GFunc) gsls_fill_func,
		   GTK_LIST_STORE (gsls));

  
}

void
gsls_tree_probe_done (GnomeScanContext *context,
		      GtkTreeSelection *tree_selection)
{
  GtkTreePath *path;
  path = gtk_tree_path_new_first ();
  gtk_tree_selection_select_path (tree_selection, path);
}


void
gsls_tree_changed (GtkTreeSelection *tree_selection,
		   GnomeScanContext *context)
{
  GtkTreeIter iter;
  GtkTreeModel *model;
  GnomeScanner *scanner;

  if (gtk_tree_selection_get_selected (tree_selection, &model, &iter)) {

    gtk_tree_model_get (GTK_TREE_MODEL (model), &iter,
			COLUMN_SCANNER, &scanner,
			-1);

    gnome_scan_context_select_scanner (context, scanner);
  }
}

/********************************
 * 	       OTHERS		*
 ********************************/


void
gsls_fill_func (GnomeScanner *scanner,
		GtkListStore *store)
{
  GtkTreeIter iter;
  GdkPixbuf *pixbuf;
  GError *error = NULL;
  gchar *icon_name;
  GnomeScannerType type = gnome_scanner_get_scanner_type (scanner);

  static const gchar *prefix = "input-scanner";

  if (type != GNOME_SCANNER_UNKNOWN) {
    icon_name = g_strconcat (prefix, "-", gs_enum_get_nick (GNOME_TYPE_SCANNER_TYPE, type), NULL);
  }
  else {
    icon_name = g_strdup (prefix);
  }

  /* Load ICON */
  pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
				     icon_name,
				     32,
				     0,
				     &error);

  gtk_list_store_append (store, &iter);
  gtk_list_store_set (store, &iter,
		      COLUMN_ICON, pixbuf,
		      COLUMN_VENDOR, gnome_scanner_get_vendor (scanner),
		      COLUMN_PRODUCT, gnome_scanner_get_product (scanner),
		      COLUMN_SCANNER, scanner,
		      -1);

}
