/* GnomeScanUI - Widgets for scan dialogs
 *
 * gnomescandialog.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gnomescanui.h>
#include "gnomescanui-intl.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#define GET_PRIVATE(obj)			(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCAN_DIALOG, GnomeScanDialogPrivate))
#define	GNOME_SCAN_DIALOG_PARENT_CLASS(klass)	(GTK_DIALOG_CLASS (g_type_class_peek_parent ((klass))))
#define GNOME_SCAN_DIALOG_ERROR			(g_type_qname (GNOME_TYPE_SCAN_DIALOG))

typedef struct _GnomeScanDialogPrivate		GnomeScanDialogPrivate;

struct _GnomeScanDialogPrivate {
  gboolean		dispose_has_run;
  GtkWidget		*acquisitiondialog;
  GtkWidget		*messagedialog;
  GtkWidget		*notebook;
  GtkWidget		*preview_label;
  GtkWidget		*preview_page;
  GtkWidget		*frontbox;
};

enum {
  PROP_0,
  PROP_CONTEXT
};

enum {
  TAB_GENERAL,
  TAB_PREVIEW,
  TAB_ADVANCED,
  N_TABS
};

void				gsd_probe_done					(GnomeScanContext *context,
										 GnomeScanDialog *dialog);

void				gsd_ok						(GtkButton *button,
										 GnomeScanDialog *dialog);

void				gsd_scan_error					(GnomeScanContext *context,
										 GError *error,
										 GtkWidget *dialog);

void				gsd_scanner_selected				(GnomeScanContext *context,
										 GnomeScanner *scanner,
										 GtkWidget *dialog);

void				gsd_acquisition_started				(GnomeScanContext *context,
										 gint size,
										 GtkWidget *dialog);

void				gsd_acquisition_terminated			(GnomeScanContext *context,
										 GtkWidget *dialog);

void				gsd_about					(GtkButton *button,
										 GtkWindow *dialog);

void				gsd_close					(GtkWidget *widget,
										 GnomeScanDialog *dialog);

void				gsd_context_changed				(GnomeScanContext *context,
										 gchar *option,
										 GnomeScanDialog *dialog);


GtkWidget*			gsd_wrap_in_frame				(const gchar* label,
										 GtkWidget *child);


/********************************
 * 	      GOBJECT		*
 ********************************/

void				gnome_scan_dialog_dispose			(GObject *obj);

void				gnome_scan_dialog_set_property 			(GObject *obj,
										 guint property_id,
										 const GValue *value,
										 GParamSpec *pspec);

void				gnome_scan_dialog_get_property 			(GObject *obj,
										 guint property_id,
										 GValue *value,
										 GParamSpec *pspec);

G_DEFINE_TYPE (GnomeScanDialog, gnome_scan_dialog, GTK_TYPE_DIALOG);

void
gnome_scan_dialog_init (GnomeScanDialog *dialog)
{
  GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);

  dialog->context 		= NULL;

  priv->dispose_has_run 	= FALSE;
  priv->acquisitiondialog	= NULL;
  priv->messagedialog		= NULL;
  priv->frontbox		= NULL;
  priv->notebook		= NULL;
  priv->preview_page		= NULL;
  priv->preview_label		= NULL;
}

void
gnome_scan_dialog_dispose (GObject *obj)
{
  GnomeScanDialog *dialog = GNOME_SCAN_DIALOG (obj);
  GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);
  GnomeScanDialogClass *b_klass = GNOME_SCAN_DIALOG_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref context */
  g_object_unref (dialog->context);
  g_object_unref (priv->preview_label);
  g_object_unref (priv->preview_page);
  priv->dispose_has_run = TRUE;

  /* chain */
  /*   GNOME_SCAN_DIALOG_PARENT_CLASS (b_klass)->dispose (obj); */
}

void
gnome_scan_dialog_class_init (GnomeScanDialogClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property	= gnome_scan_dialog_set_property;
  gobject_class->get_property	= gnome_scan_dialog_get_property;
  gobject_class->dispose 	= gnome_scan_dialog_dispose;

  g_type_class_add_private (gobject_class, sizeof (GnomeScanDialogPrivate));

  /* PROPERTIES */

  g_object_class_install_property (gobject_class,
				   PROP_CONTEXT,
				   g_param_spec_object ("context",
							"Context",
							"A GnomeScanContext the widget is connected to.",
							GNOME_TYPE_SCAN_CONTEXT,
							G_PARAM_READWRITE));

}

void
gnome_scan_dialog_set_property (GObject *obj,
				guint property_id,
				const GValue *value,
				GParamSpec *pspec)
{
  GnomeScanDialog *dialog = GNOME_SCAN_DIALOG (obj);

  switch (property_id) {
  case PROP_CONTEXT:
    dialog->context = GNOME_SCAN_CONTEXT (g_value_dup_object (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id,pspec);
    break;
  }
}

void
gnome_scan_dialog_get_property (GObject *obj,
				guint property_id,
				GValue *value,
				GParamSpec *pspec)
{
  GnomeScanDialog *dialog = GNOME_SCAN_DIALOG (obj);

  switch (property_id) {
  case PROP_CONTEXT:
    g_value_set_object (value, dialog->context);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,property_id,pspec);
    break;
  }
}



/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_dialog_new:
 * @context: a #GnomeScanContext
 * @parent: a #GtkWidget or NULL
 * 
 * Create a new #GnomeScanDialog connnected to @context. If @parent is
 * not NULL, then the new dialog will be transient for @parent. @ui is
 * the widget tree containing configuration widget showed in the body
 * of the #GnomeScanDialog.
 * 
 * Return value: a new #GnomeScanDialog
 */
GtkWidget*
gnome_scan_dialog_new (GnomeScanContext *context,
		       GtkWindow *parent)
{
  GtkWidget *dialog, *button, *box, *hbox, *label, *content, *selector, *area, *widget, *frame;
  GnomeScanDialogPrivate *priv;
  gchar *string;

  dialog = GTK_WIDGET (g_object_new (GNOME_TYPE_SCAN_DIALOG,
				     "context", context,
				     NULL));

  priv = GET_PRIVATE (dialog);

  /* customize widget */
  gtk_window_set_icon_name (GTK_WINDOW (dialog), PACKAGE);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Scan"));
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 12);

  if (GTK_IS_WINDOW (parent)) {
    gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);
    gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ON_PARENT);
  }
  else {
    gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  }


  priv->notebook = gtk_notebook_new ();
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
		      priv->notebook,
		      TRUE,
		      TRUE,
		      0);

  /* PAGE 1 : GENERAL */
  /* translator: General tab label */
  label = gtk_label_new (_("General"));
  priv->frontbox = content = gtk_vbox_new (FALSE, 6);
  gtk_container_set_border_width (GTK_CONTAINER (content),
				  12);
  gtk_notebook_insert_page (GTK_NOTEBOOK (priv->notebook),
			    content,
			    label,
			    TAB_GENERAL);


  /* selector */
  selector = gnome_scan_list_store_new_tree_view (context);
  gtk_box_pack_start (GTK_BOX (content),
		      selector,
		      TRUE,
		      TRUE,
		      0);

  /* basic options */
  box = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (content),
		      box,
		      FALSE,
		      TRUE,
		      0);

  /* source */
  /* translator: Source field label */
  frame = gsd_wrap_in_frame (_("Source"),
			     gnome_scan_source_selector_new (context));
  gtk_box_pack_start (GTK_BOX (box),
		      frame,
		      FALSE,
		      TRUE,
		      0);

  /* format */
  /* translator: Format field label */
  frame = gsd_wrap_in_frame (_("Format"),
			     gnome_scan_area_selector_new (context));
  gtk_box_pack_start (GTK_BOX (box),
		      frame,
		      FALSE,
		      TRUE,
		      0);


  /* PAGE 2 : PREVIEW */
  priv->preview_page = gtk_hbox_new (FALSE, 6);
  gtk_container_set_border_width (GTK_CONTAINER (priv->preview_page),
				  12);
  /* translator: Preview tab label */
  priv->preview_label = gtk_label_new (_("Preview"));

  gtk_notebook_insert_page (GTK_NOTEBOOK (priv->notebook),
			    priv->preview_page,
			    priv->preview_label,
			    TAB_PREVIEW);

  g_object_ref (priv->preview_page);
  g_object_ref (priv->preview_label);

  area = gnome_scan_preview_area_new (context);
  gtk_box_pack_start (GTK_BOX (priv->preview_page),
		      area,
		      TRUE,
		      TRUE,
		      0);

  box = gtk_vbutton_box_new ();
  gtk_button_box_set_layout (GTK_BUTTON_BOX (box),
			     GTK_BUTTONBOX_START);

  gtk_box_pack_start (GTK_BOX (priv->preview_page),
		      box,
		      FALSE,
		      TRUE,
		      0);

  gtk_box_pack_start (GTK_BOX (box),
		      gnome_scan_preview_area_button_refresh (GNOME_SCAN_PREVIEW_AREA (area)),
		      FALSE,
		      TRUE,
		      0);

  gtk_box_pack_start (GTK_BOX (box),
		      gnome_scan_preview_area_button_rotate_cw (GNOME_SCAN_PREVIEW_AREA (area)),
		      FALSE,
		      TRUE,
		      0);

  gtk_box_pack_start (GTK_BOX (box),
		      gnome_scan_preview_area_button_rotate_ccw (GNOME_SCAN_PREVIEW_AREA (area)),
		      FALSE,
		      TRUE,
		      0);

  /* TODO : max size, zoom in, zoom out, reset zoom */

  /* PAGE 3 : ADVANCED */
  content = gtk_hbox_new (FALSE, 6);
  gtk_container_set_border_width (GTK_CONTAINER (content),
				  12);
  /* translator: Advanced tab label */
  label = gtk_label_new (_("Advanced"));
  gtk_notebook_insert_page (GTK_NOTEBOOK (priv->notebook),
			    content,
			    label,
			    TAB_ADVANCED);

  widget = gnome_scan_resolution_selector_new (context);
  gtk_box_pack_start (GTK_BOX (content),
		      widget, TRUE, TRUE, 0);

  /* BUTTONS */

  /* close */
  button = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
  gtk_box_pack_end (GTK_BOX (GTK_DIALOG (dialog)->action_area),
		    button,
		    FALSE, FALSE, 0);

  g_signal_connect (GTK_BUTTON (button),
		    "clicked",
		    G_CALLBACK (gsd_close),
		    dialog);

  /* scan */
  button = gtk_button_new_from_stock (GS_STOCK_SCAN);
  gtk_box_pack_end (GTK_BOX (GTK_DIALOG (dialog)->action_area),
		    button,
		    FALSE, FALSE, 0);

  g_signal_connect (GTK_BUTTON (button),
		    "clicked",
		    G_CALLBACK (gsd_ok),
		    dialog);


  /* Acquisition dialog */
  priv->acquisitiondialog = gnome_scan_acquisition_dialog_new (context,
							       GTK_WINDOW (dialog));

  /* Message Dialog */
  priv->messagedialog = gtk_message_dialog_new_with_markup (GTK_WINDOW (dialog),
							    GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
							    GTK_MESSAGE_INFO,
							    GTK_BUTTONS_NONE,
							    "<big><b>%s</b></big>", _("Probing devices"));
  gtk_window_set_icon_name (GTK_WINDOW (priv->messagedialog), PACKAGE);

  gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (priv->messagedialog),
					    _("The program is probing devices. Please be patient ..."));

  /* Signals */
  g_signal_connect (context,
		    "changed",
		    (GCallback) gsd_context_changed,
		    dialog);

  g_signal_connect (context,
		    "error",
		    (GCallback) gsd_scan_error,
		    dialog);

  g_signal_connect (context,
		    "probe-done",
		    (GCallback) gsd_probe_done,
		    dialog);

  g_signal_connect (context,
		    "preview-started",
		    (GCallback) gsd_acquisition_started,
		    dialog);

  g_signal_connect (context,
		    "preview-terminated",
		    (GCallback) gsd_acquisition_terminated,
		    dialog);

  g_signal_connect (context,
		    "acquisition-started",
		    (GCallback) gsd_acquisition_started,
		    dialog);

  g_signal_connect (context,
		    "acquisition-terminated",
		    (GCallback) gsd_acquisition_terminated,
		    dialog);

  g_signal_connect (GTK_WIDGET (dialog),
		    "delete-event",
		    G_CALLBACK (gsd_close),
		    dialog);

  return (dialog);
}

/**
 * gnome_scan_dialog_add_front_widget:
 * @dialog:	a #GnomeScanDialog
 * @title: 	the title of the widget frame
 * @widget: 	the widget itself
 * 
 * Add a widget to the first page of the dialog, just below selector
 * and basic options. This is very useful if you want to avoid user to
 * switch to a custom tab that contain only one widget. However, don't
 * abuse of this function or the dialog will endup taller than screen
 * height.
 **/
void
gnome_scan_dialog_add_front_widget (GnomeScanDialog *dialog,
				    gchar *title,
				    GtkWidget *widget)
{
  g_return_if_fail (GNOME_IS_SCAN_DIALOG (dialog));
  g_return_if_fail (GTK_IS_WIDGET (widget));

  gtk_box_pack_start (GTK_BOX (GET_PRIVATE (dialog)->frontbox),
		      gsd_wrap_in_frame (title, widget),
		      FALSE,
		      TRUE,
		      0);
}

/**
 * gnome_scan_dialog_run:
 * @dialog: a #GnomeScanDialog
 * 
 * Run the dialog waiting for the use to configure the scan and
 * trigger acquisition.
 */
void
gnome_scan_dialog_run (GnomeScanDialog *dialog)
{
  GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);

  gtk_widget_set_sensitive (GTK_WIDGET (dialog), FALSE);

  /*   gtk_window_set_transient_for (GTK_WINDOW (messagedialog), */
  /* 				GTK_WINDOW (dialog)); */

  gtk_widget_show_all (GTK_WIDGET (dialog));
  gtk_widget_show (priv->messagedialog);

  while (gtk_events_pending ()) 
    gtk_main_iteration ();

  gnome_scan_context_probe_scanners (dialog->context);
}

/* CALLBACKS */
void
gsd_close (GtkWidget *widget,
	   GnomeScanDialog *dialog)
{
  gtk_widget_hide (GTK_WIDGET (dialog));
  gtk_main_quit ();
}

/* Enable UI after probe */
void
gsd_probe_done (GnomeScanContext *context,
		GnomeScanDialog *dialog)
{
  GtkWidget *mdialog;

  gtk_widget_set_sensitive (GTK_WIDGET (dialog), TRUE);
  gtk_widget_hide (GET_PRIVATE (dialog)->messagedialog);

  if (!g_slist_length (gnome_scan_context_get_scanners (context))) {
    mdialog = gtk_message_dialog_new (GTK_WINDOW (dialog),
				      GTK_DIALOG_MODAL,
				      GTK_MESSAGE_ERROR,
				      GTK_BUTTONS_CLOSE,
				      _("No devices found"));

    gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (mdialog),
					      _("Please ensure your device is properly plugged, on and configured."));

    gtk_dialog_run (GTK_DIALOG (mdialog));
    gtk_widget_destroy (mdialog);
  }
  else {
    gtk_main ();
  }
}

/* Apply the user choice */
void
gsd_ok (GtkButton *button,
	GnomeScanDialog *dialog)
{
  gnome_scan_context_start_acquisition (dialog->context);
  /*   gsd_close (GTK_WIDGET (button), dialog); */
}

void
gsd_scan_error (GnomeScanContext *context,
		GError *error,
		GtkWidget *dialog)
{
  GtkWidget *md;

  md = gtk_message_dialog_new (GTK_WINDOW (dialog),
			       GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
			       GTK_MESSAGE_ERROR,
			       GTK_BUTTONS_OK,
			       _("An error occured while accessing device !"));
  gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (md),
					    _("The scan backend returned the following error: %s"), error->message);

  gtk_dialog_run (GTK_DIALOG (md));
  gtk_widget_destroy (md);
}

void
gsd_acquisition_started (GnomeScanContext *context,
			 gint size,
			 GtkWidget *dialog)
{
  gtk_widget_set_sensitive (dialog, FALSE);
}

void
gsd_acquisition_terminated (GnomeScanContext *context,
			    GtkWidget *dialog)
{
  gtk_widget_set_sensitive (dialog, TRUE);
}

void
gsd_context_changed (GnomeScanContext *context,
		     gchar *option,
		     GnomeScanDialog *dialog)
{
  GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);

  if (g_ascii_strcasecmp (option, "source") == 0) {
    if (gnome_scan_context_get_source (context) == GNOME_SCANNER_SOURCE_FLATBED) {
      if (gtk_notebook_page_num (GTK_NOTEBOOK (priv->notebook),
				 priv->preview_page) == -1) {
	gtk_notebook_insert_page (GTK_NOTEBOOK (priv->notebook),
				  priv->preview_page,
				  priv->preview_label,
				  TAB_PREVIEW);
      }
    }
    else {
      gtk_notebook_remove_page (GTK_NOTEBOOK (priv->notebook),
				TAB_PREVIEW);
    }
  }
}

/*
 * wrap_in_frame function from gtk+/gtk/gtkprintunixdialog.c.
 */
GtkWidget*
gsd_wrap_in_frame (const gchar* label,
		   GtkWidget *child)
{
  GtkWidget *frame, *alignment, *label_widget;
  gchar *bold_text;

  label_widget = gtk_label_new ("");
  gtk_misc_set_alignment (GTK_MISC (label_widget), 0.0, 0.5);
  gtk_widget_show (label_widget);
  
  bold_text = g_markup_printf_escaped ("<b>%s</b>", label);
  gtk_label_set_markup (GTK_LABEL (label_widget), bold_text);
  g_free (bold_text);
  
  frame = gtk_vbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (frame), label_widget, FALSE, FALSE, 0);
  
  alignment = gtk_alignment_new (0.0, 0.0, 1.0, 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (alignment),
			     0, 0, 12, 0);
  gtk_box_pack_start (GTK_BOX (frame), alignment, FALSE, FALSE, 0);

  gtk_container_add (GTK_CONTAINER (alignment), child);

  gtk_widget_show (frame);
  gtk_widget_show (alignment);
  
  return frame;
}

