/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2004-2006 Nokia Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus-glib-lowlevel.h>

#include "om-dbus.h"
#include "om-utils.h"

#define d(x) x

/* Hcid communication. Talks to hcid to get device listings.
 *
 * This API can be accessed from any thread, since it creates it's own context
 * and uses that for the dbus connection.
 */

typedef struct {
	DBusConnection *dbus_conn;
	GMainContext   *context;
	GMainLoop      *loop;
} Connection;

typedef gboolean (* ForeachAdapterFunc) (Connection *conn,
					 const char *adapter,
					 gpointer user_data);

static gboolean is_obex_device (Connection *conn,
				const char *dev,
				const char *name);

static Connection *
get_system_bus_connection (void)
{
	DBusConnection *dbus_conn;
	Connection     *conn;
	DBusError       error;

	/* NOTE: We are using dbus_bus_get_private here, for the reason that
	 * need to get our own private dbus connection to avoid threading
	 * problems with other libraries or applications that use this module
	 * and dbus (the vfs daemon in particular).
	 */
	dbus_error_init (&error);
	dbus_conn = dbus_bus_get_private (DBUS_BUS_SYSTEM, &error);

	if (!dbus_conn) {
		g_printerr ("Failed to connect to the D-BUS daemon: %s", error.message);

		dbus_error_free (&error);
		return NULL;
	}

	conn = g_new0 (Connection, 1);

	conn->context = g_main_context_new ();
	conn->loop = g_main_loop_new (conn->context, FALSE);

	conn->dbus_conn = dbus_conn;

	dbus_connection_setup_with_g_main (dbus_conn, conn->context);

	return conn;
}

static void
connection_free (Connection *conn)
{
	dbus_connection_close (conn->dbus_conn);
	dbus_connection_unref (conn->dbus_conn);

	g_main_loop_unref (conn->loop);
	g_main_context_unref (conn->context);

	g_free (conn);
}

static gboolean
foreach_adapter (Connection *conn, ForeachAdapterFunc func, gpointer user_data)
{
	DBusMessage     *msg;
	DBusMessage     *ret;
        DBusMessageIter  iter;
	DBusError        error;

	msg = dbus_message_new_method_call ("org.bluez", "/org/bluez",
                                            "org.bluez.Manager",
                                            "ListAdapters");

	if (!msg) {
		return FALSE;
	}

	dbus_error_init (&error);

	ret = dbus_connection_send_with_reply_and_block (conn->dbus_conn,
							 msg, -1, &error);
	dbus_message_unref (msg);

	if (dbus_error_is_set (&error)) {
		dbus_error_free (&error);
		return FALSE;
	}

	if (dbus_message_iter_init (ret, &iter)) {
                DBusMessageIter sub;

		dbus_message_iter_recurse (&iter, &sub);

		/* Go through each entry (device) and get each paired device
		 * from the entry.
		 */
		do {
	                char *devname;

			dbus_message_iter_get_basic (&sub, &devname);

			if (!func (conn, devname, user_data)) {
				dbus_message_unref (ret);
				return FALSE;
			}
		} while (dbus_message_iter_next (&sub));
	}

	dbus_message_unref (ret);

	return TRUE;
}

struct get_display_name_t {
	const gchar    *bda;
	gchar          *name;
};

static gboolean
get_display_name (Connection *conn,
		  const char *adapter,
		  gpointer user_data)
{
	struct get_display_name_t *data;
	DBusMessage *message = NULL;
	DBusMessage *reply = NULL;
	DBusError    dbus_error;
	gboolean     ret = TRUE;
	const gchar *str = NULL;

	data = (struct get_display_name_t *)user_data;

	message = dbus_message_new_method_call ("org.bluez",
						adapter,
						"org.bluez.Adapter",
						"GetRemoteAlias");
	if (!message) {
		ret = FALSE;
		goto end;
	}

	if (!dbus_message_append_args (message,
				       DBUS_TYPE_STRING, &data->bda,
				       DBUS_TYPE_INVALID)) {
		ret = FALSE;
		goto end;
	}

	dbus_error_init (&dbus_error);
	reply = dbus_connection_send_with_reply_and_block (conn->dbus_conn,
							   message, -1,
							   &dbus_error);
	if (!dbus_error_is_set (&dbus_error)) {
		/* do we have a device alias? */
		if (dbus_message_get_args (reply, NULL,
					   DBUS_TYPE_STRING, &str,
					   DBUS_TYPE_INVALID)) {
			data->name = g_strdup(str);
			ret = FALSE;
			goto end;
		}
	}

	/* clean up for next call */
	dbus_error_free (&dbus_error);
	if (reply) {
		dbus_message_unref (reply);
	}
	dbus_message_unref (message);
	message = reply = NULL;

	message = dbus_message_new_method_call ("org.bluez",
						adapter,
						"org.bluez.Adapter",
						"GetRemoteName");
	if (!message) {
		ret = FALSE;
		goto end;
	}

	if (!dbus_message_append_args (message,
				       DBUS_TYPE_STRING, &data->bda,
				       DBUS_TYPE_INVALID)) {
		ret = FALSE;
		goto end;
	}

	reply = dbus_connection_send_with_reply_and_block (conn->dbus_conn,
							   message, -1,
							   &dbus_error);
	if (!dbus_error_is_set (&dbus_error)) {
		/* do we have a device alias? */
		if (dbus_message_get_args (reply, NULL,
					   DBUS_TYPE_STRING, &str,
					   DBUS_TYPE_INVALID)) {
			data->name = g_strdup(str);
			ret = FALSE;
			goto end;
		}
	}

 end:
	dbus_error_free (&dbus_error);
	if (reply) {
		dbus_message_unref (reply);
	}
	if (message) {
		dbus_message_unref (message);
	}
	return ret;
}

gchar *
om_dbus_get_display_name (const gchar    *bda,
			  GnomeVFSResult *result)
{
	Connection *conn;
	struct get_display_name_t data;

	if (!om_utils_check_bda (bda)) {
		*result = GNOME_VFS_ERROR_NOT_FOUND;
		return NULL;
	}

	conn = get_system_bus_connection ();
	if (!conn) {
		*result = GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
		return NULL;
	}
	data.bda = bda;
	data.name = NULL;
	foreach_adapter (conn, get_display_name, (gpointer)&data);

	/* if we don't have name, use the numeric address */
	if (!data.name) {
		data.name = g_strdup (bda);
	}
	connection_free (conn);

	return data.name;
}

static void
om_append_paired_devices (Connection   *conn,
			  DBusMessage  *msg,
                          const char   *devname,
			  GList       **list)
{
        DBusMessageIter diter;
	DBusMessageIter dsub;

        if (!dbus_message_iter_init (msg, &diter)) {
		return;
	}

	dbus_message_iter_recurse (&diter, &dsub);

	do {
		/* Add the entry to the list. */
		GnomeVFSFileInfo *info;
		char             *remote_devname;

		if (dbus_message_iter_get_arg_type (&dsub) != DBUS_TYPE_STRING) {
			continue;
		}

		dbus_message_iter_get_basic (&dsub, &remote_devname);

		if (!is_obex_device (conn, devname, (const char*) remote_devname)) {
			continue;
		}


		info = gnome_vfs_file_info_new ();

		if (!info) {
			return;
		}

		info->valid_fields =
			GNOME_VFS_FILE_INFO_FIELDS_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
			GNOME_VFS_FILE_INFO_FIELDS_ACCESS |
			GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE |
			GNOME_VFS_FILE_INFO_FIELDS_SYMLINK_NAME;

		info->name = g_strdup (remote_devname);
		info->type = GNOME_VFS_FILE_TYPE_REGULAR;
		info->permissions =
			GNOME_VFS_PERM_USER_READ |
			GNOME_VFS_PERM_GROUP_READ |
			GNOME_VFS_PERM_OTHER_READ |
			GNOME_VFS_PERM_ACCESS_READABLE;

		info->uid = 0;
		info->gid = 0;
		info->mime_type = g_strdup ("application/x-desktop");

		*list = g_list_append (*list, info);
	} while (dbus_message_iter_next (&dsub));
}

/* Leave this in for easy testing. */
#if 0
static GList *
append_fake_device (GList *list, const gchar *bda)
{
	GnomeVFSFileInfo *info;

	info = gnome_vfs_file_info_new ();

	info->valid_fields =
		GNOME_VFS_FILE_INFO_FIELDS_TYPE |
		GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS |
		GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE |
		GNOME_VFS_FILE_INFO_FIELDS_SYMLINK_NAME;

	info->flags |= GNOME_VFS_FILE_FLAGS_SYMLINK;

	info->name = g_strdup_printf ("[%s]", bda);
	info->type = GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK;
	info->permissions =
		GNOME_VFS_PERM_USER_READ |
		GNOME_VFS_PERM_GROUP_READ |
		GNOME_VFS_PERM_OTHER_READ;
	info->symlink_name = g_strdup_printf ("obex://[%s]", bda);

	info->uid = 0;
	info->gid = 0;
	info->mime_type = g_strdup ("x-directory/normal");

	/*g_print ("added fake: %s %s\n", info->name, info->symlink_name);*/

	return g_list_append (list, info);
}
#endif

static gboolean
list_bondings (Connection *conn,
	       const char *adapter,
	       gpointer user_data)
{
	GList **devlist = user_data;
	DBusMessage *msg;
	DBusMessage *ret;
	DBusError error;

	msg = dbus_message_new_method_call ("org.bluez",
					    adapter,
					    "org.bluez.Adapter",
					    "ListBondings");

	if (!msg) {
		return FALSE;
	}

	dbus_error_init (&error);

	ret = dbus_connection_send_with_reply_and_block (conn->dbus_conn,
							 msg,
							 -1,
							 &error);
	dbus_message_unref (msg);

	if (dbus_error_is_set (&error)) {
		dbus_error_free (&error);
		return FALSE;
	}


	om_append_paired_devices (conn,
				  ret,
				  adapter,
				  devlist);

	dbus_message_unref (ret);

	return TRUE;
}

GList *
om_dbus_get_dev_list (void)
{
	Connection      *conn;
	GList           *devlist = NULL;

#if 0
	if (0) {
		devlist = append_fake_device (devlist, "foo");
		devlist = append_fake_device (devlist, "bar");

		return devlist;
	}
#endif

	conn = get_system_bus_connection ();
	if (!conn) {
		return NULL;
	}

	foreach_adapter (conn, list_bondings, (gpointer) &devlist);
	connection_free (conn);

	return devlist;
}

static gboolean
is_obex_device (Connection *conn,
		const char *dev,
		const char *name)
{
	DBusMessage      *msg, *ret;
	DBusMessageIter  iter, sub;
	DBusError        error;
	gboolean         is_obex = FALSE;
	char            *class_name;

	msg = dbus_message_new_method_call ("org.bluez",
					    dev,
					    "org.bluez.Adapter",
					    "GetRemoteServiceClasses");

	dbus_message_iter_init_append (msg, &iter);

	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &name);

	dbus_error_init (&error);
	ret = dbus_connection_send_with_reply_and_block (conn->dbus_conn,
							 msg,
							 -1,
							 &error);
	dbus_message_unref (msg);

	if (dbus_error_is_set (&error)) {
		dbus_error_free (&error);
		return FALSE;
	}

	if (dbus_message_iter_init (ret, &iter)) {
		dbus_message_iter_recurse (&iter, &sub);

		do {
			if (dbus_message_iter_get_arg_type (&sub) != DBUS_TYPE_STRING) {
				continue;
			}

			dbus_message_iter_get_basic (&sub, &class_name);

			if (!strcmp (class_name, "object transfer")) {
				is_obex = TRUE;
				break;
			}


		} while (dbus_message_iter_next (&sub));
	}

	dbus_message_unref (ret);

	return is_obex;
}

