/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* gdm-main.c
 *
 * Copyright (C) 2007 David Zeuthen
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <glib-object.h>
#include <gnome.h>
#include <glib/gi18n.h>

#include <gnome-device-manager/gdm-pool.h>
#include <gnome-device-manager/gdm-info-provider.h>

#include "gdm-main.h"
#include "gdm-menus.h"
#include "gdm-device-tree.h"
#include "gdm-properties-list.h"

static GdmPool *device_pool;
static GtkTreeView *tab_properties_list;
static GtkWidget *notebook;

static GtkWidget *tab_summary_vbox;

static GtkWidget *tab_properties;
static GtkWidget *tab_properties_list_scrolled_window;

void
gdm_main_toggle_properties_view (gboolean show_properties)
{
        int page_num;

        page_num = gtk_notebook_page_num (GTK_NOTEBOOK (notebook), tab_properties_list_scrolled_window);
        if (show_properties && page_num == -1) {
                gtk_notebook_append_page (GTK_NOTEBOOK (notebook),
                                          tab_properties_list_scrolled_window,
                                          tab_properties);
                gtk_widget_show_all (tab_properties_list_scrolled_window);
                gtk_widget_show_all (tab_properties);
        } else if (page_num >= 0) {
                g_object_ref (tab_properties_list_scrolled_window);
                g_object_ref (tab_properties);
                gtk_notebook_remove_page (GTK_NOTEBOOK (notebook), page_num);
        }
}

static void
_remove_child (GtkWidget *widget, gpointer user_data)
{
        GtkContainer *container = GTK_CONTAINER (user_data);
        gtk_container_remove (container, widget);
}

static void
info_page_show_for_device (GdmDevice *device)
{
        GSList *i;
        int n;
        char *p;
        char *name;
        char *icon_name;
        GSList *kv_pairs;
        GSList *errors;
        GSList *warnings;
        GSList *notices;
        int num_pairs;
        GdkPixbuf *pixbuf;
        static GtkTable *info_table = NULL;
        GtkWidget *hbox;
        GtkWidget *image;
        GtkWidget *label;
        GtkWidget *frame;
        GtkWidget *evbox;
        GtkWidget *notif_vbox;
        struct {
                const char *icon_name;
                GSList **source;
        } notifications[] =
          {
                  {GTK_STOCK_DIALOG_ERROR, &errors},
                  {GTK_STOCK_DIALOG_WARNING, &warnings},
                  {GTK_STOCK_DIALOG_INFO, &notices},
                  {NULL, NULL}                  
          };

        /* delete all old widgets */
        gtk_container_foreach (GTK_CONTAINER (tab_summary_vbox),
                               _remove_child,
                               tab_summary_vbox);

        name = gdm_info_provider_get_long_name (device);
        icon_name = gdm_info_provider_get_icon_name (device);
        kv_pairs = gdm_info_provider_get_summary (device);
        errors = gdm_info_provider_get_errors (device);
        warnings = gdm_info_provider_get_warnings (device);
        notices = gdm_info_provider_get_notices (device);

        /* draw notifications */
        notif_vbox = gtk_vbox_new (FALSE, -1);
        for (n = 0; notifications[n].source != NULL; n++) {
                const char *icon;
                GSList *source;
                icon = notifications[n].icon_name;
                source = *(notifications[n].source);                
                if (source != NULL) {
                        for (i = source; i != NULL; i = g_slist_next (i)) {
                                GdkColor border_color = {0, 0xb800, 0xad00, 0x9d00};
                                GdkColor fill_color = {0, 0xff00, 0xff00, 0xbf00};
                                GdmInfoProviderTip *tip = i->data;
                                
                                frame = gtk_frame_new (NULL);
                                evbox = gtk_event_box_new ();
                                gtk_widget_modify_bg (frame, GTK_STATE_NORMAL, &border_color);
                                gtk_widget_modify_bg (evbox, GTK_STATE_NORMAL, &fill_color);
                                gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
                                
                                hbox = gtk_hbox_new (FALSE, 5);
                                
                                image = gtk_image_new_from_stock (notifications[n].icon_name, GTK_ICON_SIZE_MENU);
                                
                                label = gtk_label_new (NULL);
                                gtk_label_set_markup (GTK_LABEL (label), tip->text);
                                gtk_label_set_ellipsize (GTK_LABEL (label), PANGO_ELLIPSIZE_END);
                                gtk_label_set_width_chars (GTK_LABEL (label), 50);
                                gtk_label_set_selectable (GTK_LABEL (label), TRUE);
                                gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
                                
                                gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
                                gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

                                if (tip->button_text != NULL) {
                                        GtkWidget *button;
                                        button = gtk_button_new_with_label (tip->button_text);
                                        gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
                                }
                                
                                gtk_container_set_border_width (GTK_CONTAINER (hbox), 2);
                                gtk_container_add (GTK_CONTAINER (evbox), hbox);
                                gtk_container_add (GTK_CONTAINER (frame), evbox);
                                gtk_box_pack_start (GTK_BOX (notif_vbox), frame, FALSE, FALSE, 0);

                                gdm_info_provider_tip_unref (tip);
                        }

                        g_slist_free (source);
                }
        }
        gtk_box_pack_start (GTK_BOX (tab_summary_vbox), notif_vbox, FALSE, FALSE, 0);

        hbox = gtk_hbox_new (FALSE, 10);
        if (icon_name == NULL) {
                icon_name = g_strdup ("stock_unknown");
        }
        pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
                                           icon_name,
                                           48,
                                           0,
                                           NULL);
        if (pixbuf == NULL) {
                image = gtk_image_new ();
                gtk_image_clear (GTK_IMAGE (image));
        } else {
                image = gtk_image_new_from_pixbuf (pixbuf);
                g_object_unref (pixbuf);
        }
        p = g_strdup_printf ("<big><b>%s</b></big>", name);
        label = gtk_label_new (NULL);
        gtk_label_set_markup (GTK_LABEL (label), p);
        gtk_label_set_ellipsize (GTK_LABEL (label), PANGO_ELLIPSIZE_END);
        gtk_label_set_width_chars (GTK_LABEL (label), 50);
        gtk_label_set_selectable (GTK_LABEL (label), TRUE);
        gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
        g_free (p);
        gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET (image), FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET (label), FALSE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (tab_summary_vbox), hbox, FALSE, FALSE, 0);
        g_free (icon_name);
        g_free (name);

        if (kv_pairs != NULL) {
                num_pairs = g_slist_length (kv_pairs) / 2;
                info_table = GTK_TABLE (gtk_table_new (num_pairs, 2, FALSE));
                gtk_table_set_col_spacings (GTK_TABLE (info_table), 8);
                gtk_table_set_row_spacings (GTK_TABLE (info_table), 4);

                for (i = kv_pairs, n = 0; i != NULL; i = i->next, n++) {
                        char *key;
                        char *key2;
                        char *value;
                        GtkWidget *key_label;
                        GtkWidget *value_label;
                        
                        key = i->data;
                        i = i->next;
                        if (i == NULL) {
                                g_free (key);
                                break;
                        }
                        value = i->data;

                        key2 = g_strdup_printf ("<b>%s:</b>", key);
                        
                        key_label = gtk_label_new (NULL);
                        gtk_label_set_markup (GTK_LABEL (key_label), key2);
                        gtk_misc_set_alignment (GTK_MISC (key_label), 1.0, 0.5);
                        
                        value_label = gtk_label_new (NULL);
                        gtk_label_set_markup (GTK_LABEL (value_label), value);
                        gtk_misc_set_alignment (GTK_MISC (value_label), 0.0, 0.5);
                        gtk_label_set_selectable (GTK_LABEL (value_label), TRUE);
                        gtk_label_set_ellipsize (GTK_LABEL (value_label), PANGO_ELLIPSIZE_END);
                        gtk_label_set_width_chars (GTK_LABEL (value_label), 50);
                        
                        gtk_table_attach_defaults (info_table, key_label, 0, 1, n, n + 1);
                        gtk_table_attach_defaults (info_table, value_label, 1, 2, n, n + 1);

                        g_free (key);
                        g_free (key2);
                        g_free (value);
                }
                g_slist_free (kv_pairs);

                /* add property pairs */
                gtk_box_pack_start (GTK_BOX (tab_summary_vbox), GTK_WIDGET (info_table), FALSE, FALSE, 0);
        }

        gtk_widget_show_all (tab_summary_vbox);
}

static GdmDevice *now_showing = NULL;

static void
device_tree_changed (GtkTreeSelection *selection, gpointer user_data)
{
        GdmDevice *device;
        GtkTreeView *device_tree_view;

        device_tree_view = gtk_tree_selection_get_tree_view (selection);
        device = gdm_device_tree_get_selected_device (device_tree_view);

        if (device != NULL) {
                now_showing = device;
                gdm_properties_list_show_for_device (tab_properties_list, device);
                info_page_show_for_device (device);
        }
}

static void
device_removed (GdmPool *pool, GdmDevice *device, gpointer user_data)
{
        GtkTreeView *treeview = GTK_TREE_VIEW (user_data);

        /* if device we currently show is removed.. go to computer device */
        if (now_showing == device) {
                gdm_device_tree_select_device (GTK_TREE_VIEW (treeview), 
                                               gdm_pool_get_device_by_udi (device_pool, 
                                                                           "/org/freedesktop/Hal/devices/computer"));
        }
}

static void
device_property_changed (GdmPool *pool, GdmDevice *device, const char *key, gpointer user_data)
{
        //GtkTreeView *treeview = GTK_TREE_VIEW (user_data);
        if (device == now_showing) {
                info_page_show_for_device (device);
        }
}

static GtkWidget *
create_window (const gchar * geometry)
{
        GtkWidget *app;
        GtkWidget *vbox;
        GtkWidget *menubar;
        GtkWidget *toolbar;
        GtkUIManager *ui_manager;
        GtkAccelGroup *accel_group;
        GtkWidget *hpane;
        GtkWidget *treeview_scrolled_window;
        GtkWidget *treeview;
        GtkWidget *tab_summary_label;
        GtkTreeSelection *select;

        app = NULL;

        gdm_info_provider_register_builtin ();

        device_pool = gdm_pool_new ();
        if (device_pool == NULL) {
                goto out;
        }

        app = gtk_window_new (GTK_WINDOW_TOPLEVEL);
        gtk_window_set_resizable (GTK_WINDOW (app), TRUE);
        gtk_window_set_default_size (GTK_WINDOW (app), 750, 550);
        gtk_window_set_title (GTK_WINDOW (app), _("Device Manager"));

        vbox = gtk_vbox_new (FALSE, 0);
        gtk_container_add (GTK_CONTAINER (app), vbox);

        ui_manager = gdm_create_ui_manager ("GnomeDeviceManagerActions", app);
        accel_group = gtk_ui_manager_get_accel_group (ui_manager);
        gtk_window_add_accel_group (GTK_WINDOW (app), accel_group);

        menubar = gtk_ui_manager_get_widget (ui_manager, "/menubar");
        gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 0);
        toolbar = gtk_ui_manager_get_widget (ui_manager, "/toolbar");
        gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);

        /* tree view */
        treeview_scrolled_window = gtk_scrolled_window_new (NULL, NULL);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (treeview_scrolled_window),
                                        GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        treeview = GTK_WIDGET (gdm_device_tree_new (device_pool));
        gtk_container_add (GTK_CONTAINER (treeview_scrolled_window), treeview);

        /* notebook */
        notebook = gtk_notebook_new ();

        /* summary page */
        tab_summary_vbox = gtk_vbox_new (FALSE, 10);
        gtk_container_set_border_width (GTK_CONTAINER (tab_summary_vbox), 8);
        tab_summary_label = gtk_label_new (_("Summary"));
        gtk_notebook_append_page (GTK_NOTEBOOK (notebook),
                                  tab_summary_vbox,
                                  tab_summary_label);

        /* device properties */
        tab_properties = gtk_label_new (_("Properties"));
        tab_properties_list_scrolled_window = gtk_scrolled_window_new (NULL, NULL);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (tab_properties_list_scrolled_window),
                                        GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        tab_properties_list = gdm_properties_list_new ();
        gtk_container_add (GTK_CONTAINER (tab_properties_list_scrolled_window), GTK_WIDGET (tab_properties_list));

        /* setup and add pane */
        hpane = gtk_hpaned_new ();
        gtk_paned_add1 (GTK_PANED (hpane), treeview_scrolled_window);
        gtk_paned_add2 (GTK_PANED (hpane), notebook);
        gtk_paned_set_position (GTK_PANED (hpane), 260);
        gtk_box_pack_start (GTK_BOX (vbox), hpane, TRUE, TRUE, 0);

        /* geometry */
        if (geometry != NULL) {
                if (!gtk_window_parse_geometry
                    (GTK_WINDOW (app), geometry)) {
                        g_error (_("Could not parse geometry string `%s'"),
                                 geometry);
                }
        }

        select = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
        gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
        g_signal_connect (select, "changed", (GCallback) device_tree_changed, NULL);

        /* when starting up, set focus to computer entry */
        gtk_widget_grab_focus (treeview);
        gdm_device_tree_select_device (GTK_TREE_VIEW (treeview), 
                                       gdm_pool_get_device_by_udi (device_pool, 
                                                                   "/org/freedesktop/Hal/devices/computer"));
        
        g_signal_connect (device_pool, "device_removed", (GCallback) device_removed, treeview);
        g_signal_connect (device_pool, "device_property_changed", (GCallback) device_property_changed, treeview);

        g_signal_connect (app, "delete-event", gtk_main_quit, NULL);

        gtk_widget_show_all (vbox);
out:
        return app;
}

static void session_die (GnomeClient * client, gpointer client_data);
static gint save_session (GnomeClient * client, gint phase,
                          GnomeSaveStyle save_style,
                          gint is_shutdown,
                          GnomeInteractStyle interact_style, gint is_fast,
                          gpointer client_data);

static char *geometry = NULL;
static char **args = NULL;

static GOptionEntry option_entries[] = {
        {
                "geometry",
                0,
                0,
                G_OPTION_ARG_STRING,
                &geometry,
                N_("Specify the geometry of the main window"),
                N_("GEOMETRY")
        },
        {
                G_OPTION_REMAINING,
                0,
                0,
                G_OPTION_ARG_STRING_ARRAY,
                &args,
                NULL,
                NULL}
};

int
main (int argc, char **argv)
{
        GtkWidget *window;
        GOptionContext *context;
        GnomeProgram *program;
        GnomeClient *client;
        
        bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
        bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
        textdomain (GETTEXT_PACKAGE);
        
        context = g_option_context_new (_("- GNOME Device Manager"));
        g_option_context_add_main_entries (context, option_entries,
                                           GETTEXT_PACKAGE);
        
        program = gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE,
                                      argc, argv,
                                      GNOME_PARAM_GOPTION_CONTEXT, context,
                                      GNOME_PARAM_APP_DATADIR, DATADIR,
                                      NULL);

        gtk_window_set_default_icon_name ("gnome-device-manager");

        client = gnome_master_client ();
        g_signal_connect (client, "save_yourself",
                          G_CALLBACK (save_session), argv[0]);
        g_signal_connect (client, "die", G_CALLBACK (session_die), NULL);

        window = create_window (geometry);

        gtk_widget_show_all (window);
        gtk_main ();
        g_object_unref (program);
        return 0;
}

static gint
save_session (GnomeClient * client,
              gint phase,
              GnomeSaveStyle save_style,
              gint is_shutdown,
              GnomeInteractStyle interact_style,
              gint is_fast, gpointer client_data)
{
        gchar **argv;
        guint argc;

        argv = g_new0 (gchar *, 4);
        argc = 0;

        argv[argc++] = client_data;

        argv[argc] = NULL;

        gnome_client_set_clone_command (client, argc, argv);
        gnome_client_set_restart_command (client, argc, argv);

        return TRUE;
}

static void
session_die (GnomeClient * client, gpointer client_data)
{
        gtk_main_quit ();
}
