/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* gdm-info-provider-storage.c
 *
 * Copyright (C) 2007 David Zeuthen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <config.h>
#include <glib/gi18n.h>
#include <sys/statvfs.h>
#include "gdm-info-provider.h"

#define KILOBYTE_FACTOR 1024.0
#define MEGABYTE_FACTOR (1024.0 * 1024.0)
#define GIGABYTE_FACTOR (1024.0 * 1024.0 * 1024.0)

static char *
gdm_util_get_size_for_display (guint64 size, gboolean long_string)
{
        char *str;
        gdouble displayed_size;

        if (size < MEGABYTE_FACTOR) {
                displayed_size = (double) size / KILOBYTE_FACTOR;
                if (long_string)
                        str = g_strdup_printf (_("%.1f KB (%'lld bytes)"), displayed_size, size);
                else
                        str = g_strdup_printf (_("%.1f KB"), displayed_size);
        } else if (size < GIGABYTE_FACTOR) {
                displayed_size = (double) size / MEGABYTE_FACTOR;
                if (long_string)
                        str = g_strdup_printf (_("%.1f MB (%'lld bytes)"), displayed_size, size);
                else
                        str = g_strdup_printf (_("%.1f MB"), displayed_size);
        } else {
                displayed_size = (double) size / GIGABYTE_FACTOR;
                if (long_string)
                        str = g_strdup_printf (_("%.1f GB (%'lld bytes)"), displayed_size, size);
                else
                        str = g_strdup_printf (_("%.1f GB"), displayed_size);
        }
        
        return str;
}

static char *
gdm_util_get_fstype_for_display (const char *fstype, const char *fsversion, gboolean long_string)
{
        char *s;

        if (fstype == NULL) {
                fstype = "";
        }

        if (fsversion == NULL) {
                fsversion = "";
        }

        if (g_ascii_strcasecmp (fstype, "vfat") == 0) {
                /* version = FAT12 | FAT16 | FAT32 */

                if (g_ascii_strcasecmp (fsversion, "FAT12") == 0) {
                        if (long_string) {
                                s = g_strdup (_("Microsoft FAT (12-bit version)"));
                        } else {
                                s = g_strdup (_("FAT"));
                        }
                } else if (g_ascii_strcasecmp (fsversion, "FAT16") == 0) {
                        if (long_string) {
                                s = g_strdup (_("Microsoft FAT (16-bit version)"));
                        } else {
                                s = g_strdup (_("FAT"));
                        }
                } else if (g_ascii_strcasecmp (fsversion, "FAT32") == 0) {
                        if (long_string) {
                                s = g_strdup (_("Microsoft FAT (32-bit version)"));
                        } else {
                                s = g_strdup (_("FAT"));
                        }
                } else {
                        if (long_string) {
                                s = g_strdup (_("Microsoft FAT"));
                        } else {
                                s = g_strdup (_("FAT"));
                        }
                }
        } else if (g_ascii_strcasecmp (fstype, "ntfs") == 0) {
                if (long_string) {
                        s = g_strdup_printf (_("Microsoft NTFS version %s"), fsversion);
                } else {
                        s = g_strdup (_("NTFS"));
                }
        } else if (g_ascii_strcasecmp (fstype, "hfs") == 0) {
                if (long_string) {
                        s = g_strdup (_("Apple HFS"));
                } else {
                        s = g_strdup (_("HFS"));
                }
        } else if (g_ascii_strcasecmp (fstype, "hfsplus") == 0) {
                if (long_string) {
                        s = g_strdup (_("Apple HFS+"));
                } else {
                        s = g_strdup (_("HFS+"));
                }
        } else if (g_ascii_strcasecmp (fstype, "crypto_LUKS") == 0) {
                if (long_string) {
                        s = g_strdup (_("Linux Unified Key Setup"));
                } else {
                        s = g_strdup (_("LUKS"));
                }
        } else if (g_ascii_strcasecmp (fstype, "ext2") == 0) {
                if (long_string) {
                        s = g_strdup_printf (_("Linux Second Ext. FS (version %s)"), fsversion);
                } else {
                        s = g_strdup (_("ext2"));
                }
        } else if (g_ascii_strcasecmp (fstype, "ext3") == 0) {
                if (long_string) {
                        s = g_strdup_printf (_("Linux Third Ext. FS (version %s)"), fsversion);
                } else {
                        s = g_strdup (_("ext3"));
                }
        } else if (g_ascii_strcasecmp (fstype, "jbd") == 0) {
                if (long_string) {
                        s = g_strdup_printf (_("Journal for Linux ext3 (version %s)"), fsversion);
                } else {
                        s = g_strdup (_("jbd"));
                }
        } else if (g_ascii_strcasecmp (fstype, "iso9660") == 0) {
                if (long_string) {
                        s = g_strdup (_("ISO 9660"));
                } else {
                        s = g_strdup (_("iso9660"));
                }
        } else if (g_ascii_strcasecmp (fstype, "udf") == 0) {
                if (long_string) {
                        s = g_strdup (_("Universal Disk Format"));
                } else {
                        s = g_strdup (_("udf"));
                }
        } else {
                s = g_strdup (fstype);
        }

        return s;
}

static char *
get_drive_description (GdmDevice *device, gboolean short_desc)
{
        char *s;
        const char *bus;
        const char *drive_type;

        s = NULL;
        drive_type = gdm_device_get_property_string (device, "storage.drive_type");
        if (drive_type == NULL)
                goto out;

        bus = gdm_device_get_property_string (device, "storage.bus");

        if (g_ascii_strcasecmp (drive_type, "cdrom") == 0) {
                if (short_desc) {
                        s = g_strdup (_("Optical Drive"));
                } else {
                        int n;
                        GString *str;
                        struct s_disc_compat {
                                char *prop_name;
                                char *display_name;
                        } disc_compat[] = {
                                {"storage.cdrom.bd", N_("Blu-ray")},
                                {"storage.cdrom.bdr", N_("Blu-ray-R")},
                                {"storage.cdrom.bdre", N_("Blu-ray-RE")},
                                {"storage.cdrom.cdr", N_("CD-R")},
                                {"storage.cdrom.cdrw", N_("CD-RW")},
                                {"storage.cdrom.dvd", N_("DVD-ROM")},
                                {"storage.cdrom.dvdplusr", N_("DVD+R")},
                                {"storage.cdrom.dvdplusrdl", N_("DVD+R DL")},
                                {"storage.cdrom.dvdplusrw", N_("DVD+RW")},
                                {"storage.cdrom.dvdplusrwdl", N_("DVD+RW DL")},
                                {"storage.cdrom.dvdr", N_("DVD-R")},
                                {"storage.cdrom.dvdram", N_("DVD-RAM")},
                                {"storage.cdrom.dvdrw", N_("DVD-RW")},
                                {"storage.cdrom.hddvd", N_("HD DVD")},
                                {"storage.cdrom.hddvdr", N_("HD DVD-R")},
                                {"storage.cdrom.hddvdrw", N_("HD DVD-RW")},
                                {NULL, NULL}
                        };
                        
                        str = g_string_new (_("CD-ROM"));
                        for (n = 0; disc_compat[n].prop_name != NULL; n++) {
                                if (gdm_device_get_property_bool (device, disc_compat[n].prop_name)) {
                                        g_string_append (str, _("/"));
                                        g_string_append (str, disc_compat[n].display_name);
                                }
                        }
                        s = g_string_free (str, FALSE);
                }                
        } else if (g_ascii_strcasecmp (drive_type, "floppy") == 0) {
                s = g_strdup (_("Floppy Drive"));
        } else if (g_ascii_strcasecmp (drive_type, "disk") == 0) {
                if (bus != NULL && g_ascii_strcasecmp (bus, "linux_raid") == 0) {
                        s = g_strdup (_("Software RAID Drive"));
                } else {
                        s = g_strdup (_("Mass Storage Drive"));
                }
        } else if (g_ascii_strcasecmp (drive_type, "tape") == 0) {
                s = g_strdup (_("Tape Drive"));
        } else if (g_ascii_strcasecmp (drive_type, "compact_flash") == 0) {
                s = g_strdup (_("CompactFlash Drive"));
        } else if (g_ascii_strcasecmp (drive_type, "memory_stick") == 0) {
                s = g_strdup (_("MemoryStick Drive"));
        } else if (g_ascii_strcasecmp (drive_type, "smart_media") == 0) {
                s = g_strdup (_("SmartMedia Drive"));
        } else if (g_ascii_strcasecmp (drive_type, "sd_mmc") == 0) {
                s = g_strdup (_("SD/MMC Drive"));
        } else if (g_ascii_strcasecmp (drive_type, "zip") == 0) {
                s = g_strdup (_("Zip Drive"));
        } else if (g_ascii_strcasecmp (drive_type, "jaz") == 0) {
                s = g_strdup (_("Jaz Drive"));
        } else if (g_ascii_strcasecmp (drive_type, "flashkey") == 0) {
                s = g_strdup (_("Thumb Drive"));
        } 
out:
        if (s == NULL)
                s = g_strdup (_("Storage Device"));
        return s;
}

static gboolean 
get_provider_matches (GdmDevice *device)
{
        return gdm_device_test_capability (device, "storage");
}

static char *
get_icon_name (GdmDevice *device)
{
        return g_strdup ("drive-harddisk");
}

static char *
get_short_name (GdmDevice *device)
{
        return get_drive_description (device, TRUE);
}

static char *
get_long_name (GdmDevice *device)
{
        return get_drive_description (device, TRUE);
}

static char *
get_vendor (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "storage.vendor"));
}

static char *
get_product (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "storage.model"));
}

#define ADD_SUM(p, key, value)                                                       \
        do {                                                                         \
                if ((value) != NULL) {                                               \
                        p = g_slist_append (p, g_strdup (key));                      \
                        p = g_slist_append (p, (value));                             \
                }                                                                    \
        } while (FALSE)

static GSList *
get_summary (GdmDevice *device)
{
        char *s;
        const char *vendor;
        const char *model;
        const char *device_file;
        const char *serial;
        const char *firmware_version;
        const char *bus;
        const char *drive_type;
        gboolean hotpluggable;
        gboolean removable;
        gboolean media_available;
        guint64 media_size;
        const char *part_scheme;
        gboolean requires_eject;
        GSList *p = NULL;

        vendor = gdm_device_get_property_string (device, "storage.vendor");
        model = gdm_device_get_property_string (device, "storage.model");
        device_file = gdm_device_get_property_string (device, "block.device");
        serial = gdm_device_get_property_string (device, "storage.serial");
        firmware_version = gdm_device_get_property_string (device, "storage.firmware_version");
        bus = gdm_device_get_property_string (device, "storage.bus");
        drive_type = gdm_device_get_property_string (device, "storage.drive_type");
        hotpluggable = gdm_device_get_property_bool (device, "storage.hotpluggable");
        removable = gdm_device_get_property_bool (device, "storage.removable");
        media_available = gdm_device_get_property_bool (device, "storage.removable.media_available");
        media_size = gdm_device_get_property_uint64 (device, "storage.removable.media_size");
        part_scheme = gdm_device_get_property_string (device, "storage.partitioning_scheme");
        requires_eject = gdm_device_get_property_bool (device, "storage.requires_eject");

        ADD_SUM (p, _("Model"), g_strdup (model));
        ADD_SUM (p, _("Vendor"), g_strdup (vendor));
        ADD_SUM (p, _("Device File"), g_strdup (device_file));
        ADD_SUM (p, _("Serial Number"), g_strdup (serial));
        ADD_SUM (p, _("Firmware Version"), g_strdup (firmware_version));

        if (bus != NULL) {
                s = NULL;

                if (g_ascii_strcasecmp (bus, "usb") == 0) {
                        s = g_strdup (_("USB"));
                } else if (g_ascii_strcasecmp (bus, "ieee1394") == 0) {
                        s = g_strdup (_("IEEE 1394"));
                } else if (g_ascii_strcasecmp (bus, "ide") == 0) {
                        s = g_strdup (_("ATA"));
                } else if (g_ascii_strcasecmp (bus, "scsi") == 0) {
                        s = g_strdup (_("SCSI"));
                } else if (g_ascii_strcasecmp (bus, "ccw") == 0) {
                        s = g_strdup (_("CCW"));
                } else if (g_ascii_strcasecmp (bus, "linux_raid") == 0) {
                        s = g_strdup (_("Linux RAID (Logical)"));
                }

                if (s != NULL)
                        ADD_SUM (p, _("Connection Bus"), s);
        }

        ADD_SUM (p, _("Connection Type"), hotpluggable ? g_strdup (_("External")) : g_strdup (_("Internal")));
        ADD_SUM (p, _("Removable Media"), 
                 removable ? (requires_eject ? g_strdup (_("Yes (ejectable)")) : g_strdup (_("Yes")))
                                             : g_strdup (_("No")));

        if (removable && drive_type != NULL) {
                s = get_drive_description (device, FALSE);
                if (s != NULL)
                        ADD_SUM (p, _("Media Compatibility"), s);
        }

        if (drive_type != NULL && strcmp (drive_type, "cdrom") == 0) {
                int speed;
                int speed_mul;
                double speed_mbps;

                speed = gdm_device_get_property_int (device, "storage.cdrom.read_speed");
                speed_mul = speed / 150;
                speed_mbps = speed * 8.0 / 1000.0;
                ADD_SUM (p, _("Maximum Read Speed"), g_strdup_printf (_("%dx (%.2f MBps)"), speed_mul, speed_mbps));

                speed = gdm_device_get_property_int (device, "storage.cdrom.write_speed");
                if (speed > 0) {
                        speed_mul = speed / 150;
                        speed_mbps = speed * 8.0 / 1000.0;
                        ADD_SUM (p, _("Maximum Write Speed"), g_strdup_printf (_("%dx (%.2f MBps)"), speed_mul, speed_mbps));
                }
        }

        if (removable) {
                if (media_available)
                        ADD_SUM (p, _("Media Capacity"), gdm_util_get_size_for_display (media_size, TRUE));
                else
                        ADD_SUM (p, _("Media Capacity"), g_strdup (_("-")));
        } else {
                ADD_SUM (p, _("Drive Capacity"), gdm_util_get_size_for_display (media_size, TRUE));
        }

        if (part_scheme != NULL) {
                if (g_ascii_strcasecmp (part_scheme, "gpt") == 0) {
                        s = g_strdup (_("GUID Partition Table"));
                } else if (g_ascii_strcasecmp (part_scheme, "mbr") == 0) {
                        s = g_strdup (_("Master Boot Record"));
                } else if (g_ascii_strcasecmp (part_scheme, "apm") == 0) {
                        s = g_strdup (_("Apple Partition Map"));
                } else {
                        s = g_strdup (_("-"));
                }
                ADD_SUM (p, _("Partitioning"), s);
        } else {
                ADD_SUM (p, _("Partitioning"), g_strdup (_("-")));
        }

        if (bus != NULL && g_ascii_strcasecmp (bus, "linux_raid") == 0) {
                const char *raid_level;
                int raid_num_components;
                int raid_num_components_active;
                gboolean raid_is_syncing;

                raid_level = gdm_device_get_property_string (device, "storage.linux_raid.level");
                raid_num_components = gdm_device_get_property_int (device, "storage.linux_raid.num_components");
                raid_num_components_active = gdm_device_get_property_int (device, "storage.linux_raid.num_components_active");
                raid_is_syncing = gdm_device_get_property_bool (device, "storage.linux_raid.is_syncing");

                if (raid_level != NULL) {
                        if (g_ascii_strcasecmp (raid_level, "linear") == 0) {
                                s = g_strdup (_("Linear"));
                        } else if (g_ascii_strcasecmp (raid_level, "raid0") == 0) {
                                s = g_strdup (_("RAID-0 (Stripe)"));
                        } else if (g_ascii_strcasecmp (raid_level, "raid1") == 0) {
                                s = g_strdup (_("RAID-1 (Mirror)"));
                        } else if (g_ascii_strcasecmp (raid_level, "raid5") == 0) {
                                s = g_strdup (_("RAID-5"));
                        } else {
                                s = g_strdup (raid_level);
                        }
                        ADD_SUM (p, _("RAID Level"), s);
                }

                ADD_SUM (p, _("Number of RAID Disks"), g_strdup_printf (_("%d (%d active)"), 
                                                                        raid_num_components,
                                                                        raid_num_components_active));

                if (raid_is_syncing) {
                        const char *sync_action;
                        double sync_progress;
                        guint64 sync_speed;

                        sync_action = gdm_device_get_property_string (device, "storage.linux_raid.sync.action");
                        sync_progress = gdm_device_get_property_double (device, "storage.linux_raid.sync.progress");
                        sync_speed = gdm_device_get_property_uint64 (device, "storage.linux_raid.sync.speed");

                        s = NULL;
                        if (sync_action != NULL) {
                                if (g_ascii_strcasecmp (sync_action, "reshape") == 0) {
                                        s = g_strdup_printf (_("Reshaping (%2.3g%% complete @ %'ld kBps)"), 
                                                             100.0 * sync_progress, sync_speed);
                                } else if (g_ascii_strcasecmp (sync_action, "resync") == 0) {
                                        s = g_strdup_printf (_("Reshaping (%2.3g%% complete @ %'ld kBps)"), 
                                                             100.0 * sync_progress, sync_speed);
                                } else if (g_ascii_strcasecmp (sync_action, "check") == 0) {
                                        s = g_strdup_printf (_("Checking (%2.3g%% complete @ %'ld kBps)"), 
                                                             100.0 * sync_progress, sync_speed);
                                } else if (g_ascii_strcasecmp (sync_action, "repair") == 0) {
                                        s = g_strdup_printf (_("Repairing (%2.3g%% complete @ %'ld kBps)"), 
                                                             100.0 * sync_progress, sync_speed);
                                } else if (g_ascii_strcasecmp (sync_action, "recover") == 0) {
                                        s = g_strdup_printf (_("Recovering (%2.3g%% complete @ %'ld kBps)"),
                                                             100.0 * sync_progress, sync_speed);
                                }
                        }
                        if (s == NULL) {
                                s = g_strdup_printf (_("%s (%2.3g%% complete @ %'ld kB / second)"), 
                                                     sync_action, 100.0 * sync_progress, sync_speed);
                        }

                        
                } else {
                        if (raid_num_components_active < raid_num_components)
                                s = g_strdup (_("Degraded"));
                        else
                                s = g_strdup (_("Healthy"));
                }
                ADD_SUM (p, _("RAID State"), s);

                /* TODO: link to the components */
        }


        return p;
}

static GSList *
get_warnings (GdmDevice *device)
{
        GSList *p = NULL;
        const char *bus;

        bus = gdm_device_get_property_string (device, "storage.bus");

        /* warn about degraded RAID */
        if (bus != NULL && g_ascii_strcasecmp (bus, "linux_raid") == 0) {
                int raid_num_components;
                int raid_num_components_active;
                raid_num_components = gdm_device_get_property_int (device, "storage.linux_raid.num_components");
                raid_num_components_active = gdm_device_get_property_int (device, "storage.linux_raid.num_components_active");
                if (raid_num_components > raid_num_components_active) {
                        p = g_slist_append (p, 
                                            gdm_info_provider_tip_new (
                                                    -1, 
                                                    _("RAID array is running in degraded mode."),
                                                    "Resolve..."));
                }
        }

        return p;
}

static GSList *
get_notices (GdmDevice *device)
{
        GSList *p = NULL;
        const char *bus;

        bus = gdm_device_get_property_string (device, "storage.bus");

        /* put notice about RAID array being synced */
        if (bus != NULL && g_ascii_strcasecmp (bus, "linux_raid") == 0) {
                gboolean raid_is_syncing;

                raid_is_syncing = gdm_device_get_property_bool (device, "storage.linux_raid.is_syncing");
                if (raid_is_syncing) {
                        p = g_slist_append (p, 
                                            gdm_info_provider_tip_new (
                                                    -1, 
                                                    _("RAID array is currently rebuilding."),
                                                    NULL));
                }
        }

        return p;
}


GdmInfoProviderIface gdm_info_provider_storage =
{
        .get_provider_matches = get_provider_matches,
        .get_icon_name        = get_icon_name,
        .get_short_name       = get_short_name,
        .get_long_name        = get_long_name,
        .get_vendor           = get_vendor,
        .get_product          = get_product,
        .get_summary          = get_summary,
        .get_warnings         = get_warnings,
        .get_notices          = get_notices,
};

/********************* Volumes *********************/

static gboolean 
get_provider_matches_vol (GdmDevice *device)
{
        return gdm_device_test_capability (device, "volume");
}

static char *
get_icon_name_vol (GdmDevice *device)
{
        char *s;
        const char *drive_type;

        s = NULL;
        drive_type = gdm_device_get_property_string (device, "storage.drive_type");
        if (drive_type == NULL)
                goto out;

        if (g_ascii_strcasecmp (drive_type, "cdrom") == 0) {
                s = g_strdup ("media-optical");
        } else if (g_ascii_strcasecmp (drive_type, "floppy") == 0) {
                s = g_strdup ("media-floppy");
        } else if (g_ascii_strcasecmp (drive_type, "disk") == 0) {
                s = g_strdup ("drive-harddisk");
        } else if (g_ascii_strcasecmp (drive_type, "tape") == 0) {
                s = g_strdup ("media-tape");
        } else if (g_ascii_strcasecmp (drive_type, "compact_flash") == 0) {
                s = g_strdup ("media-flash-compact");
        } else if (g_ascii_strcasecmp (drive_type, "memory_stick") == 0) {
                s = g_strdup ("media-flash-memory-stick");
        } else if (g_ascii_strcasecmp (drive_type, "smart_media") == 0) {
                s = g_strdup ("media-flash-smart-media");
        } else if (g_ascii_strcasecmp (drive_type, "sd_mmc") == 0) {
                s = g_strdup ("media-flash-sd");
        } else if (g_ascii_strcasecmp (drive_type, "zip") == 0) {
                s = g_strdup ("media-zip");
        } else if (g_ascii_strcasecmp (drive_type, "jaz") == 0) {
                s = g_strdup ("media-jaz");
        } else if (g_ascii_strcasecmp (drive_type, "flashkey") == 0) {
                s = g_strdup ("media-flash-key");
        } 
out:
        if (s == NULL)
                s = g_strdup ("drive-harddisk");
        return s;
}

static char *
get_short_name_vol (GdmDevice *device)
{
        guint64 size;
        char *s;
        char *p;

        size = gdm_device_get_property_uint64 (device, "volume.size");

        p = gdm_util_get_size_for_display (size, FALSE);
        s = g_strdup_printf (_("%s Volume"), p);
        g_free (p);
        return s;
}

static char *
get_long_name_vol (GdmDevice *device)
{
        const char *fs_type;
        const char *fs_usage;
        const char *fs_version;
        guint64 size;
        char *s;
        char *p;
        char *q;

        q = NULL;
        fs_usage = gdm_device_get_property_string (device, "volume.fsusage");
        if (fs_usage != NULL && g_ascii_strcasecmp (fs_usage, "filesystem") == 0) {
                fs_type = gdm_device_get_property_string (device, "volume.fstype");
                fs_version = gdm_device_get_property_string (device, "volume.fsversion");
                q = gdm_util_get_fstype_for_display (fs_type, fs_version, FALSE);
        }

        size = gdm_device_get_property_uint64 (device, "volume.size");

        p = gdm_util_get_size_for_display (size, FALSE);
        if (q != NULL) {
                s = g_strdup_printf (_("%s %s Volume"), p, q);
        } else {
                s = g_strdup_printf (_("%s Volume"), p);
        }
        g_free (p);
        g_free (q);
        return s;
}

static char *
get_vendor_vol (GdmDevice *device)
{
        return NULL;
}

static char *
get_product_vol (GdmDevice *device)
{
        return NULL;
}

static const char *
gpt_part_type_guid_to_string (const char *guid)
{
        int n;
        /* see also http://en.wikipedia.org/wiki/GUID_Partition_Table */
        static struct {
                const char *guid;
                char *name;
        } part_type[] = {
                {"024DEE41-33E7-11D3-9D69-0008C781F39F", N_("MBR Partition Scheme")},
                {"C12A7328-F81F-11D2-BA4B-00A0C93EC93B", N_("EFI System Partition")},
                /* Microsoft */
                {"E3C9E316-0B5C-4DB8-817D-F92DF00215AE", N_("Microsoft Reserved Partition")},
                {"EBD0A0A2-B9E5-4433-87C0-68B6B72699C7", N_("Basic Data Partition")},
                {"5808C8AA-7E8F-42E0-85D2-E1E90434CFB3", N_("LDM meta data Partition")},
                {"AF9B60A0-1431-4F62-BC68-3311714A69AD", N_("LDM data Partition")},
                /* Linux */
                {"EBD0A0A2-B9E5-4433-87C0-68B6B72699C7", N_("Basic Data Partition")}, /* Same GUID as MS! */
                {"A19D880F-05FC-4D3B-A006-743F0F84911E", N_("Linux RAID Partition")},
                {"0657FD6D-A4AB-43C4-84E5-0933C84B4F4F", N_("Linux Swap Partition")},
                {"E6D6D379-F507-44C2-A23C-238F2A3DF928", N_("Linux LVM Partition")},
                {"8DA63339-0007-60C0-C436-083AC8230908", N_("Linux Reserved Partition")},
                /* Mac OS X */
                {"48465300-0000-11AA-AA11-00306543ECAC", N_("Apple HFS/HFS+ Partition")},
                {"55465300-0000-11AA-AA11-00306543ECAC", N_("Apple UFS Partition")},
                {"52414944-0000-11AA-AA11-00306543ECAC", N_("Apple RAID Partition")},
                /* TODO: add more entries */
                {NULL,  NULL}
        };

        for (n = 0; part_type[n].name != NULL; n++) {
                if (g_ascii_strcasecmp (part_type[n].guid, guid) == 0)
                        return part_type[n].name;
        }

        return guid;
}

static const char *
apm_part_type_to_string (const char *type)
{
        int n;
        /* see also http://developer.apple.com/documentation/mac/Devices/Devices-126.html
         * and http://lists.apple.com/archives/Darwin-drivers/2003/May/msg00021.html */
        static struct {
                const char *type;
                char *name;
        } part_type[] = {
                {"Apple_Unix_SVR2", N_("Apple UFS Partition")},
                {"Apple_HFS", N_("Apple HFS/HFS+ Partition")},
                {"Apple_partition_map", N_("Apple Partition Map")},
                {"DOS_FAT_12", N_("FAT 12")},
                {"DOS_FAT_16", N_("FAT 16")},
                {"DOS_FAT_32", N_("FAT 32")},
                {"Windows_FAT_16", N_("FAT 16")},
                {"Windows_FAT_32", N_("FAT 32")},
                /* TODO: add more entries */
                {NULL,  NULL}
        };

        for (n = 0; part_type[n].name != NULL; n++) {
                if (g_ascii_strcasecmp (part_type[n].type, type) == 0)
                        return part_type[n].name;
        }

        return type;
}

static const char *
msdos_part_type_to_string (int msdos_type)
{
        int n;
        /* see also http://www.win.tue.nl/~aeb/partitions/partition_types-1.html */
        static struct {
                int type;
                char *name;
        } part_type[] = {
                {0x00,  N_("Empty")},
                {0x01,  N_("FAT12")},
                {0x04,  N_("FAT16 <32M")},
                {0x05,  N_("Extended")},
                {0x06,  N_("FAT16")},
                {0x07,  N_("HPFS/NTFS")},
                {0x0b,  N_("W95 FAT32")},
                {0x0c,  N_("W95 FAT32 (LBA)")},
                {0x0e,  N_("W95 FAT16 (LBA)")},
                {0x0f,  N_("W95 Ext d (LBA)")},
                {0x10,  N_("OPUS")},
                {0x11,  N_("Hidden FAT12")},
                {0x12,  N_("Compaq diagnostics")},
                {0x14,  N_("Hidden FAT16 <32M")},
                {0x16,  N_("Hidden FAT16")},
                {0x17,  N_("Hidden HPFS/NTFS")},
                {0x1b,  N_("Hidden W95 FAT32")},
                {0x1c,  N_("Hidden W95 FAT32 (LBA)")},
                {0x1e,  N_("Hidden W95 FAT16 (LBA)")},
                {0x3c,  N_("PartitionMagic")},
                {0x82,  N_("Linux swap")},
                {0x83,  N_("Linux")},
                {0x84,  N_("Hibernation")},
                {0x85,  N_("Linux Extended")},
                {0x8e,  N_("Linux LVM")},
                {0xa0,  N_("Hibernation")},
                {0xa5,  N_("FreeBSD")},
                {0xa6,  N_("OpenBSD")},
                {0xa8,  N_("Mac OS X")},
                {0xaf,  N_("Mac OS X")},
                {0xbe,  N_("Solaris boot")},
                {0xbf,  N_("Solaris")},
                {0xeb,  N_("BeOS BFS")},
                {0xec,  N_("SkyOS SkyFS")},
                {0xee,  N_("EFI GPT")},
                {0xef,  N_("EFI (FAT-12/16/32")},
                {0xfd,  N_("Linux RAID autodetect")},
                {0x00,  NULL}
        };

        for (n = 0; part_type[n].name != NULL; n++) {
                if (part_type[n].type == msdos_type)
                        return part_type[n].name;
        }

        return _("Unknown");
}

static gint64
gdm_util_get_free_space (const char *path)
{
        struct statvfs buf;
        gint64 res;

        res = -1;

        if (statvfs (path, &buf) != -1) {
                res = ((gint64) buf.f_bsize) * ((gint64) buf.f_bfree);
        }

        return res;
}

static GSList *
get_summary_vol (GdmDevice *device)
{
        char *s;
        GSList *p = NULL;
        const char *fs_type;
        const char *fs_usage;
        const char *fs_version;
        gboolean is_part;
        const char *part_type;
        const char *part_scheme;
        int part_number;
        const char *label;
        const char *uuid;
        guint64 size;
        gint64 free_space;
        const char *device_file;
        const char *mount_point;

        fs_type = gdm_device_get_property_string (device, "volume.fstype");
        fs_usage = gdm_device_get_property_string (device, "volume.fsusage");
        fs_version = gdm_device_get_property_string (device, "volume.fsversion");
        is_part = gdm_device_get_property_bool (device, "volume.is_partition");
        part_type = gdm_device_get_property_string (device, "volume.partition.type");
        part_scheme = gdm_device_get_property_string (device, "volume.partition.scheme");
        part_number = gdm_device_get_property_int (device, "volume.partition.number");
        label = gdm_device_get_property_string (device, "volume.label");
        uuid = gdm_device_get_property_string (device, "volume.uuid");
        size = gdm_device_get_property_uint64 (device, "volume.size");
        device_file = gdm_device_get_property_string (device, "block.device");
        mount_point = gdm_device_get_property_string (device, "volume.mount_point");

        s = NULL;
        if (fs_usage != NULL) {
                if (g_ascii_strcasecmp (fs_usage, "filesystem") == 0) {
                        s = g_strdup (_("File System"));
                } else if (g_ascii_strcasecmp (fs_usage, "partitiontable") == 0) {
                        s = g_strdup (_("Partition Table"));
                } else if (g_ascii_strcasecmp (fs_usage, "raid") == 0) {
                        s = g_strdup (_("LVM / RAID member"));
                } else if (g_ascii_strcasecmp (fs_usage, "crypto") == 0) {
                        s = g_strdup (_("Encrypted Data"));
                } else if (g_ascii_strcasecmp (fs_usage, "other") == 0) {
                        if (fs_type != NULL) {
                                if (g_ascii_strcasecmp (fs_type, "swap") == 0) {
                                        s = g_strdup (_("Swap Space"));
                                }
                        }
                }
        }
        if (s == NULL) {
                s = g_strdup (_("Unknown"));
        }
        ADD_SUM (p, _("Usage"), s);
        ADD_SUM (p, _("Format"), gdm_util_get_fstype_for_display (fs_type, fs_version, TRUE));

        ADD_SUM (p, _("Capacity"), gdm_util_get_size_for_display (size, TRUE));

        free_space = gdm_util_get_free_space (mount_point);
        if (free_space >= 0 ) {
                ADD_SUM (p, _("Available"), gdm_util_get_size_for_display (free_space, TRUE));
        } else {
                ADD_SUM (p, _("Available"), g_strdup (_("-")));
        }

        ADD_SUM (p, _("Device File"), g_strdup (device_file));

        if (mount_point != NULL && mount_point[0] != '\0') {
                s = g_strdup (mount_point);
        } else {
                s = g_strdup (_("-"));
        }
        ADD_SUM (p, _("Mount Point"), s);

        if (is_part) {
                ADD_SUM (p, _("Partition Number"), g_strdup_printf ("%d", part_number));
                if (part_type != NULL) {
                        if (g_ascii_strcasecmp (part_scheme, "gpt") == 0) {
                                s = g_strdup (gpt_part_type_guid_to_string (part_type));
                        } else if (g_ascii_strcasecmp (part_scheme, "mbr") == 0 ||
                                   g_ascii_strcasecmp (part_scheme, "embr") == 0) {
                                int msdos_type;
                                msdos_type = g_ascii_strtoll (part_type, NULL, 0);
                                s = g_strdup_printf (_("%s (0x%02x)"), 
                                                     msdos_part_type_to_string (msdos_type),
                                                     msdos_type);
                        } else if (g_ascii_strcasecmp (part_scheme, "apm") == 0) {
                                s = g_strdup (apm_part_type_to_string (part_type));
                        } else {
                                s = g_strdup (part_type);
                        }
                } else {
                        s = g_strdup (_("-"));
                }

                ADD_SUM (p, _("Partition Type"), s);
        } else {
                s = _("-");
                ADD_SUM (p, _("Partition Number"), g_strdup (s));
                ADD_SUM (p, _("Partition Type"), g_strdup (s));
        }

        if (label != NULL && label[0] != '\0') {
                s = g_strdup (label);
        } else {
                s = g_strdup (_("-"));
        }
        ADD_SUM (p, _("Volume Label"), s);

        if (uuid != NULL && uuid[0] != '\0') {
                s = g_strdup (uuid);
        } else {
                s = g_strdup (_("-"));
        }
        ADD_SUM (p, _("Volume UUID"), s);

        return p;
}

GdmInfoProviderIface gdm_info_provider_volume =
{
        .get_provider_matches = get_provider_matches_vol,
        .get_icon_name        = get_icon_name_vol,
        .get_short_name       = get_short_name_vol,
        .get_long_name        = get_long_name_vol,
        .get_vendor           = get_vendor_vol,
        .get_product          = get_product_vol,
        .get_summary          = get_summary_vol,
};
