/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Nexus
 *
 * pr_tcp.c		- TCP/IP protocol module
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-packages/nexus/source/nexus/pr_tcp.c,v 1.198 2007/05/17 16:44:47 bester Exp $";

#include "internal.h"

#include <sys/uio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#if defined(HAVE_NETINET_TCP)
#   include <netinet/tcp.h>
#endif
#include <netdb.h>
#include <assert.h>
#include "globus_io.h"

#ifdef BUILD_RESOURCE
#include "nexus_resource.h"
#endif 

#if defined(TARGET_ARCH_AIX)
#define netlen_t size_t
#else
#define netlen_t int
#endif

/* ------------------------------------------------------------------------- */
/* end only build the rest if HAVE_TCP_PROTO */
/* ------------------------------------------------------------------------- */
#if  defined(HAVE_TCP_PROTO)

/*
 * NEXUS_TCP_DIRECT_CUSTOM_MAX_SIZE
 */
#ifndef NEXUS_TCP_DIRECT_CUSTOM_MIN_SIZE
#define NEXUS_TCP_DIRECT_CUSTOM_MIN_SIZE 2001
#endif

/*
 * GLOBUS_L_NEXUS_TCP_PROTOCOL_VERSION
 *
 * The version of this protocol module's wire protocol.  If you change
 * this module wire protocol, bump this version number.
 */
#define GLOBUS_L_NEXUS_TCP_PROTOCOL_VERSION (0 + GLOBUS_I_NEXUS_BUFFER_VERSION)

/*
 * GLOBUS_L_NEXUS_TCP_MI_PROTO_VERSION
 *
 * The version of this protocol module's mi_proto.  If you change
 * the contents of this module's mi_proto, bump this version number.
 */
#define GLOBUS_L_NEXUS_TCP_MI_PROTO_VERSION 0

/*
 * Only one thread is allowed to be in the tcp code (and thus
 * mucking with data structures) at a time.
 */

static nexus_mutex_t	          tcp_mutex;
#define tcp_enter()	          globus_mutex_lock(&tcp_mutex);
#define tcp_exit()	          globus_mutex_unlock(&tcp_mutex);

static globus_mutex_t	          outgoing_mutex;
static globus_mutex_t	          incoming_mutex;
static globus_cond_t              outgoing_cond;
static globus_cond_t              incoming_cond;

static nexus_bool_t	tcp_done;

#define tcp_fatal tcp_exit(); globus_fatal

/*
 * Other useful defines
 */
#define TCP_INCOMING_DEFAULT_SIZE 4096

#define CLOSE_NORMAL_FLAG	128

static nexus_byte_t close_message[] =
{
    GLOBUS_L_NEXUS_TCP_PROTOCOL_VERSION,
    CLOSE_NORMAL_FLAG
};

/*
 * Some forward typedef declarations...
 */
typedef struct _tcp_outgoing_t	tcp_outgoing_t;
typedef struct _tcp_incoming_t	tcp_incoming_t;

/*
 * tcp_outgoing_t
 *
 * This is an overload of nexus_proto_t.  It adds the
 * tcp specific information to that structure.
 */
struct _tcp_outgoing_t
{
    nexus_proto_type_t		type;
    nexus_proto_funcs_t *	funcs;
    int				version;
    unsigned long		direct_custom_min_size;
    unsigned long		direct_custom_max_size;
    unsigned long		direct_pointer_min_size;
    unsigned long		direct_pointer_max_size;
    nexus_bool_t		can_use_iovec;
    unsigned long		reserved_header_size;
    /**** end globus_nexus_proto_t *********/
    struct _tcp_outgoing_t *	next;
    char *			host;
    unsigned short		port;
    globus_io_handle_t          handle;
    int				state;
    int				new_state;
    nexus_bool_t		fault_code;
    int				reference_count;
    nexus_bool_t		write_in_progress;
    struct globus_nexus_buffer_s *	write_q_head;
    struct globus_nexus_buffer_s *	write_q_tail;
    char			read_buf[sizeof(close_message)];
#ifdef BUILD_RESOURCE
    nexus_resource_name_t	resource_name;
#endif /* BUILD_RESOURCE */
};


enum
{
    OUTGOING_STATE_UNOPENED		= 0,
    OUTGOING_STATE_OPEN			= 1,
    OUTGOING_STATE_CLOSE_PENDING	= 2,
    OUTGOING_STATE_CLOSE_POSTED		= 3,
    OUTGOING_STATE_CLOSING		= 4,
    OUTGOING_STATE_CLOSED		= 5,
    OUTGOING_STATE_WRITE_ERROR		= 6,
    OUTGOING_STATE_FAULT		= 7,
    OUTGOING_STATE_CONNECTING           = 8,
    OUTGOING_STATE_NONE                 = 9
};


/*
 * tcp_incoming_t
 *
 * One of these is allocated for each incoming file descriptor.
 * It stores partial incoming messages until they are complete
 * and can be handled or queued.
 */
struct _tcp_incoming_t
{
    globus_io_handle_t *        handle;
    int				state;
    int				format;
    unsigned long		msg_size;
    unsigned long		nbytes_parsed;
    unsigned long		nbytes_unparsed;
    unsigned long		storage_size;
    nexus_byte_t *		storage;
    nexus_byte_t *		current;
    unsigned long		save_storage_size;
    nexus_byte_t *		save_storage;
    nexus_byte_t *		save_current;
    nexus_bool_t		dispatch_in_progress;
    struct globus_nexus_buffer_s *	dispatch_q_head;
    struct globus_nexus_buffer_s *	dispatch_q_tail;
#ifdef BUILD_RESOURCE
    nexus_resource_name_t	resource_name;
#endif /* BUILD_RESOURCE */
};

enum
{
    INCOMING_STATE_VERSION,
    INCOMING_STATE_FORMAT,
    INCOMING_STATE_MSG_SIZE,
    INCOMING_STATE_BODY,
    INCOMING_STATE_CLOSED,
    INCOMING_STATE_CLOSING
};



static int                    n_outgoing_handles_open = 0;
static int                    n_incoming_handles_open = 0;
static globus_io_handle_t     globus_l_nexus_tcp_listener_handle;
static globus_list_t *        globus_l_nexus_tcp_incoming_list;
static globus_list_t *        globus_l_nexus_tcp_outgoing_list;
/*
 * My local TCP connection information
 */
static unsigned short	tcp_local_port;
static char		tcp_local_host[MAXHOSTNAMELEN];

static int		shutdown_waiting = 0;

/*
 * Outgoing table stuff.
 *
 * The outgoing table is hashed on host/port. The table itself is an
 * array of header structures pointing to a linked list of buckets.
 *
 * This table is used to avoid creating multiple tcp_outgoing_t
 * objects to the same context.  Multiple global pointers to the same
 * context share a tcp_outgoing_t.
 */
typedef struct _outgoing_table_entry_t
{
    tcp_outgoing_t *			outgoing;
    struct _outgoing_table_entry_t *	next;
} outgoing_table_entry_t;

#define OUTGOING_TABLE_SIZE 149

struct _outgoing_table_entry_t	outgoing_table[OUTGOING_TABLE_SIZE];

static void		outgoing_table_init(void);
static int		outgoing_table_hash(char *host, unsigned short port);
static void		outgoing_table_insert(tcp_outgoing_t *outgoing);
static void		outgoing_table_remove(tcp_outgoing_t *outgoing);
static tcp_outgoing_t *	outgoing_table_lookup(char *host, unsigned short port);

/*
 * Some useful queue macros
 */
#define Enqueue(Qhead, Qtail, Item) \
{ \
    if (Qhead) \
    { \
	(Qtail)->next = (Item); \
	(Qtail) = (Item); \
    } \
    else \
    { \
	(Qhead) = (Qtail) = (Item); \
    } \
}

#define Dequeue(Qhead, Qtail, Item) \
{ \
    (Item) = (Qhead); \
    (Qhead) = (Qhead)->next; \
}

#define QueueNotEmpty(Qhead)	(Qhead)


/*
 * Various forward declarations of procedures
 */
static nexus_proto_type_t	tcp_proto_type(void);
static void		        tcp_init(nexus_bool_t * add_to_my_mi_proto);
static void	     	        tcp_shutdown(void);

static int		        tcp_send_rsr(struct globus_nexus_buffer_s *buffer);

static globus_bool_t
tcp_send_rsr_outstanding(globus_nexus_proto_t *nproto);

static void                     tcp_increment_reference_count(nexus_proto_t *nproto);
static nexus_bool_t	        tcp_decrement_reference_count(nexus_proto_t *nproto);

static int                      nexusl_pr_tcp_get_my_mi_proto(
					       nexus_byte_t **array,
					       int *size,
					       void *proto_info,
					       nexus_endpoint_t *endpoint);

static nexus_bool_t	        tcp_construct_from_mi_proto(
				     nexus_proto_t **      proto,
				     nexus_mi_proto_t *    mi_proto,
				     nexus_byte_t *        proto_array,
				     int                   size);

static int		        tcp_test_proto(
                                     nexus_proto_t *        proto);
static int                      tcp_direct_info_size(void);

static void		        shutdown_write_callback(
                                     void *                 arg, 
                                     globus_io_handle_t *   handle, 
	                             globus_result_t        result,
	                             globus_byte_t *        buf,
		                     globus_size_t          nbytes);

static void		        outgoing_register_next_write(
                                     tcp_outgoing_t *       outgoing);
static void		        outgoing_write_callback(
                                     void *                 arg, 
                                     globus_io_handle_t *   handle, 
	                             globus_result_t        result,
	                             globus_byte_t *        buf,
		                     globus_size_t          nbytes);

static void		        outgoing_writev_callback(
				     void *                 arg,
				     globus_io_handle_t *   handle,
				     globus_result_t        result,
				     struct iovec *         iov,
				     globus_size_t          iovcnt,
				     globus_size_t          nbytes);

static void		        outgoing_writev_error_callback(
				     void *                 arg,
				     globus_io_handle_t *   handle,
				     globus_result_t        result,
				     struct iovec *         iov,
				     globus_size_t          iovcnt,
				     globus_size_t          nbytes);
				
static void                     outgoing_read_callback(
                                     void *                     arg,
				     globus_io_handle_t *       handle,	
				     globus_result_t            result,
				     globus_byte_t *            buf,
				     globus_size_t              nbytes);

static tcp_outgoing_t *	        outgoing_construct(
				     char *                     host,
				     unsigned short             port);

static void		        outgoing_free(
                                     tcp_outgoing_t *           outgoing);
static void		        outgoing_open(
				     tcp_outgoing_t *           outgoing);
static void		        outgoing_close(
				     tcp_outgoing_t *           outgoing,
				     int                        new_state);
static void                     globus_l_outgoing_close_callback(
                                    void *                      arg, 
                                    globus_io_handle_t *        handle,
                                    globus_result_t             result);

static void		        outgoing_close_normal(
				     tcp_outgoing_t *           outgoing);

static tcp_incoming_t *         incoming_construct(
                                     globus_io_handle_t *       handle);

static void		        incoming_close(
                                     tcp_incoming_t *           incoming);
static void		        incoming_free(
                                     tcp_incoming_t *           incoming);

static void		incoming_read_callback(
	                             void *                     arg,
                                     globus_io_handle_t *       handle,
                                     globus_result_t            result,
                                     globus_byte_t *            buf,
                                     globus_size_t              nbytes);

static void		incoming_read_error_callback(
	                             void *                     arg,
                                     globus_io_handle_t *       handle,
                                     globus_result_t            result,
                                     globus_byte_t *            buf,
                                     globus_size_t              nbytes);

static void		accept_internal_connection(
			      void *                 arg, 
			      globus_io_handle_t *   handle,
			      globus_result_t        result);

static void             internal_connection_requested(
                               void *                                    arg, 
                               globus_io_handle_t *                      handle,
                               globus_result_t                           result);


static void             globus_l_connect_callback(
                               void *                                    arg, 
                               globus_io_handle_t *                      handle,
                               globus_result_t                           result);


static void             globus_l_incoming_close_callback(
                               void *                                 arg,
                               globus_io_handle_t *                   handle,
                               globus_result_t                        result);


int globus_i_nexus_get_errno(globus_result_t *    result)
{
    globus_object_t *err;
    int save_errno = -1;

    err = globus_error_get(*result);

    if(globus_object_type_match(globus_object_get_type(err),
				        GLOBUS_IO_ERROR_TYPE_SYSTEM_FAILURE))
    {
        save_errno = globus_io_error_system_failure_get_save_errno(err);
    }
    else if(globus_object_type_match(globus_object_get_type(err),
				        GLOBUS_IO_ERROR_TYPE_EOF))
    {
        save_errno = 0;
    }
    *result = globus_error_put(err);

    return save_errno;
}



#ifdef BUILD_RESOURCE

static nexus_resource_name_t *tcp_get_resource_name_sp( nexus_startpoint_t *);
static nexus_resource_name_t *tcp_get_resource_name_ep( nexus_endpoint_t * );
static unsigned int tcp_lookup_local_addr();

/*
 * Endpoint Resources
 *
 * Since currently all endpoint share the same listener socket, there
 * is a sinlge resource for all endpoints.  This can be expanded
 * when/if we create a unique listeners for each endpoint. 
 */

nexus_resource_name_t tcp_listener_name;

#endif


#endif
/* ------------------------------------------------------------------------- */
/* end only build the rest if HAVE_TCP_PROTO */
/* ------------------------------------------------------------------------- */

static nexus_proto_type_t	tcp_proto_type(void);

/*
 *  start build always
 */
static   int                          globus_l_tcp_proto_count = 1;


static globus_bool_t    tcp_startpoint_proto_match(
					  globus_nexus_mi_proto_t *   mi_proto0,
					  int                         offset0,
					  nexus_byte_t *              subarray0,
					  int                         sub_length0,
					  globus_nexus_mi_proto_t *   mi_proto1,
					  int                         offset1,
					  nexus_byte_t *              subarray1,
					  int                         sub_length1);

static int              tcp_proto_count(void);


#if   defined(HAVE_TCP_PROTO)
static nexus_proto_funcs_t tcp_proto_funcs =
{
    tcp_proto_type,
    tcp_init,
    tcp_shutdown,
    tcp_increment_reference_count,
    tcp_decrement_reference_count,
    nexusl_pr_tcp_get_my_mi_proto,
    tcp_construct_from_mi_proto,
    NULL /* tcp_destroy_my_mi_proto */,
    tcp_test_proto,
    tcp_send_rsr,
    tcp_send_rsr_outstanding,
    tcp_direct_info_size,
    NULL /* tcp_direct_get */,
    tcp_startpoint_proto_match,
    tcp_proto_count,
#ifdef BUILD_RESOURCE
    tcp_get_resource_name_sp,
    tcp_get_resource_name_ep,
#endif
};

#else

/* if HAVE_TCP_PROTO doesn't exist only build coupla functions */
static nexus_proto_funcs_t tcp_proto_funcs =
{
    tcp_proto_type,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    tcp_startpoint_proto_match,
    tcp_proto_count,
#ifdef BUILD_RESOURCE
    GLOBUS_NULL,
    GLOBUS_NULL,
#endif
};

#endif /* end have HAVE_MPI_PROTO */

int 
tcp_l_getenv_proto_count()
{
    char *         tmp_ptr;
    int            cnt;

    tmp_ptr = globus_libc_getenv("GLOBUS_NEXUS_TCP_USE_LAN_WAN");
    if(tmp_ptr == GLOBUS_NULL)
    {
	cnt = 1;
    }
    else
    {
        cnt = 2;
    }

    return cnt;
}
/* build always */
globus_bool_t
tcp_startpoint_proto_match(globus_nexus_mi_proto_t * mi_proto0,
			   int                       offset0,
			   nexus_byte_t * subarray0,
		           int             sub_length0,
			   globus_nexus_mi_proto_t * mi_proto1,
			   int                       offset1,
			   nexus_byte_t * subarray1,
			   int             sub_length1)
{
   int           i = 0;
   int           tmp_int;
   int           version;
   char *        hostname0;
   char *        hostname1;


  UnpackMIProtoHeader((mi_proto0->array),
		      i,
		      tmp_int,
		      hostname0,
		      version);
		      i = 0;
  UnpackMIProtoHeader((mi_proto1->array),
		      i,
		      tmp_int,
		      hostname1,
		      version);

    if(offset0 != offset1)
    {
	return GLOBUS_FALSE;
    }
    else if(offset0 == 0 && tcp_proto_count() > 1)
    {
        return globus_i_is_local_area_network(hostname0, hostname1);
    }
    else
    {
        return GLOBUS_TRUE;
    }
}

int
tcp_proto_count(void)
{
   if(globus_l_tcp_proto_count == -1)
   {
       globus_l_tcp_proto_count = tcp_l_getenv_proto_count();
   }
   return globus_l_tcp_proto_count;
}

/*
 * _nx_pr_tcp_info()
 *
 * Return the nexus_proto_funcs_t function table for this protocol module.
 *
 * This procedure is used for bootstrapping the protocol module.
 * The higher level Nexus code needs to call this routine to
 * retrieve the functions it needs to use this protocol module.
 */
void *_nx_pr_tcp_info(void)
{
    return((void *) (&tcp_proto_funcs));
} /* _nx_pr_tcp_info() */

/*
 * tcp_proto_type()
 *
 * Return the nexus_proto_type_t for this protocol module.
 */
static nexus_proto_type_t tcp_proto_type(void)
{
    return (NEXUS_PROTO_TYPE_TCP);
} /* tcp_proto_type() */


/* ------------------------------------------------------------------------- */
/* only build the rest if HAVE_TCP_PROTO */
/* ------------------------------------------------------------------------- */
#if defined(HAVE_TCP_PROTO)

/*
 * tcp_init()
 *
 * Initialize the TCP protocol.
 */
static void tcp_init(nexus_bool_t * add_to_my_mi_proto)
{
    globus_result_t          res;
    globus_io_attr_t         tcp_attr;

    globus_mutex_init(&tcp_mutex, (globus_mutexattr_t *) GLOBUS_NULL);
    globus_mutex_init(&outgoing_mutex, (globus_mutexattr_t *) GLOBUS_NULL);
    globus_mutex_init(&incoming_mutex, (globus_mutexattr_t *) GLOBUS_NULL);
    globus_cond_init(&incoming_cond, (globus_condattr_t *) GLOBUS_NULL);
    globus_cond_init(&outgoing_cond, (globus_condattr_t *) GLOBUS_NULL);

    outgoing_table_init();
    tcp_done = NEXUS_FALSE;

    globus_l_nexus_tcp_incoming_list = GLOBUS_NULL;
    globus_l_nexus_tcp_outgoing_list = GLOBUS_NULL;
    n_outgoing_handles_open = 0;
    n_incoming_handles_open = 0;

    /*
     * Set up the listener for this process.
     */
    tcp_local_port = 0;

    globus_io_tcpattr_init(&tcp_attr);
    globus_io_attr_set_socket_reuseaddr(
        &tcp_attr,
        GLOBUS_TRUE);

    if (globus_module_getenv("GLOBUS_NEXUS_NO_GSI") == NULL)
    {
        res = globus_io_attr_set_secure_authentication_mode(
            &tcp_attr,
            GLOBUS_IO_SECURE_AUTHENTICATION_MODE_GSSAPI,
            GSS_C_NO_CREDENTIAL);
        if (res != GLOBUS_SUCCESS)
        {
            globus_fatal("tcp_init(): globus_io_tcp_create_listener() failed\n");
        }
        res = globus_io_attr_set_secure_authorization_mode(
            &tcp_attr,
            GLOBUS_IO_SECURE_AUTHORIZATION_MODE_SELF,
            NULL);
        if (res != GLOBUS_SUCCESS)
        {
            globus_fatal("tcp_init(): globus_io_tcp_create_listener() failed\n");
        }
    }

#ifndef BUILD_LITE        
    globus_io_attr_set_callback_space(&tcp_attr, _nx_serial_space);
#endif

    res = globus_io_tcp_create_listener(
                      &tcp_local_port,
		      -1,
		      &tcp_attr,
                      &globus_l_nexus_tcp_listener_handle);
    if (res != GLOBUS_SUCCESS)
    {
	globus_fatal("tcp_init(): globus_io_tcp_create_listener() failed\n");
    }
    globus_io_tcpattr_destroy(&tcp_attr);
    res = globus_io_tcp_register_listen(
		  &globus_l_nexus_tcp_listener_handle,
		  internal_connection_requested,
		  GLOBUS_NULL);
    if (res != GLOBUS_SUCCESS)
    {
	globus_fatal("tcp_init(): globus_io_tcp_register_listen() failed\n");
    }
    globus_libc_gethostname(tcp_local_host, MAXHOSTNAMELEN);
    nexus_debug_printf(3, ("tcp_init(): Listening on %d\n", tcp_local_port));


#ifdef BUILD_RESOURCE
	{
	  nexus_rusage_socket tcp_listener_socket;

	  nexus_resource_name(&tcp_listener_name, RESOURCE_SOCKET);

	  tcp_listener_socket.who = &tcp_listener_name;
	  tcp_listener_socket.pid = globus_libc_getpid();
	  tcp_listener_socket.fd = -1; /* can't get this from here */
	  tcp_listener_socket.mode = RESOURCE_SOCKET_LISTEN;

	  /* this is a serious kludge until we can pull the fd out of
	     the create listener call */
	  tcp_listener_socket.addr.sin_addr.s_addr = tcp_lookup_local_addr();

	  tcp_listener_socket.addr.sin_family = AF_INET;
	  tcp_listener_socket.addr.sin_port = htons(tcp_local_port);
	  tcp_listener_socket.protocol = IPPROTO_TCP;
	  tcp_listener_socket.op = RUSAGE_CREATE;

	  nexus_overseer_rusage((nexus_rusage_obj*)&tcp_listener_socket);
	}
#endif /* BUILD_RESOURCE */

    *add_to_my_mi_proto = NEXUS_TRUE;
} /* tcp_init() */


/*
 * tcp_shutdown()
 *
 * This routine is called during normal shutdown of a process.
 *
 * Shutdown the tcp protocol module by closing all off of the
 * outgoing's that have open fd's.  This will result in an orderly
 * shutdown of fd's, so that other nodes will not detect an
 * abnormal eof on the other ends of these fd's.
 *
 * It is ok if this procedure takes a little while to complete -- for
 * example to wait for another process to read off its fd and
 * free up some socket buffer space.
 *
 * An orderly closing of a connection is done as follows:
 *   1) Send the CLOSE_NORMAL_FLAG message down the fd (1 byte,
 *      signaling a close due to a normal termination) down the fd
 *	
 *   2) Then close the fd and its associated outgoing.
 */
static void tcp_shutdown(void)
{
    tcp_outgoing_t *                outgoing;
    tcp_incoming_t *                incoming;
    globus_result_t                 rc;
    globus_list_t *                 tmp_list;

    tcp_enter();

#ifdef BUILD_RESOURCE
    {
      nexus_rusage_socket tcp_listener_socket;

      tcp_listener_socket.who = &tcp_listener_name;
      tcp_listener_socket.op = RUSAGE_DESTROY;
      tcp_listener_socket.pid = globus_libc_getpid();
      tcp_listener_socket.fd = -1;
      tcp_listener_socket.mode = RESOURCE_SOCKET_LISTEN;

      /* this is a serious kludge until we can pull the fd out of
	 the create listener call */
      tcp_listener_socket.addr.sin_addr.s_addr = tcp_lookup_local_addr();

      tcp_listener_socket.addr.sin_family = AF_INET;
      tcp_listener_socket.addr.sin_port = htons(tcp_local_port);
      tcp_listener_socket.protocol = IPPROTO_TCP;

      nexus_overseer_rusage((nexus_rusage_obj*)&tcp_listener_socket);
    }
#endif /* BUILD_RESOURCE */
    
    tcp_done = NEXUS_TRUE;
    
    globus_mutex_lock(&outgoing_mutex);
    tmp_list = globus_l_nexus_tcp_outgoing_list;
    while(!globus_list_empty(tmp_list))
    {
	outgoing = (tcp_outgoing_t *)globus_list_first(tmp_list);
        tmp_list = globus_list_rest(tmp_list);

	if (outgoing->state == OUTGOING_STATE_OPEN)
	{
	    nexus_debug_printf(1, ("tcp_shutdown(): closing outgoing 0x%lx\n", 
			       (unsigned long) outgoing));
	    outgoing->state = OUTGOING_STATE_CLOSE_PENDING;
	    if (!outgoing->write_in_progress)
	    {
	        outgoing_register_next_write(outgoing);
	    }
	}
    }
    globus_mutex_unlock(&outgoing_mutex);
  
    globus_mutex_lock(&incoming_mutex);
    tmp_list = globus_l_nexus_tcp_incoming_list;
    while(!globus_list_empty(tmp_list))
    {
	incoming = (tcp_incoming_t *)globus_list_first(tmp_list);
        tmp_list = globus_list_rest(tmp_list);
        
	if(incoming->state != INCOMING_STATE_CLOSING)
	{
            rc = globus_io_register_write(
				  incoming->handle,
				  (globus_byte_t *) (close_message),
				  sizeof(close_message),
				  shutdown_write_callback,
				  (void *) incoming);

            assert(rc == GLOBUS_SUCCESS);
        }
    }
    globus_mutex_unlock(&incoming_mutex);

    tcp_exit();

    globus_mutex_lock(&outgoing_mutex);
    {
        while (n_outgoing_handles_open > 0)
        {
     	    globus_cond_wait(&outgoing_cond,
			     &outgoing_mutex);
        }
    }
    globus_mutex_unlock(&outgoing_mutex);
    
    globus_mutex_lock(&incoming_mutex);
    {
        while (n_incoming_handles_open > 0)
        {
     	    globus_cond_wait(&incoming_cond,
			     &incoming_mutex);
        }
    }
    globus_mutex_unlock(&incoming_mutex);
    
    
    globus_io_cancel(&globus_l_nexus_tcp_listener_handle, GLOBUS_FALSE); 
    globus_io_close(&globus_l_nexus_tcp_listener_handle); 

    globus_mutex_destroy(&tcp_mutex);
    globus_mutex_destroy(&outgoing_mutex);
    globus_mutex_destroy(&incoming_mutex);

    nexus_debug_printf(1, ("tcp_shutdown(): done\n"));
} /* tcp_shutdown() */


/*
 * shutdown_write_callback()
 */
static void 
shutdown_write_callback(
      void *                        arg,
      globus_io_handle_t *          handle,
      globus_result_t               result,
      globus_byte_t *               buf,
      globus_size_t                 nbytes)
{
    tcp_incoming_t *               incoming;

    incoming = (tcp_incoming_t *)arg;

    nexus_debug_printf(1, ("shutdown_write_callback(): entering\n"));

    tcp_enter();
    
    if (incoming->state != INCOMING_STATE_CLOSED && 
	incoming->state != INCOMING_STATE_CLOSING)
    {
	incoming_close(incoming);
    }
    /*
     * else, the fd must have been closed between the tcp_shutdown()
     * and the invocation of this callback
     */

    tcp_exit();
 
    nexus_debug_printf(1, ("shutdown_write_callback(): exiting\n"));

} /* shutdown_write_callback() */


/*
 * tcp_send_rsr()
 */
static int 
tcp_send_rsr(
    struct globus_nexus_buffer_s *                    buffer)
{
    int                           rc = 0;
    tcp_outgoing_t *              outgoing;

    nexus_debug_printf(2,("tcp_send_rsr(): invoked with buffer: %x\n",buffer));
    
    if (buffer->n_direct > 0)
    {
	globus_mutex_init(&(buffer->barrier.mutex),(nexus_mutexattr_t *) NULL);
	globus_cond_init(&(buffer->barrier.cond), (nexus_condattr_t *) NULL);
	buffer->barrier.count = buffer->n_direct;
	buffer->using_barrier = NEXUS_TRUE;
    }
    
    tcp_enter();
    
    outgoing = (tcp_outgoing_t *) buffer->proto;
    NexusAssert2((outgoing->type == NEXUS_PROTO_TYPE_TCP),
		 ("tcp_send_rsr(): Internal error: proto_type is not NEXUS_PROTO_TYPE_TCP\n"));

    /*
     * Open up the outgoing if it is not already open.
     */
    if(outgoing->state != OUTGOING_STATE_OPEN)
    {
	if(outgoing->state == OUTGOING_STATE_UNOPENED)
	{
            outgoing_open(outgoing);
        }
	if (outgoing->state != OUTGOING_STATE_OPEN &&
	    outgoing->state != OUTGOING_STATE_CONNECTING)
        {
	    rc = outgoing->fault_code;
	    buffer->using_barrier = NEXUS_FALSE;
	    tcp_exit();

	    goto abort_send;
	}
    }

    /* Enqueue this message on the outgoing */
    Enqueue(outgoing->write_q_head, outgoing->write_q_tail, buffer);
    
    if (!outgoing->write_in_progress &&
	outgoing->state != OUTGOING_STATE_CONNECTING)
    {
        /*
         * Nobody else has registered a write on this outgoing,
         * so register the one we just enqueued.
         *
         * If there is already a write in progress,
         * the write callback will take care of registering this
         * write (and any other ones that are enqueued) when
         * the current write completes.
         */

        outgoing_register_next_write(outgoing);
    }
    
    tcp_exit();

abort_send:
    return (rc);
} /* tcp_send_rsr() */


/*
 * tcp_send_rsr_outstanding()
 *
 * Return true if there are any sends outstanding for this proto,
 * otherwise false.
 */
static globus_bool_t
tcp_send_rsr_outstanding(globus_nexus_proto_t *nproto)
{
    globus_bool_t rc = GLOBUS_FALSE;
    tcp_outgoing_t *outgoing = (tcp_outgoing_t *) nproto;
    tcp_enter();
    if (outgoing->write_q_head)
    {
	rc = GLOBUS_TRUE;
    }
    tcp_exit();
    return(rc);
} /* tcp_send_rsr_outstanding() */


/*
 * outgoing_register_next_write()
 *
 * Register the next write operation for this outgoing.
 *
 * If the buffer at the head of the write_q list has remaining
 * direct componenents, then the next one is registered.
 * Otherwise, we go on to the next buffer in the
 * write_q (if there is one), and register it.
 *
 * If the write is complete on the buffer at the head of the write_q,
 * then this function destroys it.
 *
 * This function assumes that tcp_enter() has already been called.
 *
 * This function should not be called if 
 * outgoing->write_in_progres==NEXUS_TRUE.
 */
static void outgoing_register_next_write(tcp_outgoing_t *outgoing)
{
    struct globus_nexus_buffer_s *          buffer;
    struct globus_nexus_buffer_s *          completed_buffer = 
					      (struct globus_nexus_buffer_s *) NULL;
    nexus_bool_t                            done;
    globus_result_t                         res;

    outgoing->write_in_progress = NEXUS_FALSE;
   
    for (done = NEXUS_FALSE; !done; )
    {
	buffer = outgoing->write_q_head;
	if (!buffer)
	{
	    done = NEXUS_TRUE;
	}
	else if (buffer->current_base_segment)
	{
	    /*
	     * The first buffer in the queue as not been sent yet.
	     * So send its base segment.
	     */

	    nexus_debug_printf(
	          1, 
	          ("outgoing_register_next_write(): begin sending buffer base segment 0x%lx\n", 
		  (unsigned long) buffer));

	    /* Mark the base segment as written */
	    buffer->current_base_segment = (nexus_base_segment_t *) NULL;
	    outgoing->write_in_progress = NEXUS_TRUE;

	    /* Register the write on the base segment */
	    if (buffer->iovec_formatted)
	    {
		struct iovec *iov
		    = (struct iovec *) buffer->base_segments->current;
		size_t iovcnt = (int) buffer->base_segments->size_used;

		res = globus_io_register_writev(
					      &outgoing->handle,
					      iov,
					      iovcnt,
					      outgoing_writev_callback,
					      (void *) outgoing);
		/* TODO: Check the return code */
		    assert(res == GLOBUS_SUCCESS);
	    }
	    else
	    {
		nexus_byte_t *buf = buffer->base_segments->current;
                globus_size_t size = buffer->base_segments->size_used;
		
                if(size < _nx_tcp_mim_msg_size)
                {
                    globus_byte_t *            tmpPtr;

                    tmpPtr = &buf[2];
                    size = _nx_tcp_mim_msg_size;
                    nexus_dc_put_u_long(&tmpPtr, &size, 1);
                }

		res = globus_io_register_write(
					      &outgoing->handle,
					      (globus_byte_t *) (buf),
					      size,
					      outgoing_write_callback,
					      (void *) outgoing);
 
       	        assert(res == GLOBUS_SUCCESS);
		/* TODO: Check the return code */
	    }

	    done = NEXUS_TRUE;
	}
	else if (   buffer->direct_segments
		 && (buffer->direct_segments->n_left > 0))
	{
	    /*
	     * There is at least one more direct segment to send.
	     * So register the next direct segment.
	     */
	    nexus_direct_info_t *direct_info;
	
	    nexus_debug_printf(1, 
			       ("outgoing_register_next_write(): begin sending buffer direct segment 0x%lx\n", (unsigned long) buffer));

	    direct_info = buffer->direct_segments->current;
	    buffer->direct_segments->current++;
	    buffer->direct_segments->n_left--;
	    outgoing->write_in_progress = NEXUS_TRUE;
	    
	    /* globus_io add */
	    res = globus_io_register_write(
					  &outgoing->handle,
					  (globus_byte_t *) direct_info->data,
					  direct_info->size,
					  outgoing_write_callback,
					  (void *) outgoing);
	    /* TODO: Check the return code */
	       assert(res == GLOBUS_SUCCESS); 
	    done = NEXUS_TRUE;
	}
	else
	{
	    /*
	     * We are done with this buffer.
	     * So setup the completed buffer to be free below,
	     * and go on to the next buffer in the write_q.
	     */
	    Dequeue(outgoing->write_q_head,
		    outgoing->write_q_tail,
		    completed_buffer);
	    
	    nexus_debug_printf(1, 
		("outgoing_register_next_write(): done sending buffer 0x%lx on fd\n", 
		(unsigned long) completed_buffer)); 
	}
    }

    if (!buffer)
    {
	/*
	 * All of the buffers have been sent for this outgoing.
	 */
	if (outgoing->state == OUTGOING_STATE_CLOSE_PENDING)
	{
	    /*
	     * This outgoing is closing.
	     * So post a write of the CLOSE_NORMAL_FLAG.
	     */
	    outgoing->write_in_progress = NEXUS_TRUE;
	    outgoing->state = OUTGOING_STATE_CLOSE_POSTED;
	    nexus_debug_printf(1, ("outgoing_register_next_write(): registering close\n"));

	    res = globus_io_register_write(
		     &outgoing->handle,
		     (globus_byte_t *) (close_message),
		     sizeof(close_message),
	             outgoing_write_callback,
		     (void *) outgoing);
    assert(res == GLOBUS_SUCCESS);
	    /* TODO: Check the return code */
	}
	else if (outgoing->state == OUTGOING_STATE_CLOSE_POSTED)
	{
	    /*
	     * A CLOSE_NORMAL_FLAG was just successfully sent
	     * on outgoing->fd.
	     * So close this outgoing.
	     */
	    nexus_debug_printf(1, ("outgoing_register_next_write(): close byte sent, closing outgoing\n"));
	    outgoing_close(outgoing, OUTGOING_STATE_CLOSED);
	}
    }

    if (completed_buffer)
    {
	/* Signal the thread waiting in tcp_send_rsr(), if there is one */
	if (completed_buffer->using_barrier)
	{
	    globus_mutex_lock(&(completed_buffer->barrier.mutex));
	    completed_buffer->barrier.count--;
	    globus_cond_signal(&(completed_buffer->barrier.cond));
	    globus_mutex_unlock(&(completed_buffer->barrier.mutex));
	}
	else
	{
	    /* Destroy the buffer */
	    nexus_buffer_destroy(&completed_buffer);
	}
    }
    
} /* outgoing_register_next_write() */


/*
 * outgoing_write_callback()
 */
static void outgoing_write_callback(
	void *                     arg,
        globus_io_handle_t *       handle,
	globus_result_t            result,
	globus_byte_t *            buf,
	globus_size_t              nbytes)
{
    tcp_outgoing_t *outgoing = (tcp_outgoing_t *) arg;
   
   /* set handle user arg to outgoing */

    nexus_debug_printf(1, ("outgoing_write_callback(): entering\n"));
    if(result == GLOBUS_SUCCESS)
    {
        tcp_enter();
        outgoing_register_next_write(outgoing);
        tcp_exit();
    }
    else
    {
        globus_object_t *               err;
        int                             error;
	globus_bool_t                   canceled;

        err = globus_error_get(result);
	canceled= globus_object_type_match(
 	              globus_object_get_type(err),
	              GLOBUS_IO_ERROR_TYPE_IO_CANCELLED);
        result = globus_error_put(err);
        if(!canceled)
        {
       
            nexus_debug_printf(1, ("outgoing_write_error_callback(): entering\n"));

	    if (tcp_done)
	    {
	        tcp_enter();
	        outgoing->fault_code = NEXUS_ERROR_PROCESS_SHUTDOWN_NORMALLY;
	        outgoing_close(outgoing, OUTGOING_STATE_FAULT);
	        tcp_exit();
	        return;
	    }
	
	    error = globus_i_nexus_get_errno(&result);
	    if (error == EPIPE)
	    {
	    /*
              * The outgoing fd was closed unexpectedly.
	      * If the process at the other end died, or
	      * erroneously closed the fd.
	      */
                tcp_enter();
	        outgoing->fault_code = NEXUS_ERROR_PROCESS_DIED;
	        outgoing_close(outgoing, OUTGOING_STATE_FAULT);
	        tcp_exit();
	        if (_nx_fault_detected(outgoing->fault_code) != 0)
	        {
	            globus_fatal("outgoing_write_error_callback(): fd unexpectedly closed. Another process probably died: errno=%d: %s\n", error, globus_libc_system_error_string(error));
	        }
	    }
	    else
	    {
                globus_fatal("outgoing_write_error_callback(): Write failed (errno=%i): %s\n", error, globus_libc_system_error_string(error));
	    }

            nexus_debug_printf(1, ("outgoing_write_error_callback(): exiting\n"));

        }
	else
	{
            nexus_debug_printf(1, ("outgoing_write_error_callback(): canceled\n"));
            outgoing->write_q_head = GLOBUS_NULL;
        }
    }
    nexus_debug_printf(1, ("outgoing_write_callback(): exiting\n"));
} /* outgoing_write_callback() */


/*
 * outgoing_writev_callback()
 */
static void 
outgoing_writev_callback(
    void *                                  arg,
    globus_io_handle_t *                    handle,
    globus_result_t                         result,
    struct iovec *                          iov,
    globus_size_t                           iovcnt,
    globus_size_t                           nbytes)
{
    tcp_outgoing_t *outgoing = (tcp_outgoing_t *) arg;
    
    if(result != GLOBUS_SUCCESS)
    {
        outgoing_writev_error_callback(arg, handle, result, iov, iovcnt, nbytes);
	return;
    }

    tcp_enter();
    outgoing_register_next_write(outgoing);
    tcp_exit();
} /* outgoing_writev_callback() */

/*
 * outgoing_writev_error_callback()
 *
 * TODO: This should try to re-establish the connection and retry the send.
 */
static void		        
outgoing_writev_error_callback(
    void *                                  arg,
    globus_io_handle_t *                    handle,
    globus_result_t                         result,
    struct iovec *                          iov,
    globus_size_t                           iovcnt,
    globus_size_t                           nbytes)
{
    int                             error;    
    globus_object_t *               err;
    globus_bool_t                   canceled;
    tcp_outgoing_t *                outgoing = (tcp_outgoing_t *) arg;

    err = globus_error_get(result);
    canceled= globus_object_type_match(
 	          globus_object_get_type(err),
	          GLOBUS_IO_ERROR_TYPE_IO_CANCELLED);
    result = globus_error_put(err);

    if(canceled)
    {
        return;
    }

    if (tcp_done)
    {
	tcp_enter();
	outgoing->fault_code = NEXUS_ERROR_PROCESS_SHUTDOWN_NORMALLY;
	outgoing_close(outgoing, OUTGOING_STATE_FAULT);
	tcp_exit();
	return;
    }
   
    error = globus_i_nexus_get_errno(&result);
    if (error == EPIPE)
    {
        /*
	 * The outgoing fd was closed unexpectedly.
	 * If the process at the other end died, or
	 * erroneously closed the fd.
	 */
	tcp_enter();
	outgoing->fault_code = NEXUS_ERROR_PROCESS_DIED;
	outgoing_close(outgoing, OUTGOING_STATE_FAULT);
	tcp_exit();
	if (_nx_fault_detected(outgoing->fault_code) != 0)
	{
	    globus_fatal("outgoing_writev_error_callback(): fd unexpectedly closed. Another process probably died: errno=%d: %s\n", error, globus_libc_system_error_string(error));
	}
    }
    else
    {
        globus_fatal("outgoing_writev_error_callback(): Write failed (errno=%i): %s\n", error, globus_libc_system_error_string(error));
    }
    
} /* outgoing_writev_error_callback() */


/*
 * outgoing_read_callback()
 *
 * This function is called if a byte is received on an outgoing fd
 */
static void outgoing_read_callback(
       void *                               arg,
       globus_io_handle_t *                 handle,	
       globus_result_t                      result,
       globus_byte_t *                      buf,
       globus_size_t                        nbytes)
{
    tcp_outgoing_t *                         outgoing;
    int                                      version;
    int                                      flag;

    if(result == GLOBUS_SUCCESS)
    {    
        outgoing = (tcp_outgoing_t *) arg;

        nexus_debug_printf(1, ("outgoing_read_callback(): entering\n"));

        if (nbytes != sizeof(close_message))
        {
  	    tcp_enter();
	    outgoing->fault_code = GLOBUS_NEXUS_ERROR_BAD_PROTOCOL;
	    outgoing_close(outgoing, OUTGOING_STATE_FAULT);
	    tcp_exit();
	    _nx_fault_detected(outgoing->fault_code);
        }
    
	version = *((nexus_byte_t *) buf);
	flag = *(((nexus_byte_t *) (buf)) + 1);

	if (version != GLOBUS_L_NEXUS_TCP_PROTOCOL_VERSION)
	{
	    tcp_enter();
	    outgoing->fault_code = GLOBUS_NEXUS_ERROR_VERSION_MISMATCH;
	    outgoing_close(outgoing, OUTGOING_STATE_FAULT);
	    tcp_exit();
	    if (_nx_fault_detected(outgoing->fault_code) != 0)
	    {
	        globus_fatal("outgoing_read_callback(): Version mismatch\n");
	    }
	}
    
	if (flag == CLOSE_NORMAL_FLAG)
	{
	  /*
	   * The other end closed this fd normally.
	   * So just close this outgoing on proceed normally.
	   */
	  tcp_enter();
	  outgoing->fault_code = GLOBUS_NEXUS_ERROR_PROCESS_SHUTDOWN_NORMALLY;
	  outgoing_close(outgoing, OUTGOING_STATE_FAULT);
	  tcp_exit();
	}
	else
	{
	    tcp_enter();
	    outgoing->fault_code = GLOBUS_NEXUS_ERROR_BAD_PROTOCOL;
	    outgoing_close(outgoing, OUTGOING_STATE_FAULT);
	    tcp_exit();
	    if (_nx_fault_detected(outgoing->fault_code) != 0)
	    {
	        globus_fatal("outgoing_read_callback(): Internal error: Read unexpected data from a outgoing\n");
	    }
	}
    }
    else
    {
	globus_object_t *          err;
	int                        error;
	globus_bool_t              canceled;

        outgoing = (tcp_outgoing_t *)arg;
	nexus_debug_printf(1, ("outgoing_read_error_callback(): entering\n"));

	if (tcp_done)
	{
	    tcp_enter();
	    outgoing->fault_code = NEXUS_ERROR_PROCESS_SHUTDOWN_NORMALLY;
	    outgoing_close(outgoing, OUTGOING_STATE_FAULT);
	    tcp_exit();
	    
	    return;
	}
	 
        err = globus_error_get(result);
	canceled = globus_object_type_match(
		       globus_object_get_type(err),
		       GLOBUS_IO_ERROR_TYPE_IO_CANCELLED);
        result = globus_error_put(err);
        if(!canceled)
        {
	   error = globus_i_nexus_get_errno(&result);

	   if (   (error == 0)
	       || (error == ECONNRESET)
	       || (error == EPIPE) )
	   {
	       tcp_enter();
	       outgoing->fault_code = NEXUS_ERROR_PROCESS_DIED;
	       outgoing_close(outgoing, OUTGOING_STATE_FAULT);
	       tcp_exit();
	       if (_nx_fault_detected(outgoing->fault_code) != 0)
	       {
	       globus_fatal("outgoing_read_error_callback(): fd unexpectedly closed. Another process probably died: errno=%d: %s\n", error, globus_libc_system_error_string(error));
	       }
	   }
	   else
	   {
	       tcp_fatal("outgoing_read_error_callback(): Read failed on outgoing (errno=%d): %s\n", error, globus_libc_system_error_string(error));
	   }
    
	   nexus_debug_printf(1, ("outgoing_read_error_callback(): exiting\n"));
	}    
    }
} /* outgoing_read_callback() */


/*
 * outgoing_construct()
 *
 * Construct a tcp_outgoing_t for the given host and port. Look up in the
 * outgoing table to see if one already exists. If it does, bump its reference
 * count and return that one. Otherwise create one, insert into the
 * table with a reference count of 1 and return it.
 */
static tcp_outgoing_t *
outgoing_construct(
      char *                                host,
      unsigned short                        port)
{
    tcp_outgoing_t *                   outgoing;

    outgoing = outgoing_table_lookup(host, port);
    nexus_debug_printf(
	      3, 
              ("outgoing_construct(): Table lookup returns outgoing=%x\n", 
	       outgoing));
    if (outgoing == (tcp_outgoing_t *) NULL)
    {
	NexusMalloc(outgoing_construct(), 
		    outgoing, 
		    tcp_outgoing_t *,
		    sizeof(tcp_outgoing_t));
	
	outgoing->type = NEXUS_PROTO_TYPE_TCP;
	outgoing->funcs = &tcp_proto_funcs;
	outgoing->version = GLOBUS_L_NEXUS_TCP_PROTOCOL_VERSION;
	outgoing->direct_custom_min_size = NEXUS_TCP_DIRECT_CUSTOM_MIN_SIZE;
	outgoing->direct_custom_max_size = NEXUS_DC_MAX_U_LONG;
	outgoing->direct_pointer_min_size = NEXUS_DC_MAX_U_LONG;
	outgoing->direct_pointer_max_size = NEXUS_DC_MAX_U_LONG;
	outgoing->can_use_iovec = NEXUS_TRUE;
	outgoing->reserved_header_size = 0;
	outgoing->host = _nx_copy_string(host);
	outgoing->port = port;
	outgoing->state = OUTGOING_STATE_UNOPENED;
	outgoing->fault_code = NEXUS_ERROR_NONE;
	outgoing->reference_count = 1;
	outgoing->write_in_progress = NEXUS_FALSE;
	outgoing->write_q_head = (struct globus_nexus_buffer_s *) NULL;
	outgoing->write_q_tail = (struct globus_nexus_buffer_s *) NULL;
	
	outgoing_table_insert(outgoing);
    }
    else
    {
	outgoing->reference_count++;
    }
	
    return (outgoing);
} /* outgoing_construct() */


/*
 * outgoing_free()
 *
 * Free the passed 'outgoing'.
 */
static void outgoing_free(tcp_outgoing_t *outgoing)
{
    if (outgoing->host != (char *) NULL)
    {
	NexusFree(outgoing->host);
    }
    NexusFree(outgoing);
} /* outgoing_free() */


/*
 * outgoing_open()
 *
 * Open the connection for the passed outgoing.  If it is already
 * connected, then just return.
 *
 * Note_enqueue: This routine could cause messages to be enqueued.
 */
static void outgoing_open(tcp_outgoing_t *     outgoing)
{
    globus_result_t               rc;
    globus_io_attr_t              attr;
    globus_result_t               res;

    if (outgoing->state == OUTGOING_STATE_OPEN)
    {
	/* This outgoing is already open, so return */
	return;
    }
    outgoing->state = OUTGOING_STATE_CONNECTING;

    globus_io_tcpattr_init(&attr);
    globus_io_attr_set_tcp_nodelay(&attr, 
				   GLOBUS_TRUE);

    if (globus_module_getenv("GLOBUS_NEXUS_NO_GSI") == NULL)
    {
        res = globus_io_attr_set_secure_authentication_mode(
            &attr,
            GLOBUS_IO_SECURE_AUTHENTICATION_MODE_GSSAPI,
            GSS_C_NO_CREDENTIAL);
        if (res != GLOBUS_SUCCESS)
        {
            globus_fatal("tcp_init(): globus_io_tcp_create_listener() failed\n");
        }
        res = globus_io_attr_set_secure_authorization_mode(
            &attr,
            GLOBUS_IO_SECURE_AUTHORIZATION_MODE_SELF,
            NULL);
        if (res != GLOBUS_SUCCESS)
        {
            globus_fatal("tcp_init(): globus_io_tcp_create_listener() failed\n");
        }
    }

#ifndef BUILD_LITE        
    globus_io_attr_set_callback_space(&attr, _nx_serial_space);
#endif

    rc = globus_io_tcp_register_connect(outgoing->host, 
			       outgoing->port,
			       &attr,
			       globus_l_connect_callback,
			       (void *) outgoing,
			       &outgoing->handle);

    assert(rc == GLOBUS_SUCCESS);
    globus_io_tcpattr_destroy(&attr);

} /* outgoing_open() */


/*
 * outgoing_close()
 *
 * Close a outgoing:
 *   1) Remove the outgoing from the fd_table table
 *   2) Close the fd
 *   3) Modify outgoing data structure
 */
static void 
outgoing_close(
     tcp_outgoing_t *                outgoing,
     int                             new_state)
{

    /*
     * socket destruction notification
     */
#ifdef BUILD_RESOURCE
    {
      nexus_rusage_socket current_socket;
      globus_netlen_t len;

      current_socket.who = &outgoing->resource_name;
      current_socket.op = RUSAGE_DESTROY;
      current_socket.pid = globus_libc_getpid();
      current_socket.fd = outgoing->handle.fd;
      current_socket.mode = RESOURCE_SOCKET_WRITE;
      len = sizeof(current_socket.addr);
      getpeername(outgoing->handle.fd, (struct sockaddr *)&current_socket.addr, &len);
      current_socket.protocol = IPPROTO_TCP;

      nexus_overseer_rusage((nexus_rusage_obj*)&current_socket);
    }
#endif /* BUILD_RESOURCE */


    outgoing->state = OUTGOING_STATE_CLOSING;
    outgoing->new_state = new_state;

    globus_io_register_close(
                  &outgoing->handle,
	          globus_l_outgoing_close_callback,
	          (void *) outgoing);

} /* outgoing_close() */

static void 
globus_l_outgoing_close_callback(
       void *                                 arg,
       globus_io_handle_t *                   handle,
       globus_result_t                        result)
{
    globus_list_t *                list;
    tcp_outgoing_t *               outgoing;   

    outgoing = (tcp_outgoing_t *) arg;
    outgoing->state = outgoing->new_state;

    globus_mutex_lock(&outgoing_mutex);
    {
        list = globus_list_search(globus_l_nexus_tcp_outgoing_list,
			          (void *) outgoing);
        globus_list_remove(&globus_l_nexus_tcp_outgoing_list, list);
	n_outgoing_handles_open--;
	if(n_outgoing_handles_open <= 0 && tcp_done)
	{
            globus_cond_signal(&outgoing_cond);
	}
    }
    globus_mutex_unlock(&outgoing_mutex);

    if(outgoing->state == OUTGOING_STATE_CLOSED)
    {
        outgoing_free(outgoing);
    }
}

/*
 * outgoing_close_normal()
 */
static void outgoing_close_normal(tcp_outgoing_t *outgoing)
{
    outgoing->state = OUTGOING_STATE_CLOSE_PENDING;
    if (!outgoing->write_in_progress)
    {
	outgoing_register_next_write(outgoing); /* This calls tcp_exit() */
    }
} /* outgoing_close_normal() */

				       
static void             
globus_l_connect_callback(
        void *                                    arg, 
        globus_io_handle_t *                      handle,
        globus_result_t                           result)
{
    globus_result_t                   res;
    tcp_outgoing_t *                  outgoing = (tcp_outgoing_t *) arg;


    tcp_enter();
    if (result == GLOBUS_SUCCESS)
    {
	globus_mutex_lock(&outgoing_mutex);
	{
	    globus_list_insert(&globus_l_nexus_tcp_outgoing_list, 
			       (void *) outgoing);

            n_outgoing_handles_open++;
        }	
	globus_mutex_unlock(&outgoing_mutex);
      
        if(outgoing->reference_count==0)
        {
            outgoing->state = OUTGOING_STATE_CLOSE_PENDING;
        }
        else
        {
            outgoing->state = OUTGOING_STATE_OPEN;
        }

	res = globus_io_register_read( 
				handle,
				(globus_byte_t *) outgoing->read_buf,
				sizeof(close_message),
				sizeof(close_message),
				outgoing_read_callback,
				(void *) outgoing);
        assert(res == GLOBUS_SUCCESS);

	/* must force a read register in case a send happened 
	   while waiting for a connect */
        if(QueueNotEmpty(outgoing->write_q_head) &&
           !outgoing->write_in_progress)
        {
            outgoing_register_next_write(outgoing);
        }

	/* TODO: Check the return code */
    }
    else
    {
	outgoing->state = OUTGOING_STATE_FAULT;
	outgoing->fault_code = NEXUS_ERROR_CONNECT_FAILED;
	tcp_exit();
	if (_nx_fault_detected(outgoing->fault_code) != 0)
	{
	    globus_fatal("outgoing_open(): Failed to connect to %s:%hu\n",
			outgoing->host, outgoing->port);
	}
	tcp_enter();
    }
   
    /*
     * socket creation notification
     */
#ifdef BUILD_RESOURCE
	{
	  nexus_rusage_socket current_socket;
	  globus_netlen_t len;

	  nexus_resource_name(&outgoing->resource_name, RESOURCE_SOCKET);

	  current_socket.who = &outgoing->resource_name;
	  current_socket.pid = globus_libc_getpid();
	  current_socket.fd = outgoing->handle.fd;
	  current_socket.mode = RESOURCE_SOCKET_WRITE;
	  len = sizeof(current_socket.addr);
	  getpeername(outgoing->handle.fd, 
		      (struct sockaddr *)&current_socket.addr, &len);
	  current_socket.protocol = IPPROTO_TCP;
	  current_socket.op = RUSAGE_CREATE;

	  nexus_overseer_rusage((nexus_rusage_obj*)&current_socket);
	}
#endif /* BUILD_RESOURCE */
   
    if (!outgoing->write_in_progress && 
	 outgoing->state != OUTGOING_STATE_FAULT)
    {
	/*
	 * Nobody else has registered a write on this outgoing,
	 * so register the one we just enqueued.
	 *
	 * If there is already a write in progress,
	 * the write callback will take care of registering this
	 * write (and any other ones that are enqueued) when
	 * the current write completes.
	 */

	outgoing_register_next_write(outgoing);
    }
    tcp_exit();
}
/*
 * tcp_increment_reference_count()
 *
 * Increase the reference count on the associated proto and copy the
 * pointer to the nexus_proto_t
 *
 */
static void tcp_increment_reference_count(nexus_proto_t *nproto)
{
    tcp_outgoing_t *outgoing = (tcp_outgoing_t *) nproto;
    tcp_enter();
    outgoing->reference_count++;
    tcp_exit();
} /* tcp_increment_reference_count() */


/*
 * tcp_decrement_reference_count()
 *
 * Decrement the reference count for this proto.  If it goes to 0
 * then close the fd used by this outgoing.
 *
 * Return NEXUS_TRUE if this function frees the proto.
 */
static nexus_bool_t tcp_decrement_reference_count(nexus_proto_t *nproto)
{
    tcp_outgoing_t *outgoing = (tcp_outgoing_t *) nproto;
    nexus_bool_t proto_freed = NEXUS_FALSE;
    
    tcp_enter();
    outgoing->reference_count--;
    
    NexusAssert2((outgoing->reference_count >= 0), ("tcp_decrement_reference_count(): Internal error: Reference count < 0\n"));
	       
    if (outgoing->reference_count == 0)
    {
	outgoing_table_remove(outgoing);
	if (outgoing->state == OUTGOING_STATE_OPEN)
	{
	    outgoing_close_normal(outgoing);
	}
	proto_freed = NEXUS_TRUE;
    }
    
    tcp_exit();
    return(proto_freed);
    
} /* tcp_decrement_reference_count() */


/*
 * nexusl_pr_tcp_get_my_mi_proto()
 *
 * Return in 'array' and 'size' a byte array containing
 * enough information to enable another process to connect
 * to this one.  This byte array will be assembled with those
 * of the other protocol modules and placed in a nexus_mi_proto_t.
 *
 * Fill in 'array' with:
 *	version (1 byte)
 *	port  (4 byte, big endian integer)
 *	hostname (null terminated string)
 *
 * If the hostname is the same as my node name, then just place an
 * empty string in the hostname field.
 */
static int
nexusl_pr_tcp_get_my_mi_proto(nexus_byte_t **array,
			      int *size,
			      void *proto_info,
			      nexus_endpoint_t *endpoint)
{
    int i;
    int tmp_int;
    char *host;
    int host_length;

    if (strcmp(tcp_local_host, _nx_my_hostname) == 0)
    {
	host = "";
    }
    else
    {
	host = tcp_local_host;
    }
    
    host_length = (strlen(host) + 1);
    *size = 5 + host_length;
    NexusMalloc(tcp_get_my_mi_proto(),
		*array,
		nexus_byte_t *,
		*size);
    tmp_int = (int) tcp_local_port;
    i = 0;
    (*array)[i++] = GLOBUS_L_NEXUS_TCP_MI_PROTO_VERSION;
    /* TODO: Switch this over to using nexus_dc_*() routines */
    PackInt4(*array, i, tmp_int);
    memcpy(&((*array)[i]), host, host_length);

    return(GLOBUS_SUCCESS);
}
/* nexusl_pr_tcp_get_my_mi_proto() */


/*
 * tcp_construct_from_mi_proto()
 *
 * From the passed machine independent protocol list ('mi_proto'), plus
 * the tcp specific entry from that list ('proto_array' and 'size'),
 * see if I can use the information to create a nexus_proto_t object
 * that can be used to connect to the context:
 *	- If I cannot use this protocol to attach to the context, then
 *		return NEXUS_FALSE.  (This option is useful if two contexts
 *		both speak a particular protocol, but they cannot
 *		talk to each other via that protocol.  For example,
 *		on two MPP's, the contexts within a single MPP can
 *		talk to each other via the native messaging protocol,
 *		but cannot talk to the contexts on the other MPP
 *		using that native protocol.)
 *	- If this tcp protocol points to myself, and thus the local
 *		protocol module should be used, then set
 *		*proto=NULL, and return NEXUS_TRUE.
 *	- Otherwise, construct a tcp protocol object for this mi_proto
 *		and put it in *proto.  Then return NEXUS_TRUE.
 *
 * The 'proto_array' should contain:
 *	version (1 byte)
 *	port  (4 byte, big endian integer)
 *	hostname (null terminated string)
 *
 * If the hostname is an empty string, then use the hostname
 * from the mi_proto.
 */
static nexus_bool_t
tcp_construct_from_mi_proto(
      nexus_proto_t **                          proto,
      nexus_mi_proto_t *                        mi_proto,
      nexus_byte_t *                            proto_array,
      int                                       size)
{
    char *host;
    unsigned short port;
    int i;
    int tmp_int;
    int version;

    NexusAssert2((size >= 5),
		 ("tcp_construct_from_mi_proto(): Invalid tcp information in mi_proto\n"));
		 
    i = 0;

    /* Check the tcp mi_proto version */
    version = (int) proto_array[i++];
    if (version != GLOBUS_L_NEXUS_TCP_MI_PROTO_VERSION)
    {
	_nx_fault_detected(GLOBUS_NEXUS_ERROR_VERSION_MISMATCH);
	return(NEXUS_FALSE);
    }
    
    /* Get the port and hostname */
    /* TODO: Switch this over to using nexus_dc_*() routines */
    UnpackInt4(proto_array, i, tmp_int);
    port = (unsigned short) tmp_int;
    host = (char *) &(proto_array[i]);
    if (host[0] == '\0')
    {
	int context_differentiator;
	i = 0;
	UnpackMIProtoHeader(mi_proto->array, i,
			    context_differentiator,
			    host,
			    version);
	NexusAssert2((strlen(host) > 0),
		     ("tcp_construct_from_mi_proto(): Invalid hostname field in mi_proto\n"));
    }

	
    /*
     * Test to see if this proto points to myself.
     * If it does, then return the _nx_local_proto.
     */
    if (   (port == tcp_local_port)
	   && (strcmp(host, tcp_local_host) == 0) )
    {
	*proto = (nexus_proto_t *) NULL;
    }
    else
    {
	tcp_enter();
	*proto = (nexus_proto_t *) outgoing_construct(host, port);
	tcp_exit();
    }
    return (NEXUS_TRUE);
} /* tcp_construct_from_mi_proto() */


/*
 * tcp_test_proto()
 */
static int tcp_test_proto(nexus_proto_t *proto)
{
    tcp_outgoing_t *outgoing = (tcp_outgoing_t *) proto;
    return(outgoing->fault_code);
} /* tcp_test_proto() */


/*
 * tcp_direct_info_size()
 */
static int tcp_direct_info_size(void)
{
    /* TODO: This needs to be filled in */
    return(0);
} /* tcp_direct_info_size() */


/*
 * incoming_construct()
 *
 * Construct a tcp_incoming_t for the given file descriptor, 'fd'.
 */
static tcp_incoming_t * 
incoming_construct(globus_io_handle_t *  handle)
{
    tcp_incoming_t *           incoming;
    globus_result_t            rc;

    NexusMalloc(incoming_construct(),
		incoming,
		tcp_incoming_t *,
		sizeof(tcp_incoming_t));
    NexusMalloc(incoming_construct(),
		incoming->storage,
		nexus_byte_t *,
		TCP_INCOMING_DEFAULT_SIZE);

    incoming->handle = handle;
    incoming->state = INCOMING_STATE_VERSION;
    incoming->nbytes_parsed = 0;
    incoming->nbytes_unparsed = 0;
    incoming->storage_size = TCP_INCOMING_DEFAULT_SIZE;
    incoming->current = incoming->storage;
    incoming->save_storage_size = 0;
    incoming->dispatch_in_progress = NEXUS_FALSE;
    incoming->dispatch_q_head = (struct globus_nexus_buffer_s *) NULL;
    incoming->dispatch_q_tail = (struct globus_nexus_buffer_s *) NULL;

    globus_io_handle_set_user_pointer(incoming->handle, 
				      (void *) incoming);

    globus_mutex_lock(&incoming_mutex);
    {
        n_incoming_handles_open++;
        globus_list_insert(&globus_l_nexus_tcp_incoming_list, 
		           (void *) incoming);
    }
    globus_mutex_unlock(&incoming_mutex);
 
    rc = globus_io_register_read( 
             incoming->handle,
	     (globus_byte_t *) incoming->storage,
             incoming->storage_size,
	     1,
             incoming_read_callback,
             (void *) incoming);

    /* TODO: Check the return code */
    assert(rc == GLOBUS_SUCCESS);
    return incoming;
} /* incoming_construct() */


/*
 * incoming_close()
 *
 * Close an incoming connection:
 *   1) Remove the tcp_incoming_t from the fd_table table
 *   2) Close the fd
 *   3) Put the tcp_incoming_t back on the free list.
 */
static void 
incoming_close(
    tcp_incoming_t *                    incoming)
{
  /*
   * socket destruction notification
   */
#ifdef BUILD_RESOURCE
    {
      nexus_rusage_socket current_socket;
      globus_netlen_t len;

      current_socket.who = &incoming->resource_name;
      current_socket.op = RUSAGE_DESTROY;
      current_socket.pid = globus_libc_getpid();
      current_socket.fd = incoming->handle->fd;
      current_socket.mode = RESOURCE_SOCKET_READ;
      len = sizeof(current_socket.addr);
      getsockname(incoming->handle->fd, (struct sockaddr *)&current_socket.addr, &len);
      {
	struct in_addr *addr = &current_socket.addr.sin_addr;
	struct in_addr any_addr = {INADDR_ANY};
	if (memcmp (addr, &any_addr, sizeof(addr)) == 0) {
	  addr->s_addr = tcp_lookup_local_addr();
	}
      }
      current_socket.protocol = IPPROTO_TCP;

      nexus_overseer_rusage((nexus_rusage_obj*)&current_socket);
    }
#endif /* BUILD_RESOURCE */

    globus_io_register_close(
                  incoming->handle,
	          globus_l_incoming_close_callback,
	          (void *) incoming);

    incoming->state = INCOMING_STATE_CLOSING;
   
} /* incoming_close() */


static void
globus_l_incoming_close_callback(
       void *                                 arg,
       globus_io_handle_t *                   handle,
       globus_result_t                        result)
{
    tcp_incoming_t *                    incoming;
    globus_list_t *                     list;

    incoming = (tcp_incoming_t *) arg;

    if(result != GLOBUS_SUCCESS)
    {
        /* TODO: do something */
    }
    
    tcp_enter();
    globus_mutex_lock(&incoming_mutex);
    {
        list = globus_list_search(globus_l_nexus_tcp_incoming_list,
			          (void *) incoming);
        globus_list_remove(&globus_l_nexus_tcp_incoming_list, list);
        n_incoming_handles_open--;
	if(n_incoming_handles_open <= 0)
	{
            globus_cond_signal(&incoming_cond);
	}
    }
    globus_mutex_unlock(&incoming_mutex);

    incoming->state = INCOMING_STATE_CLOSED;
    if(!incoming->dispatch_in_progress)
    {
        incoming_free(incoming);
    }

    tcp_exit();
}

/*
 * incoming_free()
 */
static void 
incoming_free(
    tcp_incoming_t *                    incoming)
{
    NexusFree(incoming->storage);
    NexusFree(incoming->handle);
    NexusFree(incoming);
} /* incoming_free() */


/*
 * incoming_read_callback()
 *
 * EOF and the CLOSE_* flags that accompany and EOF are
 * handled by incoming_read_error_callback().
 */
static void 
incoming_read_callback(
	void *                                   arg,
        globus_io_handle_t *                     handle,
	globus_result_t                          result,
	globus_byte_t *                          buf,
	globus_size_t                            nbytes)
{
    tcp_incoming_t *                   incoming;
    globus_bool_t                      done;
    int                                sizeof_u_long;
    unsigned long                      tmp_u_long;
    struct globus_nexus_buffer_s *     buffer;
    globus_bool_t                      message_enqueued = NEXUS_FALSE;
    globus_result_t                    res;

    nexus_debug_printf(1, ("incoming_read_callback(): entering\n"));
    /* if an error occured call error callback */
    if(result != GLOBUS_SUCCESS)
    {
          incoming_read_error_callback(arg, handle, result, buf, nbytes);
          return;
    }

    incoming = (tcp_incoming_t *) arg;
    
    tcp_enter();

    if(tcp_done)
    {
        if(incoming->state != INCOMING_STATE_CLOSING &&
            incoming->state != INCOMING_STATE_CLOSED)
        {
          /* Flush incoming data */
          res = globus_io_register_read( 
    			      incoming->handle,
    			      (globus_byte_t *) incoming->storage,
    			      incoming->storage_size,
    			      0,
    			      incoming_read_callback,
    			      (void *) incoming);
    
          assert(res == GLOBUS_SUCCESS);
        }
      
        tcp_exit();
        return;
    }

    NexusAssert2(((incoming->state == INCOMING_STATE_VERSION)
		  || (incoming->state == INCOMING_STATE_FORMAT)
		  || (incoming->state == INCOMING_STATE_MSG_SIZE)
		  || (incoming->state == INCOMING_STATE_BODY) ),
		 ("incoming_read_callback(): Internal error: Invalid incoming->state = %d\n", 
		  incoming->state) );
    nexus_debug_printf(4, 
		       ("starting with %ld bytes parsed and %;d bytes unparsed and with %ld new bytes read", 
			incoming->nbytes_parsed, incoming->nbytes_unparsed, nbytes));

    
    incoming->nbytes_unparsed += nbytes;
    
    done = GLOBUS_FALSE;
    while(!done)
    {
      switch(incoming->state)
      {
          case INCOMING_STATE_VERSION:
	  {
	    nexus_debug_printf(3, ("parsing version\n"));
	    if (incoming->nbytes_unparsed < 1)
	    {
	      globus_byte_t *           tmp_ptr;
	      int                       tmp_size;
	      nexus_debug_printf(3, ("not enough bytes...posting read for more\n"));
	      /*
	       * Not enough bytes in the buffer to get the format.
	       * So post a read for it.
	       */
	      tmp_ptr = (globus_byte_t *)(incoming->current
			                             + incoming->nbytes_unparsed);
	      tmp_size = (incoming->storage_size
			  - incoming->nbytes_unparsed
			  - incoming->nbytes_parsed);
	      res = globus_io_register_read( 
				      incoming->handle,
				      tmp_ptr,
				      tmp_size,
				      1,
				      incoming_read_callback,
				      (void *) incoming);
	     
		 assert(res == GLOBUS_SUCCESS);
	      done = GLOBUS_TRUE;
	    }
	    else
	    {
	      /*
	       * Get the version from the buffer.
	       */
	      int version;
	      version = (int) *(incoming->current);
	      incoming->current++;
	      incoming->nbytes_parsed++;
	      incoming->nbytes_unparsed--;
		
	      nexus_debug_printf(
				 4,
				 ("V: nbytes_parsed is %d and nbytes_unparsed is %d\n",
				  incoming->nbytes_parsed,
				  incoming->nbytes_unparsed));
	      nexus_debug_printf(3, ("version is %d\n", version));

	      if (version == GLOBUS_L_NEXUS_TCP_PROTOCOL_VERSION)
	      {
		incoming->state = INCOMING_STATE_FORMAT;
		nexus_debug_printf(3,
				   ("state is INCOMING_STATE_FORMAT\n"));
	      }
	      else
	      {
		incoming_close(incoming);
		tcp_exit();
		_nx_fault_detected(GLOBUS_NEXUS_ERROR_VERSION_MISMATCH);
		tcp_enter();
		done = NEXUS_TRUE;
	      }
	    }
	  }
	  break;
	
          case INCOMING_STATE_FORMAT:
	  { 
	    nexus_debug_printf(3, ("parsing format\n"));
	    if (incoming->nbytes_unparsed < 1)
	    {
	      globus_byte_t *           tmp_ptr;
	      int                       tmp_size;
	      nexus_debug_printf(3, ("not enough bytes...posting read for more\n"));
	      /*
	       * Not enough bytes in the buffer to get the format.
	       * So post a read for it.
	       */
	      tmp_ptr = (globus_byte_t *)(incoming->current
					  + incoming->nbytes_unparsed);
	      tmp_size = (incoming->storage_size
			  - incoming->nbytes_unparsed
			  - incoming->nbytes_parsed);
	      res = globus_io_register_read( 
				      incoming->handle,
				      tmp_ptr,
				      tmp_size,
				      1,
				      incoming_read_callback,
				      (void *) incoming);

    assert(res == GLOBUS_SUCCESS);
	      done = GLOBUS_TRUE;
	    }
	    else
	    {
	      /*
	       * Get the format from the buffer.
	       */
	      incoming->format = (int) *(incoming->current);
	      incoming->current++;
	      incoming->nbytes_parsed++;
	      incoming->nbytes_unparsed--;
	      
	      nexus_debug_printf(
		                 4,
		                 ("F: nbytes_parsed is %d and nbytes_unparsed is %d\n",
				  incoming->nbytes_parsed,
				  incoming->nbytes_unparsed));
	      nexus_debug_printf(3, ("format is %d\n", incoming->format));
	      if (incoming->format != CLOSE_NORMAL_FLAG)
	      {
		incoming->state = INCOMING_STATE_MSG_SIZE;
		nexus_debug_printf(3,
				   ("state is INCOMING_STATE_MSG_SIZE\n"));
	      }
	      else
	      {
		incoming_close(incoming);
		done = GLOBUS_TRUE;
	      }
	    }
	  }
	  break;

          case INCOMING_STATE_MSG_SIZE:
	  {
	    nexus_debug_printf(3, ("getting message size\n"));
	    /* Get the message size */
	    sizeof_u_long = nexus_dc_sizeof_remote_u_long(1, incoming->format);
	    if (incoming->nbytes_unparsed < sizeof_u_long)
	    {
	      globus_byte_t *           tmp_ptr;
	      int                       tmp_size;
	      int                       tmp_wait_bytes;
	      nexus_debug_printf(3, ("not enough bytes...posting read for more\n"));
	      /*
	       * Not enough bytes in the buffer to get the message size.
	       * So post a read for it.
	       */
	      tmp_ptr = (globus_byte_t *) (incoming->current
					   + incoming->nbytes_unparsed);
	      tmp_size = (incoming->storage_size
			  - incoming->nbytes_unparsed
			  - incoming->nbytes_parsed);
	      tmp_wait_bytes = (sizeof_u_long
				- incoming->nbytes_unparsed);
	      res = globus_io_register_read( 
				      incoming->handle,
				      tmp_ptr,
				      tmp_size,
				      tmp_wait_bytes,
				      incoming_read_callback,
				      (void *) incoming);

    assert(res == GLOBUS_SUCCESS);
	      done = GLOBUS_TRUE;
	      
	    }
	    else
	    {
	      /* Get the message size */
	      nexus_dc_get_u_long(&(incoming->current),
				  &(incoming->msg_size),
				  1,
				  incoming->format);
	      incoming->nbytes_parsed += sizeof_u_long;
	      incoming->nbytes_unparsed -= sizeof_u_long;
	      nexus_debug_printf(4, 
				 ("B: nbytes_parsed is %d and nbytes_unparsed is %d\n", 
				  incoming->nbytes_parsed, incoming->nbytes_unparsed));
	      nexus_debug_printf(3, ("size is %lu\n", incoming->msg_size));
	      
	      /********** look at this again ************/
	      if (incoming->msg_size > incoming->storage_size)
	      {
	          globus_byte_t *           tmp_ptr;
	          int                       tmp_size;
	          int                       tmp_wait_bytes;

		nexus_debug_printf(3, ("message too big to fit in current storage\n"));
		/*
		 * The message is too big to fit in this storage.
		 * So allocate new storage that is big enough,
		 * copy the message into the new storage.
		 */
		incoming->save_storage_size = incoming->storage_size;
		incoming->save_storage = incoming->storage;
		incoming->save_current = incoming->storage;

		/* Allocate new storage */
		NexusMalloc(incoming_read_callback(),
			    incoming->storage,
			    nexus_byte_t *,
			    incoming->msg_size);
		
		incoming->storage_size = incoming->msg_size;
		
		/* Copy message to new storage */
		tmp_u_long = (incoming->nbytes_parsed
			      + incoming->nbytes_unparsed);
		memcpy(incoming->storage,
		       incoming->save_storage,
		       tmp_u_long);
		incoming->current = (incoming->storage
				     + incoming->nbytes_parsed);
		
		nexus_debug_printf(3, ("post read for rest of message\n"));
		
		/* Setup new read for rest of message */
		tmp_ptr = (globus_byte_t *) (incoming->current
					     + incoming->nbytes_unparsed);
		tmp_size = (incoming->msg_size
			    - incoming->nbytes_unparsed
			    - incoming->nbytes_parsed);
		tmp_wait_bytes = 0;
		
		res = globus_io_register_read( 
					incoming->handle,
					tmp_ptr,
					tmp_size,
					tmp_wait_bytes,
					incoming_read_callback,
					(void *) incoming);
		nbytes = 0;
	    assert(res == GLOBUS_SUCCESS);
	        done = GLOBUS_TRUE;
	      }
	      
	      /* Reset the state to get the body */
	      incoming->state = INCOMING_STATE_BODY;
	      nexus_debug_printf(3, ("state is INCOMING_STATE_BODY\n"));
	    }
	  }
	  break;

          case INCOMING_STATE_BODY:
	  {
	    nexus_debug_printf(3, ("reading body of message\n"));
	    if ((incoming->nbytes_parsed + incoming->nbytes_unparsed)
		< incoming->msg_size)
	    {
	        globus_byte_t *           tmp_ptr;
	        int                       tmp_size;
 	        int                       tmp_wait_bytes;

		nexus_debug_printf(3, 
				   ("still need more...posting read\n"));
		/*
		 * The whole message has not been received yet.
		 * So post a read for it.
		 */
		tmp_ptr = (globus_byte_t *) (incoming->current
				    + incoming->nbytes_unparsed);
		tmp_size = (incoming->storage_size
			    - incoming->nbytes_unparsed
			    - incoming->nbytes_parsed);
		tmp_wait_bytes = (incoming->msg_size
				  - incoming->nbytes_parsed
				  - incoming->nbytes_unparsed);
		res = globus_io_register_read( 
					incoming->handle,
					tmp_ptr,
					tmp_size,
					tmp_wait_bytes,
					incoming_read_callback,
					(void *) incoming);
		nbytes = 0;
		done = GLOBUS_TRUE;
	    }
	    else
	    {
	      nexus_debug_printf(3, ("got all of the message\n"));
	      if ((incoming->nbytes_parsed 
		   + incoming->nbytes_unparsed)
		  > incoming->msg_size)
	      {
		nexus_debug_printf(3, 
				   ("got part of next message\n"));
		/*
		 * We've read past the end of this message.
		 * So allocate new storage to hold the extra bytes.
		 *
		 * We know that incoming->storage is of default size,
		 * so we can just allocate another default size
		 * storage to hold the overflow.
		 * If the message was bigger than the default
		 * storage size, then storage was allocated to
		 * be exactly the right size for the message, and
		 * a read was posted only for that many bytes.
		 */
		NexusAssert2((incoming->storage_size == TCP_INCOMING_DEFAULT_SIZE), 
			     ("incoming_read_callback(): Internal error: Got a read overflow on a large message.  This shouldn't happen.\n"));
		
		NexusMalloc(incoming_read_callback(),
			    incoming->save_storage,
			    nexus_byte_t *,
			    TCP_INCOMING_DEFAULT_SIZE);
		incoming->save_storage_size = TCP_INCOMING_DEFAULT_SIZE;
		
		tmp_u_long = (incoming->nbytes_parsed
			      + incoming->nbytes_unparsed
			      - incoming->msg_size);
		memcpy(incoming->save_storage,
		       (incoming->storage + incoming->msg_size),
		       tmp_u_long);
		incoming->save_current = (incoming->save_storage
					  + tmp_u_long);
		incoming->nbytes_unparsed -= tmp_u_long;
	      }
	      /* Enqueue the message for dispatch */
	      _nx_buffer_create_from_raw(incoming->storage,
					 incoming->storage_size,
					 0,
					 incoming->msg_size,
					 NULL,
					 &buffer);
	      Enqueue(incoming->dispatch_q_head,
		      incoming->dispatch_q_tail,
		      buffer);
	      message_enqueued = NEXUS_TRUE;
	      nexus_debug_printf(3, ("message enqueued\n"));
	      
	      /* Reset the incoming state to INCOMING_STATE_VERSION */
	      if (incoming->save_storage_size > 0)
	      {
		nexus_debug_printf(3, ("next message using saved storage\n"));
		/* There is a buffer in the save_storage to use */
		incoming->state = INCOMING_STATE_VERSION;
		incoming->storage_size = incoming->save_storage_size;
		incoming->storage = incoming->save_storage;
		incoming->current = incoming->save_storage;
		incoming->save_storage_size = 0;
		incoming->nbytes_parsed = 0;
		incoming->nbytes_unparsed = (incoming->save_current
					     - incoming->save_storage);
		nexus_debug_printf(4, ("D: nbytes_parsed is %d and nbytes_unparsed is %d\n", incoming->nbytes_parsed, incoming->nbytes_unparsed));
	      }
	      else
	      {
		nexus_debug_printf(3, ("next message using new storage\n"));
		/* Must allocate new storage */
		incoming->state = INCOMING_STATE_VERSION;
		incoming->storage_size = TCP_INCOMING_DEFAULT_SIZE;
		NexusMalloc(incoming_read_callback(),
			    incoming->storage,
			    nexus_byte_t *,
			    TCP_INCOMING_DEFAULT_SIZE);
		incoming->current = incoming->storage;
		incoming->nbytes_parsed = 0;
		incoming->nbytes_unparsed = 0;
		nexus_debug_printf(4, 
				   ("E: nbytes_parsed is %d and nbytes_unparsed is %d\n", incoming->nbytes_parsed, incoming->nbytes_unparsed));
	      }
	      nexus_debug_printf(3, ("state is INCOMING_STATE_VERSION\n"));
	    } 
	    
	  }
	  break;
      }
    }
    
    if (message_enqueued && !incoming->dispatch_in_progress)
    {
      incoming->dispatch_in_progress = NEXUS_TRUE;
      while (QueueNotEmpty(incoming->dispatch_q_head))
      {
	Dequeue(incoming->dispatch_q_head,
		incoming->dispatch_q_tail,
		buffer);
	tcp_exit();
	nexus_debug_printf(3, ("dispatching buffer\n"));
	_nx_buffer_dispatch(buffer);
	tcp_enter();
      }
      incoming->dispatch_in_progress = NEXUS_FALSE;
      
      if (incoming->state == INCOMING_STATE_CLOSED)
      {
	incoming_free(incoming);
      }
    }

    tcp_exit();

    nexus_debug_printf(1, ("incoming_read_callback(): exiting\n"));
} /* incoming_read_callback() */


/*
 * incoming_read_error_callback()
 */
static void
incoming_read_error_callback(
        void *                                   args,
        globus_io_handle_t *                     handle,
	globus_result_t                          result,
	globus_byte_t *                          buf,
	globus_size_t                            nbytes)
{
    tcp_incoming_t *           incoming;
    globus_object_t *          err;
    int                        error;
    globus_bool_t              canceled;

    incoming = (tcp_incoming_t *) args;
    if (tcp_done)
    {
	incoming_close(incoming);
	return;
    }
    
    tcp_enter();
    
    nbytes += incoming->nbytes_unparsed;


    err = globus_error_get(result);
    canceled = globus_object_type_match(
		   globus_object_get_type(err),
		   GLOBUS_IO_ERROR_TYPE_IO_CANCELLED);
    result = globus_error_put(err);
    if(!canceled)
    {
	error = globus_i_nexus_get_errno(&result);

        if (error == 0)
        {
	    incoming_close(incoming);
            tcp_exit();
	    if (_nx_fault_detected(NEXUS_ERROR_BAD_PROTOCOL) != 0)
	    {
	        globus_fatal("incoming_read_error_callback(): Internal error: Got an unexpected end-of-file\n");
	    }
	    tcp_enter();
        }
        else if (   (error == ECONNRESET)
	         || (error == EPIPE) )
        {
    	   /*
	    * Got connection reset by peer on the read, so:
	    *   if fault tolerance is not enabled then die
	    *   else, close the incoming and keep on going
	    *
	    * We should never get an EOF on read under normal
	    * circumstances.  We should always get (and handle) a
	    * close message (the leading byte of a message) before
	    * a fd is closed, so we can handle the closed fd
	    * before we ever get an EOF.  This allows us to
	    * detect abnormal termination of any process we
	    * are connected to.
	    */
	    incoming_close(incoming);
	    tcp_exit();
	    if (_nx_fault_detected(NEXUS_ERROR_PROCESS_DIED) != 0)
            {
                int                         host_ip[4];
                unsigned short              port;
                globus_result_t             res;

                res = globus_io_tcp_get_remote_address(incoming->handle,
                    &host_ip,
                    &port);

                if(res == GLOBUS_SUCCESS)
                {
		            tcp_fatal(
                        "incoming_read_error_callback(): handle "
                        "0x%x connected to %d.%d.%d.%d/%d was unexpectedly "
                        "closed: n_read=%d\n",
                        incoming->handle,
                        host_ip[0], host_ip[1], host_ip[2], host_ip[3],
                        (int)port,
                        incoming->current - incoming->storage);
                }
                else
                {
	                tcp_fatal("incoming_read_error_callback(): handle 0x%x " 
                        "was unexpectedly closed: n_read=%d\n",
                        incoming->handle,
                        incoming->current - incoming->storage);
	            }

	    }
	    tcp_enter();
        }
        else /* Some other read() error */
        {
	    incoming_close(incoming);
	    tcp_exit();
	    if (_nx_fault_detected(NEXUS_ERROR_READ_FAILED) != 0)
	    {
	        globus_fatal("incoming_read_error_callback(): Internal error: Read failed with errno=%i\n", error);
	    }
	    tcp_enter();
        }
    }
    tcp_exit();
} /* incoming_read_error_callback() */

static void		
internal_connection_requested(
      void *                                    arg, 
      globus_io_handle_t *                      handle,
      globus_result_t                           result)
{
    globus_result_t                  res;
    globus_io_handle_t *             state_handle;

    if(tcp_done)
    {
        return;
    }

    if(result != GLOBUS_SUCCESS)
    {
        thread_print("internal_connection_requested() : error\n");
    }

    /* 
       allocate a new handle.  That handle will be passed
       into the accept_internal_connection() 
     */
    NexusMalloc(internal_connection_requested(),
	        state_handle,
	        globus_io_handle_t *,
	        sizeof(globus_io_handle_t));

    res = globus_io_tcp_register_accept(
             handle,
             GLOBUS_NULL,
	     state_handle,
	     accept_internal_connection,
	     arg);

    assert(res == GLOBUS_SUCCESS);
}

/*
 * accept_internal_connection()
 */
static void		
accept_internal_connection(
      void *                                    arg, 
      globus_io_handle_t *                      handle,
      globus_result_t                           result)
{
    tcp_incoming_t *                incoming;
    globus_result_t                  res;

    if(result != GLOBUS_SUCCESS)
    {
        thread_print("accept_internal_connection() : error\n");
        goto done;
    }

    tcp_enter();   
    
    incoming  = incoming_construct(handle);
    if (tcp_done)
    {
	/* globus_io add */
        res = globus_io_register_write(
                    incoming->handle,
		    (globus_byte_t *) (close_message),
		    sizeof(close_message),
		    shutdown_write_callback,
		    (void *) incoming);

        assert(res == GLOBUS_SUCCESS);
	/* TODO: Check the return code */
    }

    /*
     * socket creation notification
     */
#ifdef BUILD_RESOURCE
	{
	  nexus_rusage_socket current_socket;
	  globus_netlen_t len;

	  nexus_resource_name(&incoming->resource_name, RESOURCE_SOCKET);

	  current_socket.who = &incoming->resource_name;
	  current_socket.op = RUSAGE_CREATE;
	  current_socket.pid = globus_libc_getpid();
	  current_socket.fd = incoming->handle->fd;
	  current_socket.mode = RESOURCE_SOCKET_READ;
	  len = sizeof(current_socket.addr);
	  getsockname(incoming->handle->fd, 
		      (struct sockaddr *)&current_socket.addr, &len);
	  {
	    struct in_addr *addr = &current_socket.addr.sin_addr;
	    struct in_addr any_addr = {INADDR_ANY};
	    if (memcmp (addr, &any_addr, sizeof(addr)) == 0) {
	      addr->s_addr = tcp_lookup_local_addr();
	    }
	  }
	  current_socket.protocol = IPPROTO_TCP;

	  nexus_overseer_rusage((nexus_rusage_obj*)&current_socket);
	}
#endif /* BUILD_RESOURCE */

    tcp_exit();

done:
    res = globus_io_tcp_register_listen(
		  &globus_l_nexus_tcp_listener_handle,
		  internal_connection_requested,
		  arg);
    assert(res == GLOBUS_SUCCESS);
  

} /* accept_internal_connection() */


/*
 * outgoing_table_init()
 *
 * Initialize the outgoing table.
 */
static void outgoing_table_init(void)
{
    int i;

    for (i = 0; i < OUTGOING_TABLE_SIZE; i++)
    {
	outgoing_table[i].outgoing = (tcp_outgoing_t *) NULL;
	outgoing_table[i].next = (outgoing_table_entry_t *) NULL;
    }
} /* outgoing_table_init() */


/*
 * outgoing_table_hash()
 *
 * Hash the hostname and port for the outgoing table.
 */
static int outgoing_table_hash(char *host, unsigned short port)
{
    unsigned int hval = 0;
    char *s;

    for (s = host; *s != '\0'; s++)
    {
	hval <<= 1;
	hval += *s;
    }
    hval <<= 1;
    hval += port;
    return (hval % OUTGOING_TABLE_SIZE);
} /* outgoing_table_hash() */


/*
 * outgoing_table_insert()
 *
 * Insert the given outgoing into the table, hashing on its internal
 * hostname and port number.
 *
 * We assume that the entry is not present in the table.
 */
static void outgoing_table_insert(tcp_outgoing_t *outgoing)
{
    int bucket;
    outgoing_table_entry_t *new_ent;

    bucket = outgoing_table_hash(outgoing->host, outgoing->port);
    if (outgoing_table[bucket].outgoing == (tcp_outgoing_t *) NULL)
    {
	/* Drop it into the preallocated table entry */
	outgoing_table[bucket].outgoing = outgoing;
    }
    else
    {
	/*
	 * Need to allocate a new outgoing_table_entry_t and add it
	 * to the bucket
	 */
	NexusMalloc(outgoing_table_insert(), new_ent, outgoing_table_entry_t *,
		    sizeof(struct _outgoing_table_entry_t));

	new_ent->outgoing = outgoing;
	new_ent->next = outgoing_table[bucket].next;

	outgoing_table[bucket].next = new_ent;
    }
    nexus_debug_printf(2, ("outgoing_table_inserted(): Inserted outgoing=%x for %s/%hu bucket=%d\n",
			   outgoing, outgoing->host, outgoing->port, bucket));
} /* outgoing_table_insert() */


/*
 * outgoing_table_remove()
 *
 * Remove the given outgoing from the table.
 *
 * We assume that the entry is present in the table.
 */
static void 
outgoing_table_remove(
     tcp_outgoing_t *                               outgoing)
{
    int                                              bucket;
    outgoing_table_entry_t *                         ent;
    outgoing_table_entry_t *                         remove_ent;

    bucket = outgoing_table_hash(outgoing->host, outgoing->port);

    if (outgoing_table[bucket].outgoing == outgoing)
    {
	if (outgoing_table[bucket].next)
	{
	    outgoing_table[bucket].outgoing
		= outgoing_table[bucket].next->outgoing;
	    outgoing_table[bucket].next
		= outgoing_table[bucket].next->next;
	}
	else
	{
	    outgoing_table[bucket].outgoing = (tcp_outgoing_t *) NULL;
	    outgoing_table[bucket].next = (outgoing_table_entry_t *) NULL;
	}
    }
    else
    {
	for (ent = &(outgoing_table[bucket]);
	     ent->next->outgoing != outgoing;
	     ent = ent->next)
	    ;
	remove_ent = ent->next;
	ent->next = remove_ent->next;
	NexusFree(remove_ent);
    }
    nexus_debug_printf(2, ("outgoing_table_remove(): Removed outgoing=%x for %s/%hu from bucket=%d\n",
			   outgoing, outgoing->host, outgoing->port, bucket));
} /* outgoing_table_remove() */


/*
 * outgoing_table_lookup()
 *
 * Look up and return the tcp_outgoing_t for the given hostname
 * and port. Return NULL if none exists.
 */
static tcp_outgoing_t *outgoing_table_lookup(char *host, unsigned short port)
{
    outgoing_table_entry_t *ent;
    int bucket;

    bucket = outgoing_table_hash(host, port);

    for (ent = &(outgoing_table[bucket]);
	 ent != (outgoing_table_entry_t *) NULL;
	 ent = ent->next)
    {
	if (   (ent->outgoing != (tcp_outgoing_t *) NULL)
	    && (ent->outgoing->port == port)
	    && (strcmp(ent->outgoing->host, host) == 0) )
	{
	    nexus_debug_printf(2, ("outgoing_table_lookup(): Found entry %x outgoing=%x for %s/%hu bucket=%d\n",
				   ent, ent->outgoing, host, port, bucket));
	    return (ent->outgoing);
	}
    }
    
    nexus_debug_printf(2, ("outgoing_table_lookup(): Didn't find entry for %s/%hu bucket=%d\n",
			   host, port, bucket));

    return ((tcp_outgoing_t *) NULL);
} /* outgoing_table_lookup() */



#ifdef BUILD_RESOURCE

/*
 * tcp_get_resource_name_sp()
 *
 * Return a pointer to the resource_name of the socket with the passed
 * startpoint.
 */

nexus_resource_name_t *tcp_get_resource_name_sp(nexus_startpoint_t *sp) {
  nexus_resource_name_t * result;

  if (sp == NULL || sp->mi_proto == NULL || sp->mi_proto->proto == NULL ) {
    result = NULL;
  } else {
    tcp_outgoing_t *outgoing = (tcp_outgoing_t*)sp->mi_proto->proto;
    result = &outgoing->resource_name;
  }  

  return (result);

} /* tcp_get_resource_name_sp() */

/*
 * tcp_get_resource_name_ep()
 *
 * Return a pointer to the resource_name of the socket with the passed 
 * endpoint.
 */

nexus_resource_name_t *tcp_get_resource_name_ep(nexus_endpoint_t *ep) {
  nexus_resource_name_t * result;

  if (ep == NULL ) {
    result = NULL;
  } else {
    result = &tcp_listener_name;
  }  

  return (result);

} /* tcp_get_resource_name_ep() */

/*
 * tcp_lookup_local_addr()
 * 
 * Returns the local sockaddr bound to tcp connections.
 */

static unsigned int tcp_lookup_local_addr() {

  char *alternate_interface;
  char *interface;
  struct hostent *hp;
  struct hostent host_entry;
  char buffer[500];
  int h_errno;
  
  alternate_interface = nexus_rdb_lookup(tcp_local_host, "tcp_interface");
  interface = (alternate_interface ? alternate_interface : tcp_local_host);

  hp = globus_libc_gethostbyname_r(interface,
				   &host_entry,
				   buffer,
				   500,
				   &h_errno);
  
  if (hp == 0) return (0);
  return *((unsigned int **)hp->h_addr_list)[0];

} /* tcp_lookup_local_addr() */


/* temporary addition for mpichG QoS */
int
globus_i_nexus_startpoint_to_fd(globus_nexus_startpoint_t * sp)
{
    tcp_outgoing_t * outgoing = (tcp_outgoing_t *) sp->mi_proto->proto;
    return outgoing->handle.fd;
}

globus_bool_t
globus_i_nexus_get_host_port_pairs(
    globus_nexus_startpoint_t *            sp,
    char                                   localhost[16],
    int *                                  localport,
    char                                   remotehost[16],
    int *                                  remoteport)
{
    int                                    fd;
    struct sockaddr_in                     addr;
    netlen_t                               length;
    char *                                 host;

    fd = globus_i_nexus_startpoint_to_fd(sp);

    length = sizeof(addr);
    if(getpeername (fd, (struct sockaddr *) &addr, &length) == -1)
    {
        return GLOBUS_FALSE;
    }

    *remoteport = ntohs(addr.sin_port);
    host = (char *)inet_ntoa(addr.sin_addr);
    strcpy(remotehost, host);

    length = sizeof(addr);
    if(getsockname (fd, (struct sockaddr *) &addr, &length) == -1)
    {
        return GLOBUS_FALSE; 
    }
    
    *localport = ntohs(addr.sin_port);
    host = (char *)inet_ntoa(addr.sin_addr);
    strcpy(localhost, host);

    return GLOBUS_TRUE;
}
#endif /* BUILD_RESOURCE */
/* ------------------------------------------------------------------------- */
/* end only build the rest if HAVE_TCP_PROTO */
/* ------------------------------------------------------------------------- */
#endif /* HAVE_TCP_PROTO */
