#!/usr/bin/ruby -w
# -*- ruby -*-

module Text

  # Highlights text using either ANSI terminal codes, or HTML.

  # Note that the foreground and background sections can have modifiers
  # (attributes).
  # 
  # Examples:
  #     black
  #     blue on white
  #     bold green on yellow
  #     underscore bold magenta on cyan
  #     underscore red on cyan
  
  class Highlighter

    VERSION = "1.0.4"

    COLORS      = %w{ black red green yellow blue magenta cyan white }
    DECORATIONS = %w{ none reset bold underscore underline blink reverse }

    COLORS_RE = Regexp.new('(?: ' + 
                                # background will be in capture 0
                                'on(?:\s+|_) ( ' + COLORS.join(' | ') + ' ) | ' +
                                # foreground will be in capture 1
                                '( ' + (COLORS + DECORATIONS).join(' | ') + ' ) ' +
                            ')', Regexp::EXTENDED);

    DEFAULT_COLORS = [
      "black on yellow",
      "black on green",
      "black on magenta",
      "yellow on black",
      "magenta on black",
      "green on black",
      "cyan on black",
      "blue on yellow",
      "blue on magenta",
      "blue on green",
      "blue on cyan",
      "yellow on blue",
      "magenta on blue",
      "green on blue",
      "cyan on blue",
    ]

    attr_reader :colors
    
    def self.parse_colors(str)
      str.scan(Regexp.new(COLORS_RE)).collect do |color|
        color[0] ? "on_" + color[0] : color[1]
      end
    end

    def initialize(colors)
      @colors = colors
    end

    def highlight(str)
      pre + str + post
    end

    def to_s
      (@colors || '').join(' ')
    end

    def ==(other)
      # $stderr.puts "this: #{colors.inspect}; other: #{other.colors.inspect}"
      return @colors.sort == other.colors.sort
    end

  end

  # Highlights using ANSI escape sequences.

  class ANSIHighlighter < Highlighter

    ATTRIBUTES = Hash[
      'none'       => '0', 
      'reset'      => '0',
      'bold'       => '1',
      'underscore' => '4',
      'underline'  => '4',
      'blink'      => '5',
      'reverse'    => '7',
      'concealed'  => '8',
      'black'      => '30',
      'red'        => '31',
      'green'      => '32',
      'yellow'     => '33',
      'blue'       => '34',
      'magenta'    => '35',
      'cyan'       => '36',
      'white'      => '37',
      'on_black'   => '40',
      'on_red'     => '41',
      'on_green'   => '42',
      'on_yellow'  => '43',
      'on_blue'    => '44',
      'on_magenta' => '45',
      'on_cyan'    => '46',
      'on_white'   => '47',
    ]

    RESET = "\e[0m"

    def self.make(str)
      colors = parse_colors(str)
      ANSIHighlighter.new(colors)
    end

    def initialize(colors)
      super
      @code = nil
    end

    # Returns the escape sequence for the given name.

    def name_to_code(nm)
      "\e[#{ATTRIBUTES[nm]}m"
    end

    def highlight(str)
      @code ||= begin
                  @code = @colors.collect do |color|
                    name_to_code(color)
                  end.join("")
                end
      
      @code + str + RESET
    end

  end
  
end


