/*
 *  
 *  $Id: conexionopenssl.cpp 3830 2011-05-06 13:30:18Z carlos $
 *  Ginkgo CADx Project
 *
 *  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
 *  http://ginkgo-cadx.com
 *
 *  This file is licensed under LGPL v3 license.
 *  See License.txt for details
 *
 *
 */

#include <iostream>
#include <locale>
#include <cmath>
#ifdef _WIN32
#include <WinCrypt.h>
#endif

#include <api/api.h>
#include <endpoint/endpoint.h>

#include "conexionopenssl.h"
#include "dialogoalmacenarcertificado.h"




namespace GIL {
	namespace GnkNetwork {
		namespace GnkSSL {
			ConexionOpenSSL::ConexionOpenSSL() 
			{
				ep = NULL;
				ssl = NULL;

				/* Build our SSL context*/
				SSL_library_init();
				SSL_load_error_strings();

				/* Create our context*/
				#if (OPENSSL_VERSION_NUMBER < 0x01000000L)
				SSL_METHOD *meth = SSLv23_method();
				#else
				const SSL_METHOD *meth = SSLv23_method();
				#endif
				ctx=SSL_CTX_new(meth);

				/* Load the CAs we trust*/
				X509_STORE* almacen = SSL_CTX_get_cert_store(ctx);
				if(almacen != NULL) {
					CargarCertificadosSistema(almacen);
				}

				EndpointAddrlist::g_default_family = AF_INET;
				Endpoint::Initialize();
			}

			ConexionOpenSSL::~ConexionOpenSSL() 
			{
				if(ssl!=NULL) {
					SSL_shutdown(ssl);
					SSL_free(ssl);
				}
				if(ctx!=NULL){
					SSL_CTX_free(ctx);
				}
				if(ep!=NULL) {
					ep->Close();
					delete ep;
					ep=NULL;
				}
			}

			void ConexionOpenSSL::CrearConexionSSL(const std::string& host, const int& puerto)
			{
				if(ep!=NULL || ssl !=NULL) {
					//cerrar conexion anterior
					CerrarConexionSSL();
				}

				std::stringstream ostr;
				ostr << host;
				ostr << ':' <<puerto;
				std::string url = ostr.str();

				/*Conectamos el socket*/
				ep = new Endpoint(TCP | CLIENT, url);
				if(!ep->m_bool) {
					std::stringstream ostr;
					ostr << _Std("Error connecting") << ep->m_error_str;
					throw ConexionOpenSSLException(ostr.str());
				}

				int sock = ep->m_sockfd;

				/* Connect the SSL socket */
				ssl=SSL_new(ctx);
				BIO *sbio=BIO_new_socket(sock,BIO_NOCLOSE);
				SSL_set_bio(ssl,sbio,sbio);

				if(SSL_connect(ssl)<=0){
					throw ConexionOpenSSLException(_Std("Error when making SSL connection"));
				}

				//checkeamos certificado... //lanza excepcion
				ComprobarCertificado(ssl,host);

			}

			void ConexionOpenSSL::CerrarConexionSSL() 
			{
				if(ssl!=NULL) {
					SSL_shutdown(ssl);
				}
				if(ep!=NULL) {
					ep->Close();
					delete ep;
					ep=NULL;
				}
			}

			void ConexionOpenSSL::HacerPeticion(const std::string& peticion, std::string& respuesta)
			{
				char buf[1024];
				int r;
				int len, request_len;

				/* Find the exact request_len */
				request_len=peticion.size();

				r=SSL_write(ssl,peticion.c_str(),request_len);
				switch(SSL_get_error(ssl,r)){      
	  case SSL_ERROR_NONE:
		  if(request_len!=r)
			  throw ConexionOpenSSLException(_Std("Error when making the request, writing incomplete"));
		  break;
	  default:
		  //lanzar excepcion
		  throw ConexionOpenSSLException(_Std("Failed to perform the request, writing problem"));
				}

				/* Now read the server's response, assuming
				that it's terminated by a close */
				len=-1;
				std::stringstream ostr;
				do{
					r=SSL_read(ssl,buf,1024);
					switch(SSL_get_error(ssl,r)){
		case SSL_ERROR_NONE:
			len=r;
			break;
		default:
			throw ConexionOpenSSLException(_Std("Error in receiving the response"));
			break;
					}
					ostr.write(buf,len);
				} while (SSL_pending(ssl)>0);
				respuesta = ostr.str();
			}

#ifdef _WIN32
			//agrega los certificados de windows
			void ConexionOpenSSL::X509_STORE_load_windows_systemstore(X509_STORE *store)
			{
				HCERTSTORE hStore;
				PCCERT_CONTEXT pContext = NULL;
				X509 *x509;

				hStore = CertOpenSystemStoreA(0, "ROOT");
				if(!hStore)
					return;

				while ( (pContext = CertEnumCertificatesInStore(hStore, pContext)) != NULL )
				{
					x509 = NULL;
					x509 = d2i_X509(NULL, (const unsigned char**)(&pContext->pbCertEncoded), pContext->cbCertEncoded);
					if (x509)
					{
						X509_STORE_add_cert(store, x509);
						X509_free(x509);
					}
				}

				CertFreeCertificateContext(pContext);
				CertCloseStore(hStore, 0);
			} 
#endif

			void ConexionOpenSSL::CargarCertificadosSistema(X509_STORE *store)
			{
				X509_STORE_set_default_paths(store);
#ifdef _WIN32
				X509_STORE_load_windows_systemstore(store);
#endif
			}

			void ConexionOpenSSL::ComprobarCertificado(SSL *ssl,const std::string& host)
			{
				X509 *peer;
				char peer_CN[256];

				if(SSL_get_verify_result(ssl)!=X509_V_OK) {
					//el certificado no ha podido verificarse, se le ofrece la posibilidad al user de que lo guarde para 
					//instalarlo en el sistema
					GNC::GUI::DialogoAlmacenarCertificado dlg(NULL,ssl);
					dlg.ShowModal();
					throw ConexionOpenSSLException(_Std("The certificate has not been verified"));
				}

				/*Check the cert chain. The chain length
				is automatically checked by OpenSSL when
				we set the verify depth in the ctx */

				/*Check the common name*/
				peer=SSL_get_peer_certificate(ssl);

				X509_NAME_get_text_by_NID
					(X509_get_subject_name(peer),
					NID_commonName, peer_CN, 256);

				X509_free(peer);
				
				std::string::const_iterator it = host.begin();
				std::string::size_type i = 0;
				std::locale loc;
				unsigned int len = std::min<size_t>(strlen(peer_CN), (size_t)256);
				bool eq = true;
				for ( ; eq && i < len && it != host.end(); i++, it++) {
					if (std:: tolower(*it,loc) != std::tolower(peer_CN[i],loc)) {
						eq = false;
					}
				}
				if (!eq) {
					throw ConexionOpenSSLException(_Std("The CN field does not match the hostname"),false);
				}
			}
		};
	};
};
