/*
 * purple - Xfire Protocol Plugin
 *
 * This file is part of Gfire.
 *
 * See the AUTHORS file distributed with Gfire for a full list of
 * all contributors and this files copyright holders.
 *
 * Gfire is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Gfire.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "gf_server_query.h"

typedef struct _gfire_sq_gamespy3_player
{
	gchar *name;
	gint score;
	gint ping;
	gint deaths;
	gint kills;
	gint team;
} gfire_sq_gamespy3_player;

typedef struct _gfire_sq_gamespy3_data
{
	// Query handling
	gint32 challenge;

	// Extended data
	GData *info;
	GSList *players;
} gfire_sq_gamespy3_data;

// Prototypes
static void gfire_sq_gamespy3_query(gfire_game_server *p_server, gboolean p_full, int p_socket);
static gboolean gfire_sq_gamespy3_parse(gfire_game_server *p_server, guint16 p_ping, gboolean p_full,
									 const unsigned char *p_data, guint p_len);
static gchar *gfire_sq_gamespy3_details(gfire_game_server *p_server);
static void gfire_sq_gamespy3_free_server(gfire_game_server *p_server);

gfire_server_query_driver gf_sq_gamespy3_driver =
{
	gfire_sq_gamespy3_query,
	gfire_sq_gamespy3_parse,
	gfire_sq_gamespy3_details,
	gfire_sq_gamespy3_free_server,
	3
};

static void gfire_sq_gamespy3_query(gfire_game_server *p_server, gboolean p_full, int p_socket)
{
	static const guint8 challengeQuery[]	= { 0xfe, 0xfd, 0x09, 'G', 'F', 'S', 'Q' };
	static guint8 dataQuery[]				= { 0xfe, 0xfd, 0x00, 'G', 'F', 'S', 'Q',
												0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00,
												0x01};

	int len;
	const guint8 *query;
	if(!p_server->data)
	{
		len = sizeof(challengeQuery);
		query = challengeQuery;
	}
	else
	{
		gfire_sq_gamespy3_data *data = (gfire_sq_gamespy3_data*)p_server->data->proto_data;
		memcpy(dataQuery + 7, &data->challenge, 4);

		dataQuery[12] = p_full ? 0xff : 0x00;
		dataQuery[13] = p_full ? 0xff : 0x00;

		len = sizeof(dataQuery);
		query = dataQuery;
	}

	struct sockaddr_in addr;
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = g_htonl(p_server->ip);
	addr.sin_port = g_htons(p_server->query_port);

	sendto(p_socket, query, len, 0, (struct sockaddr*)&addr, sizeof(addr));
}

static gboolean gfire_sq_gamespy3_parse(gfire_game_server *p_server, guint16 p_ping, gboolean p_full,
									 const unsigned char *p_data, guint p_len)
{
	if(p_len < 12)
		return FALSE;

	if(!p_server->data)
	{
		if(*p_data != 0x09 || memcmp(p_data + 1, "GFSQ", 4) != 0)
			return FALSE;

		p_server->data = g_new0(gfire_game_server_data, 1);
		p_server->data->driver = &gf_sq_gamespy3_driver;
		p_server->data->ping = p_ping;

		gfire_sq_gamespy3_data *gsdata = p_server->data->proto_data = g_new0(gfire_sq_gamespy3_data, 1);

		sscanf((const char*)p_data + 5, "%d", &gsdata->challenge);
		gsdata->challenge = GINT32_TO_BE(gsdata->challenge);

		return TRUE;
	}

	if(*p_data != 0x00 || memcmp(p_data + 1, "GFSQsplitnum\0", 13) != 0)
			return FALSE;

	guint offset = 14;

	guint8 index = *(p_data + offset) & 0x7f;
	guint8 final = *(p_data + offset) & 0x80;
	offset += 2;
	// TODO: Handle split packets

	return FALSE;
}

static gchar *gfire_sq_gamespy3_details(gfire_game_server *p_server)
{
	return NULL;
}

static void gfire_sq_gamespy3_free_server(gfire_game_server *p_server)
{
	if(p_server->data && p_server->data->proto_data)
	{
		gfire_sq_gamespy3_data *data = (gfire_sq_gamespy3_data*)p_server->data->proto_data;

		g_free(data);
	}
}
