/*
** 1998-09-17 -	A module to deal with the (possible) overwriting of files, and warn the user
**		when that is about to happen. Deleting a file is considered to be an overwrite.
**		Changing protection bits and owner info is not.
** 1999-03-13 -	Changes for the new dialog module.
** 1999-06-12 -	Added "Skip All" button.
** 2001-01-01 -	Added simplistic (and most likely insufficient) recursion-detection.
** 2003-11-23 -	Now tells caller if collision was with directory or file.
*/

#include "gentoo.h"
#include "convutil.h"
#include "dialog.h"
#include "errors.h"
#include "strutil.h"
#include "fileutil.h"
#include "overwrite.h"

/* ----------------------------------------------------------------------------------------- */

static struct {
	guint		level;
	gboolean	do_all;		/* Gets set when user clicks "All". */
	gboolean	skip_all;	/* Gets set when user clicks "Skip All". */
	MainInfo	*min;
	const gchar	*fmt;
	guint32		flags;
	GtkWidget	*label;
	Dialog		*dlg;
} ovw_info = { 0U };		/* Makes sure <level> is 0 on first call. */

/* ----------------------------------------------------------------------------------------- */

/* 1998-09-17 -	Begin a "session" with overwrite protection. The <fmt> string is used in a
**		call to sprintf() with the destination file as argument if we need to inform
**		the user. Very handy. Calls to this function REALLY should nest with calls to
**		ovw_overwrite_end()! Or else.
*/
void ovw_overwrite_begin(MainInfo *min, const gchar *fmt, guint32 flags)
{
	if(ovw_info.level == 0)
	{
		ovw_info.do_all   = FALSE;
		ovw_info.skip_all = FALSE;
		ovw_info.min = min;
		ovw_info.fmt = fmt;
		ovw_info.flags = flags;
		ovw_info.label = gtk_label_new("");
		ovw_info.dlg = dlg_dialog_sync_new(ovw_info.label, _("Please Confirm"), _("_OK|A_ll|_Skip|Skip _All|_Cancel"));
	}
	else
		fprintf(stderr, "OVERWRITE: Mismatched call (level=%d)!\n", ovw_info.level);
	ovw_info.level++;
}

/* ----------------------------------------------------------------------------------------- */

static void get_info(MainInfo *min, gchar *buf, gsize buf_size, const gchar *old_file, const gchar *new_file)
{
	gchar		o_date[64], n_date[64];
	struct stat	o_st, n_st;

	if((stat(old_file, &o_st) == 0) && (stat(new_file, &n_st) == 0))
	{
		gchar	format[256];
		strftime(o_date, sizeof o_date, min->cfg.opt_overwrite.datefmt, localtime(&o_st.st_ctime));
		strftime(n_date, sizeof n_date, min->cfg.opt_overwrite.datefmt, localtime(&n_st.st_ctime));
		/* Build formatting string. This is done as an extra step since we need to use
		 * GLib macros to abstract the exact string, but we want to maintain i18n support.
		 * This happens at interactive rates, so no point in buffering the format.
		 */
		g_snprintf(format, sizeof format,
				_("\nOld: %%%s bytes, changed on %%s,\n"
				  "New: %%%s bytes, changed on %%s."), G_GUINT64_FORMAT, G_GUINT64_FORMAT);
		/* Then try again, plugging in the size and date information. */
		g_snprintf(buf, buf_size, format,  (guint64) o_st.st_size, o_date,  (guint64) n_st.st_size, n_date);
	}
}

/* 2001-01-01 -	Attempt to detect recursive overwrites, i.e. cases like copying the directory
**		"/test" to "/test/a". Incredibly heuristical, this one.
*/
static gboolean check_recursion(const gchar *f_new, const gchar *f_old)
{
	if(f_new != NULL)
	{
		gint	len_old = strlen(f_old), len_new = strlen(f_new);

		if(len_old >= len_new && f_old[len_new] == G_DIR_SEPARATOR && strncmp(f_old, f_new, len_new) == 0)
			return TRUE;
	}
	return FALSE;
}

static OvwRes proceed(const struct stat *stbuf)
{
	return S_ISDIR(stbuf->st_mode) ? OVW_PROCEED_DIR : OVW_PROCEED_FILE;
}

/* 2003-11-23 -	Inform the overwrite system that the file whose complete (absolute) name is
**		in <old_file> is about to be overwritten by <new_file> (if given). Returns:
**		OVW_SKIP	 Skip this file, but continue with the overall operation.
**		OVW_PROCEED	 No collision, proceed.
**		OVW_PROCEED_FILE Collision with file, but proceed anyway.
**		OVW_PROCEED_DIR	 Collision with directory, but proceed anyway.
**		OVW_CANCEL	 Abort the entire operation.
*/
OvwRes ovw_overwrite_file(MainInfo *min, const gchar *old_file, const gchar *new_file)
{
	if(ovw_info.level)
	{
		gchar		buf[PATH_MAX + 2048];
		gint		dlg_res, len;
		struct stat	stbuf;

		if(!(ovw_info.flags & OVWF_NO_RECURSE_TEST) && check_recursion(new_file, old_file))
		{
			errno = EINVAL;
			return OVW_CANCEL;
		}

		err_clear(min);
		if(lstat(old_file, &stbuf) == 0)
		{
			Conv	conv;
			gchar	*old_dpl;

			if(ovw_info.do_all)		/* Already answered "All"? */
				return proceed(&stbuf);
			if(ovw_info.skip_all)		/* Already answered "Skip All"? */
				return OVW_SKIP;

			old_dpl = conv_begin_reverse(&conv, old_file);
			len = g_snprintf(buf, sizeof buf, ovw_info.fmt, old_dpl);
			conv_end(&conv);
			if(new_file != NULL && min->cfg.opt_overwrite.show_info)
				get_info(min, buf + len, sizeof buf - len - 1, old_file, new_file);
			gtk_label_set_text(GTK_LABEL(ovw_info.label), buf);

			dlg_res = dlg_dialog_sync_wait(ovw_info.dlg);
			switch(dlg_res)
			{
				case DLG_POSITIVE:	/* OK ? */
					return proceed(&stbuf);
				case 1:			/* All? */
					ovw_info.do_all = TRUE;
					return proceed(&stbuf);
				case 2:			/* Skip? */
					return OVW_SKIP;
				case 3:
					ovw_info.skip_all = TRUE;
					return OVW_SKIP;
				default:
					return OVW_CANCEL;
			}
		}
		err_clear(min);			/* Don't let EEXIST leak out. */
		return OVW_PROCEED;
	}
	return OVW_CANCEL;
}

/* ----------------------------------------------------------------------------------------- */

void ovw_overwrite_end(MainInfo *min)
{
	if(--ovw_info.level == 0)
	{
		ovw_info.min = NULL;
		ovw_info.fmt = NULL;
		dlg_dialog_sync_destroy(ovw_info.dlg);
		ovw_info.dlg = NULL;
	}
}
