/*
** 1998-05-21 -	Everybody's favourite, the native delete command. Dangerous stuff if you don't
**		quite know what you're doing!
** 1998-05-29 -	Converted to use the new cmd_generic module, looks a lot better and actually IS
**		better, too. Gives user more control.
** 1998-07-28 -	Turned on the NODST flag in the cmd_generic call. Really should have done that
**		earlier! Hm. I guess I need more beta testing.
** 1998-09-18 -	Rewrote large parts of this one. Now uses the standard overwrite protection/
**		confirmation module, rather than rolling its own. No longer uses the generic
**		command interface.
** 1998-12-15 -	Commented out the progress reporting, since it didn't work well.
** 1999-03-06 -	Adjusted for the new selection/dirrow access methods.
** 1999-05-29 -	Restructured. Now exports two utility functions handy to use from elsewhere.
*/

#include "gentoo.h"
#include "cmdseq_config.h"
#include "configure.h"
#include "dialog.h"
#include "dirpane.h"
#include "errors.h"
#include "fileutil.h"
#include "gfam.h"
#include "overwrite.h"
#include "progress.h"

#include "cmd_delete.h"

#define	CMD_ID	"delete"

/* ----------------------------------------------------------------------------------------- */

enum { MODE_ASK = 0, MODE_SET, MODE_DONT_SET };

typedef struct {			/* Options used by the "Delete" command. */
	gboolean	modified;
	gint	set_mode;
} OptDelete;

static OptDelete	delete_options;
static CmdCfg	*delete_cmc = NULL;

/* ----------------------------------------------------------------------------------------- */

static gboolean get_set_mode(MainInfo *min, const gchar *filename)
{
	if(delete_options.set_mode == MODE_ASK)
	{
		gchar	buf[PATH_MAX + 256];
		GtkWidget	*vbox, *label, *check;
		Dialog	*dlg;
		gint	ans;

		vbox  = gtk_vbox_new(FALSE, 0);
		g_snprintf(buf, sizeof buf, _("%s\ncould not be deleted due to access restrictions.\nAttempt to change protection and retry?"), filename);
		label = gtk_label_new(buf);
		gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
		check = gtk_check_button_new_with_label(_("Remember the answer (alters config)"));
		gtk_box_pack_start(GTK_BOX(vbox), check, FALSE, FALSE, 0);
		dlg = dlg_dialog_sync_new(vbox, _("Access Problem"), _("Change|Leave Alone"));
		ans = dlg_dialog_sync_wait(dlg);

		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check)))
		{
			gint	arm;
			if(ans == DLG_POSITIVE)
				arm = MODE_SET;
			else
				arm = MODE_DONT_SET;
			if(delete_options.set_mode != arm)
			{
				delete_options.set_mode = arm;
				delete_options.modified = TRUE;
				cfg_modified_set(min);
			}
		}
		dlg_dialog_sync_destroy(dlg);
		return ans == DLG_POSITIVE;
	}
	return delete_options.set_mode == MODE_SET;
}

/* 2003-10-23 -	Set modes of <filename> to something that really should help with deleting.
**		The file itself must be writable, and for a directory executable, and the
**		parent must be writable. Feels a bit dirty.
*/
static gboolean set_mode(MainInfo *min, const gchar *filename, gboolean dir)
{
	if(get_set_mode(min, filename))
	{
		/* First make actual target writable, and executable if directory. */
		if(chmod(filename, S_IRUSR | S_IWUSR | (dir ? S_IXUSR : 0)) == 0)
		{
			gchar	parent[PATH_MAX];

			/* Now, compute parent filename, and see if it's a directory. */
			if(fut_path_canonicalize(filename, parent, sizeof parent))
			{
				gchar	*sep;

				if((sep = strrchr(parent, G_DIR_SEPARATOR)) != NULL)
				{
					struct stat	pstat;

					*sep = '\0';
					if(lstat(parent, &pstat) == 0)
					{
						if(S_ISDIR(pstat.st_mode))
							return chmod(parent, pstat.st_mode | S_IWUSR) == 0;
					}
				}
			}
		}
	}
	return FALSE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-05-29 -	Delete a plain file. This is capable of removing anything that is
**		not a directory. For that, use del_delete_dir().
*/
gboolean del_delete_file(MainInfo *min, const gchar *path)
{
	gint	ret;

	ret = unlink(path);

	if(ret != 0)
	{
		if(set_mode(min, path, FALSE))
		{
			if(unlink(path) == 0)
				return TRUE;
		}
		err_set(min, errno, CMD_ID, path);
		return FALSE;
	}
	return TRUE;
}

/* ----------------------------------------------------------------------------------------- */

/* 2003-10-23 -	Broken out of del_delete_dir() to facilitate retry. */
static gboolean do_delete_dir(MainInfo *min, const gchar *path, gboolean progress)
{
	gchar		old_dir[PATH_MAX];
	DIR		*dir;
	struct dirent	*de;
	struct stat	stat;

	if(rmdir(path) == 0)					/* If it's empty, this saves time. */
		return 1;
	else if((errno == ENOTEMPTY) || (errno == EEXIST))	/* Wasn't empty. Enter and traverse. */
	{
		err_clear(min);					/* Error was expected, so ignore it. */
		if(fut_cd(path, old_dir, sizeof old_dir))
		{
			if((dir = opendir(".")) != NULL)
			{
				while((de = readdir(dir)) != NULL)
				{
					if(progress)
					{
						pgs_progress_item_begin(min, de->d_name, 0);
						if(pgs_progress_item_update(min, 0) == PGS_CANCEL)
						{
							errno = EINTR;
							break;
						}
						pgs_progress_item_end(min);
					}
					if(!min->cfg.dir_filter(de->d_name))
						continue;
					if(lstat(de->d_name, &stat) == 0)
					{
						if(S_ISDIR(stat.st_mode))
						{
							if(!del_delete_dir(min, de->d_name, progress))
								break;
						}
						else
						{
							if(!del_delete_file(min, de->d_name))
								break;
						}
					}
					else
						break;
				}
				closedir(dir);
			}
			fut_cd(old_dir, NULL, 0);
			if(errno == 0)		/* Don't try to rmdir() again if anything failed above. */
				rmdir(path);
		}
	}
	if(errno)
		err_set(min, errno, CMD_ID, path);
	return (errno == 0) ? TRUE : FALSE;
}

/* 1999-05-29 -	Delete a directory. */
gboolean del_delete_dir(MainInfo *min, const gchar *path, gboolean progress)
{
	if(!do_delete_dir(min, path, progress))
	{
		if(set_mode(min, path, TRUE))
			return do_delete_dir(min, path, progress);
		return FALSE;
	}
	return TRUE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-09-18 -	A new entrypoint for the delete command. Completely replaces the old one,
**		which has been removed.
*/
gint cmd_delete(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	gchar		old_path[PATH_MAX];
	const gchar	*dest;
	gboolean	ok;
	guint		num = 0;
	OvwRes		ores;
	GSList		*slist,	 *iter;

	if(!fut_cd(src->dir.path, old_path, sizeof old_path))
		return 0;
	if((slist = dp_get_selection(src)) == NULL)
		return 1;

	err_clear(min);
	ovw_overwrite_begin(min, _("Really Delete \"%s\"?"), 0UL);
	pgs_progress_begin(min, _("Deleting..."), PFLG_BUSY_MODE);
	fam_rescan_block();
	for(iter = slist; !errno && (iter != NULL); iter = g_slist_next(iter))
	{
		dest = dp_full_name(src, DP_SEL_INDEX(src, iter));
		pgs_progress_item_begin(min, dest, 0);
		if(pgs_progress_item_update(min, 0) == PGS_CANCEL)
		{
			errno = EINTR;
			break;
		}
		pgs_progress_item_end(min);
		ores = ovw_overwrite_file(min, dest, NULL);
		if(ores == OVW_SKIP)
			continue;
		else if(ores == OVW_CANCEL)
			break;
		if(S_ISDIR(DP_SEL_LSTAT(iter).st_mode))
			ok = del_delete_dir(min, dest, TRUE);
		else
			ok = del_delete_file(min, dest);
		if(ok)
		{
			dp_unselect(src, DP_SEL_INDEX(src, iter));
			num++;
		}
		else
			break;
	}
	fam_rescan_unblock();
	pgs_progress_end(min);
	ovw_overwrite_end(min);
	if(num)
		dp_rescan_post_cmd(src);
	dp_free_selection(slist);

	fut_cd(old_path, NULL, 0);
	err_show(min);

	return errno == 0;
}

/* ----------------------------------------------------------------------------------------- */

/* 2003-10-23 -	Configuration initialization. Simple. */
void cfg_delete(MainInfo *min)
{
	if(delete_cmc == NULL)
	{
		/* Set the default values for module's options. */
		delete_options.modified	= FALSE;
		delete_options.set_mode = MODE_ASK;

		delete_cmc = cmc_config_new("Delete", &delete_options);
		cmc_field_add_boolean(delete_cmc, "modified", NULL, offsetof(OptDelete, modified));
		cmc_field_add_enum   (delete_cmc, "set_mode", _("On Access Failure"), offsetof(OptDelete, set_mode),
							  _("Ask User|Automatically Try Changing, and Retry|Fail"));
		cmc_config_register(delete_cmc);
	}
}
