"""Python module for flow properties of perfect gases with friction

In Fanno flow, friction in the passage leads to increase in entropy
and takes the process towards the choking conditions, (M = M* = 1).

Assumptions:
1) One dimensional flow with friction
2) Constant area duct
3) perfect gas with constant specific heats and molecular weights
4) adiabatic flow

"""
"""
* Copyright (C) 2006 Varun Hiremath.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * Authors: Varun Hiremath, Venkattraman A
 * Version 0.1
"""

from scipy import optimize
from math import *

class FannoFlow :
    def __init__(self, g= 1.4):
        """ g is the value of gamma (ratio of specific heats), default = 1.4
        """
        self.g = g

    def get_P_by_Pstar_from_M(self,M):
        return ((self.g+1)/2/(1+ (self.g -1)/ 2*M**2))**(0.5)/M

    def get_rho_by_rhostar_from_M(self,M):
        return ((self.g+1)/2/(1+ (self.g -1)/ 2*M**2))**(0.5)*M

    def get_T_by_Tstar_from_M(self,M):
        return (self.g+1)/2/(1+ (self.g -1)/ 2*M**2)

    def get_Po_by_Postar_from_M(self,M):
        return (2*(1+(self.g-1)/2*M**2)/(self.g+1))**((self.g+1)/(self.g-1)/2)/M

    def get_F_by_Fstar_from_M(self,M):
        return (1+self.g*M**2)/M/(2*(self.g+1)*(1+(self.g-1)/2*M**2))**0.5

    def get_4fLmax_by_D_from_M(self,M):
        return (1-M**2)/self.g/M**2 + (self.g+1)/2/self.g*log((self.g+1)*M**2/2/(1+(self.g-1)/2*M**2))

    def get_M_from_P_by_Pstar(self,P_by_Pstar):
        return optimize.fsolve(lambda M : self.get_P_by_Pstar_from_M(M) - P_by_Pstar, 0.001)

    def get_rho_by_rhostar_from_P_by_Pstar(self,P_by_Pstar):
        M = self.get_M_from_P_by_Pstar(P_by_Pstar)
        return self.get_rho_by_rhostar_from_M(M)

    def get_T_by_Tstar_from_P_by_Pstar(self,P_by_Pstar):
        M = self.get_M_from_P_by_Pstar(P_by_Pstar)
        return self.get_T_by_Tstar_from_M(M)

    def get_Po_by_Postar_from_P_by_Pstar(self,P_by_Pstar):
        M = self.get_M_from_P_by_Pstar(P_by_Pstar)
        return self.get_Po_by_Postar_from_M(M)

    def get_F_by_Fstar_from_P_by_Pstar(self,P_by_Pstar):
        M = self.get_M_from_P_by_Pstar(P_by_Pstar)
        return self.get_F_by_Fstar_from_M(M)

    def get_4fLmax_by_D_from_P_by_Pstar(self,P_by_Pstar):
        M = self.get_M_from_P_by_Pstar(P_by_Pstar)
        return self.get_4fLmax_by_D_from_M(M)

    def get_M_from_rho_by_rhostar(self,rho_by_rhostar):
        return optimize.fsolve(lambda M : self.get_rho_by_rhostar_from_M(M) - rho_by_rhostar, 0.001)

    def get_P_by_Pstar_from_rho_by_rhostar(self,rho_by_rhostar):
        M = self.get_M_from_rho_by_rhostar(rho_by_rhostar)
        return self.get_P_by_Pstar_from_M(M)

    def get_T_by_Tstar_from_rho_by_rhostar(self,rho_by_rhostar):
        M = self.get_M_from_rho_by_rhostar(rho_by_rhostar)
        return self.get_T_by_Tstar_from_M(M)

    def get_Po_by_Postar_from_rho_by_rhostar(self,rho_by_rhostar):
        M = self.get_M_from_rho_by_rhostar(rho_by_rhostar)
        return self.get_Po_by_Postar_from_M(M)

    def get_F_by_Fstar_from_rho_by_rhostar(self,rho_by_rhostar):
        M = self.get_M_from_rho_by_rhostar(rho_by_rhostar)
        return self.get_F_by_Fstar_from_M(M)

    def get_4fLmax_by_D_from_rho_by_rhostar(self,rho_by_rhostar):
        M = self.get_M_from_rho_by_rhostar(rho_by_rhostar)
        return self.get_4fLmax_by_D_from_M(M)

    def get_M_from_T_by_Tstar(self,T_by_Tstar):
        return optimize.fsolve(lambda M : self.get_T_by_Tstar_from_M(M) - T_by_Tstar, 0.0)

    def get_P_by_Pstar_from_T_by_Tstar(self,T_by_Tstar):
        M = self.get_M_from_T_by_Tstar(T_by_Tstar)
        return self.get_P_by_Pstar_from_M(M)

    def get_rho_by_rhostar_from_T_by_Tstar(self,T_by_Tstar):
        M = self.get_M_from_T_by_Tstar(T_by_Tstar)
        return self.get_rho_by_rhostar_from_M(M)

    def get_Po_by_Postar_from_T_by_Tstar(self,T_by_Tstar):
        M = self.get_M_from_T_by_Tstar(T_by_Tstar)
        return self.get_Po_By_Postar_from_M(M)

    def get_F_by_Fstar_from_T_by_Tstar(self,T_by_Tstar):
        M = self.get_M_from_T_by_Tstar(T_by_Tstar)
        return self.get_F_by_Fstar_from_M(M)

    def get_4fLmax_by_D_from_T_by_Tstar(self,T_by_Tstar):
        M = self.get_M_from_T_by_Tstar(T_by_Tstar)
        return self.get_4fLmax_by_D_from_M(M)

    def get_M_from_Po_by_Postar(self,Po_by_Postar):
        return (optimize.bisect(lambda M : self.get_Po_by_Postar_from_M(M) - Po_by_Postar, 0.001, 1.0),optimize.bisect(lambda M : self.get_Po_by_Postar_from_M(M) - Po_by_Postar, 1.0, 100))

    def get_M_from_F_by_Fstar(self,F_by_Fstar):
        return (optimize.bisect(lambda M : self.get_F_by_Fstar_from_M(M) - F_by_Fstar, 0.001, 1.0),optimize.bisect(lambda M : self.get_F_by_Fstar_from_M(M) - F_by_Fstar, 1.0, 100))

    def get_M_from_4fLmax_by_D(self,fLmax_by_D):
        return (optimize.bisect(lambda M : self.get_4fLmax_by_D_from_M(M) - fLmax_by_D, 0.001, 1.0),optimize.bisect(lambda M : self.get_4fLmax_by_D_from_M(M) - fLmax_by_D, 1.0, 100))
