(**************************************************************************)
(*                                                                        *)
(*  This file is part of Frama-C.                                         *)
(*                                                                        *)
(*  Copyright (C) 2007-2009                                               *)
(*    CEA (Commissariat  l'nergie Atomique)                             *)
(*                                                                        *)
(*  you can redistribute it and/or modify it under the terms of the GNU   *)
(*  Lesser General Public License as published by the Free Software       *)
(*  Foundation, version 2.1.                                              *)
(*                                                                        *)
(*  It is distributed in the hope that it will be useful,                 *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU Lesser General Public License for more details.                   *)
(*                                                                        *)
(*  See the GNU Lesser General Public License version 2.1                 *)
(*  for more details (enclosed in the file licenses/LGPLv2.1).            *)
(*                                                                        *)
(**************************************************************************)

(* modified by CEA *)

(* $Id: rangemap.ml,v 1.3 2008-11-04 10:05:05 uid568 Exp $ *)

module type OrderedType =
  sig
    type t
    val compare: t -> t -> int
  end

type fuzzy_order = Above | Below | Match

module type S =
  sig
    type key
    (* type +'a t *)
    type +'a t =
        Empty
      | Node of 'a t * key * 'a * 'a t * int
    val empty: 'a t
    val is_empty: 'a t -> bool
    val add: key -> 'a -> 'a t -> 'a t
    val find: key -> 'a t -> 'a
    val remove: key -> 'a t -> 'a t
    val mem:  key -> 'a t -> bool
    val iter: (key -> 'a -> unit) -> 'a t -> unit
    val map: ('a -> 'b) -> 'a t -> 'b t
    val mapi: (key -> 'a -> 'b) -> 'a t -> 'b t
    val mapii: (key -> 'a -> key*'b) -> 'a t -> 'b t
    val fold: (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
    val compare: ('a -> 'a -> int) -> 'a t -> 'a t -> int
    val equal: ('a -> 'a -> bool) -> 'a t -> 'a t -> bool
    val fold_range: (key -> fuzzy_order) -> 
                       (key -> 'a -> 'b -> 'b) -> 'a t -> 'b -> 'b
    val height : 'a t -> int
    val concerned_intervals: (key -> key -> fuzzy_order) -> 
      key -> 'a t -> (key*'a) list
    exception Empty_rangemap
    val lowest_binding : 'a t -> (key * 'a)
    exception No_such_binding
   val lowest_binding_above : (key -> bool) -> 'a t -> key * 'a
    val merge : 'a t -> 'a t -> 'a t
    val add_whole : (key -> key -> fuzzy_order) -> key -> 'a -> 'a t -> 'a t
    val remove_whole : (key -> key -> fuzzy_order) -> key -> 'a t -> 'a t
  end

module Make(Ord: OrderedType) = struct

    type key = Ord.t

    type 'a t =
        Empty
      | Node of 'a t * key * 'a * 'a t * int

    let height = function
        Empty -> 0
      | Node(_,_,_,_,h) -> h

    let create l x d r =
      let hl = height l and hr = height r in
      Node(l, x, d, r, (if hl >= hr then hl + 1 else hr + 1))

    let bal l x d r =
      let hl = match l with Empty -> 0 | Node(_,_,_,_,h) -> h in
      let hr = match r with Empty -> 0 | Node(_,_,_,_,h) -> h in
      if hl > hr + 2 then begin
        match l with
          Empty -> invalid_arg "Rangemap.bal"
        | Node(ll, lv, ld, lr, _) ->
            if height ll >= height lr then
              create ll lv ld (create lr x d r)
            else begin
              match lr with
                Empty -> invalid_arg "Rangemap.bal"
              | Node(lrl, lrv, lrd, lrr, _)->
                  create (create ll lv ld lrl) lrv lrd (create lrr x d r)
            end
      end else if hr > hl + 2 then begin
        match r with
          Empty -> invalid_arg "Rangemap.bal"
        | Node(rl, rv, rd, rr, _) ->
            if height rr >= height rl then
              create (create l x d rl) rv rd rr
            else begin
              match rl with
                Empty -> invalid_arg "Rangemap.bal"
              | Node(rll, rlv, rld, rlr, _) ->
                  create (create l x d rll) rlv rld (create rlr rv rd rr)
            end
      end else
        Node(l, x, d, r, (if hl >= hr then hl + 1 else hr + 1))

    let empty = Empty

    let is_empty = function Empty -> true | _ -> false

    let rec add x data = function
        Empty ->
          Node(Empty, x, data, Empty, 1)
      | Node(l, v, d, r, h) ->
          let c = Ord.compare x v in
          if c = 0 then
            Node(l, x, data, r, h)
          else if c < 0 then
            bal (add x data l) v d r
          else
            bal l v d (add x data r)

   let rec find x = function
        Empty ->
          raise Not_found
      | Node(l, v, d, r, _) ->
          let c = Ord.compare x v in
          if c = 0 then d
          else find x (if c < 0 then l else r)

    let rec mem x = function
        Empty ->
          false
      | Node(l, v, _d, r, _) ->
          let c = Ord.compare x v in
          c = 0 || mem x (if c < 0 then l else r)

    let rec min_binding = function
        Empty -> raise Not_found
      | Node(Empty, x, d, _r, _) -> (x, d)
      | Node(l, _x, _d, _r, _) -> min_binding l

    let rec remove_min_binding = function
        Empty -> invalid_arg "Rangemap.remove_min_elt"
      | Node(Empty, _x, _d, r, _) -> r
      | Node(l, x, d, r, _) -> bal (remove_min_binding l) x d r

    let merge t1 t2 =
      match (t1, t2) with
        (Empty, t) -> t
      | (t, Empty) -> t
      | (_, _) ->
          let (x, d) = min_binding t2 in
          bal t1 x d (remove_min_binding t2)

    let rec remove x = function
        Empty ->
          Empty
      | Node(l, v, d, r, _h) ->
          let c = Ord.compare x v in
          if c = 0 then
            merge l r
          else if c < 0 then
            bal (remove x l) v d r
          else
            bal l v d (remove x r)

    let rec iter f = function
        Empty -> ()
      | Node(l, v, d, r, _) ->
          iter f l; f v d; iter f r

    let rec map f = function
        Empty               -> Empty
      | Node(l, v, d, r, h) -> Node(map f l, v, f d, map f r, h)

    let rec mapi f = function
        Empty               -> Empty
      | Node(l, v, d, r, h) -> Node(mapi f l, v, f v d, mapi f r, h)

    let rec mapii f = function
      | Empty -> Empty
      | Node(l, v, d, r, h) -> 
	  let new_v, new_d = f v d in
	  Node(mapii f l, new_v, new_d, mapii f r, h)

    (* Modified from Caml library : fold applies [f] to bindings in order. *)
    let rec fold f m accu =
      match m with
        Empty -> accu
      | Node(l, v, d, r, _) ->
          fold f r (f v d (fold f l accu))

    type 'a enumeration = End | More of key * 'a * 'a t * 'a enumeration

    let rec cons_enum m e =
      match m with
        Empty -> e
      | Node(l, v, d, r, _) -> cons_enum l (More(v, d, r, e))

    let compare cmp m1 m2 =
      let rec compare_aux e1 e2 =
          match (e1, e2) with
          (End, End) -> 0
        | (End, _)  -> -1
        | (_, End) -> 1
        | (More(v1, d1, r1, e1), More(v2, d2, r2, e2)) ->
            let c = Ord.compare v1 v2 in
            if c <> 0 then c else
            let c = cmp d1 d2 in
            if c <> 0 then c else
            compare_aux (cons_enum r1 e1) (cons_enum r2 e2)
      in compare_aux (cons_enum m1 End) (cons_enum m2 End)

    let equal cmp m1 m2 =
      let rec equal_aux e1 e2 =
          match (e1, e2) with
          (End, End) -> true
        | (End, _)  -> false
        | (_, End) -> false
        | (More(v1, d1, r1, e1), More(v2, d2, r2, e2)) ->
            Ord.compare v1 v2 = 0 && cmp d1 d2 &&
            equal_aux (cons_enum r1 e1) (cons_enum r2 e2)
      in equal_aux (cons_enum m1 End) (cons_enum m2 End)

    let rec fold_range o f m accu =
      match m with
        Empty -> accu
      | Node(l, v, d, r, _) ->
	  let compar = o v in
	  let accu1 = 
	    if compar = Match || compar = Above
	    then fold_range o f l accu
	    else accu
	  in
	  let accu2 =
	    if compar = Match 
	    then f v d accu1
	    else accu1
	  in
	  if compar = Match || compar = Below
	  then fold_range o f r accu2
	  else accu2
    
    let cons k v l = (k,v) :: l
      
    let concerned_intervals fuzzy_order i m =       
      fold_range (fuzzy_order i) cons m []

    let remove_whole fuzzy_order i m =
      fold_range (fuzzy_order i) (fun k _v acc -> remove k acc) m m

    let add_whole fuzzy_order i v m =
      let removed = remove_whole fuzzy_order i m
      in
      add i v removed

    exception Empty_rangemap

    let rec lowest_binding m =
      match m with
	Node(Empty,k,v,_,_) -> k,v
      | Node(t,_,_,_,_) -> lowest_binding t
      | Empty -> raise Empty_rangemap

    exception No_such_binding

    let rec lowest_binding_above o m =
      match m with
      | Node(l,k,v,r,_) -> 
	  if o k
	  then begin
	    try
	      lowest_binding_above o l
	    with No_such_binding -> k,v
	  end
	  else lowest_binding_above o r
      | Empty -> raise No_such_binding

    let merge m1 m2 =
      fold (fun k v acc -> add k v acc) m1 m2
	
end
