# Skeleton FilterProxy Module

# This is an example module for those interested in writing new modules.
# It simply adds a string to the beginning of a web page, nothing more.

# Copyright (C) 2000, 2001 Bob McElrath.
# See the file COPYING for redistribution and modification terms.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# The lifetime of a single HTTP request through the proxy has *4* pieces.
#
#   1) client -> proxy    
#   2) proxy -> server    (between these two stages, $order < 0)
#   3) server -> proxy
#   4) proxy -> client    (between these two stages, $order > 0)
#
# This leaves TWO places for filters to intervene, between stages 1 and 2;
# and between stages 3 and 4.  (see $CONFIG->{order} and the $order paraemter
# below).

package FilterProxy::Skeleton;

use strict;
no strict 'subs';
use vars qw($VERSION $CONFIG);  # The $CONFIG variable MUST be exported.
                                # and should be a reference to a hash.
                                # keys to this hash are configuration
                                # variables, and values are used by this 
                                # module.

push @FilterProxy::MODULES,     # add myself to FilterProxy's list of known 
   "Skeleton";                  # modules
$VERSION = 0.01;
$CONFIG = {};                   # my $CONFIG.  This will be saved to and
                                # loaded from the config file by FilterProxy
                                # Module should define a basic CONFIG just in
                                # case this is the first time this module
                                # has ever been used.

$CONFIG->{order} = 4;           # must be a value in the range -10..10
                                # which specifies where in the chain of
                                # filters it wants to be executed.
                                # < 0: called before request is sent
                                # > 0: called after content is received
                                # -10    Outgoing header manglers
                                # -9..-1 Outgoing content manglers.
                                # -9..-7 May decode outgoing content.
                                # -6..-4 May modify outgoing content.
                                # -3..-1 May encode outgoing content.
                                # 0      not used
                                #  1..3: May decode incoming content
                                #  4..6: May modify incoming content
                                #  7..9: May encode incoming content
                                # 10: Must be called after all filters. 
                                #   (final header manipulation)
                                # This may also be an anonymous list ref if
                                # the module needs to be called more than once
                                # for a given request. (i.e. header mungers)
                                # 1-9 can expect content, 10 should not 
                                # depend on it.

$CONFIG->{mime_types} = [       # is a list of mime-types that this module
  "text/html",                  # knows how to filter.  Can contain an empty
  "text/plain",                 # string to specify all mime-types.
];

*logger = \&FilterProxy::logger;# Tells perl to use FilterProxy's logger function
                                # instead of looking for 
                                # FilterProxy::Skeleton::logger (this is because
                                # I'm lazy and don't want to type 
                                # FilterProxy::logger every time)
                                # Using this function will cause data to go to 
                                # FilterProxy's log file.

sub filter {                    # The main filtering routine.  Does the dirty work.
  my($req) = shift;             # $req is the HTTP::Request object 
                                #   This is *always* the request sent from 
                                #   client->proxy (don't modify it!)
  my($res) = shift;             # $res is the HTTP::Response object.  (your filter may modify it)
                                #   If $order < 0, this is the client->proxy request
                                #   If $order > 0, this is the server->client response
  my($siteconfig) = shift;      # $siteconfig is a reference to a list of
                                # the site-specific configuration(s) for the
                                # site(s).  non site-specific config should 
                                # be in the
                                # $CONFIG variable (see above).
  my($order) = shift;           # The order (in case your filter can be called
                                # for more than one order).  See $CONFIG->{order} above.

                                # Module must be able to handle it if any of 
                                # the above parameters are undef.
  if(defined $res) {
    ${$res->content_ref} = 
      "<h1><center>FilterProxy::Skeleton modified this file</center></h1>"
      . ${$res->content_ref};
  }
  logger(DEBUG, "FilterProxy::Skeleton::filter called.");
                                # $res->content_ref is an efficent way of
                                # accessing the document. (See HTTP::Message
                                # man page).
  return;                       # nothing needs to be returned.
}

sub Config {                    # parses the CGI parameters when the user
                                # modifies the configuration for this module.
                                # There should also be a .html file with the
                                # same name as the module, which contains the
                                # forms processed by this CGI.
  my($req, $cgi, $siteconfig) = @_;

  return "It worked.\n";        # a HTML-formatted message that will be 
                                # available to your .html file as $ENV{MESSAGE}
}

# NOTES:
#
# Is it advantageous to make FilterProxy::Module, and make everything an object?
