/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file vcard.c
 * \brief vcard functions
 */

#include <ffgtk.h>
#include <addressbook.h>
#include <vcard.h>

static GList *psVcardList = NULL;
static GList *psVcard = NULL;
static struct sCardData *psCurrentCardData = NULL;
static GString *psCurrentString = NULL;
static gint nState = STATE_NEW;
static gint nCurrentPosition = 0;
static GString *psFullName = NULL;
static GString *psFirstName = NULL;
static GString *psLastName = NULL;
static GString *psUid = NULL;
static GString *psCompany = NULL;
static GString *psTitle = NULL; 
static GString *psPrivateStreet = NULL;
static GString *psPrivateCity = NULL;
static GString *psPrivateZipCode = NULL;
static GString *psPrivateCountry = NULL;
static GString *psBusinessStreet = NULL;
static GString *psBusinessCity = NULL;
static GString *psBusinessZipCode = NULL;
static GString *psBusinessCountry = NULL;
static GString *psBusinessFax = NULL;
static GString *psPrivateFax = NULL;
static GString *psCellPhone = NULL;
static GString *psHomePhone = NULL;
static GString *psWorkPhone = NULL;
static GdkPixbuf *psImage = NULL;

/**
 * \brief Free header, options and entry line
 * \param psCard pointer to card structure
 */
static void freeData( struct sCardData *psCardData ) {
	/* if header is present, free it and set to NULL */
	if ( psCardData -> pnHeader != NULL ) {
		g_free( psCardData -> pnHeader );
		psCardData -> pnHeader = NULL;
	}

	/* if options is present, free it and set to NULL */
	if ( psCardData -> pnOptions != NULL ) {
		g_free( psCardData -> pnOptions );
		psCardData -> pnOptions = NULL;
	}

	/* if entry is present, free it and set to NULL */
	if ( psCardData -> pnEntry != NULL ) {
		g_free( psCardData -> pnEntry );
		psCardData  -> pnEntry = NULL;
	}

	g_free( psCardData );
}

/**
 * \brief Process first/last name structure
 * \param psCard pointer to card structure
 */
static void processFirstLastName( struct sCardData *psCardData ) {
	gint nLen = 0;
	gint nIndex = 0;

	if ( psCardData == NULL || psCardData -> pnEntry == NULL ) {
		return;
	}

	nLen = strlen( psCardData -> pnEntry );

	/* Create last name string */
	psLastName = g_string_new( "" );
	while ( nIndex < nLen ) {
		if (
			( psCardData -> pnEntry[ nIndex ] != 0x00 ) &&
			( psCardData -> pnEntry[ nIndex ] != ';' ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0A ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0D ) ) {
			g_string_append_c( psLastName, psCardData -> pnEntry[ nIndex++ ] );
		} else {
			break;
		}
	}

	/* Skip ';' */
	nIndex++;

	/* Create first name string */
	psFirstName = g_string_new( "" );
	while ( nIndex < nLen ) {
		if (
			( psCardData -> pnEntry[ nIndex ] != 0x00 ) &&
			( psCardData -> pnEntry[ nIndex ] != ';' ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0A ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0D ) ) {
			g_string_append_c( psFirstName, psCardData -> pnEntry[ nIndex++ ] );
		} else {
			break;
		}
	}
}

/**
 * \brief Process full name structure
 * \param psCard pointer to card structure
 */
static void processFullName( struct sCardData *psCardData ) {
	gint nLen = 0;
	gint nIndex = 0;

	if ( psCardData == NULL || psCardData -> pnEntry == NULL ) {
		return;
	}

	nLen = strlen( psCardData -> pnEntry );

	/* Create fullname string */
	psFullName = g_string_new( "" );
	for ( nIndex = 0; nIndex < nLen; nIndex++ ) {
		if (
			( psCardData -> pnEntry[ nIndex ] != 0x00 ) &&
			( psCardData -> pnEntry[ nIndex ] != ';' ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0A ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0D ) ) {
			g_string_append_c( psFullName, psCardData -> pnEntry[ nIndex ] );
		} else {
			break;
		}
	}
}

/**
 * \brief Process organization structure
 * \param psCard pointer to card structure
 */
static void processOrganization( struct sCardData *psCardData ) {
	gint nLen = 0;
	gint nIndex = 0;

	if ( psCardData == NULL || psCardData -> pnEntry == NULL ) {
		return;
	}

	nLen = strlen( psCardData -> pnEntry );

	/* Create company string */
	psCompany = g_string_new( "" );
	while ( nIndex < nLen ) {
		if (
			( psCardData -> pnEntry[ nIndex ] != 0x00 ) &&
			( psCardData -> pnEntry[ nIndex ] != ';' ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0A ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0D ) ) {
			g_string_append_c( psCompany, psCardData -> pnEntry[ nIndex++ ] );
		} else {
			break;
		}
	}
}

/**
 * \brief Process title structure
 * \param psCard pointer to card structure
 */
static void processTitle( struct sCardData *psCardData ) {
	gint nLen = 0;
	gint nIndex = 0;

	if ( psCardData == NULL || psCardData -> pnEntry == NULL ) {
		return;
	}

	nLen = strlen( psCardData -> pnEntry );

	/* Create title string */
	psTitle = g_string_new( "" );
	while ( nIndex < nLen ) {
		if (
			( psCardData -> pnEntry[ nIndex ] != 0x00 ) &&
			( psCardData -> pnEntry[ nIndex ] != ';' ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0A ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0D ) ) {
			g_string_append_c( psTitle, psCardData -> pnEntry[ nIndex++ ] );
		} else {
			break;
		}
	}
}

/**
 * \brief Process uid structure
 * \param psCard pointer to card structure
 */
static void processUid( struct sCardData *psCardData ) {
	gint nLen = 0;
	gint nIndex = 0;

	if ( psCardData == NULL || psCardData -> pnEntry == NULL ) {
		return;
	}

	nLen = strlen( psCardData -> pnEntry );

	/* Create uid string */
	psUid = g_string_new( "" );
	while ( nIndex < nLen ) {
		if (
			( psCardData -> pnEntry[ nIndex ] != 0x00 ) &&
			( psCardData -> pnEntry[ nIndex ] != ';' ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0A ) &&
			( psCardData -> pnEntry[ nIndex ] != 0x0D ) ) {
			g_string_append_c( psUid, psCardData -> pnEntry[ nIndex++ ] );
		} else {
			break;
		}
	}
}

/**
 * \brief Process address structure
 * \param psCard pointer to card structure
 */
static void processAddress( struct sCardData *psCardData ) {
	gchar *pnTmp = NULL;

	if ( psCardData == NULL || psCardData -> pnEntry == NULL ) {
		return;
	}

	if ( psCardData -> pnOptions == NULL ) {
		Debug( KERN_DEBUG, "No options for address, skipping..\n" );
		return;
	}

	pnTmp = psCardData -> pnEntry;

	/* Create address string */
	if ( g_strcasestr( psCardData -> pnOptions, "HOME" ) != NULL ) {
		/* skip pobox */
		while ( *pnTmp != ';' ) {
			pnTmp++;
		}
		pnTmp++;

		/* skip extended address */
		while ( *pnTmp != ';' ) {
			pnTmp++;
		}
		pnTmp++;

		/* read street */
		psPrivateStreet = g_string_new( "" );
		while ( *pnTmp != ';' ) {
			g_string_append_c( psPrivateStreet, *pnTmp );
			pnTmp++;
		}
		pnTmp++;

		/* read locality */
		psPrivateCity = g_string_new( "" );
		while ( *pnTmp != ';' ) {
			g_string_append_c( psPrivateCity, *pnTmp );
			pnTmp++;
		}
		pnTmp++;

		/* skip region */
		while ( *pnTmp != ';' ) {
			pnTmp++;
		}
		pnTmp++;

		/* read zip code */
		psPrivateZipCode = g_string_new( "" );
		while ( *pnTmp != ';' ) {
			g_string_append_c( psPrivateZipCode, *pnTmp );
			pnTmp++;
		}
		pnTmp++;

		/* read country */
		psPrivateCountry = g_string_new( "" );
		while ( *pnTmp != 0x00 && *pnTmp != 0x0A && *pnTmp != 0x0D ) {
			g_string_append_c( psPrivateCountry, *pnTmp );
			pnTmp++;
		}
	} else if ( g_strcasestr( psCardData -> pnOptions, "WORK" ) != NULL ) {
		/* skip pobox */
		while ( *pnTmp != ';' ) {
			pnTmp++;
		}
		pnTmp++;

		/* skip extended address */
		while ( *pnTmp != ';' ) {
			pnTmp++;
		}
		pnTmp++;

		/* read street */
		psBusinessStreet = g_string_new( "" );
		while ( *pnTmp != ';' ) {
			g_string_append_c( psBusinessStreet, *pnTmp );
			pnTmp++;
		}
		pnTmp++;

		/* read locality */
		psBusinessCity = g_string_new( "" );
		while ( *pnTmp != ';' ) {
			g_string_append_c( psBusinessCity, *pnTmp );
			pnTmp++;
		}
		pnTmp++;

		/* skip region */
		while ( *pnTmp != ';' ) {
			pnTmp++;
		}
		pnTmp++;

		/* read zip code */
		psBusinessZipCode = g_string_new( "" );
		while ( *pnTmp != ';' ) {
			g_string_append_c( psBusinessZipCode, *pnTmp );
			pnTmp++;
		}
		pnTmp++;

		/* read country */
		psBusinessCountry = g_string_new( "" );
		while ( *pnTmp != 0x00 && *pnTmp != 0x0A && *pnTmp != 0x0D ) {
			g_string_append_c( psBusinessCountry, *pnTmp );
			pnTmp++;
		}
		pnTmp++;
	}
}

/**
 * \brief Process telephone structure
 * \param psCard pointer to card structure
 */
static void processTelephone( struct sCardData *psCardData ) {
	gchar *pnTmp = psCardData -> pnEntry;

	if ( psCardData -> pnOptions == NULL ) {
		Debug( KERN_WARNING, "No option field in telephone entry\n" );
		return;
	}

	if ( g_strcasestr( psCardData -> pnOptions, "FAX" ) != NULL ) {
		if ( g_strcasestr( psCardData -> pnOptions, "WORK" ) != NULL ) {
			if ( psBusinessFax == NULL ) {
				psBusinessFax = g_string_new( "" );
				while ( *pnTmp != 0x00 && *pnTmp != 0x0A && *pnTmp != 0x0D ) {
					g_string_append_c( psBusinessFax, *pnTmp );
					pnTmp++;
				}
			}
		} else {
			if ( psPrivateFax == NULL ) {
				psPrivateFax = g_string_new( "" );
				while ( *pnTmp != 0x00 && *pnTmp != 0x0A && *pnTmp != 0x0D ) {
					g_string_append_c( psPrivateFax, *pnTmp );
					pnTmp++;
				}
			}
		}
	} else {
		/* Check for cell phone number, and create string if needed */
		if ( g_strcasestr( psCardData -> pnOptions, "CELL" ) != NULL ) {
			psCellPhone = g_string_new( "" );
			while ( *pnTmp != 0x00 && *pnTmp != 0x0A && *pnTmp != 0x0D ) {
				g_string_append_c( psCellPhone, *pnTmp );
				pnTmp++;
			}
		}

		/* Check for home phone number, and create string if needed */
		if ( g_strcasestr( psCardData -> pnOptions, "HOME" ) != NULL ) {
			psHomePhone = g_string_new( "" );
			while ( *pnTmp != 0x00 && *pnTmp != 0x0A && *pnTmp != 0x0D ) {
				g_string_append_c( psHomePhone, *pnTmp );
				pnTmp++;
			}
		}

		/* Check for work phone number, and create string if needed */
		if ( g_strcasestr( psCardData -> pnOptions, "WORK" ) != NULL ) {
			psWorkPhone = g_string_new( "" );
			while ( *pnTmp != 0x00 && *pnTmp != 0x0A && *pnTmp != 0x0D ) {
				g_string_append_c( psWorkPhone, *pnTmp );
				pnTmp++;
			}
		}
	}
}

/**
 * \brief Process photo structure
 * \param psCard pointer to card structure
 */
static void processPhoto( struct sCardData *psCardData ) {
	GdkPixbufLoader *psLoader;
	guchar *pnImage;
	gsize nLen;
	GError *psError = NULL;

	pnImage = g_base64_decode( psCardData -> pnEntry, &nLen );
	psLoader = gdk_pixbuf_loader_new();
	if ( gdk_pixbuf_loader_write( psLoader, pnImage, nLen, &psError ) ) {
		psImage = gdk_pixbuf_loader_get_pixbuf( psLoader );
	} else {
		Debug( KERN_DEBUG, "Error!! (%s)\n", psError -> message );
		g_free( pnImage );
	}
}

/**
 * \brief Create new uid
 */
GString *createUid( void ) {
	GString *psId = g_string_new( "" );
	gint nIndex = 0;

	for ( nIndex = 0; nIndex < 10; nIndex++ ) {
		int nRandom = g_random_int() % 62;
		nRandom += 48;
		if ( nRandom > 57 ) {
			nRandom += 7;
		}

		if ( nRandom > 90 ) {
			nRandom += 6;
		}

		psId = g_string_append_c( psId, ( char ) nRandom );
	}

	return psId;
}

/**
 * \brief Parse end of vcard, check for valid entry and add person
 */
static void processCardEnd( void ) {
	GHashTable *psTable = NULL;
	gchar *pnName = NULL;

	psTable = g_hash_table_new( NULL, NULL );

	if ( psUid == NULL ) {
		psUid = createUid();
	}
	AddInfo( psTable, PERSON_ID, psUid -> str );

	if ( psFirstName != NULL ) {
		AddInfo( psTable, PERSON_FIRST_NAME, psFirstName -> str );
	}
	if ( psLastName != NULL ) {
		AddInfo( psTable, PERSON_LAST_NAME, psLastName -> str );
	}
	if ( psCompany != NULL ) {
		AddInfo( psTable, PERSON_COMPANY, psCompany -> str );
	}
	if ( psWorkPhone != NULL ) {
		AddInfo( psTable, PERSON_BUSINESS_PHONE, psWorkPhone -> str );
	}
	if ( psHomePhone != NULL ) {
		AddInfo( psTable, PERSON_PRIVATE_PHONE, psHomePhone -> str );
	}
	if ( psPrivateFax != NULL ) {
		AddInfo( psTable, PERSON_PRIVATE_FAX, psPrivateFax -> str );
	}
	if ( psBusinessFax != NULL ) {
		AddInfo( psTable, PERSON_BUSINESS_FAX, psBusinessFax -> str );
	}
	if ( psCellPhone != NULL ) {
		AddInfo( psTable, PERSON_PRIVATE_MOBILE, psCellPhone -> str );
	}
	if ( psTitle != NULL ) {
		AddInfo( psTable, PERSON_TITLE, psTitle -> str );
	}
	if ( psPrivateStreet != NULL ) {
		AddInfo( psTable, PERSON_PRIVATE_STREET, psPrivateStreet -> str );
	}
	if ( psPrivateCity != NULL ) {
		AddInfo( psTable, PERSON_PRIVATE_CITY, psPrivateCity -> str );
	}
	if ( psPrivateZipCode != NULL ) {
		AddInfo( psTable, PERSON_PRIVATE_ZIPCODE, psPrivateZipCode -> str );
	}
	if ( psPrivateCountry != NULL ) {
		AddInfo( psTable, PERSON_PRIVATE_COUNTRY, psPrivateCountry -> str );
	}
	if ( psBusinessStreet != NULL ) {
		AddInfo( psTable, PERSON_BUSINESS_STREET, psBusinessStreet -> str );
	}
	if ( psBusinessCity != NULL ) {
		AddInfo( psTable, PERSON_BUSINESS_CITY, psBusinessCity -> str );
	}
	if ( psBusinessZipCode != NULL ) {
		AddInfo( psTable, PERSON_BUSINESS_ZIPCODE, psBusinessZipCode -> str );
	}
	if ( psBusinessCountry != NULL ) {
		AddInfo( psTable, PERSON_BUSINESS_COUNTRY, psBusinessCountry -> str );
	}
	if ( psImage != NULL ) {
		AddInfo( psTable, PERSON_IMAGE, ( const gchar * ) psImage );
	}

	if ( psFullName != NULL ) {
		AddInfo( psTable, PERSON_DISPLAY_NAME, psFullName -> str );
	} else if ( psFirstName != NULL && psLastName != NULL ) {
		pnName = g_strdup_printf( "%s %s", psFirstName -> str, psLastName -> str );

		AddInfo( psTable, PERSON_DISPLAY_NAME, pnName );
	}

	AddPerson( psTable, FALSE );
	if ( pnName != NULL ) {
		g_free( pnName );
		pnName = NULL;
	}

	g_hash_table_destroy( psTable );

	/* Free firstname */
	if ( psFirstName != NULL ) {
		g_string_free( psFirstName, TRUE );
		psFirstName = NULL;
	}

	/* Free lastname */
	if ( psLastName != NULL ) {
		g_string_free( psLastName, TRUE );
		psLastName = NULL;
	}

	/* Free fullname */
	if ( psFullName != NULL ) {
		g_string_free( psFullName, TRUE );
		psFullName = NULL;
	}

	/* Free company */
	if ( psCompany != NULL ) {
		g_string_free( psCompany, TRUE );
		psCompany = NULL;
	}

	/* Free cell phone */
	if ( psCellPhone != NULL ) {
		g_string_free( psCellPhone, TRUE );
		psCellPhone = NULL;
	}

	/* Free home phone */
	if ( psHomePhone != NULL ) {
		g_string_free( psHomePhone, TRUE );
		psHomePhone = NULL;
	}

	/* Free work phone */
	if ( psWorkPhone != NULL ) {
		g_string_free( psWorkPhone, TRUE );
		psWorkPhone = NULL;
	}

	/* Free private fax phone */
	if ( psPrivateFax != NULL ) {
		g_string_free( psPrivateFax, TRUE );
		psPrivateFax = NULL;
	}

	/* Free business fax phone */
	if ( psBusinessFax != NULL ) {
		g_string_free( psBusinessFax, TRUE );
		psBusinessFax = NULL;
	}

	/* Free private street */
	if ( psPrivateStreet != NULL ) {
		g_string_free( psPrivateStreet, TRUE );
		psPrivateStreet = NULL;
	}

	/* Free private city */
	if ( psPrivateCity != NULL ) {
		g_string_free( psPrivateCity, TRUE );
		psPrivateCity = NULL;
	}

	/* Free private zip code */
	if ( psPrivateZipCode != NULL ) {
		g_string_free( psPrivateZipCode, TRUE );
		psPrivateZipCode = NULL;
	}

	/* Free private country */
	if ( psPrivateCountry != NULL ) {
		g_string_free( psPrivateCountry, TRUE );
		psPrivateCountry = NULL;
	}

	/* Free business street */
	if ( psBusinessStreet != NULL ) {
		g_string_free( psBusinessStreet, TRUE );
		psBusinessStreet = NULL;
	}

	/* Free business city */
	if ( psBusinessCity != NULL ) {
		g_string_free( psBusinessCity, TRUE );
		psBusinessCity = NULL;
	}

	/* Free business zip code */
	if ( psBusinessZipCode != NULL ) {
		g_string_free( psBusinessZipCode, TRUE );
		psBusinessZipCode = NULL;
	}

	/* Free business country */
	if ( psBusinessCountry != NULL ) {
		g_string_free( psBusinessCountry, TRUE );
		psBusinessCountry = NULL;
	}

	/* Free title */
	if ( psTitle != NULL ) {
		g_string_free( psTitle, TRUE );
		psTitle = NULL;
	}

	/* Free uid */
	if ( psUid != NULL ) {
		g_string_free( psUid, TRUE );
		psUid = NULL;
	}

	/* set image to null */
	if ( psImage != NULL ) {
		psImage = NULL;
	}
}

/**
 * \brief Process new data structure (header/options/entry)
 * \param psCardData pointer to card data structure
 */
static void processData( struct sCardData *psCardData ) {
	if ( psCardData -> pnHeader != NULL && psCardData -> pnEntry != NULL ) {

		if ( strcasecmp( psCardData -> pnHeader, "BEGIN" ) == 0 ) {
			/* Begin of vcard */
			psVcard = g_list_append( NULL, psCardData );
			psVcardList = g_list_append( psVcardList, psVcard );
		} else {
			psVcard = g_list_append( psVcard, psCardData );
		}

		if ( strcasecmp( psCardData -> pnHeader, "END" ) == 0 ) {
			processCardEnd();
		} else {
			if ( strcasecmp( psCardData -> pnHeader, "FN" ) == 0 ) {
				/* Full name */
				processFullName( psCardData );
			} else if ( strcasecmp( psCardData -> pnHeader, "END" ) == 0 ) {
				/* End of vcard */
				processCardEnd();
			} else if ( strcasecmp( psCardData -> pnHeader, "N" ) == 0 ) {
				/* First and Last name */
				processFirstLastName( psCardData );
			} else if ( strcasecmp( psCardData -> pnHeader, "TEL" ) == 0 ) {
				/* Telephone */
				processTelephone( psCardData );
			} else if ( strcasecmp( psCardData -> pnHeader, "ORG" ) == 0 ) {
				/* Organization */
				processOrganization( psCardData );
			} else if ( strcasecmp( psCardData -> pnHeader, "TITLE" ) == 0 ) {
				/* Title */
				processTitle( psCardData );
			} else if ( strcasecmp( psCardData -> pnHeader, "ADR" ) == 0 ) {
				/* Address */
				processAddress( psCardData );
			} else if ( strcasecmp( psCardData -> pnHeader, "PHOTO" ) == 0 ) {
				/* Photo */
				processPhoto( psCardData );
			} else if ( strcasecmp( psCardData -> pnHeader, "UID" ) == 0 ) {
				/* UID */
				processUid( psCardData );
			}
		}
	}
}

/**
 * \brief Parse one char and add it to internal card structure
 * \param nChar current char
 */
static void parseChar( int nChar ) {
	switch ( nState ) {
		case STATE_NEW:
			psCurrentCardData = g_malloc0( sizeof( struct sCardData ) );
			nState = STATE_TAG;
			/* fall-through */
		case STATE_TAG:
			switch ( nChar ) {
				case '\r':
					break;
				case '\n':
					if ( psCurrentString != NULL ) {
						g_string_free( psCurrentString, TRUE );
					}
					psCurrentString = NULL;
					freeData( psCurrentCardData );
					nState = STATE_NEW;
					break;
				case ':':
					psCurrentCardData -> pnHeader = g_string_free( psCurrentString, FALSE );
					psCurrentString = NULL;
					nState = STATE_ENTRY;
					break;
				case ';':
					psCurrentCardData -> pnHeader = g_string_free( psCurrentString, FALSE );
					psCurrentString = NULL;
					nState = STATE_OPTIONS;
					break;
				default:
					if ( psCurrentString == NULL ) {
						psCurrentString = g_string_new( "" );
					}
					g_string_append_c( psCurrentString, nChar );
					break;
			}
			break;
		case STATE_OPTIONS:
			switch ( nChar ) {
				case '\r':
					break;
				case '\n':
					g_string_free( psCurrentString, TRUE );
					psCurrentString = NULL;
					g_free( psCurrentCardData -> pnHeader );
					g_free( psCurrentCardData );
					freeData( psCurrentCardData );
					nState = STATE_NEW;
					break;
				case ':':
					psCurrentCardData -> pnOptions = g_string_free( psCurrentString, FALSE );
					psCurrentString = NULL;
					nState = STATE_ENTRY;
					break;
				default:
					if ( psCurrentString == NULL ) {
						psCurrentString = g_string_new( "" );
					}
					g_string_append_c( psCurrentString, nChar );
					break;
			}
			break;
		case STATE_ENTRY:
			switch ( nChar ) {
				case '\r':
					break;
				case '\n':
					if ( psCurrentString != NULL ) {
						psCurrentCardData -> pnEntry = g_string_free( psCurrentString, FALSE );
						processData( psCurrentCardData );
					}
					psCurrentString = NULL;
					nState = STATE_NEW;
					break;
				default:
					if ( psCurrentString == NULL ) {
						psCurrentString = g_string_new( "" );
					}
					g_string_append_c( psCurrentString, nChar );
					break;
			}
			break;
	}
}

/**
 * \brief Load card file information
 * \param pnFileName file name to read
 */
void loadCardFile( char *pnFileName ) {
	FILE *psFile;
	gint nChar;
	gboolean bStartOfLine = TRUE;
	gboolean bFold = FALSE;

	/* Open file */
	psFile = fopen( pnFileName, "r" );
	if ( psFile == NULL ) {
		Debug( KERN_WARNING, "Could not open '%s', abort!\n", pnFileName );
		return;
	}

	nState = STATE_NEW;

	while ( !feof( psFile ) ) {
		nChar = fgetc( psFile );

		if ( bStartOfLine == TRUE ) {
			if ( nChar == '\r' ) {
				/* simple empty line */
				continue;
			}
			if ( nChar == '\n' ) {
				/* simple empty line */
				continue;
			}

			if ( bFold == FALSE && isspace( nChar ) ) {
				/* Ok, we have a fold case, mark it and continue */
				bFold = TRUE;
				continue;
			}

			bStartOfLine = FALSE;
			if ( bFold == TRUE ) {
				bFold = FALSE;
			} else {
				parseChar( '\n' );
			}
		}

		if ( nChar == '\n' ) {
			bStartOfLine = TRUE;
		} else {
			parseChar( nChar );
		}
	}

	/* Ensure we get a '\n' */
	parseChar( '\n' );

	fclose( psFile );
}

/**
 * \brief Put char to vcard structure
 * \param psData data string
 * \param nChar put char
 */
static void vcardPutChar( GString *psData, gint nChar ) {
	if ( nCurrentPosition == 74 && nChar != '\r' ) {
		g_string_append( psData, "\r\n " );
		nCurrentPosition = 1;
	} else if ( nChar == '\n' ) {
		nCurrentPosition = 0;
	}

	psData = g_string_append_c( psData, nChar );
	nCurrentPosition++;
}

/**
 * \brief printf to data structure
 * \param psData data string
 * \param pnFormat format string
 */
static void vcardPrint( GString *psData, gchar *pnFormat, ... ) {
	va_list args;
	int nLen;
	int nSize = 100;
	gchar *pnPtr = NULL;

	if ( ( pnPtr = g_malloc( nSize ) ) == NULL ) {
		return;
	}

	while ( 1 ) {
		va_start( args, pnFormat );

		nLen = vsnprintf( pnPtr, nSize, pnFormat, args );

		va_end( args );

		if ( nLen > -1 && nLen < nSize ) {
			int nIndex;

			for ( nIndex = 0; nIndex < nLen; nIndex++ ) {
				vcardPutChar( psData, pnPtr[ nIndex ] );
			}

			break;
		}

		if ( nLen > -1 ) {
			nSize = nLen + 1;
		} else {
			nSize *= 2;
		}

		if ( ( pnPtr = g_realloc( pnPtr, nSize ) ) == NULL ) {
			break;
		}
	}

	if ( pnPtr != NULL ) {
		g_free( pnPtr );
	}
}

/**
 * \brief Find vcard entry via uid
 * \param pnUid uid
 * \param vcard entry list pointer or NULL
 */
static GList *findVcardEntry( const gchar *pnUid ) {
	GList *psList1 = NULL;
	GList *psList2 = NULL;
	GList *psCard = NULL;
	struct sCardData *psData;
	gchar *pnCurrentUid;

	for ( psList1 = psVcardList; psList1 != NULL && psList1 -> data != NULL; psList1 = psList1 -> next ) {
		psCard = psList1 -> data;

		for ( psList2 = psCard; psList2 != NULL && psList2 -> data != NULL; psList2 = psList2 -> next ) {
			psData = psList2 -> data;

			if ( strcmp( psData -> pnHeader, "UID" ) == 0 ) {
				pnCurrentUid = psData -> pnEntry;
				if ( pnCurrentUid != NULL && !strcmp( pnCurrentUid, pnUid ) ) {
					return psCard;
				}
			}
		}
	}

	return NULL;
}

/**
 * \brief Find card data via header and option
 * \param psList vcard list structure
 * \param pnHeader header string
 * \param pnOption optional option string
 * \return card data structure or NULL
 */
struct sCardData *findCardData( GList *psList, gchar *pnHeader, gchar *pnOption ) {
	GList *psTmp = NULL;
	struct sCardData *psData = NULL;

	for ( psTmp = psList; psTmp != NULL && psTmp -> data != NULL; psTmp = psTmp -> next ) {
		psData = psTmp -> data;

		if( strcmp( psData -> pnHeader, pnHeader ) == 0 ) {
			if ( pnOption != NULL && psData -> pnOptions != NULL && strstr( psData -> pnOptions, pnOption ) ) {
				return psData;
			}
		}
	}

	return NULL;
}

/**
 * \brief Write card file information
 * \param pnFileName file name to read
 */
void writeCardFile( char *pnFileName ) {
	GString *psData = NULL;
	GList *psList = NULL;
	struct sPerson *psPerson = NULL;
	GList *psEntry = NULL;
	gchar *pnTmp = NULL;
	GList *psList2 = NULL;
	struct sCardData *psCardData = NULL;

	psData = g_string_new( "" );

	nCurrentPosition = 0;

	for ( psList = psPersonsList; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psPerson = psList -> data;

again:
		switch ( psPerson -> nFlags ) {
			case PERSON_FLAG_DELETED:
				break;
			case PERSON_FLAG_NEW:
				psCardData = g_malloc0( sizeof( struct sCardData ) );
				psCardData -> pnHeader = g_strdup( "UID" );
				psUid = createUid();
				psPerson -> pnId = g_string_free( psUid, FALSE );
				psUid = NULL;
				psCardData -> pnEntry = g_strdup( psPerson -> pnId );
				psVcard = g_list_append( NULL, psCardData );
				psVcardList = g_list_append( psVcardList, psVcard );
				/* fall-through */
			case PERSON_FLAG_CHANGED:
				psEntry = findVcardEntry( psPerson -> pnId );
				if ( psEntry == NULL ) {
					if ( psPerson -> nFlags == PERSON_FLAG_CHANGED ) {
						psPerson -> nFlags = PERSON_FLAG_NEW;
						goto again;
					}
					continue;
				}

				/* Fullname */
				psCardData = findCardData( psEntry, "FN", NULL );
				if ( psCardData == NULL ) {
					psCardData = g_malloc0( sizeof( struct sCardData ) );
					psCardData -> pnHeader = g_strdup( "FN" );
					psEntry = g_list_append( psEntry, psCardData );
				} else {
					g_free( psCardData -> pnEntry );
				}
				if ( psPerson -> pnDisplayName != NULL ) {
					g_free( psPerson -> pnDisplayName );
					psPerson -> pnDisplayName = NULL;
				}
				psPerson -> pnDisplayName = g_strdup_printf( "%s %s", psPerson -> pnFirstName, psPerson -> pnLastName );
				psCardData -> pnEntry = g_strdup_printf( "%s", psPerson -> pnDisplayName );

				/* Name */
				psCardData = findCardData( psEntry, "N", NULL );
				if ( psCardData == NULL ) {
					psCardData = g_malloc0( sizeof( struct sCardData ) );
					psCardData -> pnHeader = g_strdup( "N" );
					psEntry = g_list_append( psEntry, psCardData );
				} else {
					g_free( psCardData -> pnEntry );
				}
				psCardData -> pnEntry = g_strdup_printf( "%s;%s;;;", psPerson -> pnLastName, psPerson -> pnFirstName );

				/* Title */
				psCardData = findCardData( psEntry, "TITLE", NULL );
				if ( psCardData == NULL ) {
					psCardData = g_malloc0( sizeof( struct sCardData ) );
					psCardData -> pnHeader = g_strdup( "TITLE" );
					psEntry = g_list_append( psEntry, psCardData );
				} else {
					g_free( psCardData -> pnEntry );
				}
				psCardData -> pnEntry = g_strdup_printf( "%s", psPerson -> pnTitle );

				/* Organization */
				psCardData = findCardData( psEntry, "ORG", NULL );
				if ( psCardData == NULL ) {
					psCardData = g_malloc0( sizeof( struct sCardData ) );
					psCardData -> pnHeader = g_strdup( "ORG" );
					psEntry = g_list_append( psEntry, psCardData );
				} else {
					g_free( psCardData -> pnEntry );
				}
				psCardData -> pnEntry = g_strdup_printf( "%s", psPerson -> pnCompany );
 	
				/* Photo */
				if ( psPerson -> pnNewImage != NULL ) {
					gchar *pnData = NULL;
					gsize nLen = 0;

					psCardData = findCardData( psEntry, "PHOTO", NULL );
					if ( psCardData == NULL ) {
						psCardData = g_malloc0( sizeof( struct sCardData ) );
						psCardData -> pnHeader = g_strdup( "PHOTO" );
						psEntry = g_list_append( psEntry, psCardData );
					} else {
						g_free( psCardData -> pnEntry );
					}

					if ( g_file_get_contents( psPerson -> pnNewImage, &pnData, &nLen, NULL ) ) {
						gchar *pnBase64 = g_base64_encode( ( const guchar * ) pnData, nLen );
						if ( psCardData -> pnOptions != NULL ) {
							g_free( psCardData -> pnOptions );
						}
						psCardData -> pnOptions = g_strdup( "ENCODING=b" );
						psCardData -> pnEntry = g_strdup( pnBase64 );
						g_free( pnTmp );
						g_free( pnBase64 );
					}
				} else if ( psPerson -> psImage == NULL ) {
					psCardData = findCardData( psEntry, "PHOTO", NULL );
					if ( psCardData != NULL ) {
						psEntry = g_list_remove( psEntry, psCardData );
					}
				}

				/* Telephone */
				if ( psPerson -> pnPrivatePhone != NULL && strlen( psPerson -> pnPrivatePhone ) > 0 ) {
					psCardData = findCardData( psEntry, "TEL", "TYPE=HOME,VOICE" );
					if ( psCardData == NULL ) {
						psCardData = g_malloc0( sizeof( struct sCardData ) );
						psCardData -> pnHeader = g_strdup( "TEL" );
						psCardData -> pnOptions = g_strdup( "TYPE=HOME,VOICE" );
						psEntry = g_list_append( psEntry, psCardData );
					} else {
						g_free( psCardData -> pnEntry );
					}
					psCardData -> pnEntry = g_strdup( psPerson -> pnPrivatePhone );
				}
				if ( psPerson -> pnBusinessPhone != NULL && strlen( psPerson -> pnBusinessPhone ) > 0 ) {
					psCardData = findCardData( psEntry, "TEL", "TYPE=WORK,VOICE" );
					if ( psCardData == NULL ) {
						psCardData = g_malloc0( sizeof( struct sCardData ) );
						psCardData -> pnHeader = g_strdup( "TEL" );
						psCardData -> pnOptions = g_strdup( "TYPE=WORK,VOICE" );
						psEntry = g_list_append( psEntry, psCardData );
					} else {
						g_free( psCardData -> pnEntry );
					}
					psCardData -> pnEntry = g_strdup( psPerson -> pnBusinessPhone );
				}
				if ( psPerson -> pnPrivateMobile != NULL && strlen( psPerson -> pnPrivateMobile ) > 0 ) {
					psCardData = findCardData( psEntry, "TEL", "TYPE=CELL" );
					if ( psCardData == NULL ) {
						psCardData = g_malloc0( sizeof( struct sCardData ) );
						psCardData -> pnHeader = g_strdup( "TEL" );
						psCardData -> pnOptions = g_strdup( "TYPE=CELL" );
						psEntry = g_list_append( psEntry, psCardData );
					} else {
						g_free( psCardData -> pnEntry );
					}
					psCardData -> pnEntry = g_strdup( psPerson -> pnPrivateMobile );
				}
				if ( psPerson -> pnPrivateFax != NULL && strlen( psPerson -> pnPrivateFax ) > 0 ) {
					psCardData = findCardData( psEntry, "TEL", "TYPE=HOME,FAX" );
					if ( psCardData == NULL ) {
						psCardData = g_malloc0( sizeof( struct sCardData ) );
						psCardData -> pnHeader = g_strdup( "TEL" );
						psCardData -> pnOptions = g_strdup( "TYPE=HOME,FAX" );
						psEntry = g_list_append( psEntry, psCardData );
					} else {
						g_free( psCardData -> pnEntry );
					}
					psCardData -> pnEntry = g_strdup( psPerson -> pnPrivateFax );
				}
				if ( psPerson -> pnBusinessFax != NULL && strlen( psPerson -> pnBusinessFax ) > 0 ) {
					psCardData = findCardData( psEntry, "TEL", "TYPE=WORK,FAX" );
					if ( psCardData == NULL ) {
						psCardData = g_malloc0( sizeof( struct sCardData ) );
						psCardData -> pnHeader = g_strdup( "TEL" );
						psCardData -> pnOptions = g_strdup( "TYPE=WORK,FAX" );
						psEntry = g_list_append( psEntry, psCardData );
					} else {
						g_free( psCardData -> pnEntry );
					}
					psCardData -> pnEntry = g_strdup( psPerson -> pnBusinessFax );
				}

				/* Address */
				psCardData = findCardData( psEntry, "ADR", "TYPE=HOME" );
				if ( psCardData == NULL ) {
					psCardData = g_malloc0( sizeof( struct sCardData ) );
					psCardData -> pnHeader = g_strdup( "ADR" );
					psCardData -> pnOptions = g_strdup( "TYPE=HOME" );
					psEntry = g_list_append( psEntry, psCardData );
				} else {
					g_free( psCardData -> pnEntry );
				}
				psCardData -> pnEntry = g_strdup_printf( ";;%s;%s;;%s;%s",
					psPerson -> pnPrivateStreet ? psPerson -> pnPrivateStreet : "",
				    psPerson -> pnPrivateCity ? psPerson -> pnPrivateCity : "",
				    psPerson -> pnPrivateZipCode ? psPerson -> pnPrivateZipCode : "",
				    psPerson -> pnPrivateCountry ? psPerson -> pnPrivateCountry : "" );

				psCardData = findCardData( psEntry, "ADR", "TYPE=WORK" );
				if ( psCardData == NULL ) {
					psCardData = g_malloc0( sizeof( struct sCardData ) );
					psCardData -> pnHeader = g_strdup( "ADR" );
					psCardData -> pnOptions = g_strdup( "TYPE=WORK" );
					psEntry = g_list_append( psEntry, psCardData );
				} else {
					g_free( psCardData -> pnEntry );
				}
				psCardData -> pnEntry = g_strdup_printf( ";;%s;%s;;%s;%s",
					psPerson -> pnBusinessStreet ? psPerson -> pnBusinessStreet : "",
				    psPerson -> pnBusinessCity ? psPerson -> pnBusinessCity : "",
				    psPerson -> pnBusinessZipCode ? psPerson -> pnBusinessZipCode : "",
				    psPerson -> pnBusinessCountry ? psPerson -> pnBusinessCountry : "" );

				/* Fall-through */
			case PERSON_FLAG_UNCHANGED: {
				psEntry = findVcardEntry( psPerson -> pnId );
				if ( psEntry == NULL ) {
					Debug( KERN_WARNING, "UNCHANGED: Entry is NULL!!\n" );
					continue;
				}

				vcardPrint( psData, "BEGIN:VCARD\r\n" );

				for ( psList2 = psEntry; psList2 != NULL && psList2 -> data != NULL; psList2 = psList2 -> next ) {
					psCardData = psList2 -> data;

					if ( strcmp( psCardData -> pnHeader, "BEGIN" ) && strcmp( psCardData -> pnHeader, "END" ) ) {
						if ( psCardData -> pnOptions != NULL ) {
							vcardPrint( psData, "%s;%s:%s\r\n", psCardData -> pnHeader, psCardData -> pnOptions, psCardData -> pnEntry );
						} else {
							vcardPrint( psData, "%s:%s\r\n", psCardData -> pnHeader, psCardData -> pnEntry );
						}
					}
				}

				vcardPrint( psData, "END:VCARD\r\n" );
				break;
			}
		}
	}

	saveData( pnFileName, psData -> str, psData -> len );

	g_string_free( psData, TRUE );
}
