/*
 * $Id: chip_fau_camera_controller.c,v 1.22 2010-12-15 08:04:56 vrsieh Exp $
 *
 * Copyright (C) 2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG	0

#define BUFFER_SHIFT	6
#define BUFFER_SIZE	(1<<BUFFER_SHIFT)

#include "config.h"
#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "chip_fau_camera_controller.h"

#define COMP_(x) chip_fau_camera_controller_ ## x

struct cpssp {
	struct sig_boolean_or *port_intA;
	unsigned int state_power;
	unsigned int state_resetXhashX;
	struct sig_pci_bus_main *port_pci_bus;

	/* Config Space Registers */
	uint32_t base;
	unsigned int master_enabled;
	unsigned int io_enabled;
	uint8_t interrupt_line;

	/* I/O Registers */
	int received;

	int dma_enabled;
	uint64_t dma_pagelist;

	unsigned int width;
	unsigned int height;

	/* Internal Registers */
	int cur_enabled;
	uint64_t cur_pagelist;
	uint64_t cur_address;
	uint16_t cur_offset;

	uint8_t buf[BUFFER_SIZE];
	unsigned int count;

	unsigned int x;
	unsigned int y;
};

static void
COMP_(reset)(struct cpssp *cpssp)
{
#if DEBUG
	fprintf(stderr, "%s\n", __FUNCTION__);
#endif

	cpssp->base = 0x00000000;
	cpssp->master_enabled = 0;
	cpssp->io_enabled = 0;
	cpssp->interrupt_line = 0x00;

	cpssp->received = 0;
	cpssp->dma_enabled = 0;
	cpssp->dma_pagelist = 0x0000000000000000;
	cpssp->width = 0;
	cpssp->height = 0;

	cpssp->cur_enabled = 0;
	cpssp->cur_pagelist = 0x0000000000000000;
	cpssp->cur_address = 0x0000000000000000;
	cpssp->cur_offset = 0x000;
	cpssp->count = 0;
	cpssp->x = 0;
	cpssp->y = 0;
}

static void
COMP_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

#if DEBUG
	fprintf(stderr, "%s %u\n", __FUNCTION__, val);
#endif

	cpssp->state_power = val;
	COMP_(reset)(cpssp);
}

static void
COMP_(reset_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

#if DEBUG
	fprintf(stderr, "%s %u\n", __FUNCTION__, val);
#endif

	cpssp->state_resetXhashX = val;
	COMP_(reset)(cpssp);
}

static void
COMP_(irq_update)(struct cpssp *cpssp)
{
#if DEBUG
	fprintf(stderr, "%s (%d)\n", __FUNCTION__, cpssp->received);
#endif

	sig_boolean_or_set(cpssp->port_intA, cpssp, cpssp->received);
}

static void
COMP_(mr64)(struct cpssp *cpssp, uint64_t addr, unsigned int bs, uint64_t *valp)
{
	uint32_t tmp0;
	uint32_t tmp1;

	sig_pci_bus_mr(cpssp->port_pci_bus, cpssp,
			addr + 0, (bs >> 0) & 0xf, &tmp0);
	sig_pci_bus_mr(cpssp->port_pci_bus, cpssp,
			addr + 4, (bs >> 4) & 0xf, &tmp1);

	*valp = ((uint64_t) tmp1 << 32) | ((uint64_t) tmp0 << 0);
}

static void
COMP_(mw64)(struct cpssp *cpssp, uint64_t addr, unsigned int bs, uint64_t val)
{
	sig_pci_bus_mw(cpssp->port_pci_bus, cpssp,
			addr + 0, (bs >> 0) & 0xf, (uint32_t) (val >> 0));
	sig_pci_bus_mw(cpssp->port_pci_bus, cpssp,
			addr + 4, (bs >> 4) & 0xf, (uint32_t) (val >> 32));
}

static void
COMP_(dma_page)(struct cpssp *cpssp)
{
	uint64_t tmp;

	COMP_(mr64)(cpssp, cpssp->cur_pagelist, 0xff, &tmp);

#if DEBUG
	fprintf(stderr, "%s: got 0x%016" PRIx64 " from 0x%016" PRIx64 "\n",
			__FUNCTION__, tmp, cpssp->cur_pagelist);
#endif
	cpssp->cur_pagelist += 8;
	cpssp->cur_address = tmp & ~0xfffUL;
	cpssp->cur_enabled = tmp & 0x1;
}

static void
COMP_(flush)(struct cpssp *cpssp)
{
	unsigned int x;

#if DEBUG
	fprintf(stderr, "%s\n", __FUNCTION__);
#endif

	if (cpssp->master_enabled
	 && cpssp->cur_enabled) {
		for (x = 0; x < BUFFER_SIZE; x += 8) {
			if (cpssp->cur_enabled
			 && x < cpssp->count) {
				unsigned int count;
				uint64_t val;
				unsigned int bs;

				val = ((uint64_t) cpssp->buf[x + 0] << 0)
					| ((uint64_t) cpssp->buf[x + 1] << 8)
					| ((uint64_t) cpssp->buf[x + 2] << 16)
					| ((uint64_t) cpssp->buf[x + 3] << 24)
					| ((uint64_t) cpssp->buf[x + 4] << 32)
					| ((uint64_t) cpssp->buf[x + 5] << 40)
					| ((uint64_t) cpssp->buf[x + 6] << 48)
					| ((uint64_t) cpssp->buf[x + 7] << 56);
				count = cpssp->count - x;
				switch (count) {
				case 1: bs = 0x01; break;
				case 2: bs = 0x03; break;
				case 3: bs = 0x07; break;
				case 4: bs = 0x0f; break;
				case 5: bs = 0x1f; break;
				case 6: bs = 0x3f; break;
				case 7: bs = 0x7f; break;
				default: bs = 0xff; break;
				}
#if DEBUG
				fprintf(stderr, "%s: 0x%016" PRIx64 "(0x%x) <- 0x%016" PRIx64 "\n",
						__FUNCTION__,
						cpssp->cur_address | cpssp->cur_offset, bs, val);
#endif
				COMP_(mw64)(cpssp, 
					cpssp->cur_address | cpssp->cur_offset,
					bs, val);
			}

			cpssp->cur_offset += 8;
			cpssp->cur_offset &= 0xff8;

			if (! (cpssp->cur_offset & 0xff8)) {
				/*
				 * Read next page address.
				 */
				COMP_(dma_page)(cpssp);
			}
		}
	}
	cpssp->count = 0;
}

static void
COMP_(buf)(struct cpssp *cpssp, uint8_t byte)
{
#if DEBUG
	fprintf(stderr, "%s 0x%02x\n", __FUNCTION__, byte);
#endif

	cpssp->buf[cpssp->count++] = byte;
	if (cpssp->count == sizeof(cpssp->buf)) {
		COMP_(flush)(cpssp);
	}
}

static void
COMP_(video_out)(void *_cpssp, uint8_t r, uint8_t g, uint8_t b)
{
	struct cpssp *cpssp = _cpssp;

#if DEBUG
	fprintf(stderr, "%s\n", __FUNCTION__);
#endif

	if (! cpssp->state_power) return;

	COMP_(buf)(cpssp, r);
	COMP_(buf)(cpssp, g);
	COMP_(buf)(cpssp, b);

	cpssp->x++;
}

static void
COMP_(video_hor_retrace)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

#if DEBUG
	fprintf(stderr, "%s\n", __FUNCTION__);
#endif

	if (! cpssp->state_power) return;

	cpssp->width = cpssp->x;
	cpssp->x = 0;
	cpssp->y++;
}

static void
COMP_(video_vert_retrace)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

#if DEBUG
	fprintf(stderr, "%s %d 0x016%" PRIx64 "\n", __FUNCTION__,
			cpssp->cur_enabled, cpssp->cur_pagelist);
#endif

	if (! cpssp->state_power) return;

	cpssp->height = cpssp->y;
	cpssp->y = 0;

	COMP_(flush)(cpssp);

	cpssp->received |= cpssp->cur_enabled;
	COMP_(irq_update)(cpssp);

	cpssp->cur_enabled = cpssp->dma_enabled;
	cpssp->cur_pagelist = cpssp->dma_pagelist;
	cpssp->cur_offset = 0; /* cpssp->dma_offset - FIXME */

	cpssp->dma_enabled = 0;
	cpssp->dma_pagelist = 0;
	/* cpssp->dma_offset = 0; FIXME */

#if DEBUG
	fprintf(stderr, "%s %d 0x%016" PRIx64 " 0x%03x\n",
			__FUNCTION__,
			cpssp->cur_enabled,
			cpssp->cur_pagelist,
			cpssp->cur_offset);
#endif

	if (cpssp->cur_enabled) {
		COMP_(dma_page)(cpssp);
	}
}

static void
COMP_(video_no_sync)(void *_cpssp)
{
#if DEBUG
	fprintf(stderr, "%s\n", __FUNCTION__);
#endif

	/* Nothing to do... */
}

static int
COMP_(c0r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	if (! cpssp->state_power) return -1;

	addr &= 0xfc;

	switch (addr) {
	case 0x00: /* Device ID, Vendor ID */
		*valp = (0xcafe << 16) /* Device ID */
			| (0xaffe << 0); /* Vendor ID */
		break;

        case 0x04: /* Status, Command */
		*valp = (0 << 16) /* Status */
			/* Command */
			| (cpssp->master_enabled << 2)
			| (0 << 1)
			| (cpssp->io_enabled << 0);
		break;

        case 0x08: /* Class Code, Revision ID */
		*valp = (0x04 << 24) /* Multimedia Device */
			| (0x00 << 16) /* Video Device */
			| (0x00 << 8) /* Interface */
			| (0x01 << 0); /* Revision ID */
		break;

        case 0x0c: /* BIST, Header Type, Latency Timer, Cache Line Size */
		*valp = (0x00 << 24) /* No BIST */
			| (0 << (16 + 7)) /* Not a Multifunction Device */
			| (0x00 << 16) /* Standard Header Format */
			| (0x00 << 8) /* Latency Timer FIXME */
			| (16 << 0); /* Cache Line Size */
		break;

        case 0x10: /* Base Address 0 */
		*valp = cpssp->base
			| 0x1; /* I/O Space */
		break;

        case 0x3c: /* Max_Lat, Min_Gnt, Interrupt Pin, Interrupt Line */
		*valp = (1 << 24)
			| (1 << 16)
			| (1 << 8) /* Int#A */
			| (cpssp->interrupt_line << 0);
		break;

        case 0x28: /* Cardbus CIS Pointer */
        case 0x2c: /* Subsystem ID, Subsystem Vendor ID */
        case 0x34: /* Reserved */
        case 0x38: /* Reserved */
        case 0x14: /* Base Address 1 */
        case 0x18: /* Base Address 2 */
        case 0x1c: /* Base Address 3 */
        case 0x20: /* Base Address 4 */
        case 0x24: /* Base Address 5 */
        case 0x30: /* Expansion ROM Base Address */
		*valp = 0x00000000;
		break;
	}

#if DEBUG
	fprintf(stderr, "%s 0x%08x 0x%x -> 0x%08x\n", __FUNCTION__, addr, bs, *valp);
#endif

	return 0;
}

static int
COMP_(c0w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

#if DEBUG
	fprintf(stderr, "%s 0x%08x 0x%x <- 0x%08x\n", __FUNCTION__, addr, bs, val);
#endif

	if (! cpssp->state_power) return -1;

	addr &= 0xfc;

	switch (addr) {
        case 0x04: /* Status, Command */
		if ((bs >> 0) & 1) {
			cpssp->master_enabled = (val >> 2) & 1;
			cpssp->io_enabled = (val >> 0) & 1;
		}
		break;

        case 0x0c: /* BIST, Header Type, Latency Timer, Cache Line Size */
		if ((bs >> 1) & 1) {
			/* Latency Timer FIXME */
		}
		break;

        case 0x10: /* Base Address 0 */
		if ((bs >> 3) & 1) {
			cpssp->base &= ~0xff000000;
			cpssp->base |= val & 0xff000000;
		}
		if ((bs >> 2) & 1) {
			cpssp->base &= ~0x00ff0000;
			cpssp->base |= val & 0x00ff0000;
		}
		if ((bs >> 1) & 1) {
			cpssp->base &= ~0x0000ff00;
			cpssp->base |= val & 0x0000ff00;
		}
		if ((bs >> 0) & 1) {
			cpssp->base &= ~0x000000ff;
			cpssp->base |= val & 0x000000ff;
		}
		cpssp->base &= ~0xf;
		break;

        case 0x3c: /* Max_Lat, Min_Gnt, Interrupt Pin, Interrupt Line */
		if ((bs >> 0) & 1) {
			cpssp->interrupt_line = val & 0x000000ff;
		}
		break;

	case 0x00: /* Device ID, Vendor ID */
        case 0x08: /* Class Code, Revision ID */
        case 0x28: /* Cardbus CIS Pointer */
        case 0x2c: /* Subsystem ID, Subsystem Vendor ID */
        case 0x34: /* Reserved */
        case 0x38: /* Reserved */
        case 0x14: /* Base Address 1 */
        case 0x18: /* Base Address 2 */
        case 0x1c: /* Base Address 3 */
        case 0x20: /* Base Address 4 */
        case 0x24: /* Base Address 5 */
        case 0x30: /* Expansion ROM Base Address */
		/* Read-only/Reserved */
		break;
	}

	return 0;
}

static int
COMP_(ior)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	if (! cpssp->state_power
	 || ! cpssp->io_enabled
	 || addr < cpssp->base
	 || cpssp->base + 0x10 <= addr) {
		return -1;
	}

	addr &= 0x0c;

	switch (addr) {
	case 0x00: /* Status Register */
		*valp = cpssp->received << 0;
		cpssp->received = 0;
		COMP_(irq_update)(cpssp);
		break;

	case 0x04: /* DMA Register */
		*valp = cpssp->dma_pagelist
			| cpssp->dma_enabled;
		break;
	case 0x08: /* Width Register */
		*valp = cpssp->width;
		break;
	case 0x0c: /* Height Register */
		*valp = cpssp->height;
		break;
	}

#if DEBUG
	fprintf(stderr, "%s 0x%08x 0x%x -> 0x%08x\n", __FUNCTION__, addr, bs, *valp);
#endif

	return 0;
}

static int
COMP_(iow)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	if (! cpssp->state_power
	 || ! cpssp->io_enabled
	 || addr < cpssp->base
	 || cpssp->base + 0x10 <= addr) {
		return -1;
	}

#if DEBUG
	fprintf(stderr, "%s 0x%08x 0x%x <- 0x%08x\n", __FUNCTION__, addr, bs, val);
#endif

	addr &= 0x0c;

	switch (addr) {
	case 0x04: /* DMA page list, DMA enable */
		if ((bs >> 3) & 1) {
			cpssp->dma_pagelist &= ~0xff000000;
			cpssp->dma_pagelist |= val & 0xff000000;
		}
		if ((bs >> 2) & 1) {
			cpssp->dma_pagelist &= ~0x00ff0000;
			cpssp->dma_pagelist |= val & 0x00ff0000;
		}
		if ((bs >> 1) & 1) {
			cpssp->dma_pagelist &= ~0x0000ff00;
			cpssp->dma_pagelist |= val & 0x0000ff00;
		}
		if ((bs >> 0) & 1) {
			cpssp->dma_pagelist &= ~0x000000ff;
			cpssp->dma_pagelist |= val & 0x000000fc;

			cpssp->dma_enabled = val & 1;
		}
		break;

	case 0x00: /* Status Register */
	case 0x08: /* Width Register */
	case 0x0c: /* Height Register */
		/* Read-only */
		break;
	}

	return 0;
}

void *
COMP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_boolean *port_power,
	struct sig_boolean *port_resetXhashX,
	struct sig_pci_bus_idsel *port_idsel,
	struct sig_pci_bus_main *port_pci_bus,
	struct sig_boolean_or *port_intA,
	struct sig_video *port_video
)
{
	static const struct sig_boolean_funcs power_funcs = {
		.set = COMP_(power_set),
	};
	static const struct sig_boolean_funcs resetXhashX_funcs = {
		.set = COMP_(reset_set),
	};
	static const struct sig_pci_bus_idsel_funcs idsel_funcs = {
		.c0r = COMP_(c0r),
		.c0w = COMP_(c0w),
	};
	static const struct sig_pci_bus_main_funcs pci_bus_funcs = {
		.ior = COMP_(ior),
		.iow = COMP_(iow),
	};
	static const struct sig_video_funcs video_funcs = {
		.out = COMP_(video_out),
		.hor_retrace = COMP_(video_hor_retrace),
		.vert_retrace = COMP_(video_vert_retrace),
		.no_sync = COMP_(video_no_sync),
	};
	struct cpssp *cpssp;

#if DEBUG
	fprintf(stderr, "%s\n", __FUNCTION__);
#endif

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	/* Call */
	sig_pci_bus_idsel_connect(port_idsel, cpssp, &idsel_funcs);

	cpssp->port_pci_bus = port_pci_bus;
	sig_pci_bus_main_connect(port_pci_bus, cpssp, &pci_bus_funcs);

	sig_video_connect(port_video, cpssp, &video_funcs);

	/* Out */
	cpssp->port_intA = port_intA;
	sig_boolean_or_connect_out(port_intA, cpssp, 0);

	/* In */
	cpssp->state_power = 0;
	sig_boolean_connect_in(port_power, cpssp, &power_funcs);

	cpssp->state_resetXhashX = 0;
	sig_boolean_connect_in(port_resetXhashX, cpssp, &resetXhashX_funcs);

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

#if DEBUG
	fprintf(stderr, "%s\n", __FUNCTION__);
#endif

	free(cpssp);
}
