/****************************************************************\
*                                                                *
*  Library for HSP sets (high-scoring segment pairs)             *
*                                                                *
*  Guy St.C. Slater..   mailto:guy@ebi.ac.uk                     *
*  Copyright (C) 2000-2006.  All Rights Reserved.                *
*                                                                *
*  This source code is distributed under the terms of the        *
*  GNU Lesser General Public License. See the file COPYING       *
*  or http://www.fsf.org/copyleft/lesser.html for details        *
*                                                                *
*  If you use this code, please keep this notice intact.         *
*                                                                *
\****************************************************************/

/* Version 0.4
 */

#ifndef INCLUDED_HSPSET_H
#define INCLUDED_HSPSET_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include <glib.h>

#include "match.h"
#include "argument.h"
#include "recyclebin.h"
#include "pqueue.h"
#include "matrix.h"

typedef struct {
    gint filter_threshold;
    /**/
    gint dna_wordlen;
    gint protein_wordlen;
    gint codon_wordlen;
    /**/
    gint dna_hsp_dropoff;
    gint protein_hsp_dropoff;
    gint codon_hsp_dropoff;
    /**/
    gint dna_hsp_threshold;
    gint protein_hsp_threshold;
    gint codon_hsp_threshold;
    /**/
    gint dna_wordlimit;
    gint protein_wordlimit;
    gint codon_wordlimit;
    /**/
} HSPset_ArgumentSet;

HSPset_ArgumentSet *HSPset_ArgumentSet_create(Argument *arg);

/**/

typedef struct {
        guint  query_start;
        guint  target_start;
        guint  length;   /* Length is number of match state visits */
  Match_Score  score;
        guint  cobs;     /* cobs == Centre Offset By Score         */
struct HSPset *hsp_set;  /* Never included in hsp_set->ref_count   */
} HSP;
/* FIXME: remove hsp_set from HSP to save space ? */

void HSP_destroy(HSP *hsp);

#define HSP_query_advance(hsp) \
    ((hsp)->hsp_set->param->match->query->advance)

#define HSP_target_advance(hsp) \
    ((hsp)->hsp_set->param->match->target->advance)

#define HSP_query_end(hsp) \
    ((hsp)->query_start    \
  + ((hsp)->length*HSP_query_advance(hsp)))

#define HSP_target_end(hsp) \
    ((hsp)->target_start    \
  + ((hsp)->length*HSP_target_advance(hsp)))

#define HSP_query_cobs(hsp) \
    ((hsp)->query_start     \
   +((hsp)->cobs*HSP_query_advance(hsp)))

#define HSP_target_cobs(hsp) \
    ((hsp)->target_start     \
  + ((hsp)->cobs*HSP_target_advance(hsp)))

#define HSP_diagonal(hsp)                           \
    (((hsp)->target_start*HSP_query_advance(hsp))   \
    -((hsp)->query_start*HSP_target_advance(hsp)))
/* advance_{query,target} are swapped for position on diagonal */

#define HSP_get_score(hsp, query_pos, target_pos)    \
     ((hsp)->hsp_set->param->match->score_func(      \
      (hsp)->hsp_set->param->match,                  \
      (hsp)->hsp_set->query, (hsp)->hsp_set->target, \
      (query_pos), (target_pos)))

#define HSP_get_display(hsp, query_pos, target_pos, display_str)  \
     ((hsp)->hsp_set->param->match->display_func(                 \
      (hsp)->hsp_set->param->match,                               \
      (hsp)->hsp_set->query, (hsp)->hsp_set->target,              \
      (query_pos), (target_pos), display_str))

#define HSP_query_masked(hsp, query_pos)             \
    ((hsp)->hsp_set->param->match->query->mask_func( \
     (hsp)->hsp_set->param->match->query,            \
     (hsp)->hsp_set->query, (query_pos)))

#define HSP_target_masked(hsp, target_pos)            \
    ((hsp)->hsp_set->param->match->target->mask_func( \
     (hsp)->hsp_set->param->match->target,            \
     (hsp)->hsp_set->target, (target_pos)))

#define HSP_query_self(hsp, query_pos)               \
    ((hsp)->hsp_set->param->match->query->self_func( \
     (hsp)->hsp_set->param->match->query,            \
     (hsp)->hsp_set->query, (query_pos)))

#define HSP_target_self(hsp, target_pos)               \
    ((hsp)->hsp_set->param->match->target->self_func(  \
     (hsp)->hsp_set->param->match->target,             \
     (hsp)->hsp_set->target, (target_pos)))

typedef struct HSP_Param {
              gint  ref_count;
HSPset_ArgumentSet   *has;
             Match  *match;
              gint   wordlen;
              gint   seedlen;
       Match_Score   dropoff;
       Match_Score   threshold;
       Match_Score   wordlimit;
          gboolean   use_horizon;
} HSP_Param;

HSP_Param *HSP_Param_create(Match *match, gboolean use_horizon);
     void  HSP_Param_destroy(HSP_Param *hsp_param);
HSP_Param *HSP_Param_share(HSP_Param *hsp_param);

typedef struct HSPset {
              gint   ref_count;
          Sequence   *query;
          Sequence   *target;
         HSP_Param   *param;
        RecycleBin   *hsp_recycle;
              gint ***horizon;
         GPtrArray   *hsp_list;
          /**/
          gboolean    is_finalised;
            PQueue  **filter;
          gboolean    is_empty;
         PQueueSet   *pqueue_set;
} HSPset;

HSPset *HSPset_create(Sequence *query, Sequence *target,
                      HSP_Param *hsp_param);
HSPset *HSPset_share(HSPset *hsp_set);
  void  HSPset_destroy(HSPset *hsp_set);
  void  HSPset_seed_hsp(HSPset *hsp_set,
                        guint query_start, guint target_start);
  void  HSPset_seed_all_hsps(HSPset *hsp_set,
                        guint *seed_list, guint seed_list_len);
/* HSPset_seed_all_hsps() can only be called once on the HSPset.
 * It automatically finalises the HSPset
 * position_list should contain (qpos,tpos) pairs,
 * and may be sorted in place.
 */

HSPset *HSPset_finalise(HSPset *hsp_set);

void HSP_print(HSP *hsp, gchar *name);
void HSPset_print(HSPset *hsp_set);

void HSPset_filter_ungapped(HSPset *hsp_set);
/* Remove HSPs which overlap by more than 50% of the score.
 * This is used for 3:3 ungapped alignments.
 */

#define HSPset_is_empty(hspset) ((hspset)->is_empty)

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* INCLUDED_HSPSET_H */

