/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-vfolder.c,v 1.18 2005/01/31 00:40:57 hoa Exp $
 */

#include "etpan-cfg-vfolder.h"

#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <libetpan/libetpan.h>

#include "etpan-db-read.h"
#include "etpan-errors.h"
#include "etpan-cfg-account.h"
#include "etpan-cfg-storage.h"
#include "etpan-imf-helper.h"

static struct etpan_vfolder_property *
get_prop(chash * folder_prop, struct mailfolder * folder)
{
  int r;
  chashdatum key;
  chashdatum value;
  
  key.data = &folder;
  key.len = sizeof(folder);
  r = chash_get(folder_prop, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}

struct etpan_vfolder_property *
etpan_vfolder_get_prop(struct etpan_vfolder_config * config,
    struct mailfolder * folder)
{
  return get_prop(config->folder_prop, folder);
}

struct mailfolder *
etpan_vfolder_get_sent_folder(struct etpan_vfolder_config * config,
      struct mailfolder * folder)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(config->folder_prop, folder);
  
  if (prop != NULL) {
    if (prop->sent_folder != NULL)
      return prop->sent_folder;
  }
  
  return config->sent_folder;
}

struct mailfolder *
etpan_vfolder_get_draft_folder(struct etpan_vfolder_config * config,
      struct mailfolder * folder)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(config->folder_prop, folder);
  
  if (prop != NULL) {
    if (prop->draft_folder != NULL)
      return prop->draft_folder;
  }
  
  return config->draft_folder;
}

struct etpan_account_info *
etpan_vfolder_get_account(struct etpan_vfolder_config * vfolder_config,
                    struct etpan_account_config * account_config,
                    struct mailfolder * folder)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(vfolder_config->folder_prop, folder);
  
  if (prop != NULL) {
    if (prop->account != NULL)
      return prop->account;
  }
  
  return account_config->default_account;
}

struct mailimf_address_list *
etpan_vfolder_get_to(struct etpan_vfolder_config * vfolder_config,
    struct mailfolder * folder)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(vfolder_config->folder_prop, folder);
  
  if (prop != NULL)
    return prop->mail_to;
  
  return NULL;
}

int etpan_vfolder_do_poll(struct etpan_vfolder_config * vfolder_config,
    struct mailfolder * folder)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(vfolder_config->folder_prop, folder);
  
  if (prop != NULL)
    return prop->poll;
  
  return 0;
}

uint32_t etpan_vfolder_get_max(struct etpan_vfolder_config * vfolder_config,
    struct mailfolder * folder)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(vfolder_config->folder_prop, folder);
  
  if (prop != NULL)
    return prop->max;
  
  return 0;
}

void etpan_vfolder_set_max(struct etpan_vfolder_config * vfolder_config,
    struct mailfolder * folder, uint32_t max)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(vfolder_config->folder_prop, folder);
  
  if (prop != NULL)
    prop->max = max;
}

void etpan_vfolder_set_poll(struct etpan_vfolder_config * vfolder_config,
    struct mailfolder * folder, int poll)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(vfolder_config->folder_prop, folder);
  
  if (prop != NULL)
    prop->poll = poll;
}

void etpan_vfolder_set_sent_folder(struct etpan_vfolder_config *
    vfolder_config,
    struct mailfolder * folder, struct mailfolder * sent_folder)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(vfolder_config->folder_prop, folder);
  
  if (prop != NULL)
    prop->sent_folder = sent_folder;
}

void etpan_vfolder_set_draft_folder(struct etpan_vfolder_config *
    vfolder_config,
    struct mailfolder * folder, struct mailfolder * draft_folder)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(vfolder_config->folder_prop, folder);
  
  if (prop != NULL)
    prop->draft_folder = draft_folder;
}

void etpan_vfolder_set_account(struct etpan_vfolder_config * vfolder_config,
    struct mailfolder * folder, struct etpan_account_info * account)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(vfolder_config->folder_prop, folder);
  
  if (prop != NULL)
    prop->account = account;
}

/*
  etpan_vfolder_set_mail_to()

  duplicates mail_to before setting so that the given argument can be
  destroyed later.
*/

void etpan_vfolder_set_mail_to(struct etpan_vfolder_config * vfolder_config,
    struct mailfolder * folder, struct mailimf_address_list * mail_to)
{
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(vfolder_config->folder_prop, folder);
  
  if (prop != NULL) {
    if (mail_to != NULL) {
      mail_to = etpan_dup_address_list(mail_to);
      if (mail_to == NULL)
        return;
    }
    prop->mail_to = mail_to;
  }
}

static void recursive_folder_free(struct mailfolder * folder)
{
  while (carray_count(folder->fld_children) > 0) {
    struct mailfolder * child;
    
    child = carray_get(folder->fld_children, 0);
    recursive_folder_free(child);
  }

  mailfolder_free(folder);
}

struct etpan_vfolder_config *
etpan_vfolder_config_new(struct mailfolder * root,
    chash * folder_prop,
    struct mailfolder * default_folder,
    struct mailfolder * sent_folder,
    struct mailfolder * draft_folder)
{
  struct etpan_vfolder_config * config;
  
  config = malloc(sizeof(struct etpan_vfolder_config));
  if (config == NULL)
    return NULL;

  config->root = root;
  config->folder_prop = folder_prop;
  config->default_folder = default_folder;
  config->sent_folder = sent_folder;
  config->draft_folder = draft_folder;

  return config;
}

static void vfolder_property_free(struct etpan_vfolder_property * prop);

void etpan_vfolder_config_free(struct etpan_vfolder_config * config)
{
  chashiter * cur;
  
  for(cur = chash_begin(config->folder_prop) ; cur != NULL ;
      cur = chash_next(config->folder_prop, cur)) {
    chashdatum value;
    struct etpan_vfolder_property * prop;
    
    chash_value(cur, &value);
    prop = value.data;
    vfolder_property_free(prop);
  }
  chash_free(config->folder_prop);
  recursive_folder_free(config->root);
  free(config);
}

struct etpan_offline_config * etpan_offline_config_new(chash * folder_hash,
    clist * folder_list)
{
  struct etpan_offline_config * config;
  
  config = malloc(sizeof(struct etpan_offline_config));
  if (config == NULL)
    return NULL;

  config->folder_hash = folder_hash;
  config->folder_list = folder_list;

  return config;
}

void etpan_offline_config_free(struct etpan_offline_config * config)
{
  clist_foreach(config->folder_list, (clist_func) mailfolder_free, NULL);
  clist_free(config->folder_list);

  chash_free(config->folder_hash);
  
  free(config);
}

char * etpan_folder_get_virtual_path(struct mailfolder * folder)
{
  char * virtual_path;
  size_t len;

  len = 0;
  virtual_path = NULL;

  do {
    char * new_vp;
    size_t name_len;

    if (folder->fld_virtual_name == NULL)
      break;
    
    if (virtual_path == NULL) {
      virtual_path = strdup(folder->fld_virtual_name);
      if (virtual_path == NULL)
        goto err;
      len = strlen(virtual_path);
    }
    else {
      name_len = strlen(folder->fld_virtual_name);
      new_vp = malloc(name_len + len + 2);
      if (new_vp == NULL)
        goto free;
      
      strcpy(new_vp, folder->fld_virtual_name);
      new_vp[name_len] = '/';
      strcpy(new_vp + name_len + 1, virtual_path);

      len += name_len + 1;
      free(virtual_path);
      virtual_path = new_vp;
      
    }

    folder = folder->fld_parent;
    
  } while (folder != NULL);
  
  if (virtual_path == NULL)
    return strdup("/");
  
  return virtual_path;
  
 free:
  free(virtual_path);
 err:
  return NULL;
}


static struct etpan_vfolder_property * vfolder_property_new(void)
{
  struct etpan_vfolder_property * prop;
  
  prop = malloc(sizeof(* prop));
  if (prop == NULL)
    return NULL;
  
  prop->account = NULL;
  prop->sent_folder = NULL;
  prop->draft_folder = NULL;
  prop->poll = 0;
  prop->max = 0;
  prop->mail_to = NULL;
  
  prop->auto_build = 0;
  prop->auto_build_type = 0;
  prop->build_storage = NULL;
  prop->build_path = NULL;
  prop->generated = 0;
  
  return prop;
}

static void vfolder_property_free(struct etpan_vfolder_property * prop)
{
  if (prop->mail_to != NULL)
    mailimf_address_list_free(prop->mail_to);
  free(prop);
}

static struct mailfolder * get_folder(chash * folder_hash, char * name)
{
  int r;
  chashdatum key;
  chashdatum value;
  
  key.data = name;
  key.len = strlen(name);
  r = chash_get(folder_hash, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}



static struct mailfolder * vfolder_new_folder(chash * folder_hash,
    struct mailstorage * storage, char * virtualpath, char * location)
{
  char * p;
  char * name;
  int r;
  struct mailfolder * folder;
  chashdatum key;
  chashdatum value;
  
  /* get base name */
  p = virtualpath;
  while (1) {
    name = p;
    p = strchr(name, '/');
    if (p == NULL)
      break;
    p ++;
  }
  
  folder = mailfolder_new(storage, location, name);
  if (folder == NULL) {
    goto err;
  }
  
  key.data = virtualpath;
  key.len = strlen(virtualpath);
  value.data = folder;
  value.len = 0;
  
  r = chash_set(folder_hash, &key, &value, NULL);
  if (r < 0) {
    mailfolder_free(folder);
    goto err;
  }
  
  return folder;
  
 err:
  return NULL;
}


static int add_prop(chash * folder_prop,
    struct mailfolder * folder)
{
  struct etpan_vfolder_property * prop;
  int r;
  chashdatum key;
  chashdatum value;
  
  prop = vfolder_property_new();
  if (prop == NULL) {
    goto err;
  }
  
  key.data = &folder;
  key.len = sizeof(folder);
  value.data = prop;
  value.len = 0;
  r = chash_set(folder_prop, &key, &value, NULL);
  if (r < 0) {
    vfolder_property_free(prop);
    goto err;
  }
  
  return NO_ERROR;
  
 err:
  return ERROR_MEMORY;
}

int etpan_cfg_vfolder_add_prop(struct etpan_vfolder_config * config,
    struct mailfolder * folder)
{
  return add_prop(config->folder_prop, folder);
}

int etpan_cfg_vfolder_remove_prop(struct etpan_vfolder_config * config,
    struct mailfolder * folder)
{
  struct etpan_vfolder_property * prop;
  int r;
  chashdatum key;
  chashdatum value;
  
  key.data = &folder;
  key.len = sizeof(folder);
  r = chash_get(config->folder_prop, &key, &value);
  if (r < 0) {
    goto err;
  }
  
  prop = value.data;
  vfolder_property_free(prop);
  
  chash_delete(config->folder_prop, &key, NULL);
  
  return NO_ERROR;
  
 err:
  return ERROR_INVAL;
}

static int vfolder_collect_folders(struct etpan_db * db,
    struct etpan_storage_config * storage_config,
    struct etpan_account_config * account_config,
    chash ** pfolder_hash,
    chash ** pfolder_prop,
    struct mailfolder ** pdefault_folder,
    struct mailfolder ** psent_folder,
    struct mailfolder ** pdraft_folder)
{
  chashiter * cur;
  int r;
  int res;
  char * default_folder_name;
  char * default_sent_name;
  char * default_draft_name;
  struct mailfolder * default_folder;
  struct mailfolder * sent_folder;
  struct mailfolder * draft_folder;
  unsigned int i;
  chash * folder_prop;
  chash * folder_hash;

  folder_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_hash == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }
  
  folder_prop = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (folder_prop == NULL) {
    res = ERROR_MEMORY;
    goto free_folder_hash;
  }
  
  default_sent_name = NULL;
  default_folder_name = NULL;
  default_draft_name = NULL;
  
  for(i = 0 ; i < carray_count(db->data) ; i ++) {
    chash * entry;
    char * id;
    
    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;
    
    id = etpan_db_entry_get_value(entry, "id");
    if (id != NULL) {
      if (strcasecmp(id, "default") == 0) {
        /* old version compatibility */
        if (default_folder_name == NULL)
          default_folder_name = etpan_db_entry_get_value(entry, "folder");
        if (default_sent_name == NULL)
          default_sent_name = etpan_db_entry_get_value(entry, "sent");
        if (default_draft_name == NULL)
          default_draft_name = etpan_db_entry_get_value(entry, "draft");
      }
    }
    
    if (default_folder_name == NULL)
      default_folder_name = etpan_db_entry_get_value(entry, "default-folder");
    if (default_sent_name == NULL)
      default_sent_name = etpan_db_entry_get_value(entry, "default-sent");
    if (default_draft_name == NULL)
      default_draft_name = etpan_db_entry_get_value(entry, "default-draft");
  }
  
  /* build initial folder hash */
  
  for(i = 0 ; i < db->data->len ; i ++) {
    char * storage_name;
    chash * entry;
    struct mailstorage * storage;
    char * virtualpath;
    char * location;
    struct mailfolder * folder;
    
    entry = carray_get(db->data, i);
    
    /* virtual path */
    virtualpath = etpan_db_entry_get_value(entry, "name");
    if (virtualpath == NULL)
      continue;
    
    storage_name = etpan_db_entry_get_value(entry, "storage");
    
    storage = NULL;
    if (storage_name != NULL) {
      storage = etpan_storage_get(storage_config, storage_name);
      if (storage == NULL)
        fprintf(stderr, "storage unknown - %s\n", storage_name);
    }
    
    location = etpan_db_entry_get_value(entry, "location");
    
    folder = vfolder_new_folder(folder_hash, storage, virtualpath, location);
    
    /* don't create property for folder if no storage */
    if (storage == NULL)
      continue;
    
    /* property of the folder */
    
    r = add_prop(folder_prop, folder);
    if (r != NO_ERROR) {
      res = r;
      goto free_prop;
    }
  }
  
  sent_folder = NULL;
  if (default_sent_name != NULL)
    sent_folder = get_folder(folder_hash, default_sent_name);
  draft_folder = NULL;
  if (default_draft_name != NULL)
    draft_folder = get_folder(folder_hash, default_draft_name);
  default_folder = NULL;
  if (default_folder_name != NULL)
    default_folder = get_folder(folder_hash, default_folder_name);
  
  /* setting properties value */
  
  for(i = 0 ; i < db->data->len ; i ++) {
    char * account_name;
    char * mail_to;
    char * do_poll_str;
    char * max_str;
    char * virtualpath;
    char * sent_name;
    char * draft_name;
    struct mailfolder * folder;
    struct etpan_vfolder_property * prop;
    chash * entry;
    
    entry = carray_get(db->data, i);
    
    virtualpath = etpan_db_entry_get_value(entry, "name");
    if (virtualpath == NULL)
      continue;
    
    folder = get_folder(folder_hash, virtualpath);
    
    prop = get_prop(folder_prop, folder);
    if (prop == NULL)
      continue;
    
    account_name = etpan_db_entry_get_value(entry, "account");
    if (account_name != NULL) {
      prop->account =  etpan_account_get(account_config, account_name);
      if (prop->account == NULL)
        fprintf(stderr, "account unknown - %s\n", account_name);
    }
    
    mail_to = etpan_db_entry_get_value(entry, "mail-to");
    if (mail_to != NULL) {
      size_t cur_token;
      struct mailimf_address_list * addr_list;
      
      cur_token = 0;
      r = mailimf_address_list_parse(mail_to, strlen(mail_to),
          &cur_token, &addr_list);
      if (r == MAILIMF_NO_ERROR) {
        prop->mail_to = addr_list;
      }
      else {
        fprintf(stderr, "error parsing address - %s\n", mail_to);
      }
    }
    
    do_poll_str = etpan_db_entry_get_value(entry, "poll");
    if (do_poll_str != NULL) {
      prop->poll = strtoul(do_poll_str, NULL, 10);
    }
    
    max_str = etpan_db_entry_get_value(entry, "max");
    if (max_str != NULL) {
      prop->max = strtoul(max_str, NULL, 10);
    }
    
    sent_name = etpan_db_entry_get_value(entry, "sent");
    if (sent_name != NULL) {
      prop->sent_folder = get_folder(folder_hash, sent_name);
      if (prop->sent_folder == NULL)
        fprintf(stderr, "unknown folder - %s\n", sent_name);
    }

    draft_name = etpan_db_entry_get_value(entry, "draft");
    if (draft_name != NULL) {
      prop->draft_folder = get_folder(folder_hash, draft_name);
      if (prop->draft_folder == NULL)
        fprintf(stderr, "unknown folder - %s\n", draft_name);
    }
  }
  
  * pfolder_hash = folder_hash;
  * pfolder_prop = folder_prop;
  * pdefault_folder = default_folder;
  * psent_folder = sent_folder;
  * pdraft_folder = draft_folder;

  return NO_ERROR;

 free_prop:
  for(cur = chash_begin(folder_prop) ; cur != NULL ;
      cur = chash_next(folder_prop, cur)) {
    chashdatum value;
    struct etpan_vfolder_property * prop;
    
    chash_value(cur, &value);
    prop = value.data;
    vfolder_property_free(prop);
  }
  chash_free(folder_prop);
 free_folder_hash:
  for(cur = chash_begin(folder_hash) ; cur != NULL ;
      cur = chash_next(folder_hash, cur)) {
    chashdatum value;
    struct mailfolder * folder;
    
    chash_value(cur, &value);
    folder = value.data;
    mailfolder_free(folder);
  }
  chash_free(folder_hash);
 err:
  return res;
}


int etpan_vfolder_config_read(char * filename,
    struct etpan_storage_config * storage_config,
    struct etpan_account_config * account_config,
    struct etpan_vfolder_config ** result)
{
  struct etpan_db * db;
  struct etpan_vfolder_config * config;
  int r;
  int res;
  chashiter * cur;
  struct mailfolder * root;
  struct mailfolder * default_folder;
  struct mailfolder * default_sent_folder;
  struct mailfolder * default_draft_folder;
  chash * folder_hash;
  chash * folder_prop;
  unsigned int i;

  r = etpan_read_config(filename, &db);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  r = vfolder_collect_folders(db,
      storage_config, account_config,
      &folder_hash, &folder_prop,
      &default_folder, &default_sent_folder,
      &default_draft_folder);
  if (r != NO_ERROR) {
    res = r;
    goto free_db;
  }
  
  /* build the tree */
  
  for(i = 0 ; i < carray_count(db->data) ; i ++) {
    char * virtualpath;
    struct mailfolder * folder;
    struct mailfolder * child;
    char path[PATH_MAX];
    char * end;
    size_t len;
    chash * entry;
    
    entry = carray_get(db->data, i);
    
    virtualpath = etpan_db_entry_get_value(entry, "name");
    if (virtualpath == NULL)
      continue;
    
    strncpy(path, virtualpath, PATH_MAX);
    path[PATH_MAX - 1] = 0;

    len = strlen(path);
    child = NULL;
    
    while (len > 0) {
      chashdatum key;
      chashdatum value;
      
      key.data = path;
      key.len = len;
      
      r = chash_get(folder_hash, &key, &value);
      if (r == 0) {
        folder = value.data;
      }
      else {
        folder = vfolder_new_folder(folder_hash, NULL, path, NULL);
        if (folder == NULL) {
          res = ERROR_MEMORY;
          goto free_prop;
        }
      }
      
      if (child != NULL) {
        r = mailfolder_add_child(folder, child);
        if (r != MAIL_NO_ERROR) {
          res = r;
          goto free_prop;
        }
      }
      
      if (folder->fld_parent != NULL)
        break;
      
      end = path + len - 1;
      while ((end != path) && (* end != '/')) {
	end --;
	len --;
      }
      len --;
      * end = 0;
      
      child = folder;
    }
  }
  
  root = mailfolder_new(NULL, NULL, NULL);
  if (root == NULL) {
    res = r;
    goto free_prop;
  }

  /* add folders that have no parent to the root folder */
  
  for(i = 0 ; i < db->data->len ; i ++) {
    char * virtualpath;
    struct mailfolder * folder;
    chash * entry;
    chashdatum key;
    chashdatum value;
    
    entry = carray_get(db->data, i);
    
    virtualpath = etpan_db_entry_get_value(entry, "name");
    if (virtualpath == NULL)
      continue;
    
    key.data = virtualpath;
    key.len = strlen(virtualpath);
    
    r = chash_get(folder_hash, &key, &value);
    if (r < 0)
      continue;
    
    folder = value.data;
    
    if (folder->fld_parent == NULL) {
      r = mailfolder_add_child(root, folder);
      if (r != MAIL_NO_ERROR) {
        mailfolder_free(root);
	res = r;
	goto free_prop;
      }
    }
  }
  
  config = etpan_vfolder_config_new(root, folder_prop,
      default_folder, default_sent_folder, default_draft_folder);
  if (config == NULL) {
    res = ERROR_MEMORY;
    goto free_prop;
  }
  
  chash_free(folder_hash);
  
  etpan_db_free(db);
  
  * result = config;
  
  return NO_ERROR;
  
 free_prop:
  for(cur = chash_begin(folder_prop) ; cur != NULL ;
      cur = chash_next(folder_prop, cur)) {
    chashdatum value;
    struct etpan_vfolder_property * prop;
    
    chash_value(cur, &value);
    prop = value.data;
    vfolder_property_free(prop);
  }
  chash_free(folder_prop);
  for(cur = chash_begin(folder_hash) ; cur != NULL ;
      cur = chash_next(folder_hash, cur)) {
    chashdatum value;
    struct mailfolder * folder;
    
    chash_value(cur, &value);
    folder = value.data;
    mailfolder_free(folder);
  }
  chash_free(folder_hash);
 free_db:
  etpan_db_free(db);
 err:
  return res;
}

#define MAX_LINE 1024

static void folder_write(FILE * f, char * id,
    struct etpan_vfolder_config * vfolder_config,
    struct etpan_account_config * account_config,
    struct mailfolder * folder)
{
  struct mailimf_address_list * mail_to;
  struct etpan_account_info * account;
  struct mailfolder * sent_folder;
  struct mailfolder * draft_folder;
  
  if (folder->fld_virtual_name != NULL) {
    char * virtualname;
    
    virtualname = etpan_folder_get_virtual_path(folder);
    if (virtualname != NULL) {
      fprintf(f, "name = %s\n", virtualname);
      free(virtualname);
    }
  }
  
  if (folder->fld_storage != NULL)
    fprintf(f, "storage = %s\n", folder->fld_storage->sto_id);
  
  if (folder->fld_pathname != NULL)
    fprintf(f, "location = %s\n", folder->fld_pathname);
  
  account = etpan_vfolder_get_account(vfolder_config, account_config, folder);
  if (account == account_config->default_account)
    account = NULL;
  if (account != NULL)
    fprintf(f, "account = %s\n", account->id);
  
  sent_folder = etpan_vfolder_get_sent_folder(vfolder_config, folder);
  if (sent_folder == vfolder_config->sent_folder)
    sent_folder = NULL;
  if (sent_folder != NULL) {
    char * virtualname;
    
    virtualname = etpan_folder_get_virtual_path(sent_folder);
    if (virtualname != NULL) {
      fprintf(f, "sent = %s\n", virtualname);
      free(virtualname);
    }
  }

  draft_folder = etpan_vfolder_get_draft_folder(vfolder_config, folder);
  if (draft_folder == vfolder_config->draft_folder)
    draft_folder = NULL;
  if (draft_folder != NULL) {
    char * virtualname;
    
    virtualname = etpan_folder_get_virtual_path(draft_folder);
    if (virtualname != NULL) {
      fprintf(f, "draft = %s\n", virtualname);
      free(virtualname);
    }
  }
  
  mail_to = etpan_vfolder_get_to(vfolder_config, folder);
  if (mail_to != NULL) {
    char * mail_to_str;
    
    mail_to_str = mailimf_address_list_to_string(mail_to);
    if (mail_to_str != NULL)
      fprintf(f, "mail-to = %s\n", mail_to_str);
    free(mail_to_str);
  }
  
  if (folder->fld_storage != NULL) {
    int storage_type;
    int poll;
    
    storage_type =
      etpan_cfg_storage_get_type(folder->fld_storage->sto_driver->sto_name);
    
    poll = etpan_vfolder_do_poll(vfolder_config, folder);
    fprintf(f, "poll = %i\n", poll);
    
    if (storage_type == STORAGE_TYPE_NNTP) {
      uint32_t max;
      
      max = etpan_vfolder_get_max(vfolder_config, folder);
      fprintf(f, "max = %i\n", max);
    }
  }
  
  fprintf(f, "\n");
}

static void recursive_folder_write(FILE * f, char * id,
    struct etpan_vfolder_config * vfolder_config,
    struct etpan_account_config * account_config,
    struct mailfolder * folder)
{
  unsigned int i;
  char child_id[MAX_LINE];
  
  if (folder->fld_virtual_name != NULL)
    folder_write(f, id, vfolder_config, account_config, folder);
  
  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    struct mailfolder * child;
    
    if (* id == '\0')
      snprintf(child_id, sizeof(child_id), "%i", i + 1);
    else
      snprintf(child_id, sizeof(child_id), "%s.%i", id, i + 1);
    child = carray_get(folder->fld_children, i);
    recursive_folder_write(f, child_id,
        vfolder_config, account_config, child);
  }
}

int etpan_cfg_vfolder_write(char * filename,
    struct etpan_vfolder_config * vfolder_config,
    struct etpan_account_config * account_config)
{
  FILE * f;
  mode_t old_mask;

  old_mask = umask(0077);
  f = fopen(filename, "w");
  umask(old_mask);
  if (f == NULL)
    goto err;
  
  if (vfolder_config->default_folder != NULL) {
    char * virtualname;
    
    virtualname =
      etpan_folder_get_virtual_path(vfolder_config->default_folder);
    if (virtualname != NULL) {
      fprintf(f, "default-folder = %s\n", virtualname);
      free(virtualname);
    }
  }
  if (vfolder_config->sent_folder != NULL) {
    char * virtualname;
    
    virtualname =
      etpan_folder_get_virtual_path(vfolder_config->sent_folder);
    if (virtualname != NULL) {
      fprintf(f, "default-sent = %s\n", virtualname);
      free(virtualname);
    }
  }
  if (vfolder_config->draft_folder != NULL) {
    char * virtualname;
    
    virtualname =
      etpan_folder_get_virtual_path(vfolder_config->draft_folder);
    if (virtualname != NULL) {
      fprintf(f, "default-draft = %s\n", virtualname);
      free(virtualname);
    }
  }
  fprintf(f, "\n");
  
  recursive_folder_write(f, "", vfolder_config, account_config,
      vfolder_config->root);
  
  fclose(f);
  
  return NO_ERROR;
  
 err:
  return ERROR_FILE;
}


/*
  etpan_vfolder_config_remove_generated()
  
  before running this function, we should be sure all folders
  are disconnected.
*/

void etpan_vfolder_config_remove_generated(struct etpan_vfolder_config *
    vfolder_config, struct mailfolder * root)
{
  unsigned int i;
  struct etpan_vfolder_property * prop;
  
  prop = get_prop(vfolder_config->folder_prop, root);
  
  if (prop->generated && (root->fld_parent != NULL)) {
    if (carray_count(root->fld_children) == 0) {
      etpan_cfg_vfolder_remove_prop(vfolder_config, root);
      mailfolder_detach_parent(root);
      mailfolder_free(root);
    }
    else {
      root->fld_storage = NULL;
    }
  }
  
  for(i = 0 ; i < carray_count(root->fld_children) ; i ++) {
    struct mailfolder * child;
    
    child = carray_get(root->fld_children, i);
    
    etpan_vfolder_config_remove_generated(vfolder_config, child);
  }
}

void etpan_vfolder_config_mark_storage(struct etpan_storage_config *
    storage_config,
    struct mailfolder * root)
{
  unsigned int i;
  
  if (root->fld_storage != NULL)
    etpan_storage_config_set_generated(storage_config,
        root->fld_storage, 0);
  
  for(i = 0 ; i < carray_count(root->fld_children) ; i ++) {
    struct mailfolder * child;
    
    child = carray_get(root->fld_children, i);
  }
}



static struct mailfolder * find_folder(char * path_prefix, size_t len,
    struct mailfolder * folder, char * name)
{
  char path[PATH_MAX];
  unsigned int i;

  if (strcmp(path_prefix, name) == 0)
    return folder;

  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    struct mailfolder *  child;
    struct mailfolder *  found;
    size_t path_len;

    child = carray_get(folder->fld_children, i);

    if (len == 0)
      snprintf(path, PATH_MAX, "%s", child->fld_virtual_name);
    else
      snprintf(path, PATH_MAX - len, "%s/%s",
	       path_prefix, child->fld_virtual_name);

    path_len = len + strlen(child->fld_virtual_name);
    found = find_folder(path, path_len, child, name);
    if (found != NULL)
      return found;
  }

  return NULL;
}

struct mailfolder *
etpan_vfolder_get_folder(struct etpan_vfolder_config * config,
    char * name)
{
  return find_folder("", 0, config->root, name);
}
