# -*- coding: utf-8 -*-
#
# Graphical interface for the eSpeak speech synthesizer
#
# Copyright © 2009-2010 Siegfried-Angel Gevatter Pujals <siegfried@gevatter.com>
# Copyright © 2009 Joe Burmeister <joe.a.burmeister@googlemail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import os
import gtk
import gobject
import gettext
import locale
from optparse import OptionParser
from espeak import espeak

from resources import *

gtk.gdk.threads_init()
gettext.install('espeak-gui', unicode=True)
locale.setlocale(locale.LC_ALL, "") # for unicode sorting with locale.strcoll

_gnulicense = \
"This program is free software: you can redistribute it and/or modify\n\
it under the terms of the GNU General Public License as published by\n\
the Free Software Foundation, either version 3 of the License, or\n\
(at your option) any later version.\n\
\n\
This program is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
GNU General Public License for more details.\n\
\n\
You should have received a copy of the GNU General Public License\n\
along with this program.  If not, see <http://www.gnu.org/licenses/>."

class Application:
    
    def _create_about_dialog(self):
        
        dialog = gtk.AboutDialog()
        dialog.set_version(version)
        dialog.set_program_name('espeak-gui')
        dialog.set_icon(self._main.get_icon())
        dialog.set_license(_gnulicense)
        dialog.set_copyright(
            'Copyright © 2009-2010 Siegfried-Angel Gevatter Pujals\n' \
            'Copyright © 2009 Joe Burmeister')
        dialog.set_authors([
            'Siegfried-Angel Gevatter Pujals <siegfried@gevatter.com>',
            'Joe Burmeister <joe.a.burmeister@googlemail.com>'])
        dialog.set_logo(dialog.get_icon())
        
        def close(w, res=None):
            w.hide()
        
        dialog.connect('response', close)
        dialog.connect('destroy', close)
        
        return dialog
    
    def __init__(self):
        
        self._builder = gtk.Builder()
        self._builder.set_translation_domain('espeak-gui')
        self._builder.add_from_file(interface_file)
        self._builder.connect_signals(self)
        
        self._main = self._builder.get_object('main_window')
        self._main.set_icon_from_file(icon_file)
        
        self._inputbox = self._builder.get_object('inputbox')
        self._scrollbox = self._builder.get_object('scrolledwindow1')
        self._voicebox = self._builder.get_object('voicebox')
        self._playbutton = self._builder.get_object('button_read')
        self._speedspin = self._builder.get_object('speedspin')
        self._about_dialog = self._create_about_dialog()
        
        self._playing = False
        
        store = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING)
        for voice in self._get_voice_list():
            store.append([voice.name.capitalize(), voice.identifier])
        self._voicebox.set_model(store)
        cellr = gtk.CellRendererText()
        self._voicebox.pack_start(cellr)
        self._voicebox.add_attribute(cellr, 'text', 0)
        self._set_voice(
            settings.get_string(settings_selected_voice) or \
            'English') # Do not translate the "English" string
        self._voicebox.connect('key-press-event', self._on_voicebox_keypress)
        
        self._speedspin.set_value(settings.get_int(settings_speed) or 160)
        
        espeak.set_SynthCallback(self._e_eventCB)
        
        self._speaking_tag = self._inputbox.get_buffer().create_tag('speaking')
        self._speaking_tag.set_property('background', 'black')
        self._speaking_tag.set_property('foreground', 'white')
        
        self._speaking_text_offset = 0
        
        if arguments:
            self._load_files(*arguments)
        
        self._main.show_all()
        gtk.main()
    
    def _get_voice_list(self):
        """ Returns the output of espeak.list_voices() after sorting it. """
        return sorted(espeak.list_voices(), cmp=locale.strcoll,
            key=lambda voice: voice.name)
    
    def _find_voice_position(self, start=None, complete=None):
        """ Returns the position of the first voice whose name matches
            the value provided in `complete` or starts with the character
            given in `start`.
            
            Only one of `start` and `complete` can be used at the same time. """
        
        store = self._voicebox.get_model()
        needle = start or complete
        try:
            return [(x[0][0] if start else x[0]) for x in store].index(needle)
        except ValueError:
            return -1
    
    def _set_voice(self, voice):
        self._voicebox.set_active(self._find_voice_position(
            complete=voice.capitalize()))
    
    def _on_voicebox_keypress(self, voicebox, event):
        """ Iterates the selection over the languages starting with the
            typed character. """
        position = self._voicebox.get_active()
        store = self._voicebox.get_model()
        letter = event.string.upper()
        if not letter:
            return
        if store[position][0].startswith(letter) and len(store) > position and \
            store[position + 1][0].startswith(letter):
                self._voicebox.set_active(position + 1)
        else:
            self._voicebox.set_active(self._find_voice_position(start=letter))
    
    def _e_voicebox_changed(self, widget):
        language = widget.get_model()[self._voicebox.get_active()]
        self._active_voice = language[0]
        espeak.set_voice(language=language[1])
        settings.set_string(settings_selected_voice, language[0])
    
    def cleanup_speech_end(self):
        self._playing = False
        buffer = self._inputbox.get_buffer()
        buffer.remove_tag(self._speaking_tag, buffer.get_start_iter(),
            buffer.get_end_iter())
        self._inputbox.show_all()
        self._inputbox.set_editable(True)
        self._playbutton.set_label(_('Play'))
        self._playbutton.show()
    
    def _e_eventCB(self, event, pos, length):
        
        gtk.gdk.threads_enter()
        
        if event == espeak.core.event_WORD:
            pos += (self._speaking_text_offset - 1)
            buffer = self._inputbox.get_buffer()
            buffer.remove_tag(self._speaking_tag, buffer.get_start_iter(),
                buffer.get_end_iter())
            
            s = buffer.get_iter_at_offset(pos)
            e = buffer.get_iter_at_offset(length+pos)
            pos = self._inputbox.get_iter_location(s)
            
            adj = self._scrollbox.get_vadjustment()
            halfPage = adj.page_size / 2
            if (pos.y + pos.height) > (adj.value + halfPage):
                adj.set_value(min(adj.value+halfPage/2, adj.upper))
            elif pos.y < adj.value:
                adj.set_value(pos.y)
            
            buffer.apply_tag(self._speaking_tag, s, e)
            self._inputbox.show_all()
        
        if event == espeak.event_MSG_TERMINATED:
            self.cleanup_speech_end()
        
        gtk.gdk.threads_leave()
        
        return True
    
    def _play(self):
        if self._playing:
            return
        self._playing = True
        
        self._inputbox.set_editable(False)
        buffer = self._inputbox.get_buffer()
        text = ''
        
        if buffer.get_has_selection():
            selection = buffer.get_selection_bounds()
            
            s = selection[0]
            e = selection[1]
            
            self._speaking_text_offset = s.get_offset()
            text = buffer.get_text(s, e)
            buffer.select_range(buffer.get_start_iter(),
                buffer.get_start_iter())
        
        else:
            self._speaking_text_offset = 0
            text = buffer.get_text(buffer.get_start_iter(),
                buffer.get_end_iter())
        
        self._playbutton.set_label(_('Stop'))
        
        if not espeak.synth(text):
            self._playbutton.set_label(_('Play'))
            self._inputbox.set_editable(True)
            self._playing = False
    
    def _e_button_read_cd(self, *discard):
        if not self._playing:
            self._play()
        else:
            espeak.cancel()
            self.cleanup_speech_end()
        
        self._playbutton.show()
    
    def _e_play(self, *discard):
        self._play()
    
    def _e_speed_changed(self, widget):
        speed = int(widget.get_value())
        espeak.set_parameter(espeak.parameter_RATE, speed, False)
        settings.set_int(settings_speed, speed)
    
    def _e_new(self, *discard):
        buffer = self._inputbox.get_buffer()
        buffer.delete(buffer.get_start_iter(), buffer.get_end_iter())
    
    def _e_new_window(self, *discard):
        new_instance()
    
    def _load_files(self, *filenames):
        text = ''
        multiple_files = len(filenames) > 1
        for filename in filenames:
            if multiple_files:
                if text:
                    text += '\n' * (3 - len(text[-2:]) + len(text[-2:].strip()))
                text += _('File:') + ' %s\n\n' % os.path.basename(filename)
            try:
                f = open(filename)
            except IOError, error:
                text += _('This file couldn\'t be opened: %(strerror)s.') % error
            else:
                content = f.read()
                if content:
                    text += content
                elif multiple_files:
                    text += _('This file is empty.')
                del content
                f.close()
        self._inputbox.get_buffer().set_text(text)
    
    @staticmethod
    def _set_file_filters(dialog):
        filter = gtk.FileFilter()
        filter.set_name(_('All files'))
        filter.add_pattern('*')
        dialog.add_filter(filter)
        
        filter = gtk.FileFilter()
        filter.set_name(_('Plain text'))
        filter.add_mime_type('text/plain')
        filter.add_pattern('*.txt')
        dialog.add_filter(filter)
    
    def _e_open(self, *discard):
        dialog = gtk.FileChooserDialog(_('Open...'), self._main,
            gtk.FILE_CHOOSER_ACTION_OPEN, (gtk.STOCK_CANCEL,
            gtk.RESPONSE_CANCEL, gtk.STOCK_OPEN, gtk.RESPONSE_OK))
        dialog.set_select_multiple(True)
        dialog.set_default_response(gtk.RESPONSE_OK)
        self._set_file_filters(dialog)
        
        response = dialog.run()
        if response == gtk.RESPONSE_OK:
            self._load_files(*dialog.get_filenames())
        
        dialog.destroy()
    
    def _e_save(self, *discard):
        # TRANSLATORS: This is the title of the file selection dialog
        dialog = gtk.FileChooserDialog(_('Save as...'), self._main,
            gtk.FILE_CHOOSER_ACTION_SAVE, (gtk.STOCK_CANCEL,
            gtk.RESPONSE_CANCEL, gtk.STOCK_SAVE, gtk.RESPONSE_OK))
        dialog.set_default_response(gtk.RESPONSE_OK)
        self._set_file_filters(dialog)
        
        response = dialog.run()
        if response == gtk.RESPONSE_OK:
            f = open(dialog.get_filename(), 'w')
            if f != None:
                buffer = self._inputbox.get_buffer()
                f.write(buffer.get_text(buffer.get_start_iter(),
                    buffer.get_end_iter()) + '\n')
                f.close()
            else:
                msg = gtk.MessageDialog(self._main, gtk.DIALOG_MODAL,
                    gtk.MESSAGE_ERROR, gtk.BUTTONS_OK,
                    _('Could not open file "%s" for saving.') % dialog.get_filename())
                msg.run()
                msg.destroy()
        
        dialog.destroy()
    
    def _e_cut(self, *discard):
        if self._inputbox.get_editable():
            self._inputbox.get_buffer().cut_clipboard(gtk.Clipboard(), True)
        else:
            self._e_copy()

    def _e_copy(self, *discard):
        self._inputbox.get_buffer().copy_clipboard(gtk.Clipboard())

    def _e_paste(self, *discard):
        if self._inputbox.get_editable():
            self._inputbox.get_buffer().paste_clipboard(gtk.Clipboard(), None, True)
    
    def _e_delete(self, *discard):
        if self._inputbox.get_editable():
            self._inputbox.get_buffer().delete_selection(True,True)
    
    def _e_quit(self, *discard):
        espeak.set_SynthCallback(None)
        gtk.main_quit()
    
    def _e_about(self, *discard):
        self._about_dialog.run()

def main():
        
    app = Application()
