/* $Id: er_main.c,v 1.80 2010/11/22 14:05:59 kbanse Exp $
 *
 *   This file is part of the ESO Common Pipeline Library
 *   Copyright (C) 2001-2009 European Southern Observatory
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: kbanse $
 * $Date: 2010/11/22 14:05:59 $
 * $Revision: 1.80 $
 * $Name: esorex-3_9_0 $
 */

#ifdef HAVE_CONFIG_H
#   include <config.h>
#endif

#include <unistd.h>
#include <sys/types.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <dlfcn.h>

/*
#include <fitsio.h>
*/

#include <dirent.h>
#include <fcntl.h>
#include <sys/stat.h>

#include <cpl.h>
#include <cpl_version.h>		/* should be fixed */

# ifndef S_SPLINT_S
#include "ltdl.h"
# endif

#include "er_main.h"
#include "er_help.h"
#include "er_stringarray.h"
#include "er_params.h"
#include "er_paramutils.h"
#include "er_plugin.h"
#include "er_fileutils.h"
#include "er_stringutils.h"

#include "er_prefix.h"

int  MyOS = -1;

static char trace_msg[80];


/**
 * @defgroup er_main Main function for EsoRex
 *
 * Module containing the top-level function for the EsoRex application.
 *
 */


/**@{*/

/**********************************************************************/
/**
 * @brief   Fills a list with EsoRex-specific parameters.
 *
 * @param param_list  List of parameters
 */
/**********************************************************************/

# ifndef S_SPLINT_S

static void er_init_parameters (cpl_parameterlist * param_list)
{
cpl_parameter *p;

  /*
   * Processing different EsoRex options
   */

  /*
   *  help
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".help",
                             CPL_TYPE_BOOL,
                             "Display this help and exit. "
                             "If a recipe name is also given, then "
                             "help will be given for it as well.",
                             PACKAGE_RESOURCE, FALSE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "help");
/*
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV, PACKAGE_ENV "_HELP");
*/
cpl_parameterlist_append (param_list, p);

  /*
   *  version
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".version",
                             CPL_TYPE_BOOL,
                             "Display version information and exit.",
                             PACKAGE_RESOURCE, FALSE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "version");
/*
cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_ENV,PACKAGE_ENV "_VERSION");
*/
cpl_parameterlist_append (param_list, p);

/*
 *      check-sof-exist
 */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".check-sof-exist",
                             CPL_TYPE_BOOL,
                             "When TRUE, all the input files must exist  "
                             "and be readable before calling the recipe. ",
                             PACKAGE_RESOURCE, FALSE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "check-sof-exist");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_CHECK_SOF_EXIST");
cpl_parameterlist_append (param_list, p);

  /*
   *  config
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".config",
                             CPL_TYPE_STRING,
                             "Configuration file to be used for EsoRex.",
                             PACKAGE_RESOURCE, NULL);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "config");
cpl_parameter_disable (p, CPL_PARAMETER_MODE_CFG);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_CONFIG");
cpl_parameterlist_append (param_list, p);

  /*
   *  create-config
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".create-config",
/*
                             CPL_TYPE_BOOL,
*/
                             CPL_TYPE_STRING,
                             "Creates a configuration file for Esorex. "
                             "If set to TRUE a config file 'esorex.rc' is created "
                             "in the '.esorex' directory in $HOME of the user."
                             "If a filename is specified, a config file will be "
                             "created accordingly. "
                             "If a recipe is specified in the command line, then the "
                             "configuration file will "
                             "be created for the recipe instead (called "
                             "'recipename.rc')"
                             " Note that an existing file will be "
                             "overwritten, but a backup file will be "
                             "copied to 'filename.rc.bak' in "
                             "the same directory.", 
                             PACKAGE_RESOURCE, "FALSE");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "create-config");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_CREATE_CONFIG");
cpl_parameterlist_append (param_list, p);

  /*
   *  link-dir
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".link-dir",
                             CPL_TYPE_STRING,
                             "The directory in which a symbolic link "
                             "to each of the product files should be "
                             "written. The "
                             "enable/disable switch to control "
                             "whether the link is actually made is "
                             "the '--suppress-link' option.",
                             PACKAGE_RESOURCE, "/tmp");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "link-dir");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_LINK_DIR");
cpl_parameterlist_append (param_list, p);

  /*
   *      log-dir
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".log-dir",
                             CPL_TYPE_STRING,
                             "Directory where to place the logfile.",
                             PACKAGE_RESOURCE, ".");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "log-dir");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_LOG_DIR");
cpl_parameterlist_append (param_list, p);

  /*
   *      log-file
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".log-file",
                             CPL_TYPE_STRING,
                             "Filename of logfile.",
                             PACKAGE_RESOURCE, PACKAGE ".log");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "log-file");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_LOG_FILE");
cpl_parameterlist_append (param_list, p);

  /*
   *      log-level
   */

p = cpl_parameter_new_enum (PACKAGE_RESOURCE ".log-level",
                            CPL_TYPE_STRING,
                            "Controls the severity level of messages "
                            "that will be printed to the logfile.",
                            PACKAGE_RESOURCE,
                            "info", 5, "debug", "info", "warning",
                            "error", "off");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "log-level");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_LOG_LEVEL");
cpl_parameterlist_append (param_list, p);

  /*
   *  man-page
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".man-page",
                             CPL_TYPE_BOOL,
                             "Display a manual page for the specified "
                             "recipe, and then exit. Note that this option "
                             "only applies to recipes, and that it does "
                             "nothing for " PACKAGE
                             " by itself. See also "
                             "the '--help' option.", PACKAGE_RESOURCE,
                             FALSE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "man-page");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_MAN_PAGE");
cpl_parameterlist_append (param_list, p);

  /*
   *  memcheck
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".mem-check",
                             CPL_TYPE_BOOL,
                             "Report on memory status at completion "
                             "of recipe execution.",
                             PACKAGE_RESOURCE, FALSE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "mem-check");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_MEM_CHECK");
cpl_parameterlist_append (param_list, p);

  /*
   *      msg-level
   */

p = cpl_parameter_new_enum (PACKAGE_RESOURCE ".msg-level",
                            CPL_TYPE_STRING,
                            "Controls the severity level of messages "
                            "that will be printed to the terminal.",
                            PACKAGE_RESOURCE,
                            "info", 5, "debug", "info", "warning",
                            "error", "off");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "msg-level");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_MSG_LEVEL");
cpl_parameterlist_append (param_list, p);

  /*
   *  output-dir
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".output-dir",
                             CPL_TYPE_STRING,
                             "The directory where the product "
                             "files should be finally moved to "
                             "(all products are first created in the "
                             "current dir).",
                             PACKAGE_RESOURCE, ".");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "output-dir");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_OUTPUT_DIR");
cpl_parameterlist_append (param_list, p);

  /*
   *  output-prefix
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".output-prefix",
                             CPL_TYPE_STRING,
                             "Prefix applied to any output file. "
                             "For example, specifying 'pre' would translate "
                             "'filename.fits' to 'pre_0000.fits'. See also "
                             "the '--suppress-prefix' option.",
                             PACKAGE_RESOURCE, "out");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "output-prefix");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_OUTPUT_PREFIX");
cpl_parameterlist_append (param_list, p);

  /*
   *      output-readonly  (replaces in some ways the old "output-overwrite")
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".output-readonly",
                             CPL_TYPE_BOOL,
                             "When TRUE, any existing product files "
                             "in the specified output directory will be "
                             "set to read-only, for user, group and other. "
                             "If FALSE, then EsoRex will use the default "
                             "permissions for that account/directory. "
                             "destroy any pre-existing files. "
                             "This option exists for the Paranal operations "
                             "environment. This option can additionally be "
                             "used to prevent EsoRex from overwriting "
                             "pre-existing files.",
                             PACKAGE_RESOURCE, FALSE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "output-readonly");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_OUTPUT_READONLY");
cpl_parameterlist_append (param_list, p);

  /*
   *  paf-config
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".paf-config",
                           CPL_TYPE_STRING,
                           "Configuration file for creation of PAF files.",
                           PACKAGE_RESOURCE, "");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "paf-config");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                       PACKAGE_ENV "_PAF_CONFIG");
cpl_parameterlist_append (param_list, p);

  /*
   *  params
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".params",
                             CPL_TYPE_BOOL,
                             "List the input parameters and their current "
                             "settings (whether from the command line or a "
                             "configuration file) for the " PACKAGE
                             " application. Parameters are "
                             "labelled using the parameter's alias. "
                             "If a recipe is also specified, then the "
                             "list of its parameters will also be generated "
                             "in the same way.",
                             PACKAGE_RESOURCE, FALSE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "params");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_PARAMS");
cpl_parameterlist_append (param_list, p);

  /*
   *  recipes
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".recipes",
                             CPL_TYPE_BOOL,
                             "Display a list of all available recipes "
                             "(that are available in the directory tree "
                             "specified with '--recipe-dir').",
                             PACKAGE_RESOURCE, FALSE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "recipes");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_RECIPES");
cpl_parameterlist_append (param_list, p);

  /*
   *  recipe-config
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".recipe-config",
                             CPL_TYPE_STRING,
                             "Configuration file for any selected recipe.",
                             PACKAGE_RESOURCE, NULL);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "recipe-config");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_RECIPE_CONFIG");
cpl_parameterlist_append (param_list, p);

  /*
   *  recipe-dir
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".recipe-dir",
                             CPL_TYPE_STRING,
                             "Directory containing recipe libraries. Note "
                             "that " PACKAGE
                             " will recursively search not "
                             "only the specified directory, but all "
                             "sub-directories below it as well. "
                             "Multiple directory heads may be "
                             "specified, by separating the "
                             "starting paths with colons (:).",
                             PACKAGE_RESOURCE, ".");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "recipe-dir");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_PLUGIN_DIR");
cpl_parameterlist_append (param_list, p);

/*
 *      suppress_link
 */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".suppress-link",
                             CPL_TYPE_BOOL,
                             "When TRUE, no symbolic link is created "
                             "to the output product. However, "
                             "if FALSE, then a symbolic link is created "
                             "in the directory specified with the "
                             "option '--link-dir' for each product "
                             "that is created by the recipe.",
                             PACKAGE_RESOURCE, TRUE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "suppress-link");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                       PACKAGE_ENV "_SUPPRESS_LINK");
cpl_parameterlist_append (param_list, p);

  /*
   *      suppress_prefix
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".suppress-prefix",
                             CPL_TYPE_BOOL,
                             "When TRUE, the original name of the "
                             "output product, as produced by the "
                             "recipe, is maintained. "
                             "If FALSE, then the name of the output "
                             "file is changed to the \"prefix_number\" "
                             "format. The prefix can be altered using the "
                             "'--output-prefix' option.",
                             PACKAGE_RESOURCE, FALSE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "suppress-prefix");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_SUPPRESS_PREFIX");
cpl_parameterlist_append (param_list, p);

  /*
   *  time
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".time",
                             CPL_TYPE_BOOL,
                             "Measure and show the recipe's execution time.",
                             PACKAGE_RESOURCE, FALSE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "time");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                       PACKAGE_ENV "_TIME");
cpl_parameterlist_append (param_list, p);

  /*
   *      unload-plugin
   */

p = cpl_parameter_new_value (PACKAGE_RESOURCE ".unload-plugin",
                             CPL_TYPE_BOOL,
                             "When TRUE, the plugin is unloaded "
                             "after execution. "
                             "If FALSE, the plugin is not unloaded "
                             "after processing, so that a software "
			     "like, e.g. valgrind, can be used "
			     "for debugging the executed recipe. ",
                             PACKAGE_RESOURCE, TRUE);
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_CLI, "unload-plugin");
cpl_parameter_set_alias (p, CPL_PARAMETER_MODE_ENV,
                         PACKAGE_ENV "_UNLOAD_PLUGIN");
cpl_parameterlist_append (param_list, p);


/*
 * The parameter list is now filled. The remaining step is to set
 * the tag to indicate that the sources of the current values for
 * all parameters in this list are the EsoRex defaults.
 */

p = cpl_parameterlist_get_first (param_list);
while (p != NULL)
   {
   char *cptr, def_str[] = PACKAGE " default";
 
   cptr = def_str;				/* save source of this param */
   er_manage_sources(1,cpl_parameter_get_name(p),&cptr);

   /* Get the next parameter in the list */
   p = cpl_parameterlist_get_next (param_list);
   }                            /* End of loop through all parameters in the list */



}                               /* End of er_init_parameters() */

# endif

/*

*/

/**********************************************************************/
/**
 * @brief
 *   Prints Program Information header to stdout
 */
/**********************************************************************/

static void er_print_header (void)
{


printf ("\n     ***** %s, version %s  *****\n", PACKAGE_NAME, PACKAGE_VERSION);

/* printf ("\n           as of 091217\n"); */

fflush(stdout);

/* 
MyOS indicates the host system:
	MyOS = 1	Linux
	MyOS = 0	Mac OSX
*/

/* poor man's way to get OS used... 
   currently not needed 
   fortunately, because it fails when Esorex cannot write into /tmp !

char buf[10];
int src;

system("uname -s > /tmp/esorex-HostOs");
src = open ("/tmp/esorex-HostOs", O_RDONLY);
if ((int) read (src, buf, 8) > 0)
   {
   if (strncmp(buf,"Linu",4) != 0)
   MyOS = 0;				/. should be a Mac ./
   }
close(src);
*/


/* so we just default to Linux */

MyOS = 1;

}

/*

*/

/**********************************************************************/
/**
 * @brief      Top-level function of the EsoRex application.
 *
 * @param argc Count of commandline arguments
 * @param argv Handle to commandline aguments
 */
/**********************************************************************/

int main(int argc, char *argv[])
{
const char *cpp, *fn = "EsoRex";

char  plugin_name[FILEMAX+2];
char  *conf_file_global=NULL, *conf_file_local=NULL;

cpl_parameter *p = NULL;
cpl_parameterlist *caller_parameter_list = NULL;
cpl_msg_severity msg_level = CPL_MSG_ERROR;

er_stringarray_t *set_of_frames_filenames = NULL;
 
int e_code = 0;
int e_code2 = 0;
int flag_mem_check = 1;
int  n;

unsigned  init_flag = CPL_INIT_DEFAULT;

FILE  *rcfp;



/* check that we're not somebody else... */

if (getuid() != geteuid()) 
   {
   (void) printf("effective user ID not the same as real user ID => no EsoRex...\n");
   exit(EXIT_FAILURE);
   }


/* Initialization and setup */

cpl_init (init_flag);                    /* also does cpl_msg_init() stuff */

cpl_msg_set_domain (PACKAGE);
cpl_msg_set_time_off ();
cpl_msg_set_domain_on ();

er_print_header ();



/* Main processing */

if (lt_dlinit () != 0)
   {
   cpl_msg_error (fn, "Unable to initialise ltdl; aborting program");
   cpl_end ();                  /* stop subsystems of CPL */
   e_code = -1;
   goto clean_up;
   }


caller_parameter_list = cpl_parameterlist_new ();
set_of_frames_filenames = er_stringarray_new ();

#ifdef USE_LOCAL_ESOREX_RC		/* Specific configuration file */
n = ((int) strlen(LOCAL_RC_PATH)) +
    ((int) strlen(GLOBAL_RC_NAME)) + ((int) strlen(GLOBAL_RC_EXTENSION));
n += 4;

conf_file_local = (char *) cpl_malloc((size_t) n);
if ( conf_file_local == NULL)
   {
   cpl_msg_error (er_func, "Could not allocate %d bytes for conf_file_local",n);
   e_code = -1;
   goto clean_up;
   }

(void) strcpy(conf_file_local,LOCAL_RC_PATH);
(void) strcat(conf_file_local, "/");
(void) strcat(conf_file_local, GLOBAL_RC_NAME);
(void) strcat(conf_file_local, GLOBAL_RC_EXTENSION);
#endif

#ifdef USE_GLOBAL_ESOREX_RC		/* User configuration file */
cpp = getenv ("HOME");
if (cpp == NULL) 
   {
   n = 0;
   cpl_msg_warning (er_func, "Env. variable HOME not set");
   }
else
   {
   n = (int) strlen(cpp);
   }

n = n + ((int) strlen(GLOBAL_RC_DIR)) +
    ((int) strlen(GLOBAL_RC_NAME)) + ((int) strlen(GLOBAL_RC_EXTENSION));
n += 4;

conf_file_global = (char *) cpl_malloc((size_t) n);
if ( conf_file_global == NULL)
   {
   cpl_msg_error (er_func, "Could not allocate %d bytes for conf_file_global",n);
   e_code = -1;
   goto clean_up;
   }

if (cpp == NULL)
   {
   (void) strcpy(conf_file_global, "/");
   }
else
   {
   (void) strcpy(conf_file_global,cpp);
   (void) strcat(conf_file_global, "/");
   }
(void) strcat(conf_file_global, GLOBAL_RC_DIR);
(void) strcat(conf_file_global, "/");
(void) strcat(conf_file_global, GLOBAL_RC_NAME);
(void) strcat(conf_file_global, GLOBAL_RC_EXTENSION);


/* check if resource file exists, 
   use .rc file from --prefix dir in configuration of Esorex if necessary,
   this is needed for Scisoft  */

rcfp = fopen(conf_file_global, "r");
if (rcfp != NULL)
   {
   fclose(rcfp);
   }
else
   {
   char save_name[1024], new_name[1024];

   (void) strcpy(save_name,conf_file_global);

   (void) strcpy(new_name,ESOREX_PREFIX);	/* replace by .rc name in --prefix dir */
   (void) strcat(new_name, GLOBAL_RC_NAME);
   (void) strcat(new_name, GLOBAL_RC_EXTENSION);

   rcfp = fopen(new_name, "r");
   if (rcfp == NULL)				/* no .rc file in --prefix dir, */
      {						/* keep original name for compatibility */
      (void) strcpy(conf_file_global,save_name);
      }
   else
      {
      fclose(rcfp);

      cpl_free(conf_file_global);
      n = (int) strlen(new_name) + 4;
      conf_file_global = (char *) cpl_malloc((size_t) n);
      if ( conf_file_global == NULL)
         {
         cpl_msg_error (er_func, "Could not allocate %d bytes for conf_file_global",n);
         e_code = -1;
         goto clean_up;
         }
      (void) strcpy(conf_file_global,new_name);
      }
   }
#endif


/* Create list of parameters for EsoRex */

er_init_parameters (caller_parameter_list);

if (argc == 1)		/* entering just Esorex yields the library versions */
   {
   const char *cdescr;

   cdescr = cpl_get_description(CPL_DESCRIPTION_DEFAULT);
   (void) printf("\nLibraries used: %s\n\n",cdescr);
   goto clean_up;
   }

/* if ((argc > 1) && ((int)strcmp(argv[1],"--V") == 0)) */
if ((argc == 2) && ((int)strcmp(argv[1],"--V") == 0)) 
   {
   const char *cdescr;

   (void) printf("           last modif - 101118 \n");

   cdescr = cpl_get_description(CPL_DESCRIPTION_DEFAULT);
   (void) printf("\nLibraries used: %s\n\n",cdescr);

   (void) printf("using %s as resource file\n",conf_file_global);

   /* 
   n = er_fileutils_file_is_fits(argv[2]);
   printf("er_fileutils_file_is_fits(%s) returned %d\n",argv[2],n);
   */

   goto end_of_it;
   }

(void) printf("\n");

/* Process caller configuration information */

plugin_name[0] = '\0';
e_code = params_process_configuration (caller_parameter_list,
                                       conf_file_global, conf_file_local,
                                       argc, argv,
                                       plugin_name, set_of_frames_filenames);

(void) sprintf(trace_msg,"param_process_configuration returned %d",e_code);
ER_TRACEX (trace_msg)


/* Process Plugin */

if (e_code == 0)
   {
   e_code = plugin_process_plugin (caller_parameter_list,
                                   plugin_name, set_of_frames_filenames,
                                   argc, argv);
 
   (void) sprintf(trace_msg,"plugin_process_plugin returned %d",e_code); 
   ER_TRACEX(trace_msg)
   }



msg_level = cpl_msg_get_level();	/* test, if we are in debug mode */
if (msg_level == CPL_MSG_DEBUG) 
   {						/* if debug level set, */
   e_code2 = cpl_error_get_code ();		/* check the CPL_error subsystem */
   if (e_code2 != CPL_ERROR_NONE)
      {
      cpl_msg_error (fn, "CPL-error '%s' was set in '%s'",
                     cpl_error_get_message (), cpl_error_get_where ());
      if (e_code == 0)
         {
         cpl_msg_error (fn, "recipe %s returned status: %d\n",plugin_name,e_code);
         }
      }
   }

if (lt_dlexit () != 0) cpl_msg_error (fn, "Unable to deinitialize ltdl");

p = cpl_parameterlist_find (caller_parameter_list, PACKAGE_RESOURCE ".mem-check");
flag_mem_check = cpl_parameter_get_bool (p);


/* cleanup allocated structures and memory */

clean_up:
cpl_parameterlist_delete (caller_parameter_list);
er_stringarray_delete (set_of_frames_filenames);

if (conf_file_local != NULL) 
   {
   cpl_free(conf_file_local);
   }
if (conf_file_global != NULL) 
   {
   cpl_free(conf_file_global);
   }

(void) er_help_free();
(void) er_manage_sources(3," ",&conf_file_local);	/* free buffer of sources */


if (flag_mem_check != 0)	/* see if we should check the memory */
   {
   cpl_memory_dump(); 
   }

end_of_it:
cpl_end ();			/* stop subsystems of CPL */

if (e_code == -99999) e_code = 0;		/* -99999 for non-executing options like --version */
return (e_code);                /* exit the program */
}


/**@}*/


/* End of file */
