MLoef <- function(robj, splitcr="median")
{
# performs the Martin-Lf LR-test
# robj... object of class RM
# splitcr... splitting criterion for two groups. "median" (default) and "mean"
#            split items in two groups according to the median/mean or item raw
#            scores.
#            a vector of length k (number of items) containing two different
#            elements signifying group membership of items can be supplied.

  if(all(class(robj) != "Rm")) stop("robj must be of class \"Rm\".")

  wrning <- NULL   # initialize an object for warnings

  if(length(splitcr) == 1){   # generate split-vector if "mean" or "median"
    if(splitcr=="median"){
      raw.scores <- colSums(robj$X01,na.rm=T)
      numsplit <- as.numeric(raw.scores > median(raw.scores,na.rm=T))
      if( any(raw.scores == median(raw.scores,na.rm=T)) ){   # Only if one item's raw score == the median, a warning is issued
        wrning <- which(raw.scores == median(raw.scores,na.rm=T))   # append a warning-slot to the object for print and summary methods
        cat("Item(s)",paste(names(wrning),collapse=", "),"with raw score equal to the median assigned to the lower raw score group!\n")
      }
    }
    if(splitcr=="mean"){
      raw.scores <- colSums(robj$X01,na.rm=T)
      numsplit <- as.numeric(raw.scores > mean(raw.scores,na.rm=T))
      if( any(raw.scores == mean(raw.scores,na.rm=T)) ){   # Only if one item's raw score == the mean, a warning is issued
        wrning <- which(raw.scores == mean(raw.scores,na.rm=T))   # append a warning-slot to the object for print and summary methods
        cat("Item(s)",paste(names(wrning),collapse=", "),"with raw score equal to the mean assigned to the lower raw score group!\n")
      }
    }
  }
  else{   # check if the submitted split-vector is appropriate
    if(length(splitcr) != ncol(robj$X01)) stop("Split vector too long/short.")
    if(length(unique(splitcr)) > 2) stop("Only two groups allowed.")
    if(length(unique(splitcr)) < 2) stop("Split vector must contain two groups.")
    numsplit <- splitcr
  }
  sp.groups <- unique(numsplit)
  i.groups <- list(which(numsplit == sp.groups[1]), which(numsplit == sp.groups[2]))

  # check if one group countains less than 2 items
  if( (length(i.groups[[1]]) < 2) | (length(i.groups[[2]]) < 2) ){
    stop("Each group of items must contain at least 2 items.")
  }

  # check if one group contains subject with <=1 valid responses
  if(any(rowSums(is.na(robj$X01[,i.groups[[1]]])) >= (length(i.groups[[1]]) - 1)))stop("Group 1 contains subjects with less than two valid responses.")
  if(any(rowSums(is.na(robj$X01[,i.groups[[2]]])) >= (length(i.groups[[2]]) - 1)))stop("Group 2 contains subjects with less than two valid responses.")

  ### no test with missing values rh 19-05-10
  if (any(is.na(robj$X)))stop("Martin-Loef Test with NA currently not available\n")

  ### possible missing patterns and classification of persons into groups
  MV.X <- apply(matrix(as.numeric(is.na(robj$X01)),ncol=ncol(robj$X01)),1,paste,collapse="")
  MV.p <- sort(unique(MV.X))
  MV.g <- numeric(length=length(MV.X))
  g <- 1
  for(i in MV.p){
    MV.g[MV.X == i] <- g;
    g <- g + 1
  }
  na.X01 <- list()
  for(i in 1:length(MV.p)){
    na.X01[[i]] <- matrix(robj$X01[which(MV.g == i),], ncol=ncol(robj$X01))
  }

  res1 <- RM(robj$X01[,i.groups[[1]]])
  res2 <- RM(robj$X01[,i.groups[[2]]])

  ### calculating the numerator and denominator
  ml.num <- ml.den <- numeric()

  for(i in 1:length(MV.p)){
    .temp.num <- table(rowSums(na.X01[[i]],na.rm=T))
    ml.num[i] <- sum(log((.temp.num/sum(.temp.num))^.temp.num))

    if(nrow(na.X01[[i]]) > 1){
      .temp.den <- table(rowSums(na.X01[[i]][,i.groups[[1]]],na.rm=T),
                         rowSums(na.X01[[i]][,i.groups[[2]]],na.rm=T))
    }
    else{
      .temp.den <- table(sum(na.X01[[i]][,i.groups[[1]]],na.rm=T),
                         sum(na.X01[[i]][,i.groups[[2]]],na.rm=T))
    }
    ml.den[i] <- sum(log((.temp.den/sum(.temp.den))^.temp.den))
  }

  a <- sum(ml.num)
  b <- sum(ml.den)
  k <- c(length(i.groups[[1]]),length(i.groups[[2]]))

  ML.LR <- -2*( (a + robj$loglik) - (b + res1$loglik + res2$loglik) )
  DF <- prod(k) - 1
  p.value <- 1 - pchisq(ML.LR, DF)

  result <- list(X01=robj$X01, model=robj$model, LR=ML.LR,
                 df=DF, p.value=p.value, L0=robj$loglik,  L1=res1$loglik,  L2=res2$loglik,
                 theta.table.RM=table(rowSums(robj$X01)),                        # both used for the plotting
                 theta.table.MLoef=table(rowSums(res1$X01),rowSums(res2$X01)),   # routine plot.MLoef
                 items1=i.groups[[1]], items2=i.groups[[2]], k=k,
                 splitcr=splitcr, split.vector=numsplit, warning=wrning, call=match.call())
  class(result) <- "MLoef"
  return(result)
}
