#-------------------------------------------------------------------------------
#  
#  Defines the OMTrackContactManager class of the Enable 'om' (Object Model) 
#  package. 
#
#  The OMTrackContactManager is a subclass of OMContactManager that implements
#  a specific style of contact management based upon the notion of 'tracks'
#  (implemented by the OMTack class). A track is a rectangular horizontally or
#  vertically oriented region that can contain zero, one or more Contacts, all
#  of which share a common vertical (or horizontal) 'center' point. An
#  OMTrackContactManager object can contain/manage several tracks that are
#  arranged around the inner component of an OMComponent object, as shown in
#  the following diagram:
#
#            +-----------------------------------------------------+
#            |     |                 Track                   |     |
#            |     +-----+-----------------------------+-----+     |
#            |     |Inner|        Inner Track          |Inner|     |
#            |     |     +-----------------------------+     |     |
#            |  T  |  T  |                             |  T  |  T  |
#            |  r  |  r  |                             |  r  |  r  |
#            |  a  |  a  |      'Inner' component      |  a  |  a  |
#            |  c  |  c  |                             |  c  |  c  |
#            |  k  |  k  |                             |  k  |  k  |
#            |     |     +-----------------------------+     |     |
#            |     |     |        Inner Track          |     |     |
#            |     +-----+-----------------------------+-----+     |
#            |     |                 Track                   |     |
#            +-----------------------------------------------------+
#
#  Tracks have associated with them several characteristics used by the 
#  OMTrackContactManager and the component's associated controller object:
#    - position (left, right, top, bottom, inner left, inner right, ...)
#    - name     (a string identifying the track)
#    - category (a string specifying the category of Contacts the track should
#                be used for)
#    - group    (a string specifying the group of Contacts the track should
#                be used for)
#  
#  Written by: David C. Morrill
#  
#  Date: 02/01/2005
#  
#  (c) Copyright 2005 by Enthought, Inc.
#  
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:  
#-------------------------------------------------------------------------------

from om_component       import OMComponent
from om_contact_manager import OMContactManager
from om_track           import OMTrack, OUTER_LEFT, OUTER_RIGHT, OUTER_TOP, \
                               OUTER_BOTTOM, INNER_LEFT, INNER_RIGHT, \
                               INNER_TOP, INNER_BOTTOM
from om_traits          import TOP, CENTER, BOTTOM                               

from enthought.traits.api   import HasPrivateTraits, List, Instance, Property

#-------------------------------------------------------------------------------
#  Trait definitions:  
#-------------------------------------------------------------------------------

AnOMTrack = Instance( OMTrack, allow_none = True )

#-------------------------------------------------------------------------------
#  'OMTrackContactManager' class:  
#-------------------------------------------------------------------------------

class OMTrackContactManager ( HasPrivateTraits, OMContactManager ):
   
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------

    # The component that we are managing tracks and contacts for:
    component = Instance( OMComponent )
    
    # Set of tracks being managed:
    tracks    = List( AnOMTrack, 
                       [ None, None, None, None, None, None, None, None ],
                       minlen = 8, maxlen = 8 )
                       
    # List of all links connected to the associated component:                       
    links     = Property  
    
    # List of all contacts for the associated component:
    contacts  = Property
    
#-- Property Implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  'links' property:  
    #---------------------------------------------------------------------------

    def _get_links ( self ):
        result = []
        for track in self.tracks:
            if track is not None:
                result.extend( track.links )
        return result
        
    #---------------------------------------------------------------------------
    #  'contacts' property  
    #---------------------------------------------------------------------------
                
    def _get_contacts ( self ):
        contacts = []
        for track in self.tracks:
             if track is not None:
                 contacts.extend( track.contacts )
        return contacts
    
#-- Mandatory Interface --------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Adds one or more Contact objects to the associated Component:
    #---------------------------------------------------------------------------

    def add_contacts ( self, *contacts ):
        """ Adds one or more Contact objects to the associated Component.
        """
        # Loop over all of the contacts specified:
        for i in range( len( contacts ) - 1, -1, -1 ):
            contact = contacts[i]
                      
            # Search all tracks whose category matches the contact and find the
            # one with the fewest number of contacts:
            best_track   = None
            min_contacts = 999999
            for track in self.get_tracks( category = contact.category ):
                n = len( track.get_contacts() )
                if n < min_contacts:
                    min_contacts = n
                    best_track   = track
                    
            # If no match was found, report the error:
            if best_track is None:
                raise ValueError, (
                      "No track defined for contact with category '%s'" % 
                      contact.category)
                      
            # And add the contact to the best track found:
            best_track.add_contacts( contact )
            
            # Otherwise, connect the contact to the component:
            contact.container = self.component
        
    #---------------------------------------------------------------------------
    #  Removes one or more Contact objects from the associated Component:  
    #---------------------------------------------------------------------------
                
    def remove_contacts ( self, *contacts ):
        """ Removes one or more Contact objects from the associated Component.
        """
        for track in self.tracks:
            if track is not None:
                track.remove_contacts( *contacts )
        
    #---------------------------------------------------------------------------
    #  Gets the list of Contacts associated with a Component that match an 
    #  optionally specified name, category and/or group. If no arguments are 
    #  specified, it returns all Contacts associated with the Component;
    #  otherwise it returns all Contacts whose name, category or group match
    #  the specified value.
    #---------------------------------------------------------------------------
                
    def get_contacts ( self, name = None, category = None, group = None ):
        """ Gets the list of Contacts associated with a Component that match an 
            optionally specified name, category and/or group. If no arguments 
            are specified, it returns all Contacts associated with the 
            Component; otherwise it returns all Contacts whose name, category
            or group match the specified value.
        """
        result = []
        for track in self.tracks:
            if track is not None:
                result.extend( track.get_contacts( name, category, group ) )
        return result
        
    #---------------------------------------------------------------------------
    #  Lays out all of the Contacts being managed within the specified region:  
    #---------------------------------------------------------------------------
            
    def layout_contacts ( self, width = None, height = None ):
        """ Lays out all of the Contacts being managed.
        """
        # Get the component we are associated with:
        component = self.component
        
        # Get the current size of the component (before the following code
        # changes it incidentally):
        old_dx, old_dy = component.size
        
        # Gather up all the layout info we need from its 'outer component':
        oc  = component.outer_component
        ar  = oc.aspect_ratio
        chs = oc.horizontally_symmetric
        cvs = oc.vertically_symmetric
        ml  = oc.left_margin
        mr  = oc.right_margin
        mt  = oc.top_margin
        mb  = oc.bottom_margin
        pl  = oc.left_padding
        pr  = oc.right_padding
        pt  = oc.top_padding
        pb  = oc.bottom_padding
        
        # Get the position and minimum size of the component's 
        # 'inner component':
        icp  = component.inner_component_position_ 
        ic   = component.inner_component
        t_dx = t_dy = b_dy = 0
        if ic is None:
            ic_dx = ic_dy = 0
        else:
            ic_dx, ic_dy = ic.min_width, ic.min_height
            if icp != CENTER:
                t_dx = ic_dx + pl + pr
                if icp == TOP:
                    t_dy = ic_dy
                elif icp == BOTTOM:
                    b_dy = ic_dy
            
        # Get the minimum size of all of the tracks:
        itl_dx, itl_dy = self._get_min_track_size( INNER_LEFT )
        itr_dx, itr_dy = self._get_min_track_size( INNER_RIGHT )
        itt_dx, itt_dy = self._get_min_track_size( INNER_TOP )
        itb_dx, itb_dy = self._get_min_track_size( INNER_BOTTOM )
        otl_dx, otl_dy = self._get_min_track_size( OUTER_LEFT )
        otr_dx, otr_dy = self._get_min_track_size( OUTER_RIGHT )
        ott_dx, ott_dy = self._get_min_track_size( OUTER_TOP )
        otb_dx, otb_dy = self._get_min_track_size( OUTER_BOTTOM )
                      
        # Get the center points of the outer tracks:
        otl_cx, otl_cy = self._get_track_center( OUTER_LEFT )
        otr_cx, otr_cy = self._get_track_center( OUTER_RIGHT )
        ott_cx, ott_cy = self._get_track_center( OUTER_TOP )
        otb_cx, otb_cy = self._get_track_center( OUTER_BOTTOM )
        
        # Adjust the width/height of the inner tracks:
        if icp == CENTER:
            itt_dx = itb_dx = max( itt_dx, itb_dx, ic_dx )
            itl_dy = itr_dy = max( itl_dy, itr_dy, ic_dy )
        
        # Compute the 'padding' sizes for the component:
        pl_dx  = max( otl_dx - otl_cx, pl )
        pr_dx  = max( otr_cx, pr )
        pt_dy  = max( ott_cy, pt )
        pb_dy  = max( otb_dy - otb_cy, pb )
        
        # Compute the 'margin' sizes for the component:
        ml_dx = max( otl_cx, ml )
        mr_dx = max( otr_dx - otr_cx, mr )
        mt_dy = max( ott_dy - ott_cy, mt )
        mb_dy = max( otb_cy, mb )
        
        # Compute the width/height of the component and outer tracks:
        if chs:
            # Horizontally symmetric case:
            c_dx   = max( max( ott_dx, otb_dx ) + pl_dx + pr_dx, t_dx,
                          2 * max( pl_dx + itl_dx, pr_dx + itr_dx ) + itt_dx )
            ott_dx = otb_dx = c_dx - pl_dx - pr_dx
            itt_dx = itb_dx = itt_dx + (ott_dx - itt_dx - 
                                        2 * max( itl_dx, itr_dx )) / 2
            h_dx   = ml_dx + (c_dx / 2)
        else:
            # Horizontally asymmetric case:
            dx     = itt_dx + itl_dx + itr_dx
            c_dx   = max( max( ott_dx, otb_dx, dx ) + pl_dx + pr_dx, t_dx )
            ott_dx = otb_dx = c_dx - pl_dx - pr_dx
            itt_dx = itb_dx = itt_dx + (ott_dx - dx) / 2
            h_dx   = ml_dx + pl_dx + itl_dx + (itt_dx / 2)
            
        if cvs:
            # Vertically symmetric case:
            c_dy   = (max( max( otl_dy, otr_dy ) + pb_dy + pt_dy,
                           2 * max( pt_dy + itt_dy, pb_dy + itb_dy ) + itl_dy )
                           + t_dy + b_dy)
            otl_dy = otr_dy = c_dy - pt_dy - pb_dy - t_dy
            itl_dy = itr_dy = itl_dy + (otl_dy - itl_dy - 
                                        2 * max( itt_dy, itb_dy )) / 2
            h_dy   = mb_dy + (c_dy / 2) + b_dy
        else:
            # Vertically asymmetric case:
            otl_dy = otr_dy = max( otl_dy, otr_dy, itl_dy + itt_dy + itb_dy )
            itl_dy = itr_dy = itl_dy + (otl_dy - dy) / 2
            c_dy   = otl_dy + pt_dy + pb_dy + t_dy + b_dy
            h_dy   = otb_dy + itb_dy + (itl_dy / 2) + b_dy
            
        # Compute the overall (and also minimum) size of the component:
        component.min_width  = tc_dx = c_dx + ml_dx + mr_dx
        component.min_height = tc_dy = c_dy + mt_dy + mb_dy
        
        ec_dx = ec_dy = 0
        oc_dx = c_dx + ml + mr
        oc_dy = c_dy + mt + mb
        if (width is None) and (height is None):
            if ar > 0.0:
                ec_dx = max( oc_dx, int( ar * oc_dy ) ) - oc_dx
                ec_dy = max( oc_dy, int( round( oc_dx / ar ) ) ) - oc_dy
        else:
            # Compute any 'extra' space left over:
            width   = width  or tc_dx
            height  = height or tc_dy
            ec_dx   = max( 0, width  - tc_dx )
            ec_dy   = max( 0, height - tc_dy )
        
        # Apportion the extra space to the elements of the component:
        tc_dx  += ec_dx
        oc_dx  += ec_dx
        ott_dx += ec_dx
        otb_dx += ec_dx
        itt_dx += ec_dx / 2
        itb_dx += ec_dx / 2
        h_dx   += ec_dx / 2
        
        tc_dy  += ec_dy
        oc_dy  += ec_dy
        otl_dy += ec_dy
        otr_dy += ec_dy    
        itl_dy += ec_dy / 2
        itr_dy += ec_dy / 2
        h_dy   += ec_dy / 2
        
        # Set the computed size and position of each of the tracks:
        self._set_track_position( OUTER_LEFT, ml_dx - otl_cx, 
                                  mb_dy + pb_dy + b_dy, otl_dx, otl_dy )
        self._set_track_position( OUTER_RIGHT, tc_dx - mr_dx - otr_cx,
                                  mb_dy + pb_dy + b_dy, otr_dx, otr_dy )
        self._set_track_position( OUTER_TOP, ml_dx + pl_dx, 
                                  tc_dy - mt_dy - t_dy - ott_cy, 
                                  ott_dx, ott_dy )
        self._set_track_position( OUTER_BOTTOM, ml_dx + pl_dx, 
                                  mb_dy - otb_cy + b_dy, otb_dx, otb_dy )
        self._set_track_position( INNER_LEFT, h_dx - (itt_dx / 2) - itl_dx, 
                                  h_dy - (itl_dy / 2), itl_dx, itl_dy )
        self._set_track_position( INNER_RIGHT, h_dx + (itt_dx / 2), 
                                  h_dy - (itr_dy / 2), itr_dx, itr_dy )
        self._set_track_position( INNER_TOP, h_dx - (itt_dx / 2), 
                                  h_dy + (itl_dy / 2), itt_dx, itt_dy )
        self._set_track_position( INNER_BOTTOM, h_dx - (itb_dx / 2), 
                                  h_dy - (itl_dy / 2) - itb_dy, itb_dx, itb_dy )
                                  
        # Set the position of the inner component:
        if ic is not None:
            if icp == CENTER:
                ic.origin = ( int( h_dx - (ic_dx / 2) ), 
                              int( h_dy - (ic_dy / 2) ) )
                ic.size   = ( int( ic_dx ), int( ic_dy ) )
            else:
                ic.size = ( int( oc_dx - pl - pr - ml - mr ), int( ic_dy ) )
                if icp == TOP:
                    ic.origin = ( int( ml_dx + pl ),
                                  int( mb_dy - mb + oc_dy - mt_dy - ic_dy ) )
                else:
                    ic.origin = ( int( ml_dx + pl ), int( mb_dy - mb ) )
        
        # Set the size and position of the outer component:
        oc.origin = ( int( ml_dx - ml ), int( mb_dy - mb ) )
        oc.size   = ( int( oc_dx ), int( oc_dy ) )
        
        # Set the size of the main component:
        new_dx, new_dy = int( tc_dx ), int( tc_dy )
        component.size = ( new_dx, new_dy )
        if old_dx != 0:
            old_x, old_y     = component.origin
            component.origin = ( max( 0, old_x + ((old_dx - new_dx) / 2) ),
                                 max( 0, old_y + ((old_dy - new_dy) / 2) ) )
                       
    #---------------------------------------------------------------------------
    #  Return the components that contain a specified (x,y) point:
    #---------------------------------------------------------------------------
       
    def components_at ( self, x, y ):
        result = []
        for track in self.tracks:
            if track is not None:
                result.extend( track.components_at( x, y ) )
        return result
        
#-- Optional Interface: --------------------------------------------------------    

    #---------------------------------------------------------------------------
    #  Adds one or more Tracks to the associated Component:  
    #---------------------------------------------------------------------------

    def add_tracks ( self, *tracks ):
        """ Adds one or more Tracks to the associated Component.
        """
        for track in tracks:
            if not isinstance( track, OMTrack ):
                raise ValueError, 'Each argument must be an OMTrack object'
            old_track = self.tracks[ track.position_ ]
            if old_track is not track:
                if old_track is not None:
                    old_track.remove_contacts()
                self.tracks[ track.position_ ] = track
                
            # Track is managing contacts for the same component we are:
            track.component = self.component
        
    #---------------------------------------------------------------------------
    #  Removes one or more Tracks from the associated Component:  
    #---------------------------------------------------------------------------
                
    def remove_tracks ( self, *tracks ):
        """ Removes one or more Tracks from the associated Component.
        """
        the_tracks = self.tracks
        
        if len( tracks ) is None:
            tracks = the_tracks
            
        for track in tracks:
            if track is not None:
                track.remove_contacts()
                the_tracks[ track.position_ ] = None
        
    #---------------------------------------------------------------------------
    #  Gets the list of Tracks associated with a Component that match an 
    #  optionally specified name and/or category. If no arguments are 
    #  specified, it returns all Tracks associated with the Component; 
    #  otherwise it returns all Tracks whose name or category match the
    #  specified value.
    #---------------------------------------------------------------------------
                
    def get_tracks ( self, name = None, category = None ):
        """ Gets the list of Tracks associated with a Component that match an 
            optionally specified name and/or category. If no arguments are 
            specified, it returns all Tracks associated with the Component; 
            otherwise it returns all Tracks whose name or category match the
            specified value.
        """
        return [ track for track in self.tracks
                 if ((track is not None) and
                     ((name is None) or (name == track.name)) and
                     ((category is None) or (category == track.category))) ]

#-- Event Handlers -------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Handles the 'component' trait being changed:  
    #---------------------------------------------------------------------------
    
    def _component_changed ( self ):
        """ Handles the 'component' trait being changed.
        """
        self.component.contact_manager = self

#-- Public Methods -------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Draws all of the tracks:  
    #---------------------------------------------------------------------------
    
    def draw ( self, gc ):
        """ Draws all of the tracks.
        """
        for track in self.tracks:
            if track is not None:
                track.draw( gc )

#-- Private Methods ------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Gets the minimum size of a specified track:  
    #---------------------------------------------------------------------------
            
    def _get_min_track_size ( self, track_index ):
        track = self.tracks[ track_index ]
        if track is None:
            return ( 0, 0 )
        return track.min_size
        
    #---------------------------------------------------------------------------
    #  Gets the center point of a specified track:  
    #---------------------------------------------------------------------------
            
    def _get_track_center ( self, track_index ):
        track = self.tracks[ track_index ]
        if track is None:
            return ( 0, 0 )
        return track.center
        
    #---------------------------------------------------------------------------
    #  Sets the size and position of the specified track:  
    #---------------------------------------------------------------------------
                
    def _set_track_position ( self, track_index, x, y, dx, dy ):
        track = self.tracks[ track_index ]
        if track is not None:
            track.origin = ( int( x  ), int( y  ) )
            track.size   = ( int( dx ), int( dy ) )
        
