#-------------------------------------------------------------------------------
#
#  Define container classes that implement component layout.
#
#  Written by: David C. Morrill
#
#  Date: 11/02/2003
#
#  (c) Copyright 2003 by Enthought, Inc.
#
#  Classes defined: Layout
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

from base                import LEFT, RIGHT, TOP, BOTTOM
from component           import Component
from container           import FilledContainer

from enable_traits       import layout_style_trait, layout_mode_trait, \
                                spacing_trait, padding_trait, \
                                clear_color_trait, left_position_trait
from enthought.traits.ui.api import Group, View, Include

#-------------------------------------------------------------------------------
#  'Layout' class:
#-------------------------------------------------------------------------------

class Layout ( FilledContainer ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    bg_color       = clear_color_trait
    style          = layout_style_trait
    mode           = layout_mode_trait
    alignment      = left_position_trait
    spacing_height = spacing_trait
    spacing_width  = spacing_trait
    padding_left   = padding_trait
    padding_right  = padding_trait
    padding_top    = padding_trait
    padding_bottom = padding_trait

    #---------------------------------------------------------------------------
    #  Trait view definitions:
    #---------------------------------------------------------------------------
    
    traits_view = View( Group( '<component>', id = 'component' ),
                        Group( '<links>',     id = 'links' ),
                        Group( '<container>', id = 'container' ),
                        Group( 'style', '_', 'mode', '_', 'alignment',
                               id    = 'layout', 
                               style = 'custom' ),
                        Group( 'spacing_height', 'spacing_width', '_',
                               'padding_left', 'padding_right', 
                               'padding_top', 'padding_bottom',
                               id = 'padding' ) )
    
    #---------------------------------------------------------------------------
    #  Initialize the object: 
    #---------------------------------------------------------------------------
    
    def __init__ ( self, *components, **traits ):
        self._needs_layout = False
        self.components    = []   ### TEMPORARY ###
        self.on_trait_change( self._components_changed, 'components_items' ) 
        FilledContainer.__init__( self, *components, **traits )
        self.mode = 'immediate'
            
    #---------------------------------------------------------------------------
    #  Handle changes that should cause a new layout to occur:
    #---------------------------------------------------------------------------
    
    def _components_changed ( self ):
        self._needs_layout = True
        if self.mode[0] == 'i':
            self.layout()
     
    def _style_changed ( self ):
        self._components_changed()
     
    def _alignment_changed ( self ):
        self._components_changed()
        
    def _spacing_height_changed ( self ):    
        self._components_changed()
        
    def _spacing_width_changed ( self ):    
        self._components_changed()
        
    def _padding_left_changed ( self ):    
        self._components_changed()
        
    def _padding_right_changed ( self ):    
        self._components_changed()
        
    def _padding_top_changed ( self ):    
        self._components_changed()
        
    def _padding_bottom_changed ( self ):    
        self._components_changed()
        
    def _border_size_changed ( self ):
        self._components_changed()
        
    #---------------------------------------------------------------------------
    #  Handle the layout update mode being changed: 
    #---------------------------------------------------------------------------
    
    def _mode_changed ( self ):
        if self.mode[0] == 'i':
            self.layout()
            
    #----------------------------------------------------------------------------
    #  Handle the container changing its location or size: 
    #----------------------------------------------------------------------------
    
    def _bounds_changed ( self, old, new ):
        Component._bounds_changed( self, old, new )
        self._components_changed()
                
    #---------------------------------------------------------------------------
    #  Handle a component that has been added to the container: 
    #---------------------------------------------------------------------------

    def _add ( self, component ):
        component.on_trait_change( self._components_changed, 'min_width'  )
        component.on_trait_change( self._components_changed, 'min_height' )
                
    #---------------------------------------------------------------------------
    #  Remove one or more components from the container:
    #---------------------------------------------------------------------------
    
    def _remove ( self, component ):
        component.on_trait_change( self._components_changed, 'min_width', 
                                   remove = True )
        component.on_trait_change( self._components_changed, 'min_height', 
                                   remove = True )
    
    #---------------------------------------------------------------------------
    #  Layout all of the contained components: 
    #---------------------------------------------------------------------------
    
    def layout ( self ):
        if self._needs_layout:
            self._needs_layout = False
            components         = self.components
            x, y, dx, dy       = self.bounds
            pl                 = self.padding_left
            pr                 = self.padding_right
            pt                 = self.padding_top
            pb                 = self.padding_bottom
            sw                 = self.spacing_width
            sh                 = self.spacing_height
            bs2                = 2.0 * self.border_size
            align              = self.alignment_
            if self.style[0] == 'v':
               self.min_width = mdx = (reduce( 
                    lambda a, b: max( a, b.min_width ), components, 0 ) +
                    pl + pr + bs2)
               self.min_height = mdy = (reduce( 
                    lambda a, b: a + b.min_height, components, 0 ) + 
                    pt + pb + ((max( 0, len( components ) - 1 )) * sh) + bs2)
               dx = max( dx, mdx )
               dy = max( dy, mdy )
               self.dimensions( dx, dy )
               xdy = dy - mdy
               tsy = reduce( lambda a, b: a + b.stretch_height, components, 0 ) 
               x  += pl
               y  += dy - pt
               dx -= (pl + pr)
               for component in components:
                   cdy = component.min_height
                   if tsy > 0.0:
                      csh  = component.stretch_height
                      ady  = min( round( (xdy * csh) / tsy ), 
                                  component.max_height - cdy )
                      cdy += ady
                      xdy -= ady
                      tsy -= csh
                   cmw = component.min_width
                   cdx = min( cmw + round( component.stretch_width * (dx-cmw) ),
                              component.max_width )
                   if (align & LEFT):
                       cx = x
                   elif (align & RIGHT):
                       cx = x + dx - cdx
                   else:
                       cx = x + round( ((dx - cdx) / 2.0) )
                   y -= cdy
                   component.bounds = ( cx, y, cdx, cdy )
                   y -= sh
            else:
               self.min_width = mdx = (reduce( 
                    lambda a, b: a + b.min_width, components, 0 ) +
                    pl + pr + ((max( 0, len( components ) - 1 )) * sw) + bs2)
               self.min_height = mdy = (reduce( 
                    lambda a, b: max( a, b.min_height ), components, 0 ) +
                    pt + pb + bs2)
               dx = max( dx, mdx )
               dy = max( dy, mdy )
               self.dimensions( dx, dy )
               xdx = dx - mdx
               tsx = reduce( lambda a, b: a + b.stretch_width, components, 0 ) 
               x  += pl
               y  += pb
               dy -= (pt + pb)
               for component in components:
                   cdx = component.min_width
                   if tsx > 0.0:
                      csw  = component.stretch_width
                      adx  = min( round( (xdx * csw) / tsx ),
                                  component.max_width - cdx )
                      cdx += adx
                      xdx -= adx
                      tsx -= csw
                   cmh = component.min_height
                   cdy = min( cmh + round( component.stretch_height * (dy-cmh)),
                              component.max_height )
                   if (align & BOTTOM):
                       cy = y
                   elif (align & TOP):
                       cy = y + dy - cdy
                   else:
                       cy = y + round( (dy - cdy) / 2.0 )
                   component.bounds = ( x, cy, cdx, cdy )
                   x += cdx + sw
            self.redraw()

