#------------------------------------------------------------------------------
# Copyright (c) 2005-2007 by Enthought, Inc.
# All rights reserved.
#
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
#
#------------------------------------------------------------------------------

""" Defines the font editor factory for Kiva fonts, for the wxPython user
interface toolkit.
"""

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

import wx

from enthought.kiva.constants import NORMAL, BOLD, ITALIC, DEFAULT, ROMAN, \
    DECORATIVE, SCRIPT, SWISS, MODERN
from enthought.kiva.fonttools import Font
from enthought.traits.trait_base import SequenceTypes
from enthought.traits.ui.wx.font_editor import ToolkitEditorFactory \
    as EditorFactory
from enthought.kiva.fonttools.font_manager import fontManager

#-------------------------------------------------------------------------------
#  'ToolkitEditorFactory' class:
#-------------------------------------------------------------------------------

class ToolkitEditorFactory ( EditorFactory ):
    """ wxPython editor factory for Kiva fonts.
    """
    #---------------------------------------------------------------------------
    #  Returns a Font's 'face name':
    #---------------------------------------------------------------------------

    def face_name ( self, font ):
        """ Returns a Font's typeface name.
        """
        face_name = font.face_name
        if type( face_name ) in SequenceTypes:
            face_name = face_name[0]
        return face_name

    #---------------------------------------------------------------------------
    #  Returns a wxFont object corresponding to a specified object's font trait:
    #---------------------------------------------------------------------------

    def to_wx_font ( self, editor ):
        """ Returns a wxFont object corresponding to a specified object's font
            trait.
        """
        font   = editor.value
        weight = ( wx.NORMAL, wx.BOLD   )[ font.weight == BOLD ]
        style  = ( wx.NORMAL, wx.ITALIC )[ font.style  == ITALIC ]
        family = { DEFAULT:    wx.DEFAULT,
                   DECORATIVE: wx.DECORATIVE,
                   ROMAN:      wx.ROMAN,
                   SCRIPT:     wx.SCRIPT,
                   SWISS:      wx.SWISS,
                   MODERN:     wx.MODERN }.get( font.family, wx.SWISS )
        return wx.Font( font.size, family, style, weight,
                        (font.underline != 0), self.face_name( font ) )

    #---------------------------------------------------------------------------
    #  Gets the application equivalent of a wxPython value:
    #---------------------------------------------------------------------------

    def from_wx_font ( self, font ):
        """ Gets the application equivalent of a wxPython value.
        """
        return Font( size = font.GetPointSize(),
                     family = { wx.DEFAULT:    DEFAULT,
                                wx.DECORATIVE: DECORATIVE,
                                wx.ROMAN:      ROMAN,
                                wx.SCRIPT:     SCRIPT,
                                wx.SWISS:      SWISS,
                                wx.MODERN:     MODERN }.get(font.GetFamily(), SWISS),
                    weight = ( NORMAL, BOLD   )[ font.GetWeight() == wx.BOLD ],
                    style = ( NORMAL, ITALIC )[ font.GetStyle()  == wx.ITALIC ],
                    underline = font.GetUnderlined() - 0, #convert Bool to an int type
                    face_name = font.GetFaceName() )

    #---------------------------------------------------------------------------
    #  Returns the text representation of the specified object trait value:
    #---------------------------------------------------------------------------

    def str_font ( self, font ):
        """ Returns the text representation of the specified object trait value.
        """
        weight    = { BOLD:   ' Bold'   }.get( font.weight, '' )
        style     = { ITALIC: ' Italic' }.get( font.style,  '' )
        underline = [ '', ' Underline' ][ font.underline != 0 ]
        return '%s point %s%s%s%s' % (
               font.size, self.face_name( font ), style, weight, underline )

    #---------------------------------------------------------------------------
    #  Returns a list of all available font facenames:
    #---------------------------------------------------------------------------

    def all_facenames ( self ):
        """ Returns a list of all available font typeface names.
        """
        facenames = fontManager.ttfdict.keys()
        return facenames


def KivaFontEditor ( *args, **traits ):
    return ToolkitEditorFactory( *args, **traits )

