
#include <assert.h>
#include "agg_path_storage.h"
#include "kiva_graphics_context_base.h"


using namespace kiva;

graphics_context_base::graphics_context_base(unsigned char *data, int width,
                       int height, int stride, interpolation_e interp):
                       buf(),
                       _image_interpolation(interp),
#ifdef KIVA_USE_FREETYPE
                       font_engine(),
#endif
#ifdef KIVA_USE_WIN32
                       font_engine(hdc),
#endif
                       font_manager(font_engine),
                       _is_font_initialized(false)
{
   this->buf.attach(data, width, height, stride);
}

graphics_context_base::~graphics_context_base()
{
}


int graphics_context_base::width()
{
    return this->buf.width();
}


int graphics_context_base::height()  
{ 
    return this->buf.height(); 
}


int graphics_context_base::stride()  
{ 
    return this->buf.stride(); 
}


int graphics_context_base::bottom_up() 
{ 
    return (this->stride() > 0 ? 0 : 1); 
}


agg::rendering_buffer* graphics_context_base::rendering_buffer_ptr() 
{ 
    return &(this->buf); 
}



kiva::interpolation_e graphics_context_base::get_image_interpolation() 
{ 
    return this->_image_interpolation; 
}


void graphics_context_base::set_image_interpolation(kiva::interpolation_e interpolation) 
{ 
    this->_image_interpolation = interpolation; 
}


//---------------------------------------------------------------
// set graphics_state values
//---------------------------------------------------------------


void graphics_context_base::set_stroke_color(agg::rgba& value) 
{ 
    this->state.line_color = value; 
}


agg::rgba& graphics_context_base::get_stroke_color() 
{ 
    return this->state.line_color; 
}


void graphics_context_base::set_line_width(double value) 
{ 
    this->state.line_width = value; 
}


void graphics_context_base::set_line_join(kiva::line_join_e value) 
{ 
    this->state.line_join = value; 
}


void graphics_context_base::set_line_cap(kiva::line_cap_e value) 
{ 
    this->state.line_cap = value; 
}


void graphics_context_base::set_line_dash(double* pattern, int n, double phase)
{
    this->state.line_dash = kiva::dash_type(phase, pattern, n);
}

kiva::blend_mode_e graphics_context_base::get_blend_mode()
{
    return this->state.blend_mode;
}

void graphics_context_base::set_blend_mode(kiva::blend_mode_e value) 
{ 
    this->state.blend_mode = value; 
}

void graphics_context_base::set_fill_color(agg::rgba& value)
{ 
    this->state.fill_color = value; 
}

agg::rgba& graphics_context_base::get_fill_color() 
{
    return this->state.fill_color; 
}

void graphics_context_base::set_alpha(double value)
{
    // alpha should be between 0 and 1, so clamp:
    if (value < 0.0)
    {
        value = 0.0;
    }
    else if (value > 1.0)
    {
        value = 1.0;
    }
    this->state.alpha = value;
}


double graphics_context_base::get_alpha() 
{ 
    return this->state.alpha; 
}

void graphics_context_base::set_antialias(int value) 
{ 
    this->state.should_antialias = value; 
}


int graphics_context_base::get_antialias() 
{ 
    return this->state.should_antialias; 
}


void graphics_context_base::set_miter_limit(double value) 
{ 
    this->state.miter_limit = value; 
}


void graphics_context_base::set_flatness(double value) 
{ 
    this->state.flatness = value; 
}


//---------------------------------------------------------------
// text and font functions
//---------------------------------------------------------------


void graphics_context_base::set_text_position(double tx, double ty)
{
    double temp[6];
    this->text_matrix.store_to(temp);
    temp[4] = tx;
    temp[5] = ty;
    this->text_matrix.load_from(temp);
}


void graphics_context_base::get_text_position(double* tx, double* ty)
{
    double temp[6];
    agg::trans_affine result = this->get_text_matrix();
    result.store_to(temp);            
    *tx = temp[4];
    *ty = temp[5];
}

bool graphics_context_base::is_font_initialized()
{
    return _is_font_initialized;
}

void graphics_context_base::set_text_matrix(agg::trans_affine& value) 
{ 
    this->text_matrix = value; 
}


agg::trans_affine graphics_context_base::get_text_matrix() 
{ 
    return this->text_matrix; 
}


void graphics_context_base::set_character_spacing(double value) 
{ 
    this->state.character_spacing = value; 
}


double graphics_context_base::get_character_spacing() 
{ 
    return this->state.character_spacing; 
}


void graphics_context_base::set_text_drawing_mode(kiva::text_draw_mode_e value)
{
    this->state.text_drawing_mode = value;
}




//---------------------------------------------------------------
// save/restore graphics state
//---------------------------------------------------------------


void graphics_context_base::save_state()
{
    this->state_stack.push(this->state);
    this->path.save_ctm();
}

//---------------------------------------------------------------
// coordinate transform matrix transforms
//---------------------------------------------------------------


void graphics_context_base::translate_ctm(double x, double y)
{
    this->path.translate_ctm(x, y);
}


void graphics_context_base::rotate_ctm(double angle)
{
    this->path.rotate_ctm(angle);
}


void graphics_context_base::scale_ctm(double sx, double sy)
{
    this->path.scale_ctm(sx, sy);
}


void graphics_context_base::concat_ctm(agg::trans_affine& m)
{
    this->path.concat_ctm(m);
}


void graphics_context_base::set_ctm(agg::trans_affine& m)
{
    this->path.set_ctm(m);
}


agg::trans_affine graphics_context_base::get_ctm()
{
    return this->path.get_ctm();
}


void graphics_context_base::get_freetype_text_matrix(double* out)
{            
    agg::trans_affine result =  this->get_ctm();
    result.multiply(this->get_text_matrix());
    result.store_to(out);            
    // freetype and agg transpose their matrix conventions
    double temp = out[1];
    out[1] = out[2];
    out[2] = temp;
}
//---------------------------------------------------------------
// Sending drawing data to a device
//---------------------------------------------------------------

void graphics_context_base::flush()
{
    // TODO-PZW: clarify this and other "not sure if anything is needed" functions
    // not sure if anything is needed.
}


void graphics_context_base::synchronize()
{
    // not sure if anything is needed.
}

//---------------------------------------------------------------
// Page Definitions
//---------------------------------------------------------------


void graphics_context_base::begin_page()
{
    // not sure if anything is needed.
}


void graphics_context_base::end_page()
{
    // not sure if anything is needed.
}

//---------------------------------------------------------------
// Path operations
//---------------------------------------------------------------


void graphics_context_base::begin_path()
{
    this->path.begin_path();
}


void graphics_context_base::move_to(double x, double y)
{
    this->path.move_to(x, y);
}


void graphics_context_base::line_to( double x, double y)
{
    this->path.line_to(x, y);
}


void graphics_context_base::curve_to(double cpx1, double cpy1,
              double cpx2, double cpy2,
              double x, double y)
{
    this->path.curve_to(cpx1, cpy1, cpx2, cpy2, x, y);
}


void graphics_context_base::quad_curve_to(double cpx, double cpy,
                   double x, double y)
{
    this->path.quad_curve_to(cpx, cpy, x, y);
}

void graphics_context_base::arc(double x, double y, double radius,
                                double start_angle, double end_angle,
                                bool cw)
{
    this->path.arc(x, y, radius, start_angle, end_angle, cw);
}

void graphics_context_base::arc_to(double x1, double y1, double x2, double y2,
                                   double radius)
{
    this->path.arc_to(x1, y1, x2, y2, radius);
}

void graphics_context_base::close_path()
{
    this->path.close_polygon();
}


void graphics_context_base::add_path(kiva::compiled_path& other_path)
{
    this->path.add_path(other_path);
}


void graphics_context_base::lines(double* pts, int Npts)
{
    this->path.lines(pts, Npts);
}

void graphics_context_base::line_set(double* start, int Nstart, double* end, int Nend)
{
    this->path.line_set(start, Nstart, end, Nend);
}

void graphics_context_base::rect(double x, double y, double sx, double sy)
{
    this->path.rect(x, y, sx, sy);
}

void graphics_context_base::rect(kiva::rect_type &rect)
{
    this->path.rect(rect);
}


void graphics_context_base::rects(double* all_rects, int Nrects)
{
    this->path.rects(all_rects,Nrects);
}

void graphics_context_base::rects(kiva::rect_list_type &rectlist)
{
    this->path.rects(rectlist);
}

kiva::compiled_path graphics_context_base::_get_path()
{
    return this->path;
}


agg::path_storage graphics_context_base::boundary_path(agg::trans_affine& affine_mtx)
{
    // Return the path that outlines the image in device space
    // This is used in _draw to specify the device area
    // that should be rendered.
    agg::path_storage clip_path;
    double p0x = 0;
    double p0y = 0;
    double p1x = this->width();
    double p1y = 0;
    double p2x = this->width();
    double p2y = this->height();
    double p3x = 0;
    double p3y = this->height(); 
            
    affine_mtx.transform(&p0x, &p0y);
    affine_mtx.transform(&p1x, &p1y);
    affine_mtx.transform(&p2x, &p2y);
    affine_mtx.transform(&p3x, &p3y);
    //printf("clip path: %g, %g, %g, %g\n",lx,ly,hx,hy);
    
    clip_path.move_to(p0x, p0y);
    clip_path.line_to(p1x, p1y);
    clip_path.line_to(p2x, p2y);
    clip_path.line_to(p3x, p3y);
    clip_path.close_polygon();
    return clip_path;
}

/////////////////////////////////////////////////////////////////////////////
// Text methods
/////////////////////////////////////////////////////////////////////////////

bool graphics_context_base::set_font(kiva::font_type& font) 
{ 
    bool success = false;
    
    // See if the font is the same; if it is, then do nothing:
    if (font == this->state.font)
    {
        return true;
    }
    
    this->state.font = font;
    this->_is_font_initialized = false;

    // short-circuit: if the font didn't even load properly, then this call can't succeed.
    if (!this->state.font.is_loaded())
    {
        return false;
    }
    
#ifdef KIVA_USE_FREETYPE
    if (this->state.font.filename != "")
    {
        success = this->font_engine.load_font(this->state.font.filename.c_str(), 0, 
                                agg::glyph_ren_agg_gray8);
    }
    else
    {
        success = this->font_engine.load_font(this->state.font.name.c_str(), 0, 
                                    agg::glyph_ren_agg_gray8);
    }
#endif

#ifdef KIVA_USE_WIN32
    success = this->font_engine.create_font(this->state.font.name,
                                  agg::glyph_ren_native_gray8,
                                  this->state.font.size);
#endif

    if (success)
    {
        this->font_engine.hinting(1);
        this->font_engine.resolution(72);
        this->set_font_size(font.size);
        this->_is_font_initialized = true;
    }
    
    return success;
}


kiva::font_type& graphics_context_base::get_font() 
{ 
    return this->state.font; 
}


bool graphics_context_base::set_font_size(int size) 
{ 
    // just make sure the font is loaded; don't check is_font_initialized
    if (!this->state.font.is_loaded())
    {
        return false;
    }

    this->state.font.size = size;

//    The following is a more laborious but more "correct" way of determining
//    the correct font size to use under Win32.  Unfortunately, it doesn't
//    work exactly right either; characters come out just a few pixels larger.
//    Thus, for the time being, we're going to punt and leave the hard-coded
//    adjustment factor.
//
//    this->font_engine.height(12.0);
//    this->font_engine.width(12.0);
//    kiva::rect_type tmp(this->get_text_extent("X"));
//    this->font_engine.height(font.size*12.0/tmp.h);
//    this->font_engine.width(font.size*12.0/tmp.w);

    this->font_engine.height(size);
    this->font_engine.width(size);
    return true;
}


bool graphics_context_base::show_text_at_point(char *text, 
                                               double tx, double ty)
{
    double oldx, oldy;
    
    if (!is_font_initialized())
    {
        return false;
    }
    
    this->get_text_position(&oldx, &oldy);
    this->set_text_position(tx, ty);
    bool retval = this->show_text(text);
    this->set_text_position(oldx, oldy);
    return retval;
}

kiva::rect_type graphics_context_base::get_text_extent(char *text)
{
    const agg::glyph_cache *glyph = NULL;
    char *p = text;
    double x1 = 0.0, x2 = 0.0, y1 = 0.0, y2= 0.0;

    //typedef agg::glyph_raster_bin<agg::rgba8> GlyphGeneratorType;
    //GlyphGeneratorType glyphGen(this->font_manager.glyph(*p)->data);
    
    while (*p)
    {
        glyph = this->font_manager.glyph(*p);
        if (glyph == NULL)
        {
            p++;
            continue;
        }
        this->font_manager.add_kerning(&x2, &y2);
        x1 = kiva::min(x1, glyph->bounds.x1);
        x2 += glyph->advance_x;
        y1 = kiva::min(y1, glyph->bounds.y1);
        y2 = kiva::max(y2, glyph->bounds.y2);
        p++;
    }
    
    return kiva::rect_type(x1, y1, x2-x1, y2 - y1);
}


bool graphics_context_base::get_text_bbox_as_rect(char *text)
{
    return false;
}

int graphics_context_base::draw_image(kiva::graphics_context_base* img)
{
    double tmp[] = {0, 0, img->width(), img->height()};
    return this->draw_image(img, tmp);
}
