#-------------------------------------------------------------------------------
#
#  Define the base Enable container class for Enable components that have
#  'container' functionality.
#
#  Written by: David C. Morrill
#
#  Date: 09/22/2003
#
#  (c) Copyright 2003 by Enthought, Inc.
#
#  Classes defined: AbstractContainer
#                   Container
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

from enthought.traits.api    import Trait, List
from enthought.traits.ui.api import View, Group, Include

from base                import bounds_to_coordinates, coordinates_to_bounds, \
                                transparent_color, filled_rectangle
from base_container      import BaseContainer, default_container
from component           import Component
from enable_traits       import white_color_trait, black_color_trait, \
                                border_size_trait

#-------------------------------------------------------------------------------
#  'AbstractContainer' class:
#-------------------------------------------------------------------------------

class AbstractContainer ( BaseContainer ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    components = List( Component )
    
    #---------------------------------------------------------------------------
    #  Trait view definitions:
    #---------------------------------------------------------------------------
    
    abstract_container_view = View( Group( '<clinks>', 'components', 
                                           id = 'clinks' ) )
    
    #---------------------------------------------------------------------------
    #  Initialize the object:
    #---------------------------------------------------------------------------
    
    def __init__ ( self, *components, **traits ):
        if len( self.components ) == 0:   ### TEMPORARY ###
            self.components = []          ### TEMPORARY ###
        self.add( *components )
        
    #---------------------------------------------------------------------------
    #  Add one or more components to the container:
    #---------------------------------------------------------------------------
    
    def add ( self, *components ):
        the_components = self.components
        for component in components:
            self.add_at( len( the_components ), component )
        
    #----------------------------------------------------------------------------
    #  Add a component at a specified index:
    #----------------------------------------------------------------------------
    
    def add_at ( self, index, component ):
        if index < 0:
            raise IndexError
        the_components = self.components
        container = component.container
        if container is self:
            try:
                the_components.remove( component )
            except:
                pass
        else:
            container.remove( component )
        the_components[ index: index ] = [ component ]
        component.container = self
        self._add( component )
                
    #---------------------------------------------------------------------------
    #  Handle a component that has been added to the container: 
    #---------------------------------------------------------------------------
    
    def _add ( self, component ):
        self._check_bounds( component )
                
    #---------------------------------------------------------------------------
    #  Remove one or more components from the container:
    #---------------------------------------------------------------------------
    
    def remove ( self, *components ):
        the_components = self.components
        for component in components:
            try:
                the_components.remove( component )
                self._remove( component )
                
                # If the mouse owner is either the component, or one of its
                # children, then release its ownership:
                owner = self.window.mouse_owner
                if owner is not None:
                    while True:
                        if owner is component:
                            self.window.mouse_owner = None
                            break
                        if not hasattr( owner, 'container' ):
                            break
                        owner = owner.container
                            
                component.container = default_container
            except:
                pass
                
    #---------------------------------------------------------------------------
    #  Remove all components from the container:
    #---------------------------------------------------------------------------
    
    def remove_all ( self ):
        self.remove( *self.components[:] )
                
    #---------------------------------------------------------------------------
    #  Handle a component that has been removed from the container: 
    #---------------------------------------------------------------------------
    
    def _remove ( self, component ):
        if component.min_width == self.min_width:
            self.min_width = reduce( 
                lambda a, b: max( a, b.min_width ), the_components, 0 )
        if component.min_height == self.min_height:
            self.min_height = reduce( 
                lambda a, b: max( a, b.min_height ), the_components, 0 )
            
    #---------------------------------------------------------------------------
    #  Check that the bounds of a specified component fits within the
    #  container:
    #---------------------------------------------------------------------------

    def _check_bounds ( self, component ):
        xl,  yb,  xr,  yt  = bounds_to_coordinates( self.bounds )
        cxl, cyb, cxr, cyt = bounds_to_coordinates( component.bounds )
        dx = xl - cxl
        if dx > 0.0:
            cxl += dx
            cxr += dx
        dx = cxr - xr
        if dx > 0.0:
            cxl -= dx
            cxr -= dx
        if cxl < xl:
            cxl = xl
        dy = yb - cyb
        if dy > 0.0:
            cyb += dy
            cyt += dy
        dy = cyt - yt
        if dy > 0.0:
            cyb -= dy
            cyt -= dy
        if cyb < yb:
            cyb = yb
        component.bounds = coordinates_to_bounds( ( cxl, cyb, cxr, cyt ) )
        self.min_width   = max( self.min_width,  component.min_width  )
        self.min_height  = max( self.min_height, component.min_height )

#-------------------------------------------------------------------------------
#  'Container' class:
#-------------------------------------------------------------------------------

class Container ( Component, AbstractContainer ):
    
    #---------------------------------------------------------------------------
    #  Trait view definitions:
    #---------------------------------------------------------------------------
    
    traits_view = View( Group( '<component>',         id = 'component' ),
                        Group( '<links>', '<clinks>', id = 'links' ) )
    
    #---------------------------------------------------------------------------
    #  Initialize the object:
    #---------------------------------------------------------------------------
    
    def __init__ ( self, *components, **traits ):
        Component.__init__( self, **traits )
        AbstractContainer.__init__( self, *components )
                       
    #---------------------------------------------------------------------------
    #  Return the components that contain a specified (x,y) point:
    #---------------------------------------------------------------------------
       
    def _components_at ( self, x, y ):
        result = [ self ]
        for component in self.components:
            result.extend( component.components_at( x, y ) )
        return result

    #---------------------------------------------------------------------------
    #  Draw the container and its contained components in a specified graphics 
    #  context:
    #---------------------------------------------------------------------------
    
    def _draw ( self, gc ):
        self._draw_container( gc )
        gc.save_state()
        gc.clip_to_rect(*self.bounds)
        for component in self.components:
            component.draw( gc )
        gc.restore_state()

    #---------------------------------------------------------------------------
    #  Draw the container background in a specified graphics context:
    #  (This method should normally be overridden by a subclass)
    #---------------------------------------------------------------------------
    
    def _draw_container ( self, gc ):
        pass
    
    #----------------------------------------------------------------------------
    #  Handle the container changing its location or size: 
    #----------------------------------------------------------------------------
    
    def _bounds_changed ( self, old, new ):
        Component._bounds_changed( self, old, new )
        ox, oy, odx, ody = old
        nx, ny, ndx, ndy = new
        cx = nx - ox
        cy = ny - oy
        if (cx != 0) or (cy != 0):
            for component in self.components:
                x, y = component.location()
                component.location( x + cx, y + cy )
        for component in self.components:
            self._check_bounds( component ) 
                
    #-------------------------------------------------------------------------------
    #  Handle being dropped on by a WindowFrame:
    #-------------------------------------------------------------------------------

    def dropped_on_by_windowframe ( self, frame, event ):
        event.handled = True
        frame.location( frame.x + event.x - event.x0, 
                        frame.y + event.y - event.y0 )
        try:
            components = self.components
            i = components.index( frame )
            if i < (len( components ) - 1):
                del components[i]
                components.append( frame )
        except:
            pass

#-------------------------------------------------------------------------------
#  'FilledContainer' class:
#-------------------------------------------------------------------------------

class FilledContainer ( Container ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    bg_color     = white_color_trait
    border_color = black_color_trait
    border_size  = border_size_trait

    #---------------------------------------------------------------------------
    #  Trait view definition:
    #---------------------------------------------------------------------------
    
    traits_view = View( Group( '<component>', id = 'component' ),
                        Group( '<links>',     id = 'links' ),
                        Group( 'bg_color{Background color}', '_',
                               'border_color', '_', 'border_size',
                               id    = 'container', 
                               style = 'custom' ) )
    
    colorchip_map = {
        'bg_color':  'bg_color',
        'alt_color': 'border_color'
    }

    #---------------------------------------------------------------------------
    #  Draws the container background in a specified graphics context:
    #---------------------------------------------------------------------------
    
    def _draw_container ( self, gc ):
        filled_rectangle( gc, self.bounds, self.bg_color_, self.border_color_,
                              self.border_size )
                
