#include <glib.h>
#include "../../include/string.h"
#include "../../include/disk.h"
#include "edv_context.h"
#include "edv_notify.h"
#include "config.h"


void EDVNotifyQueueObjectAdded(
	edv_context_struct *ctx, const gchar *path
);
void EDVNotifyQueueObjectModified(
	edv_context_struct *ctx, const gchar *path, const gchar *new_path
);
void EDVNotifyQueueObjectRemoved(
	edv_context_struct *ctx, const gchar *path
);

void EDVNotifyQueueObjectMounted(
	edv_context_struct *ctx, const gchar *path
);
void EDVNotifyQueueObjectUnmounted(
	edv_context_struct *ctx, const gchar *path
);

void EDVNotifyQueueRecycledObjectAdded(
	edv_context_struct *ctx, guint index
);
void EDVNotifyQueueRecycledObjectRemoved(
	edv_context_struct *ctx, guint index
);

void EDVNotifyQueueReconfiguredNotify(
	edv_context_struct *ctx
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Queues an "object_added_notify" command.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to the object that was
 *	added.
 */
void EDVNotifyQueueObjectAdded(
	edv_context_struct *ctx,
	const gchar *path
)
{
	gchar *cmd;
	if(ctx == NULL)
	    return;
	cmd = g_strdup_printf(
	    "object_added_notify \"%s\"",
	    path
	);
	EDVContextQueueCommand(ctx, cmd);
	g_free(cmd);
}

/*
 *	Queues an "object_modified_notify" command.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to the object that was
 *	modified (it may not exist in cases of renaming or moving).
 *
 *	The new_path specifies the full path to the new path of the
 *	modified object. If new_path is NULL then only path will be
 *	sent (hinting that the name of the object was not modified).
 */
void EDVNotifyQueueObjectModified(
	edv_context_struct *ctx,
	const gchar *path,
	const gchar *new_path
)
{
	gchar *cmd;
	if(ctx == NULL)
	    return;
	if(new_path != NULL)
	    cmd = g_strdup_printf(
		"object_modified_notify \"%s\" \"%s\"",
		path, new_path
	    );
	else
	    cmd = g_strdup_printf(
		"object_modified_notify \"%s\"",
		path
	    );
	EDVContextQueueCommand(ctx, cmd);
	g_free(cmd);
}

/*
 *	Queues an "object_removed_notify" command.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to the object that was
 *	removed.
 */
void EDVNotifyQueueObjectRemoved(
	edv_context_struct *ctx,
	const gchar *path
)
{
	gchar *cmd;
	if(ctx == NULL)
	    return;
	cmd = g_strdup_printf(
	    "object_removed_notify \"%s\"",
	    path
	);
	EDVContextQueueCommand(ctx, cmd);
	g_free(cmd);
}


/*
 *	Queues an "object_mounted_notify" command.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to the mount path.
 */
void EDVNotifyQueueObjectMounted(
	edv_context_struct *ctx,
	const gchar *path
)
{
	gchar *cmd;
	if(ctx == NULL)
	    return;
	cmd = g_strdup_printf(
	    "object_mounted_notify \"%s\"",
	    path
	);
	EDVContextQueueCommand(ctx, cmd);
	g_free(cmd);
}

/*
 *	Queues an "object_unmounted_notify" command.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to the mount path.
 */
void EDVNotifyQueueObjectUnmounted(
	edv_context_struct *ctx,
	const gchar *path
)
{
	gchar *cmd;
	if(ctx == NULL)
	    return;
	cmd = g_strdup_printf(
	    "object_unmounted_notify \"%s\"",
	    path
	);
	EDVContextQueueCommand(ctx, cmd);
	g_free(cmd);
}


/*
 *	Queues a "recycled_object_added_notify" command.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The index specifies the index of the recycled object.
 */
void EDVNotifyQueueRecycledObjectAdded(
	edv_context_struct *ctx,
	const guint index
)
{
	gchar *cmd;
	if(ctx == NULL)
	    return;
	cmd = g_strdup_printf(
	    "recycled_object_added_notify %i",
	    index
	);
	EDVContextQueueCommand(ctx, cmd);
	g_free(cmd);
}

/*
 *	Queues a "recycled_object_removed_notify" command.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The index specifies the index of the recycled object.
 */
void EDVNotifyQueueRecycledObjectRemoved(
	edv_context_struct *ctx,
	const guint index
)
{
	gchar *cmd;
	if(ctx == NULL)
	    return;
	cmd = g_strdup_printf(
	    "recycled_object_removed_notify %i",
	    index
	);
	EDVContextQueueCommand(ctx, cmd);
	g_free(cmd);
}


/*
 *	Queues a "reconfigured_notify" command.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
void EDVNotifyQueueReconfiguredNotify(edv_context_struct *ctx)
{
	if(ctx == NULL)
	    return;

	EDVContextQueueCommand(ctx, "reconfigured_notify");
}

