#include <sys/stat.h>
#include <glib.h>

#include "edv_types.h"
#include "edv_obj.h"
#include "edv_recycled_obj.h"
#include "edv_utils.h"


edv_recycled_object_struct *EDVRecycledObjectNew(void);
edv_recycled_object_struct *EDVRecycledObjectCopy(
	edv_recycled_object_struct *obj
);
void EDVRecycledObjectSetStat(
	edv_recycled_object_struct *obj,
	struct stat *lstat_buf
);
void EDVRecycledObjectSetToStat(
	struct stat *lstat_buf,
	edv_recycled_object_struct *obj
);
void EDVRecycledObjectClear(edv_recycled_object_struct *obj);
void EDVRecycledObjectDelete(edv_recycled_object_struct *obj);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new Recycled Object.
 */
edv_recycled_object_struct *EDVRecycledObjectNew(void)
{
	return(EDV_RECYCLED_OBJECT(g_malloc0(
	    sizeof(edv_recycled_object_struct)
	)));
}

/*
 *	Coppies the Recycled Object.
 */
edv_recycled_object_struct *EDVRecycledObjectCopy(
	edv_recycled_object_struct *obj
)
{
	edv_recycled_object_struct *new_obj;

	if(obj == NULL)
	    return(NULL);

	new_obj = EDVRecycledObjectNew();
	if(new_obj == NULL)
	    return(NULL);

	new_obj->type = obj->type;

	new_obj->index = obj->index;

	new_obj->name = STRDUP(obj->name);
	new_obj->original_path = STRDUP(obj->original_path);

	new_obj->deleted_time = obj->deleted_time;

	new_obj->size = obj->size;

	new_obj->link_target = STRDUP(obj->link_target);

	new_obj->permissions = obj->permissions;

	new_obj->access_time = obj->access_time;
	new_obj->modify_time = obj->modify_time;
	new_obj->change_time = obj->change_time;

	new_obj->owner_id = obj->owner_id;
	new_obj->group_id = obj->group_id;

	return(new_obj);
}

/*
 *	Sets/converts the struct stat information to the recycled
 *	object.
 *
 *	The obj specifies the target recycled object containing the
 *      information to set/convert to.
 *
 *	The lstat_buf specifies the source struct stat to set/convert
 *	the information from.
 *
 *	Note that the struct stat contains less information than the
 *	object so not all the information will be set/converted.
 */
void EDVRecycledObjectSetStat(
	edv_recycled_object_struct *obj,
	struct stat *lstat_buf
)
{
	mode_t m;

	if((lstat_buf == NULL) || (obj == NULL))
            return;

	m = lstat_buf->st_mode;

	obj->type = EDVStatModeToObjectType(m);

	obj->index = (guint)lstat_buf->st_ino;

	obj->size = (gulong)lstat_buf->st_size;

	obj->permissions = EDVStatModeToEDVPermissions(m);

        obj->access_time = (gulong)lstat_buf->st_atime;
        obj->modify_time = (gulong)lstat_buf->st_mtime;
        obj->change_time = (gulong)lstat_buf->st_ctime;

	obj->owner_id = (gint)lstat_buf->st_uid;
	obj->group_id = (gint)lstat_buf->st_gid;
}

/*
 *	Sets/converts the recycled object information to the struct
 *	stat.
 *
 *	The lstat_buf specifies the target struct stat to set/convert
 *	the information to.
 *
 *	The obj specifies the source recycled object containing the
 *	information to set/convert from.
 *
 *	Note that the struct stat contains less information than the
 *	object so not all the information will be set/converted.
 */
void EDVRecycledObjectSetToStat(
	struct stat *lstat_buf,
	edv_recycled_object_struct *obj
)
{
	if((obj == NULL) || (lstat_buf == NULL))
            return;

        lstat_buf->st_dev = (dev_t)0;
        lstat_buf->st_ino = (ino_t)obj->index;

        lstat_buf->st_mode = EDVObjectTypeToStatType(obj->type) |
	    EDVEDVPermissionsToStatMode(obj->permissions);

        lstat_buf->st_nlink = (nlink_t)0;

        lstat_buf->st_uid = (uid_t)obj->owner_id;
        lstat_buf->st_gid = (gid_t)obj->group_id;

        lstat_buf->st_rdev = (dev_t)0;

        lstat_buf->st_size = (off_t)obj->size;

#if !defined(_WIN32)
        lstat_buf->st_blksize = (unsigned long)0;
        lstat_buf->st_blocks = (unsigned long)0;
#endif

        lstat_buf->st_atime = (time_t)obj->access_time;
        lstat_buf->st_mtime = (time_t)obj->modify_time;
        lstat_buf->st_ctime = (time_t)obj->change_time;
}

/*
 *	Clears the Recycled Object.
 */
void EDVRecycledObjectClear(edv_recycled_object_struct *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->original_path);
	g_free(obj->link_target);

	memset(obj, 0x00, sizeof(edv_recycled_object_struct));
}

/*
 *	Deletes the Recycled Object.
 */
void EDVRecycledObjectDelete(edv_recycled_object_struct *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->original_path);

	g_free(obj->link_target);

	g_free(obj);
}
