#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>
#include <errno.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "pdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "edv_obj_create.h"
#include "endeavour2.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.                        
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.    
 *	-4	User responded with "Cancel".                     
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.    
 */


/* Error Message */  
const gchar *EDVObjCreateGetError(edv_core_struct *core);
static void EDVObjCreateCopyErrorMessage(
	edv_core_struct *core, const gchar *msg
);

/* New Path */
static gchar *EDVObjCreateGenerateNewPath(
	const gchar *path, const edv_object_type type
);

/* Create New File Query Name */
gint EDVObjCreateFileQueryName(
	edv_core_struct *core,
	const gchar *parent_path,
        const gchar *suggested_name,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	gboolean *yes_to_all
);

/* Create New Object Nexus */
static gint EDVObjCreateNexus(
	edv_core_struct *core,
	const gchar *path,
	const edv_object_type type,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);

/* Create New File */
gint EDVObjCreateFile(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New Directory */
gint EDVObjCreateDirectory(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New Link */
gint EDVObjCreateLink(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New FIFO Pipe */
gint EDVObjCreateFifo(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New Block Device */
gint EDVObjCreateDeviceBlock(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New Character Device */
gint EDVObjCreateDeviceCharacter(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);
/* Create New Socket */
gint EDVObjCreateSocket(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *      Returns the last error message or NULL if there was no error.
 *
 *      The returned pointer must not be deallocated.
 */
const gchar *EDVObjCreateGetError(edv_core_struct *core)
{
	return((core != NULL) ? core->obj_last_error : NULL);
}

/*
 *	Coppies the error message to the core's obj_last_error_buf
 *	and sets the core's obj_last_error to point to it.
 */
static void EDVObjCreateCopyErrorMessage(
	edv_core_struct *core, const gchar *msg
)
{
	if(core == NULL)
	    return;

	core->obj_last_error = NULL;

	g_free(core->obj_last_error_buf);
	core->obj_last_error_buf = STRDUP(msg);

	core->obj_last_error = core->obj_last_error_buf;
}


/*
 *	Generates a path to a new object.
 *
 *	The path specifies the full path to the directory at where
 *	the new object is to be created.
 *
 *	The type specifies the type of object to be created. The type
 *	will be used as part of the generated path's name.
 *
 *	Returns a dynamically allocated string describing the full
 *	path to the new object.
 */
static gchar *EDVObjCreateGenerateNewPath(
	const gchar *path, const edv_object_type type
)
{
	struct stat lstat_buf;
	guint16 i;
	gchar *type_str, *new_path;

	if(STRISEMPTY(path))
	    return(NULL);

	/* Get the type string from the type code */
	type_str = STRDUP(EDVGetObjectTypeName(type));
	if(type_str == NULL)
	    return(NULL);

	strtolower(type_str);

	/* Generate a new path */
	new_path = g_strconcat(
	    path,
	    G_DIR_SEPARATOR_S,
	    "new_",
	    type_str,
	    NULL
	);
	if(new_path == NULL)
	{
	    g_free(type_str);
	    return(NULL);
	}

	/* Does the new path refer to a non-existant object? */
	if(lstat((const char *)new_path, &lstat_buf))
	{
	    const gint error_code = (gint)errno;
#ifdef ENOENT
	    if(error_code == ENOENT)
#else
	    if(FALSE)
#endif
	    {
		g_free(type_str);
		return(new_path);
	    }
	}

	/* An object already exists at the generated new path, so
	 * regenerate a new pth with a number appended to it and
	 * regenerate in the same way as needed until a path is
	 * generated that does not refer to an existing object
	 */
	for(i = 2; i < (guint16)-1; i++)
	{
	    /* Delete the previously generated new path */
	    g_free(new_path);

	    /* Generate a new path with a number appended to it */
	    new_path = g_strdup_printf(
		"%s%cnew_%s%u",
		path, G_DIR_SEPARATOR, type_str, i
	    );
	    if(new_path == NULL)
		break;

	    /* Does the new path refer to a non-existant object? */
	    if(lstat((const char *)new_path, &lstat_buf))
	    {
		const gint error_code = (gint)errno;
#ifdef ENOENT
		if(error_code == ENOENT)
#else
		if(FALSE)
#endif
		{
		    g_free(type_str);
		    return(new_path);
		}
	    }
	}

	/* If this point is reached then we failed to generate a new
	 * path to a non-existent object
	 *
	 * Delete the last generated new path and return NULL
	 */
	g_free(new_path);
	g_free(type_str);

	return(NULL);
}


/*
 *	Prompts the user to create a new file.
 *
 *	The parent_path specifies the full path to the directory at
 *	which the new object is to be created at.
 *
 *	The suggested_name specifies the suggested name for the new
 *	file or NULL.
 *
 *	If new_path_rtn is not NULL then a dynamically allocated
 *	string describing the full path to the new object will be set
 *	to it.
 *
 *	If an error occures then the last error will be set to the
 *	error message that describes what error occured.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint EDVObjCreateFileQueryName(
	edv_core_struct *core,
	const gchar *parent_path,
        const gchar *suggested_name,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	gboolean *yes_to_all
)
{
	const gboolean interactive = TRUE;
	struct stat lstat_buf;
	gint status, fd, strc;
	guint m;
	gulong time_start;
	gchar *name, *path, **strv;

	if(new_path_rtn != NULL)
	    *new_path_rtn = NULL;

	/* Clear the last error message */
	EDVObjCreateCopyErrorMessage(core, NULL);

	if((core == NULL) || STRISEMPTY(parent_path) ||
	   (yes_to_all == NULL)
	)
	{
	    EDVObjCreateCopyErrorMessage(core, "Invalid value.");
	    return(-2);
	}

        /* Is there an operation already in progress? */
        if(core->op_level > 0)
        {
            core->obj_last_error =
"An operation is already in progress, please try again later.";
            return(-6);
        }

        core->op_level++;

	if(!g_path_is_absolute(parent_path))
	{
	    core->obj_last_error = "Invalid value.";
	    core->op_level--;
	    return(-2);
	}

	/* Prompt the user for a file name */
        PDialogDeleteAllPrompts();
        PDialogSetTransientFor(toplevel);
        PDialogAddPrompt(NULL, "New Name:", suggested_name);
        PDialogSetSize(320, -1);
        strv = PDialogGetResponse(
            "Create New File",
            NULL, NULL,
            PDIALOG_ICON_FILE,
            "Create", "Cancel",
            PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
            PDIALOG_BTNFLAG_SUBMIT,
            &strc
        );
        PDialogSetTransientFor(NULL);
	if((strv == NULL) || (strc < 1))
	{
	    PDialogDeleteAllPrompts();
	    core->op_level--;
	    return(-4);
	}

	/* Get a copy of the name */
	name = STRDUP(strv[0]);

	PDialogDeleteAllPrompts();

	/* Unable to get a copy of the name? */
	if(name == NULL)
	{
	    core->obj_last_error = "Memory allocation error.";
	    core->op_level--;
            return(-3);
	}

	/* Generate the full path to the new object to be created */
	path = g_strconcat(
	    parent_path,
	    G_DIR_SEPARATOR_S,
	    name,
	    NULL
	);
	if(path == NULL)
	{
	    g_free(name);
	    core->obj_last_error = "Memory allocation error.";
	    core->op_level--;
            return(-3);
	}

	EDVSimplifyPath(path);

	/* Check if the object already exists */
	if(!stat((const char *)path, &lstat_buf))
	{
	    if(interactive)
	    {
		/* Confirm overwrite */
		gint response;
                gchar *msg = g_strdup_printf(
"Overwrite existing file:\n\
\n\
    %s",
		    path
		);
                EDVPlaySoundWarning(core);
                CDialogSetTransientFor(toplevel);
                response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Escriba Para Reemplazar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer craser"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie berschreibt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Sovrascrivere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Beschrijft"   
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Overwrite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Overskriver"
#else
"Confirm Overwrite"
#endif
                    , msg,
                    NULL,
                    CDIALOG_ICON_WARNING,
                    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
                    CDIALOG_BTNFLAG_NO
                );
                g_free(msg);
                CDialogSetTransientFor(NULL);
                if((response != CDIALOG_RESPONSE_YES) &&
		   (response != CDIALOG_RESPONSE_YES_TO_ALL)
		)
		{
		    g_free(name);
		    g_free(path);
	            core->op_level--;
	            return(-5);
		}
		if(response == CDIALOG_RESPONSE_YES_TO_ALL)
		    *yes_to_all = TRUE;
	    }

	    /* Remove the existing file */
	    if(unlink((const char *)path))
	    {
		const gint error_code = (gint)errno;
		gchar *msg = g_strconcat(
		    "Unable to create a new ",
		    EDVGetObjectTypeNameLower(EDV_OBJECT_TYPE_FILE),
		    ", ",
		    g_strerror(error_code),
		    ".",
		    NULL
		);
		EDVObjCreateCopyErrorMessage(core, msg);
		g_free(msg);
		g_free(name);
		g_free(path);
	        core->op_level--;
	        return(-1);
	    }
	}

	time_start = (gulong)time(NULL);
	status = 0;

	/* Create the new file */
	m = EDVGetUMask();
	fd = (gint)open(
	    (const char *)path,
	    O_WRONLY | O_CREAT | O_EXCL,
	    (mode_t)(~m) &
		(S_IRUSR | S_IWUSR |
		 S_IRGRP | S_IWGRP |
		 S_IROTH | S_IWOTH)
	);
	if(fd > -1)
	{
	    close((int)fd);
	}
	else
	{
	    const gint error_code = (gint)errno;
	    gchar *msg = g_strconcat(
		"Unable to create a new ",
		EDVGetObjectTypeNameLower(EDV_OBJECT_TYPE_FILE),
		", ",
		g_strerror(error_code),
		".",
	        NULL
	    );
	    EDVObjCreateCopyErrorMessage(core, msg);
	    g_free(msg);
	    status = -1;
	}

	/* Update new path return if creation of object was successful */
	if((status == 0) && (new_path_rtn != NULL))
	{
	    g_free(*new_path_rtn);
	    *new_path_rtn = STRDUP(path);
	}

	/* Record history */
	EDVAppendHistory(
	    core,
	    EDV_HISTORY_VFS_OBJECT_CREATE,
	    time_start, (gulong)time(NULL),
	    status,
	    parent_path,			/* Source */
	    path,				/* Target */
	    core->obj_last_error		/* Comment */
	);

	g_free(name);
	g_free(path);

	core->op_level--;

	return(status);
}


/*
 *	Creates a new object of the specified type.
 *
 *	The path specifies the full path to either a non-existant
 *	object to be created or an existing directory at which to
 *	create the new object in.
 *
 *	If new_path_rtn is not NULL then a dynamically allocated
 *	string describing the full path to the new object will be set
 *	to it.
 *
 *	If an error occures then the last error will be set to the
 *	error message that describes what error occured.
 *
 *	Returns 0 on success and non-zero on error.
 */
static gint EDVObjCreateNexus(
	edv_core_struct *core,
	const gchar *path,
	const edv_object_type type,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	const gulong time_start = (gulong)time(NULL);
	struct stat stat_buf;
	gint status;
	gchar *lpath = NULL, *new_path = NULL;

	if(new_path_rtn != NULL)
	    *new_path_rtn = NULL;

	/* Clear the last error message */
	EDVObjCreateCopyErrorMessage(core, NULL);

	if((core == NULL) || STRISEMPTY(path) ||
	   (yes_to_all == NULL)
	)
	{
	    EDVObjCreateCopyErrorMessage(core, "Invalid value.");
	    return(-2);
	}

        /* Is there an operation already in progress? */
        if(core->op_level > 0)
        {
            core->obj_last_error =
"An operation is already in progress, please try again later.";
            return(-6);
        }

        core->op_level++;

#define CLEANUP_RETURN(_v_)	{	\
 g_free(lpath);				\
 g_free(new_path);			\
					\
 return(_v_);				\
}

	/* Copy the specified path */
	lpath = STRDUP(path);

	/* Simplify the path */
	EDVSimplifyPath(lpath);

	/* Get the statistics of the object at the specified path */
	if(stat((const char *)lpath, &stat_buf))
	{
	    const gint error_code = (gint)errno;
#ifdef ENOENT
	    if(error_code == ENOENT)
#else
	    if(FALSE)
#endif
	    {
		/* No such object exists at the specified path's
		 * location so use it as the path to the new object
		 */
		new_path = STRDUP(lpath);
	    }
	    else
	    {
		gchar *msg = g_strconcat(
		    "Unable to create a new ",
		    EDVGetObjectTypeNameLower(type),
		    ", ",
		    g_strerror(error_code),
		    ".",
		    NULL
		);
		EDVObjCreateCopyErrorMessage(core, msg);
		g_free(msg);
		core->op_level--;
		CLEANUP_RETURN(-1);
	    }
	}
	else
	{
	    /* The specified path exists
	     *
	     * Check if its destination is not a directory, in which
	     * case no new object can be created
	     */
#ifdef S_ISDIR
	    if(!S_ISDIR(stat_buf.st_mode))
#else
	    if(TRUE)
#endif
	    {
		core->obj_last_error =
"The location to create the new object at is not a directory.";
		core->op_level--;
		CLEANUP_RETURN(-1);
	    }

	    /* Generate a new name based on the specified path */
	    new_path = EDVObjCreateGenerateNewPath(lpath, type);
	}
	/* Unable to generate the full path to the new object? */
	if(new_path == NULL)
	{
	    core->obj_last_error = "Unable to generate the full path for the new object.";
	    core->op_level--;
	    CLEANUP_RETURN(-1);
	}

	/* At this point new_path specifies the full path to the new
	 * object that is to be created
	 *
	 * Begin creating by type
	 */
	status = 0;

	/* File? */
	if(type == EDV_OBJECT_TYPE_FILE)
	{
	    const guint m = EDVGetUMask();
	    const gint fd = (gint)open(
		(const char *)new_path,
		O_WRONLY | O_CREAT | O_EXCL,
		(mode_t)(~m) &
		    (S_IRUSR | S_IWUSR |
		     S_IRGRP | S_IWGRP |
		     S_IROTH | S_IWOTH)
	    );
	    if(fd > -1)
	    {
		close((int)fd);
	    }
	    else
	    {
		const gint error_code = (gint)errno;
		gchar *msg = g_strconcat(
		    "Unable to create a new ",
		    EDVGetObjectTypeNameLower(type),
		    ", ",
		    g_strerror(error_code),
		    ".",
		    NULL
		);
		EDVObjCreateCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }
	}
	/* Directory */
	else if(type == EDV_OBJECT_TYPE_DIRECTORY)
	{
	    const guint m = EDVGetUMask();
	    if(mkdir(
		(const char *)new_path,
		(mode_t)(~m) &
		    (S_IRUSR | S_IWUSR | S_IXUSR |
		     S_IRGRP | S_IWGRP | S_IXGRP |
		     S_IROTH | S_IWOTH | S_IXOTH)
	    ))
	    {
		const gint error_code = (gint)errno;
		gchar *msg = g_strconcat(
		    "Unable to create a new ",
		    EDVGetObjectTypeNameLower(type),
		    ", ",
		    g_strerror(error_code),
		    ".",
		    NULL
		);
		EDVObjCreateCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }
	}
	/* Link */
	else if(type == EDV_OBJECT_TYPE_LINK)
	{
	    if(symlink("undefined", (const char *)new_path))
	    {
		const gint error_code = (gint)errno;
		gchar *msg = g_strconcat(
		    "Unable to create a new ",
		    EDVGetObjectTypeNameLower(type),
		    ", ",
		    g_strerror(error_code),
		    ".",
		    NULL
		);
		EDVObjCreateCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }
	}
	/* FIFO Pipe */
	else if(type == EDV_OBJECT_TYPE_FIFO)
	{
#if defined(S_IFFIFO) || defined(S_IFIFO)
	    const guint m = EDVGetUMask();
	    if(mknod(
		(const char *)new_path,
#if defined(S_IFFIFO)
		S_IFFIFO | ((mode_t)(~m) &
		    (S_IRUSR | S_IWUSR |
		     S_IRGRP | S_IWGRP |
		     S_IROTH | S_IWOTH)
		),
#else
		S_IFIFO | ((mode_t)(~m) &
		    (S_IRUSR | S_IWUSR |
		     S_IRGRP | S_IWGRP |
		     S_IROTH | S_IWOTH)
		),
#endif
		0
	    ))
	    {
		const gint error_code = (gint)errno;
		gchar *msg = g_strconcat(
		    "Unable to create a new ",
		    EDVGetObjectTypeNameLower(type),
		    ", ",
		    g_strerror(error_code),
		    ".",
		    NULL
		);
		EDVObjCreateCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }
#else
	    core->obj_last_error = "Unsupported object type.";
	    status = -2;
#endif
	}
	/* Block Device */
	else if(type == EDV_OBJECT_TYPE_DEVICE_BLOCK)
	{
#if defined(S_IFBLK)
	    const guint m = EDVGetUMask();
	    const gint dev_num = EDVFormatDeviceNumbers(1, 1);
	    if(mknod(
		(const char *)new_path,
		S_IFBLK | ((mode_t)(~m) &
		    (S_IRUSR | S_IWUSR |
		     S_IRGRP | S_IWGRP |
		     S_IROTH | S_IWOTH)
		),
		(dev_t)dev_num
	    ))
	    {
		const gint error_code = (gint)errno;
		gchar *msg = g_strconcat(
		    "Unable to create a new ",
		    EDVGetObjectTypeNameLower(type),
		    ", ",
		    g_strerror(error_code),
		    ".",
		    NULL
		);
		EDVObjCreateCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }
#else
	    core->obj_last_error = "Unsupported object type.";
	    status = -2;
#endif
	}
	/* Character Device */
	else if(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	{
#if defined(S_IFCHR)
	    const guint m = EDVGetUMask();
	    const gint dev_num = EDVFormatDeviceNumbers(1, 1);
	    if(mknod(
		(const char *)new_path,
		S_IFCHR | ((mode_t)(~m) &
		    (S_IRUSR | S_IWUSR |
		     S_IRGRP | S_IWGRP |
		     S_IROTH | S_IWOTH)
		),
		(dev_t)dev_num
	    ))
	    {
		const gint error_code = (gint)errno;
		gchar *msg = g_strconcat(
		    "Unable to create a new ",
		    EDVGetObjectTypeNameLower(type),
		    ", ",
		    g_strerror(error_code),
		    ".",
		    NULL
		);
		EDVObjCreateCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }
#else
	    core->obj_last_error = "Unsupported object type.";
	    status = -2;
#endif
	}
	/* Socket */
	else if(type == EDV_OBJECT_TYPE_SOCKET)
	{
#if defined(S_IFSOCK)
	    const guint m = EDVGetUMask();
	    if(mknod(
		(const char *)new_path,
		S_IFSOCK | ((mode_t)(~m) &
		    (S_IRUSR | S_IWUSR |
		     S_IRGRP | S_IWGRP |
		     S_IROTH | S_IWOTH)
		),
		0
	    ))
	    {
		const gint error_code = (gint)errno;
		gchar *msg = g_strconcat(
		    "Unable to create a new ",
		    EDVGetObjectTypeNameLower(type),
		    ", ",
		    g_strerror(error_code),
		    ".",
		    NULL
		);
		EDVObjCreateCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }
#else
	    core->obj_last_error = "Unsupported object type.";
	    status = -2;
#endif
	}
	else
	{
	    core->obj_last_error = "Unsupported object type.";
	    status = -2;
	}

	/* Update new path return if creation of object was successful */
	if((status == 0) && (new_path_rtn != NULL))
	{
	    g_free(*new_path_rtn);
	    *new_path_rtn = STRDUP(new_path);
	}


	/* Record history */
	EDVAppendHistory(
	    core,
	    EDV_HISTORY_VFS_OBJECT_CREATE,
	    time_start, (gulong)time(NULL),
	    status,
	    path,				/* Source */
	    new_path,				/* Target */
	    core->obj_last_error		/* Comment */
	);

	core->op_level--;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}


/*
 *	Create a new file.
 */
gint EDVObjCreateFile(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVObjCreateNexus(
	    core,
	    path, EDV_OBJECT_TYPE_FILE,
	    new_path_rtn,
	    toplevel, show_progress, interactive,
	    yes_to_all
	));
}

/*
 *	Create a new directory.
 */
gint EDVObjCreateDirectory(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVObjCreateNexus(
	    core,
	    path, EDV_OBJECT_TYPE_DIRECTORY, new_path_rtn,
	    toplevel, show_progress, interactive,
	    yes_to_all
	));
}

/*
 *	Create a new link.
 */
gint EDVObjCreateLink(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVObjCreateNexus(
	    core,
	    path, EDV_OBJECT_TYPE_LINK,
	    new_path_rtn,
	    toplevel, show_progress, interactive,
	    yes_to_all
	));
}

/*
 *	Create a new FIFO pipe.
 */
gint EDVObjCreateFifo(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVObjCreateNexus(
	    core,
	    path, EDV_OBJECT_TYPE_FIFO,
	    new_path_rtn,
	    toplevel, show_progress, interactive,
	    yes_to_all
	));
}

/*
 *	Create a new block device.
 */
gint EDVObjCreateDeviceBlock(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVObjCreateNexus(
	    core, path,
	    EDV_OBJECT_TYPE_DEVICE_BLOCK,
	    new_path_rtn,
	    toplevel, show_progress, interactive,
	    yes_to_all
	));
}

/*
 *	Create a new character device.
 */
gint EDVObjCreateDeviceCharacter(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVObjCreateNexus(
	    core,
	    path, EDV_OBJECT_TYPE_DEVICE_CHARACTER,
	    new_path_rtn,
	    toplevel, show_progress, interactive,
	    yes_to_all
	));
}

/*
 *	Create a new socket.
 */
gint EDVObjCreateSocket(
	edv_core_struct *core,
	const gchar *path,
	gchar **new_path_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all
)
{
	return(EDVObjCreateNexus(
	    core,
	    path, EDV_OBJECT_TYPE_SOCKET,
	    new_path_rtn,
	    toplevel, show_progress, interactive,
	    yes_to_all
	));
}
