#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "../include/string.h"

#include "guiutils.h"

#include "tlist.h"
#include "edv_list_seek.h"


gboolean EDVCListSeekCharacter(
	GtkCList *clist,
	const gint column,
	const gint start_row,
	const gboolean backwards,
	const gint cidx,
	const gchar c
);
gboolean EDVTListSeekCharacter(
	tlist_struct *tlist,
	const gint start_thumb_num,
	const gboolean backwards,
	const gint cidx,
	const gchar c
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Seeks to a row who's cell text character matches the
 *	specified character at the specified index.
 *
 *	The clist specifies the GtkCList.
 *
 *	The column specifies the column index at which each cell's
 *	text will be tested against the specified character.
 *
 *	The start_row specifies the starting row index.
 *
 *	The cidx specifies the string index of the cell's text to
 *	test the specified character at.
 *
 *	The c specifies the character to to seek to.
 *
 *	If a match is made, then the clist will be scrolled to the
 *	position of the first match and the return will be TRUE.
 */
gboolean EDVCListSeekCharacter(
	GtkCList *clist,
	const gint column,
	const gint start_row,
	const gboolean backwards,
	const gint cidx,
	const gchar c
)
{

	if((clist == NULL) || (cidx < 0))
	    return(FALSE);

	if((column < 0) || (column >= clist->columns))
	    return(FALSE);

	if(backwards)
	{
	    const gint nrows = clist->rows;
	    gint row;
	    gchar *text;
	    guint8 spacing;
	    GdkBitmap *mask;
	    GdkPixmap *pixmap;

	    for(row = MIN(start_row, (nrows - 1)); row >= 0; row--)
	    {
		/* Handle by cell type */
		text = NULL;
		switch(gtk_clist_get_cell_type(clist, row, column))
		{
		  case GTK_CELL_TEXT:
		    gtk_clist_get_text(
			clist, row, column, &text
		    );
		    break;
		  case GTK_CELL_PIXTEXT:
		    gtk_clist_get_pixtext(
			clist, row, column, &text,
			&spacing, &pixmap, &mask
		    );
		    break;
		  case GTK_CELL_PIXMAP:
		  case GTK_CELL_WIDGET:
		  case GTK_CELL_EMPTY:
		    break;
		}
		if(!STRISEMPTY(text))
		{
		    /* Is the specified character index valid on this
		     * text's string?
		     */
		    const gint len = STRLEN(text);
		    if(cidx < len)
		    {
			/* Check if this character matches the
			 * specified character
			 */
			if(text[cidx] == c)
			{
			    /* Got a match, scroll to this row */
			    gtk_clist_moveto(
				clist,
				row, column,	/* Row, column */
				0.5f, 0.0f	/* Row, column */
			    );
			    return(TRUE);
			}
		    }
		}
	    }
	}
	else
	{
	    const gint nrows = clist->rows;
	    gint row;
	    gchar *text;
	    guint8 spacing;
	    GdkBitmap *mask;
	    GdkPixmap *pixmap;

	    for(row = MAX(start_row, 0); row < nrows; row++)
	    {
		/* Handle by cell type */
		text = NULL;
		switch(gtk_clist_get_cell_type(clist, row, column))
		{
		  case GTK_CELL_TEXT:
		    gtk_clist_get_text(
			clist, row, column, &text
		    );
		    break;
		  case GTK_CELL_PIXTEXT:
		    gtk_clist_get_pixtext(
			clist, row, column, &text,
			&spacing, &pixmap, &mask
		    );
		    break;
		  case GTK_CELL_PIXMAP:
		  case GTK_CELL_WIDGET:
		  case GTK_CELL_EMPTY:
		    break;
		}
		if(!STRISEMPTY(text))
		{
		    /* Is the specified character index valid on this
		     * text's string?
		     */
		    const gint len = STRLEN(text);
		    if(cidx < len)
		    {
			/* Check if this character matches the
			 * specified character
			 */
			if(text[cidx] == c)
			{
			    /* Got a match, scroll to this row */
			    gtk_clist_moveto(
				clist,
				row, column,	/* Row, column */
				0.5f, 0.0f	/* Row, column */
			    );
			    return(TRUE);
			}
		    }
		}
	    }
	}

	return(FALSE);
}

/*
 *      Searches all thumbs of the given tlist who has a the character
 *      at index cidx that matches c.
 *
 *      If a match is made, then the tlist will be scrolled to the
 *      position of the first match and the return will be TRUE.
 */
gboolean EDVTListSeekCharacter(
	tlist_struct *tlist,
	const gint start_thumb_num,
	const gboolean backwards,
	const gint cidx,
	const gchar c
)
{
	if((tlist == NULL) || (cidx < 0))
	    return(FALSE);

	if(backwards)
	{
	    const gint nthumbs = tlist->total_thumbs;
	    gint i;
	    const gchar *text;
	    tlist_thumb_struct *thumb;

	    for(i = MIN(start_thumb_num, (nthumbs - 1)); i >= 0; i--)
	    {
		thumb = tlist->thumb[i];
		if(thumb == NULL)
		    continue;

		text = thumb->text;
		if(!STRISEMPTY(text))
		{
		    /* Is the specified character index valid on this
		     * text's string?
		     */
		    const gint len = STRLEN(text);
		    if(cidx < len)
		    {
			/* Check if this character matches the
			 * specified character
			 */
			if(text[cidx] == c)
			{
			    /* Got a match, scroll to this thumb */
			    TListMoveTo(tlist, i, 0.5f);
			    return(TRUE);
			}
		    }
		}
	    }
	}
	else
	{
	    const gint nthumbs = tlist->total_thumbs;
	    gint i;
	    const gchar *text;
	    tlist_thumb_struct *thumb;

	    for(i = MAX(start_thumb_num, 0); i < nthumbs; i++)
	    {
		thumb = tlist->thumb[i];
		if(thumb == NULL)
		    continue;

		text = thumb->text;
		if(!STRISEMPTY(text))
		{
		    /* Is the specified character index valid on this
		     * text's string?
		     */
		    const gint len = STRLEN(text);
		    if(cidx < len)
		    {
			/* Check if this character matches the
			 * specified character
			 */
			if(text[cidx] == c)
			{
			    /* Got a match, scroll to this thumb */
			    TListMoveTo(tlist, i, 0.5f);
			    return(TRUE);
			}
		    }
		}
	    }
	}

	return(FALSE);
}
