#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <glib.h>

#include "edv_types.h"
#include "edv_id.h"


/* User ID functions */
edv_uid_struct *EDVUIDNew(void);
void EDVUIDDelete(edv_uid_struct *uid);

GList *EDVUIDGetSystem(void);

edv_uid_struct *EDVUIDMatchByUID(
	GList *uids_list,
	const gint user_id, gint *n
);
edv_uid_struct *EDVUIDMatchByGID(
	GList *uids_list,
	const gint group_id, gint *n
);
gint EDVUIDNameToUID(GList *uids_list, const gchar *user_name);
gchar *EDVUIDGetNameFromUID(
	GList *uids_list,
	const gint user_id, gint *n
);
gchar *EDVUIDGetNameFromGID(
	GList *uids_list,
	const gint group_id, gint *n
);

/* Group ID functions */
edv_gid_struct *EDVGIDNew(void);
void EDVGIDDelete(edv_gid_struct *gid);

GList *EDVGIDGetSystem(void);

edv_gid_struct *EDVGIDMatchByGID(
	GList *gids_list,
	const gint group_id, gint *n
);
gint EDVGIDNameToGID(GList *gids_list, const gchar *group_name);
gchar *EDVGIDGetNameFromGID(
	GList *gids_list,
	const gint group_id, gint *n
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new UID.
 */
edv_uid_struct *EDVUIDNew(void)
{
	return(EDV_UID(g_malloc0(sizeof(edv_uid_struct))));
}

/*
 *	Deletes the UID.
 */
void EDVUIDDelete(edv_uid_struct *uid)
{
	if(uid == NULL)
	    return;

	g_free(uid->name);
	g_free(uid->password);

	g_free(uid->government_name);
	g_free(uid->home_directory);
	g_free(uid->shell_program);

	g_free(uid);
}

/*
 *	Gets a list of UIDs from the system.
 *
 *	Returns a GList of edv_uid_struct * or NULL on error. The
 *	calling function must delete the returned list and each
 *	uid.
 */
GList *EDVUIDGetSystem(void)
{
	struct passwd *pwent;
	GList *uids_list = NULL;
	edv_uid_struct *uid;

	/* Open the passwords file and get the first entry */
	pwent = getpwent();

	/* Iterate through each entry */
	while(pwent != NULL)
	{
	    /* Create a new UID */
	    uid = EDVUIDNew();
	    if(uid == NULL)
		break;

	    /* Get this UID */
	    uid->name = STRDUP(pwent->pw_name);
	    uid->password = STRDUP(pwent->pw_passwd);
	    uid->user_id = pwent->pw_uid;
	    uid->group_id = pwent->pw_gid;
	    uid->government_name = STRDUP(pwent->pw_gecos);
	    uid->home_directory = STRDUP(pwent->pw_dir);
	    uid->shell_program = STRDUP(pwent->pw_shell);

	    uids_list = g_list_append(uids_list, uid);

	    /* Get the next entry */
	    pwent = getpwent();
	}

	/* Close the passwords file */
	endpwent();

	return(uids_list);
}


/*
 *	Matches the UID with the specified user ID.
 */
edv_uid_struct *EDVUIDMatchByUID(
	GList *uids_list,
	const gint user_id, gint *n
)
{
	gint i;
	GList *glist;
	edv_uid_struct *uid;

	if(n != NULL)
	    *n = -1;

	for(glist = uids_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
	    uid = EDV_UID(glist->data);
	    if(uid == NULL)
		continue;

	    if(uid->user_id == user_id)
	    {
		if(n != NULL)
		    *n = i;
		return(uid);
	    }
	}

	return(NULL);
}

/*
 *	Matches the UID with the specified group ID.
 */
edv_uid_struct *EDVUIDMatchByGID(
	GList *uids_list,
	const gint group_id, gint *n
)
{
	gint i;
	GList *glist;
	edv_uid_struct *uid;

	if(n != NULL)
	    *n = -1;

	for(glist = uids_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
	    uid = EDV_UID(glist->data);
	    if(uid == NULL)
		continue;

	    if(uid->group_id == group_id)
	    {
		if(n != NULL)
		    *n = i;
		return(uid);
	    }
	}

	return(NULL);
}

/*
 *	Matches the UID with the specified user name.
 */
gint EDVUIDNameToUID(GList *uids_list, const gchar *user_name)
{
	if(STRISEMPTY(user_name))
	    return(0);

	/* Match by number? */
	if(isdigit(*user_name))
	{
	    return(ATOI(user_name));
	}
	else
	{
	    GList *glist;
	    edv_uid_struct *uid;

	    for(glist = uids_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		uid = EDV_UID(glist->data);
		if(uid == NULL)
		    continue;

		if(uid->name == NULL)
		    continue;

		if(!strcmp((const char *)uid->name, (const char *)user_name))
		    return(uid->user_id);
	    }
	}

	return(0);
}


/*
 *	Matches the user name with the specified user ID.
 *
 *	Returns a dynamically allocated string describing the user
 *	name.
 */
gchar *EDVUIDGetNameFromUID(
	GList *uids_list,
	const gint user_id, gint *n
)
{
	edv_uid_struct *uid = EDVUIDMatchByUID(uids_list, user_id, n);
	if((uid != NULL) ? (uid->name != NULL) : FALSE)
	{
	    return(STRDUP(uid->name));
	}
	else if(user_id < 0)
	{
	    return(STRDUP(""));
	}
	else
	{
	    return(g_strdup_printf(
		"%i",
		user_id
	    ));
	}
}

/*
 *	Matches the user name with the specified group ID.
 *
 *	On failed matched a number string describing the user ID is
 *	returned, so this function never returns NULL.
 */
gchar *EDVUIDGetNameFromGID(
	GList *uids_list,
	const gint group_id, gint *n
)
{
	edv_uid_struct *uid = EDVUIDMatchByGID(uids_list, group_id, n);
	if((uid != NULL) ? (uid->name != NULL) : FALSE)
	{
	    return(STRDUP(uid->name));
	}
	else if(group_id < 0)
	{
	    return(STRDUP(""));
	}
	else
	{
	    return(g_strdup_printf(
		"%i",
		group_id
	    ));
	}
}


/*
 *	Creates a new GID.
 */
edv_gid_struct *EDVGIDNew(void)
{
	return(EDV_GID(g_malloc0(sizeof(edv_gid_struct))));
}

/*
 *	Deletes the GID.
 */
void EDVGIDDelete(edv_gid_struct *gid)
{
	if(gid == NULL)
	    return;

	g_free(gid->name);
	g_free(gid->password);
	if(gid->group_members_list != NULL)
	{
	    g_list_foreach(gid->group_members_list, (GFunc)g_free, NULL);
	    g_list_free(gid->group_members_list);
	}
	g_free(gid);
}

/*
 *	Gets a list of GIDs from the system.
 *
 *	Returns a GList of edv_gid_struct * or NULL on error. The
 *	calling function must delete the returned list and each
 *	gid.
 */
GList *EDVGIDGetSystem(void)
{
	struct group *grent;
	GList *gids_list = NULL;
	edv_gid_struct *gid;

	/* Open the groups file and get the first entry */
	grent = getgrent();

	/* Iterate through each entry */
	while(grent != NULL)
	{
	    /* Create a new GID */
	    gid = EDVGIDNew();
	    if(gid == NULL)
		break;

	    /* Get this GID */
	    gid->name = STRDUP(grent->gr_name);
	    gid->password = STRDUP(grent->gr_passwd);
	    gid->group_id = grent->gr_gid;

	    /* Any group members to add? */
	    if(grent->gr_mem != NULL)
	    {
		gint i;
		for(i = 0; grent->gr_mem[i] != NULL; i++)
		    gid->group_members_list = g_list_append(
			gid->group_members_list,
			STRDUP(grent->gr_mem[i])
		    );
	    }

	    gids_list = g_list_append(gids_list, gid);

	    /* Get the next entry */
	    grent = getgrent();
	}

	/* Close the groups file */
	endgrent();

	return(gids_list);
}

/*
 *	Matches the GID with the specified group ID.
 */
edv_gid_struct *EDVGIDMatchByGID(
	GList *gids_list,
	const gint group_id, gint *n
)
{
	gint i;
	GList *glist;
	edv_gid_struct *gid;

	if(n != NULL)
	    *n = -1;

	for(glist = gids_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
	    gid = EDV_GID(glist->data);
	    if(gid == NULL)
		continue;

	    if(gid->group_id == group_id)
	    {
		if(n != NULL)
		    *n = i;
		return(gid);
	    }
	}

	return(NULL);
}

/*
 *	Matches the GID with the specified group name.
 */
gint EDVGIDNameToGID(GList *gids_list, const gchar *group_name)
{
	if(STRISEMPTY(group_name))
	    return(0);

	/* Match by number? */
	if(isdigit(*group_name))
	{
	    return(ATOI(group_name));
	}
	else
	{
	    GList *glist;
	    edv_gid_struct *gid;

	    for(glist = gids_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		gid = EDV_GID(glist->data);
		if(gid == NULL)
		    continue;

		if(gid->name == NULL)
		    continue;

		if(!strcmp((const char *)gid->name, (const char *)group_name))
		    return(gid->group_id);
	    }
	}

	return(0);
}


/*
 *	Matches the group name from the specified group ID.
 *
 *	On failed matched, a number string containing the group ID is
 *      returned, so this function never returns NULL.
 */
gchar *EDVGIDGetNameFromGID(
	GList *gids_list,
	const gint group_id, gint *n
)
{
	edv_gid_struct *gid = EDVGIDMatchByGID(gids_list, group_id, n);
	if((gid != NULL) ? (gid->name != NULL) : FALSE)
	{
	    return(STRDUP(gid->name));
	}
	else if(group_id < 0)
	{
	    return(STRDUP(""));
	}
	else
	{
	    return(g_strdup_printf(
		"%i",
		group_id
	    ));
	}
}
