/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Author: Stéphane Démurget  <stephane.demurget@enst-bretagne.fr>
 */

#include <glib.h>

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>

#include <errno.h>

#include "ef-msg-queue.h"
#include "ef-utils.h"

static key_t
get_ipc_key ()
{
	key_t k = ftok (IPC_PATHNAME, IPC_PROJECT_ID);

	if (k != -1)
		return k;

	return IPC_FALLBACK_KEY;
}

EFMessageQueue *
ef_msg_queue_new (gboolean server_side)
{
	
	int flags = server_side ? IPC_CREAT : 0;
        int queue_id = msgget (get_ipc_key (), flags | 0666);

        if (queue_id != -1) {
		EFMessageQueue *queue;

		queue = (EFMessageQueue *) g_new0 (EFMessageQueue, 1);
		queue->id = queue_id;
		queue->server_side = server_side;

		return queue;
	}
		
	return NULL;
}

gboolean
ef_msg_queue_is_alive (EFMessageQueue *queue)
{
	int id;

	g_return_val_if_fail (queue != NULL, FALSE);

	if ((id = msgget (get_ipc_key (), 0666)) == -1 && errno == ENOENT)
		return FALSE;

	/* Note: it's important to update the value has the daemon may have
		 been restarted. Okay, I am paranoiac ;) */
	queue->id = id;

	return TRUE;
}

gboolean
ef_msg_queue_send (EFMessageQueue *queue,
		   EFMessage 	  *msg)
{
	g_return_val_if_fail (queue != NULL, -1);
	g_return_val_if_fail (msg != NULL, -1);

	return (msgsnd (queue->id, msg, sizeof(msg->frequency), 0) != -1) ? TRUE : FALSE;
}

EFMessage *
ef_msg_queue_receive (EFMessageQueue *queue)
{
	EFMessage *msg;

	g_return_val_if_fail (queue != NULL, NULL);

	msg = g_new0 (EFMessage, 1);

	if (msgrcv (queue->id, msg, sizeof (msg->frequency), 0, 0) > 0)
		return msg;

	g_free (msg);

	return NULL;
}

void
ef_msg_queue_free (EFMessageQueue *queue)
{
	g_return_if_fail (queue != NULL);

	/* We can't do something if the result goes bad, so let's take a nap :) */
	if (queue->server_side)
		msgctl (queue->id, IPC_RMID, NULL);

	g_free (queue);
}
