/* $Id: e2_fs.c 568 2007-07-25 23:08:53Z tpgww $

Copyright (C) 2005-2007 tooar <tooar@gmx.net>

This file is part of emelFM2.
emelFM2 is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

emelFM2 is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emelFM2; see the file GPL. If not, contact the Free Software
Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

/**
 @file src/filesystem/e2_fs.c
 @brief filesystem I/O functions
*/
/**
\page filesystem interactions with the file system

ToDo - describe how this happens

\section vfs the virtual filesystem

Not implemented yet.
*/

#include "emelfm2.h"
#include <string.h>
#include <fcntl.h>
#include <dirent.h>
#include <pthread.h>
#include <langinfo.h>
#include <pwd.h>
#include <grp.h>
#if defined(__FreeBSD__) || defined(__OpenBSD__)
# include <sys/param.h>
# include <sys/mount.h>
#else
# include <sys/statfs.h>
#endif
#ifndef MNT_LOCAL
# include <sys/statvfs.h>
#endif
#include "e2_fs.h"
#include "e2_dialog.h"
#if defined(__linux__) || defined(__FreeBSD__)
#include "e2_complete.h"
#endif

typedef struct _E2_DRFuncArgs
{
	gchar *localpath;	//absolute path of dir to process, localised string FIXME
	E2_FSType dirtype;	//dirtype code for type of dir being processed
	gpointer callback;	//function to call for each processed item, or NULL to append item's name to list
	gpointer cb_data;	//pointer to data to send to @a callback
	GDestroyNotify free_data_func;	//function to call at end of scan, to clean @a cb_data, or NULL
} E2_DRFuncArgs;

#if 0
/**
@brief check if any x permission flag is set in @a mode
@param mode
@return TRUE if any x flag is set
*/
//FIXME this test should be for current user only, cf writable test
static gboolean _e2_fs_S_ISEXE (mode_t mode)
{
	if ((S_IXUSR & mode) || (S_IXGRP & mode) || (S_IXOTH & mode))
		return TRUE;
	return FALSE;
}
#endif //0

#ifdef E2_VFSTMP
	//FIXME may need to check coding for each displayed dir, if that's possible
#endif
#ifndef E2_FILES_UTF8ONLY
//check whether a character encoding is compatible with utf
static gboolean _e2_fs_charset_is_utf (gchar *filecoding)
{
	return (!strcmp (filecoding, "UTF-8") ||
			strstr (filecoding, "ASCII") != NULL ||
			!strcmp (filecoding, "ANSI_X3.4-1968") ||
			!strcmp (filecoding, "646") ||
			!strcmp (filecoding, "ISO646") ||
			!strcmp (filecoding, "ISO_646.IRV"));
}
//check whether the locale language is one of the english variants
static gboolean _e2_fs_language_is_english (void)
{
	const gchar *lang = g_getenv ("LANGUAGE");
	if (lang == NULL)
		lang = g_getenv ("LANG");
	return (lang != NULL && strstr (lang, "en_") != NULL);
}
/**
@brief check whether the native filesystem encoding is utf-8 (or ascii), and setup for conversion accordingly

@return
*/
void e2_fs_check_coding (void)
{
	gchar *filecoding;
	app.utf8_filenames = FALSE;
	//1st, check what the user specified about encoding
	if (e2_cl_options.encoding != NULL)
		app.utf8_filenames = _e2_fs_charset_is_utf (e2_cl_options.encoding);
	//otherwise, try to find something from the environment ...
	else
	{
		//for glib, $G_FILENAME_ENCODING has priority over $G_BROKEN_FILENAMES
		filecoding = (gchar *) g_getenv ("G_FILENAME_ENCODING");
		if (filecoding == NULL)
		{
			//check what the locale data knows about the encoding
			filecoding = (gchar *) g_getenv ("G_BROKEN_FILENAMES");
			if (filecoding != NULL)
			{
				filecoding = nl_langinfo (CODESET);
				app.utf8_filenames = _e2_fs_charset_is_utf (filecoding);
			}
			else
				app.utf8_filenames = _e2_fs_language_is_english ();
		}
		else if (g_str_equal (filecoding, "@locale"))
		{
			const gchar *charset;
			app.utf8_filenames = g_get_charset (&charset)
			//we assume locales using english language have ascii filenames
			//(which are compatible with utf)
				|| _e2_fs_language_is_english ();
		}
		else
		{
		//FIXME maybe a list, check only its 1st member
			app.utf8_filenames = _e2_fs_charset_is_utf (filecoding)
				|| _e2_fs_language_is_english ();
		}
	}
	//set pointers to conversion functions
	if (app.utf8_filenames)
	{
		e2_display_from_locale = e2_fname_to_locale = e2_fname_from_locale
			= e2_utf8_not_converted;
 		e2_fname_dupto_locale = e2_fname_dupfrom_locale = g_strdup;
		e2_fname_free = e2_utf8_not_freed;
	}
	else
	{
		e2_display_from_locale = g_filename_display_name;
		e2_fname_to_locale = e2_fname_dupto_locale = e2_utf8_filename_to_locale;
		e2_fname_from_locale = e2_fname_dupfrom_locale = e2_utf8_filename_from_locale;
		e2_fname_free = g_free;
	}
}
#endif //ndef E2_FILES_UTF8ONLY

/**
@brief get the "work" dir for the filelist associated with @a view
@param view pointer to view data struct

@return newly-allocated path string, localised
*/
gchar *e2_fs_get_localdir (ViewInfo *view)
{
	gchar *localdir;
	if (view->dirtype == FS_LOCAL || view->dirtype == FS_FUSE)
	{
		localdir = D_FILENAME_TO_LOCALE (view->dir);
		*(localdir + strlen (localdir) - sizeof (gchar)) = '\0';	//strip ascii trailer
	}
	else
	{
#ifdef E2_VFS
#ifdef E2_VFSTMP
		FIXME
#else
		localdir = view->spacedata->workplace;	//FIXME BIGTIME
#endif
#else
		localdir = "/tmp";
#endif
	}
	return localdir;
}
/**
@brief check whether @a localpath exists
Can't use the function access(), which (for glibc at least) looks
through links
Errors other than non-readable are ignored
NOTE this does not work properly on FAT32 (etc??) FAT32 is case-insensitive
@param localpath item to test, absolute localised string

@return 0 (FALSE) if @a localpath exists (like access(F_OK)), else -1
*/
gint e2_fs_access2 (gchar *localpath E2_ERR_ARG())
{
	struct stat statbuf;
	return (e2_fs_lstat (localpath, &statbuf E2_ERR_SAMEARG()));
}
/**
@brief check whether @a localpath can be 'used' in a manner consistent with flags in @a howflags
This mimics the stdio function access(), except that it does not look through
links, as does access() (in glibc at least).
@param localpath localised name of file to test
@param howflags or'd combination of R_OK, W_OK, X_OK and/or F_OK

@return 0 (FALSE) if all @a howflags condition(s) are satisfied (like access()), else -1
*/
gint e2_fs_access3 (gchar *localpath, gint howflags E2_ERR_ARG())
{
	struct stat statbuf;
	if (e2_fs_lstat (localpath, &statbuf E2_ERR_SAMEARG()))
		return -1;
	else if (! S_ISLNK (statbuf.st_mode))
		return (e2_fs_access (localpath, howflags E2_ERR_SAMEARG()));

	static uid_t myuid = -1;
	static gid_t mygid = -1;

	if (myuid == -1)
		myuid = getuid ();
	if (mygid == -1)
		mygid = getgid ();
	gboolean result = FALSE;

	if (howflags & R_OK)
	{
		if (statbuf.st_mode & S_IROTH)
			result = TRUE;
		else if (statbuf.st_mode & S_IRUSR)
			result = ((myuid == 0) || (myuid == statbuf.st_uid));
		else if (statbuf.st_mode & S_IRGRP)
			result = ((myuid == 0) || e2_fs_ingroup (statbuf.st_gid));
		if (!result)
			return -1;
	}
	if (howflags & W_OK)
	{
		if (statbuf.st_mode & S_IWOTH)
			result = TRUE;
		else if (statbuf.st_mode & S_IWUSR)
			result = ((myuid == 0) || (myuid == statbuf.st_uid));
		else if (statbuf.st_mode & S_IWGRP)
			result = ((myuid == 0) || e2_fs_ingroup (statbuf.st_gid));
		if (!result)
			return -1;
	}
	if (howflags & X_OK)
	{
		if (statbuf.st_mode & S_IXOTH)
			result = TRUE;
		else if (statbuf.st_mode & S_IXUSR)
			result = ((myuid == 0) || (myuid == statbuf.st_uid));
		else if (statbuf.st_mode & S_IXGRP)
			result = ((myuid == 0) || e2_fs_ingroup (statbuf.st_gid));
		if (!result)
			return -1;
	}
	//we already know the link exists, no need for F_OK test
	return 0;
}
/**
@brief check whether the current user may modify @a localpath
This is more than just a simple X-flag check
This does not look through links (target may not exist)

@param localpath localised string with full path of item to check

@return TRUE if the user has write authority for @a localpath
*/
gboolean e2_fs_check_write_permission (gchar *localpath E2_ERR_ARG())
{
#ifdef E2_VFS
	if (!e2_fs_item_is_mounted (localpath, E2_ERR_NAME))
	{
# ifdef E2_VFSTMP
	//detect virtual items that are never writable by any user
	//e.g. a local archive without write/change functionality
	//FIXME handle archive where can write file but cannot chmod etc
# endif
		return FALSE;
	}
#endif
	if (!e2_fs_access3 (localpath, W_OK E2_ERR_SAMEARG()))
		return TRUE;	//write permission is actually set
	//otherwise, check whether we can change it anyway
	struct stat statbuf;
	if (e2_fs_lstat (localpath, &statbuf E2_ERR_SAMEARG()))
		return FALSE;
	else if (S_ISLNK (statbuf.st_mode))
		return FALSE;	//link permissions are thoroughly evaluated in access3(), no point in repeating
	//CHECKME which of these are actually checked in access() ??
	uid_t id = getuid ();
	if (id == 0 || id == statbuf.st_uid)
		return TRUE;
	//NB for GRP and OTH this does not work, for dirs at least
	else if ((statbuf.st_mode & S_IWGRP) //must already be group-writable for a member to change it
		&& e2_fs_ingroup (statbuf.st_gid))
		return TRUE;
	else if (statbuf.st_mode & S_IWOTH)//must already be other-writable for another to change it
		return TRUE;
	return FALSE;
}
/**
@brief determine whether the current user is a member of group whose id is @a gid
This is mainly to check whether to manipulate USR, GRP or OTH permissions
@param gid a group id
@return TRUE if current user is part of the group
*/
gboolean e2_fs_ingroup (gid_t gid)
{
	gid_t mygid = getgid ();
	if (mygid == gid)
		return TRUE;
	struct passwd *pw_buf;
	struct group *grp_buf;
	gboolean ingroup = FALSE;
	pw_buf = getpwuid (getuid ());
	if (pw_buf != NULL)
	{
		grp_buf = getgrgid (gid);
		if (grp_buf != NULL && grp_buf->gr_mem != NULL)
		{
			gchar *myname = g_strdup (pw_buf->pw_name);
			gint i = 0;
	        while (grp_buf->gr_mem [i] != NULL)
			{
				if (g_str_equal (grp_buf->gr_mem [i], myname))
				{
					ingroup = TRUE;
					break;
				}
				i++;
            }
			g_free (myname);
		}
	}
	return ingroup;
}
/**
@brief execute the file(1) command on @a localpath, and search each line of its output for @a string

Used in e2_fs_is_text and e2_fs_is_executable

@param localpath localised name of file to test
@param string ascii string for which to search, in the output from 'find'

@return TRUE if @a string is found
*/
static gboolean _e2_fs_grep_file_output (gchar *localpath, gchar *string E2_ERR_ARG())
{
#ifdef E2_VFS
	if (e2_fs_item_is_mounted (localpath, E2_ERR_NAME))
	{
#endif
		gchar line[PATH_MAX + 20];
		//CHECKME weirdness when > 1 opening of the same file,
		//avoided by setting the cwd before opening the pipe
		gchar *local = F_FILENAME_TO_LOCALE (curr_view->dir);
		if (e2_fs_chdir_local (local E2_ERR_NONE()))
		{
		// FIXME warn user
			F_FREE (local);
			return FALSE;
		}
		F_FREE (local);

		size_t bsize = E2_MAX_LEN;
#ifdef USE_GLIB2_10
		gchar *buf = (gchar *) g_slice_alloc ((gulong) bsize);
#else
		gchar *buf = (gchar *) g_try_malloc ((gulong) bsize);
#endif
		CHECKALLOCATEDWARN (buf, return FALSE;);

		//not g_strdup_printf or g_snprintf (in case they don't like localised text??)
		snprintf (line, sizeof (line), ("file \"%s\""), localpath);	//do not translate
		E2_FILE *pipe = e2_fs_open_pipe (line);
		if (pipe == NULL)
		{
#ifdef USE_GLIB2_10
			g_slice_free1 (bsize, buf);
#else
			g_free (buf);
#endif
			return FALSE;
		}
		gboolean retval = FALSE;
#ifdef __USE_GNU
		while (getdelim (&buf, &bsize, '\n', pipe) > 1)	//not an empty line
#else
		while (fgets (buf, bsize, pipe) != NULL && *buf != '\n')
#endif
		{
			if (strstr (buf, string))
			{
				retval = TRUE;
				break;
			}
		}
		e2_fs_pipe_close (pipe);
#ifdef USE_GLIB2_10
		g_slice_free1 (bsize, buf);
#else
		g_free (buf);
#endif
		return retval;
#ifdef E2_VFS
	}
	else	//item is virtual
	{
//		g_error_free (*E2_ERR_NAME);	//just in case?
# ifdef E2_VFSTMP
		//get information about virtual item, how ??
# endif
		return FALSE;
	}
#endif
}
/**
@brief test whether @a localpath is a text file

@param localpath localised name of file to test

@return TRUE if it is a text file
*/
gboolean e2_fs_is_text (gchar *localpath E2_ERR_ARG())
{
#ifdef E2_VFS
	if (e2_fs_item_is_mounted (localpath, E2_ERR_NAME))
	{
#endif
		return _e2_fs_grep_file_output (localpath, "text" E2_ERR_SAMEARG()); //do not translate
#ifdef E2_VFS
	}
	else	//item is virtual
	{
//		g_error_free (*E2_ERR_NAME);	//just in case?
# ifdef E2_VFSTMP
	//FIXME do this some other way for virtual items e.g. test mimetype
# endif
		return FALSE;
	}
#endif
}
/**
@brief test whether @a info belongs to an executable item
If the item is a symlink, its target will be tested
@param info pointer to a FileInfo data struct
@param view pointer to view data struct from which to get path info for @a info

@return TRUE if system reports x permission, and the file (1) command says it's executable
*/
gboolean e2_fs_is_executable (FileInfo *info, ViewInfo *view)
{
	gchar *dir = e2_fs_get_localdir (view);
	if (dir == NULL)
		return FALSE;
	gboolean retval = FALSE;
	gchar *localpath = g_build_filename (dir, info->filename, NULL);
	if (!e2_fs_access (localpath, X_OK E2_ERR_NONE()))
	{	//permission is ok
		if (S_ISLNK (info->statbuf.st_mode))
			retval = TRUE;
		else
		{
			//non-link, double check ...
#ifdef E2_VFS
			if (e2_fs_item_is_mounted (localpath, NULL))
			{
#endif
				retval = _e2_fs_grep_file_output (localpath, "executable" E2_ERR_NONE ());  //string not to be translated
#ifdef E2_VFS
			}
			else	//item is virtual
			{
//				g_error_free (*E2_ERR_NAME);	//just in case?
# ifdef E2_VFSTMP
			//FIXME full path for vfs item
			//FIXME do this some other way for virtual items e.g. test mimetype
# endif
				retval = FALSE;
			}
#endif
		}
	}
	g_free (localpath);
	g_free (dir);
	return retval;
}
/* *
@brief check whether @a path is a link

@param path localised path string

@return TRUE if @a info belongs to a directory or a link to a directory
*/
/*gboolean e2_fs_is_link (gchar *local_path)
{
	struct stat statbuf;
	return (!e2_fs_lstat (local_path, &statbuf E2_ERR_NONE()) && S_ISLNK (statbuf.st_mode));
} */
/**
@brief check whether item in active pane is a directory or a link to one
This assumes that the item's mode flags have been gathered using lstat(),
not stat() (i.e. no link look-through)
@param info ptr to FileInfo struct for the item
@param view pointer to view data struct from which to get path info for @a info

@return TRUE if @a info belongs to a directory or a link to a directory
*/
gboolean e2_fs_is_dir (FileInfo *info, ViewInfo *view)
{
	//CHECKME want to return any vfs error ?
	if ( S_ISDIR (info->statbuf.st_mode) )
		return TRUE;
	if ( S_ISLNK(info->statbuf.st_mode) )
	{
		gchar *dir = e2_fs_get_localdir (view);
		if (dir != NULL)
		{
			struct stat statbuf;
			gchar *localpath = g_build_filename (dir, info->filename, NULL);
			gboolean ok = !e2_fs_stat (localpath, &statbuf E2_ERR_NONE());
			g_free (localpath);
			g_free (dir);
			if (ok && S_ISDIR (statbuf.st_mode))
			  return TRUE;
		}
	}
	return FALSE;
}
/* *
@brief inline version of func that checks whether item in active pane is a directory or a link to one
NOW CONVERTED TO AN EQUIVALENT DEFINE
This assumes that the mode flags have been gathered using
lstat, not stat (ie no link look-through)

Item needs to be in active pane because no path is added to info->filename

@param info FileInfo structure
@param statbuf pointer to statbuf to use for statting @a info ->filename

@return TRUE if @a info belongs to a directory or a link to a directory
*/
/*inline gboolean e2_fs_is_dir_fast (FileInfo *info, struct stat *statbuf)
{
	if (S_ISDIR (info->statbuf.st_mode) )
		return TRUE;
	if (S_ISLNK(info->statbuf.st_mode) )
	{
FIXME absolute path for stat()
		if (!stat (info->filename, statbuf)	//look thru the link
			&& S_ISDIR (statbuf->st_mode))
		  return TRUE;
	}
	return FALSE;
} */
/**
@brief check whether @a localpath is a directory or a link to one

@param localpath localised absolute path string

@return TRUE if @a localpath is a directory or a link to a directory
*/
gboolean e2_fs_is_dir3 (gchar *localpath E2_ERR_ARG())
{
	struct stat statbuf;
	return (!e2_fs_stat (localpath, &statbuf E2_ERR_SAMEARG())
			&& S_ISDIR (statbuf.st_mode));
}
/**
@brief make directory @a localpath regardless of whether all ancestors exist
This only works for single-byte path-separator characters
@param localpath localised absolute path string
@param mode octal permissions of new dir e.g. 0777 or 0644

@return FALSE if @a creation succeeds (like mkdir)
*/
gboolean e2_fs_recurse_mkdir (gchar *localpath, gint mode E2_ERR_ARG())
{
	struct stat sb;
	gchar c;
	gchar *s = localpath;	//skips check on fs root dir
	while ((s = strchr (s+1, G_DIR_SEPARATOR)) != NULL)
	{
		c = *s;
		*s = '\0';
		if (e2_fs_stat (localpath, &sb E2_ERR_SAMEARG()))	//thru links, not e2_fs_, FIXME vfs
		{
			if (E2_ERR_PISNOT (ENOENT) || e2_fs_mkdir (localpath, mode E2_ERR_NONE()))	//FIXME vfs
			{
				*s = c;
#ifdef E2_VFS
				g_error_free (*E2_ERR_NAME);
#endif
				return TRUE;
			}
		}
		else if (! S_ISDIR (sb.st_mode))
		{
			*s = c;
			return TRUE;
		}
		*s = c;
	}
	//check last or only segment
	if (e2_fs_stat (localpath, &sb E2_ERR_SAMEARG()))	//thru links, not e2_fs_,
	{
		if (E2_ERR_PISNOT (ENOENT) || e2_fs_mkdir (localpath, mode E2_ERR_NONE()))	//FIXME vfs
		{
#ifdef E2_VFS
			g_error_free (*E2_ERR_NAME);
#endif
			return TRUE;
		}
	}
	return FALSE;
}
/**
@brief change system directory to @a utfpath, if possible

@a utfpath may be a link, in which case it will be traversed
Error message expects BGL to be closed
@param path utf-8 string with path to directory to be opened

@return TRUE if @a path is opened ok
*/
gboolean e2_fs_chdir (gchar *utfpath E2_ERR_ARG())
{
#ifdef E2_VFSTMP
	//what about incomplete mount of fuse-dir
#endif
	gboolean retval;
	gchar *local = F_FILENAME_TO_LOCALE (utfpath);
#ifdef E2_VFS
	GError *localerr = (E2_ERR_NAME == NULL) ? NULL : *E2_ERR_NAME ;
#endif
	if (e2_fs_chdir_local (local
#ifdef E2_VFS
		, &localerr
#endif
	))
	{	//go there failed
#ifdef E2_VFS
		e2_output_print_error (localerr->message, FALSE);
#else
		e2_output_print_strerrno ();
#endif
		retval = FALSE;
	}
	else
		retval = TRUE;

#ifdef E2_VFS
	if (E2_ERR_NAME == NULL)
		g_error_free (localerr);
#endif
	F_FREE (local);
	return retval;
}
/**
@brief check whether the user is able to change CWD to the directory @a utfpath

This tests @a utfpath for existence, is-a-directory, is-executable and readable
If @a utfpath is a link, it will be traversed
Error message expects BGL to be closed
@param utfpath utf-8 string with path of directory to be checked

@return TRUE if @a path is ok to cd to, or is not mounted-local
*/
gboolean e2_fs_cd_isok (gchar *utfpath E2_ERR_ARG())
{
	gchar *message;
#ifdef E2_VFSTMP
	//FIXME relevant path to open ?
#endif
	gchar *local = F_FILENAME_TO_LOCALE (utfpath);
	gboolean success = !e2_fs_access (local, F_OK E2_ERR_SAMEARG());
	if (success)
	{
		if (!e2_fs_is_dir3 (local E2_ERR_SAMEARG()))
		{
			message = g_strdup_printf (_("'%s' is not a directory"), utfpath);
			success = FALSE;
		}
		else if (e2_fs_access (local, R_OK | X_OK E2_ERR_SAMEARG()))
		{
			message = g_strdup_printf (_("Cannot access directory '%s' - No permission"), utfpath);
			success = FALSE;
		}
	}
	else if (e2_fs_dir_is_native (utfpath))
		message = g_strdup_printf (_("Directory '%s' does not exist"), utfpath);
	else
	{
#ifdef E2_VFS
		if (E2_ERR_NAME != NULL && *E2_ERR_NAME != NULL)
		{
			g_error_free (*E2_ERR_NAME);
			*E2_ERR_NAME = NULL;
		}
#endif
		success = TRUE;	//missing path ok when it's non-mounted
	}

	if (!success)
	{
		e2_output_print_error (message, TRUE);
	}
	F_FREE (local);
	return success;
}
/**
@brief check whether directory @a utfpath is valid, and optionally, accessible, and if not, pick a fallback
If @a utfpath refers to a link, it will be traversed.
If test(s) are not satisfied, @a path is set to a different path string,
pointing to a place up the tree branch that does pass the test(s)
If a different path string is returned, the old one is freed, of course

@param utfpath store for utf-8 path string which may be invalid, must be freeable
@param accessible TRUE to find a path that has X and R permissions, FALSE if don't care about that

@return TRUE if @a path meets the specified test(s), FALSE if a replacement has been provided
*/
gboolean e2_fs_get_valid_path (gchar **utfpath, gboolean accessible E2_ERR_ARG())
{
#ifdef E2_VFSTMP
	//FIXME relevant path to open ? walkup *utfpath, past its root if vdir,
	//revert to local space if vfs fails
	//caller will want a PlaceInfo * and a message about any change of place
#endif
	gchar *local = D_FILENAME_TO_LOCALE (*utfpath);
	gchar *freeme;
#ifdef E2_VFSTMP
	//what about incomplete mount of fuse-dir
#endif
	g_strchug (local);
	if (*local != '\0')
	{
		//CHECKME do a full interpretation here ?
		if (local[0] == '~') //home dir check
		{
			freeme = local;
			if (local[1] == '\0')
			{
				local = g_strdup (g_get_home_dir ());
				g_free (freeme);
			}
			else
				if (local[1] == G_DIR_SEPARATOR)
			{
				local = g_strconcat (g_get_home_dir (), local + 1, NULL);
				g_free (freeme);
			}
		}

		if (e2_fs_is_dir3 (local E2_ERR_NONE())
			&& (!accessible || !e2_fs_access (local, R_OK | X_OK E2_ERR_NONE())))
		{
			g_free (local);
			return TRUE;
		}
	}
	//walk back up the tree branch until we find an acceptable dir
	gchar *p;
	while ((p = strrchr (local, G_DIR_SEPARATOR)) != NULL)
	{
		*p = '\0';	//CHECKME ok to change local (maybe == *path) ??
#ifdef E2_VFSTMP
	//FIXME new path may be virtual item eg archive ?
#endif
		if (e2_fs_is_dir3 (local E2_ERR_NONE())
			&& (!accessible || !e2_fs_access (local, R_OK | X_OK E2_ERR_NONE())))
		{	//replace the invalid path string with the valid one
			p = D_FILENAME_FROM_LOCALE (local);	//copy in case not changed by macro
			g_free (*utfpath);
			*utfpath = p;
			g_free (local);
			return FALSE;
		}
	}
	//whole branch bad, revert to default
#ifdef E2_VFSTMP
	//FIXME relevant default path to open ?
#endif
//	p = g_strdup (G_DIR_SEPARATOR_S);
	p = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, NULL);
	g_free (*utfpath);
	*utfpath = D_FILENAME_FROM_LOCALE (p);	//copy in case not changed by macro
	g_free (p);
	g_free (local);
	return FALSE;
}
/**
@brief complete a directory string in @a entry after a keypress @a pressed_char
This works only for native directories
@param entry the entry widget to be updated
@param pressed_char utf8-string form of pressed key which triggered the completion
@param pane enumerator of pane to use for default dir, 1,2,0=current

@return TRUE if completion was performed
*/
static gboolean _e2_fs_complete_dir (GtkWidget *entry, gchar *pressed_char, guint pane)
{
	gboolean retval;
	//start is characters, not bytes
	gint start = gtk_editable_get_position (GTK_EDITABLE (entry));
	gint pos = start + 1;	//+1 to include the key just pressed
	gchar *entrytext = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, start);
	gchar *text = g_strconcat (entrytext, pressed_char, NULL);
	gchar *utfdir;
	if (g_path_is_absolute (text))
		utfdir = g_path_get_dirname (text);
	else
	{
		switch (pane)
		{
			case E2PANE1:
				utfdir = app.pane1_view.dir;
				break;
			case E2PANE2:
				utfdir = app.pane2_view.dir;
				break;
//			case E2PANECUR:
			default:
				utfdir = curr_view->dir;
				break;
		}
		gchar *freeme = g_strconcat (utfdir, text, NULL);
		utfdir = g_path_get_dirname (freeme);
		g_free (freeme);
	}
#ifdef E2_VFSTMP
	//CHECKME enable vfs completion ??
#endif
	if (e2_fs_dir_is_native (utfdir))
	{
		GList *found = NULL;
		retval = (e2_complete_str (&text, &pos, &found, E2_COMPLETE_FLAG_DIRS, pane) > 0);
		if (retval)
		{
			g_free (entrytext);
			entrytext = gtk_editable_get_chars (GTK_EDITABLE (entry), start, -1);
			gchar *newtext = g_strconcat (text, entrytext, NULL);
			gtk_entry_set_text (GTK_ENTRY (entry), newtext);
			gtk_editable_set_position (GTK_EDITABLE (entry), pos);
			g_free (newtext);
			e2_list_free_with_data (&found);
		}
	}
	else
		retval = FALSE;

	g_free (utfdir);
	g_free (entrytext);
	g_free (text);
	return retval;
}
/**
@brief auto-complete a directory path string in @a entry after a keypress @a keyval
Completion is performed (or not) according to the relevant config option
This does not check for a valid key (non-modifier < 0xF000, >0xFFFF) - that
should be done before calling here
@param entry the entry widget to be updated
@param keyval code of pressed key which triggered the completion
@param pane enumerator of pane to use for default dir, 1,2,0=current

@return TRUE if completion was performed
*/
gboolean e2_fs_complete_dir (GtkWidget *entry, guint keyval, guint pane)
{
	guint32 unikey = gdk_keyval_to_unicode (keyval);
	if (unikey > 0)
	{
		gchar keystring[8];
		gint bytes = g_unichar_to_utf8 (unikey, keystring);
		*(keystring+bytes) = '\0';
		gint type = e2_option_sel_get ("dir-line-completion");
		switch (type)
		{
			case 1:	//insert completed text, and move after it
				if (_e2_fs_complete_dir (entry, keystring, pane))
					return TRUE;
				break;
			case 2:	//select completed text
			{
				gint start, end, newpos, oldpos;
				if (gtk_editable_get_selection_bounds (GTK_EDITABLE (entry), &start, &end))
				{
					//when there's a selection, cursor position = end, fix that
					oldpos = start;
					gtk_editable_delete_text (GTK_EDITABLE (entry), start, end);
				}
				else
					oldpos = gtk_editable_get_position (GTK_EDITABLE (entry));

				if (_e2_fs_complete_dir (entry, keystring, pane))
				{
					newpos = gtk_editable_get_position (GTK_EDITABLE (entry));
					gtk_editable_select_region (GTK_EDITABLE (entry), oldpos+1, newpos);
					return TRUE;
				}
			}
				break;
			default:	//no auto-completion
				break;
		}
	}
	return FALSE;
}
/**
@brief evaluate whether @a utfpath is a native dir
Native means local and mounted, not fuse or other vfs
This is for tailoring pre-change-dir and pre-refresh-dir checking
@param utfpath absolute path of dir to check, utf-8 string

@return TRUE if @a utfpath is or seems to be native
*/
gboolean e2_fs_dir_is_native (gchar *utfpath)
{
#if defined(__linux__) || defined(__FreeBSD__)
	GList *fusemounts, *member;
	fusemounts = e2_fs_mount_get_fusemounts_list ();
	for (member = fusemounts ; member != NULL; member = member->next)
	{
		//member->data is utf with no trailing "/"
		if (g_str_has_prefix (utfpath, (gchar *)member->data))
			break;	//path is a descendant of a fuse mountpoint
	}
	if (fusemounts != NULL)
		e2_list_free_with_data (&fusemounts);
	if (member != NULL)
		return FALSE;
#endif
#ifdef MNT_LOCAL
	gint result;
	gchar *localpath;
	struct statfs fstatus;

	localpath = F_FILENAME_TO_LOCALE (utfpath);
	result = statfs (localpath, &fstatus);
	F_FREE (localpath);
	if (result)
	{
		if (result == ESTALE)
		{
			printd (DEBUG, "%s resides in an unmounted VFS", utfpath);
			//CHECKME mount it ?
		}
		return TRUE;	//any operation on utfpath will fail ASAP
	}
	if (!(fstatus.f_flags & MNT_LOCAL))
		return FALSE;
#else
	struct statvfs fstatus;
	gchar *localpath;
	localpath = F_FILENAME_TO_LOCALE (utfpath);
retry:
	if (statvfs (localpath, &fstatus))
	{
		if (errno == EINTR)
			goto retry;
//		if (errno == ENOLINK)
//			re-mount it ??
		F_FREE (localpath);
		return FALSE;
	}
	F_FREE (localpath);
#endif
	//FIXME MORE TESTS HERE e.g.
	//from string contents e.g. "//:"
	//from plugin data
	//from vfs history data
	//from other e2 data
	return TRUE;
}
#ifdef E2_VFS
/**
@brief thread-function to read all of a virtual directory
@a view ->dir has the target path
To eliminate BGL-racing, no UI-change from here
@param view ptr to data struct for the view to be updated

@return list of FileInfo's for entries (maybe NULL), or error code if a problem occurred
*/
static gpointer _e2_fs_read_virtual_dir (E2_DRFuncArgs *data)
{
	e2_utils_block_thread_signals ();	//block all allowed signals to this thread

	gint oldtype = 0;
	pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, &oldtype);

	gchar *itemname;
	gboolean (*processor) (gchar *, gchar *, GList **, gpointer) = data->callback;
	GList *member, *entries, *wanted;
	//this will leak list data and cb_data if that's supposed to be cleared here
	pthread_cleanup_push ((gpointer)g_list_free, (gpointer)entries);
//	E2_BLOCK	//don't want this interrupted

#ifdef E2_VFSTMP
	THIS NEEDS WORK
	//populate list using vfs read dir function
	entries = (GList *)e2_gvfs_dir_foreach (data->localpath, data->callback,
		data->cb_data);
#else
	entries = NULL;
#endif
	if (!E2DREAD_FAILED(entries))
	{
		wanted = NULL;
		for (member = entries; member != NULL; member = member->next)
		{
			itemname = (gchar *)member->data;
			//one item we're not interested in
			if (itemname[0] != '.' || itemname[1] != '\0')
			{
				if (processor != NULL)
				{
					if (!processor (data->localpath, itemname, &entries, data->cb_data))
						break;
				}
				else
					wanted = g_list_append (entries, g_strdup (itemname));
			}
		}
	}
//	E2_UNBLOCK
	e2_list_free_with_data (&entries);

	if (data->free_data_func != NULL)
		data->free_data_func (data->cb_data);

	pthread_cleanup_pop (0);	//free entries list
	pthread_setcanceltype (oldtype, NULL);

	return wanted;
}
#endif
/**
@brief synchronous or thread-function to read all of a mounted directory
This is called synchronously when @a view ->fstype is FS_LOCAL, or as a thread
when @a view ->fstype is FS_FUSE
To eliminate BGL-racing, no UI-change from here
@a view ->dir has the target path
@param view ptr to data struct for the view to be updated

@return list of FileInfo's for entries (maybe NULL), or error code if a problem occurred
*/
static gpointer _e2_fs_read_mounted_dir (E2_DRFuncArgs *data)
{
/* FIXME ASYNC CANCELLATION OR TEXT-CANCELS. CANCELLATION CLEANUPS
	if (1)	//debugging
	{
		pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
		pthread_t ID = pthread_self ();
		while (TRUE)
		{
			sleep (10);
			printd (DEBUG,"dir-read-thread (ID=%lu) loop continues", (gulong) ID);
		}
	}
*/

/*	would be nice to prevent access-time updates purely due to
	refreshing, but the atime really does change as a result
	of the refresh process, and if we were to revert it, the
	ctime would be changed instead
	times are immediately shown in the other pane if it's the
	parent of this one
	if we could suspend monitoring of such parent dir, we
	must not lose any prior reports for there
	QUERY can the changed atime itself trigger a report which
	causes a subsequent refresh, creating an endless cycle?
	especially with multiple threads and/or processors ?
	Assuming it can do so, we clear the reports 'queue'
	after the dir is opened
	Would be nice if we could just clear any report(s) due to
	the refresh per se - but the reporting is not that detailed */
/*#ifdef E2_FAM_KERNEL
	if (view->refresh)
		e2_fs_FAM_cancel_monitor_dir (view->dir);
#endif */

/*#if defined (E2_FAM_DNOTIFY)
	//cut down on spurious context changes
	//block DNOTIFY_SIGNAL
	sigprocmask (SIG_BLOCK, &dnotify_signal_set, NULL);
#endif */

	gboolean threaded = (data->dirtype == FS_FUSE);	//monitoring needed, so this func is a thread
	if (threaded)
		e2_utils_block_thread_signals ();	//block all allowed signals to this thread

	DIR *dp = opendir (data->localpath);
	if (dp == NULL)
	{
		printd (WARN, "Unable to open directory: %s", data->localpath);
		return GINT_TO_POINTER (E2DREAD_DNR);
	}

	gint oldtype = 0;
	if (threaded)
	{
		pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, &oldtype);
//		pthread_cleanup_push ((gpointer)closedir, (gpointer)dp); can't be conditional or inside braces
	}

	struct dirent entry;
	struct dirent *entryptr;
	gboolean (*processor) (gchar *, gchar *, GList **, gpointer) = data->callback;
	GList *entries = NULL;
//	if (threaded)
//		pthread_cleanup_push ((gpointer)g_list_free, (gpointer)entries);	//this will leak list data
//	E2_BLOCK	//don't want this interrupted
	while (!readdir_r (dp, &entry, &entryptr) && entryptr != NULL)
	{
		//one item we're not interested in
		if (entry.d_name[0] != '.' || entry.d_name[1] != '\0')
		{
			if (processor != NULL)
			{
				if (!processor (data->localpath, entry.d_name, &entries, data->cb_data))
					break;
			}
			else
				entries = g_list_append (entries, g_strdup (entry.d_name));
		}
	}
	closedir (dp);
//	E2_UNBLOCK

	if (data->free_data_func != NULL)
		data->free_data_func (data->cb_data);

	if (threaded)	//this func is a thread
	{
//		pthread_cleanup_pop (0);	//free entries list
//		pthread_cleanup_pop (0);	//close dp
		pthread_setcanceltype (oldtype, NULL);
	}

	return entries;
}
/**
@brief callback for responses from too-slow-dialog @a dialog
This approach eliminates gtk_main() (which hates being aborted) from the dialog
@param dialog the dialog from which the response was initiated
@param response the response enumerator
@param data task-data specified when the callback was connected
@return
*/
static void _e2_fs_slowread_response_cb (GtkDialog *dialog, gint response,
	E2_DRead *data)
{
	pthread_t ID;
	GtkWidget *wid;
	switch (response)
	{
		case GTK_RESPONSE_NO:	//abort the operation
			ID = data->aid;
			data->aid = 0;
			if (ID > 0)	//operation still not finished
			{
				pthread_cancel (ID); //shutdown action thread
				printd (DEBUG,"dir read thread aborted by user");
				//after this cancellation, the main thread will cleanup
			}
			break;
		case E2_RESPONSE_YESTOALL:	//no more reminders
//CHECKME pthread_mutex_lock (&?_mutex);
			wid = data->dialog; 	//race-management
			data->dialog = NULL;
			if (wid != NULL && GTK_IS_WIDGET (wid))
				gtk_widget_destroy (wid);	//== (GTK_WIDGET(dialog)
//			pthread_mutex_unlock (&?_mutex);
			//if the racy action-thread ends about now, signal to main
			//thread that it doesn't need to abort the monitor thread
			ID = data->mid;
			data->mid = 0;
			if (ID > 0) //race-management
				pthread_cancel (ID);
			break;
		default:
//		case GTK_RESPONSE_YES:	//keep waiting
//			pthread_mutex_lock (&?_mutex);
			wid = data->dialog; //race-management
			data->dialog = NULL;
			if (wid != NULL)
			{
				gtk_widget_hide (wid);
				data->dialog = wid;
//				WAIT_FOR_EVENTS;	//make the hide work
			}
//			pthread_mutex_unlock (&?_mutex);
			break;
	}
}
/**
@brief thread function to manage timeout when reading a directory
Assumes BGL is open, and @a data ->dialog is NULLED before 1st use
@param data pointer to data struct with thread id's etc

@return never happens - this must be cancelled by some other thread
*/
static gpointer _e2_fs_progress_monitor (E2_DRead *data)
{
	struct timespec timeout; //data for timeout value for the wait function

	e2_utils_block_thread_signals ();	//block all allowed signals to this thread

	//these are not used externally, but are needed for the wait
	pthread_mutex_t condition_mutex = PTHREAD_MUTEX_INITIALIZER;	//no recurse
	pthread_cond_t wait_cond = PTHREAD_COND_INITIALIZER;

	gint secs = 10;	//initial wait = 10 secs

	while (TRUE)
	{
//		sleep (secs);	//CHECKME more effective to use cond wait with timeout ?
		pthread_mutex_lock (&condition_mutex);
		clock_gettime (CLOCK_REALTIME, &timeout);
		timeout.tv_sec += secs;
		pthread_cond_timedwait (&wait_cond, &condition_mutex, &timeout);
		pthread_mutex_unlock (&condition_mutex);
		printd (DEBUG,"dir-monitor-thread (ID=%lu) %d-sec sleep ended", (gulong) data->mid, secs);

		gboolean shown = (data->dialog != NULL && GTK_WIDGET_VISIBLE (data->dialog));
		if (data->dialog == NULL) //once-only, create the dialog
			data->dialog = e2_dialog_slow (_("Reading directory data"),
				_("directory read"), _e2_fs_slowread_response_cb, data);

		if (data->dialog != NULL && !GTK_WIDGET_VISIBLE (data->dialog))
		{
			gdk_threads_enter ();
			gtk_widget_show (data->dialog);
			gtk_window_present (GTK_WINDOW (data->dialog));
			WAIT_FOR_EVENTS;
			gdk_threads_leave ();
			printd (DEBUG,"dir-monitor-thread dialog presented");
		}

		//initially, progressively lengthen interval between dialog popups
		if (!shown //dialog was not shown already when this loop was traversed
				&& secs < 30)
			secs += 10;
	}
	//never get to here
	return NULL;
}
/**
@brief read into memory the contents of the directory associated with @a view
CWD is temporarily changed to that directory, can be for the inactive pane.
Threads cannot assume that CWD always matches curr_view->dir.
The directory timestamp is upated, for refresh-management purposes
If the read is lengthy, a downstream dialog will be shown. That locks gdk threads mutex ?
This assumes BGL is closed
@param localpath absolute path of dir to process, localised string FIXME
@param dirtype code for type of dir being processed
@param callback function to call for each processed item, or NULL to append item's name to list
@param cb_data pointer to data to send to @a callback
@param free_data_func function to call at end of scan, to clean @a cb_data, or NULL

@return list of FileInfo's for entries (maybe NULL), or error code if a problem occurred
*/
gpointer e2_fs_dir_foreach (gchar *localpath, E2_FSType dirtype, gpointer callback,
	gpointer cb_data, GDestroyNotify free_data_func E2_ERR_ARG())
{
//	mounted = FALSE;	//for debugging

	E2_DRFuncArgs args = { localpath, dirtype, callback, cb_data, free_data_func };
	if (dirtype == FS_LOCAL)	//no timeout-monitoring needed
//		return _e2_fs_read_mounted_dir (view);
		return (_e2_fs_read_mounted_dir (&args));

	//FUSE dirs use same func as native dirs, but with timeout checking
#ifdef E2_VFS
	gpointer readfunc = (dirtype == FS_FUSE) ?
		_e2_fs_read_mounted_dir : _e2_fs_read_virtual_dir ;
#else
	gpointer readfunc = _e2_fs_read_mounted_dir;
#endif
	//create joinable read thread
	//we don't use glib thread funcs, as we need to kill thread(s)
	pthread_t ID;
//	if (!pthread_create (&ID, NULL, readfunc, view))
	if (!pthread_create (&ID, NULL, readfunc, &args))
		printd (DEBUG,"read-dir-thread (ID=%lu) started", ID);
	else
	{
		//FIXME message to user
		printd (WARN,"read-dir-thread create error!");
		printd (DEBUG,"exiting function e2_fs_dir_foreach () with error result");
		return GINT_TO_POINTER (E2DREAD_ENOTH);
	}

	//FIXME use a E2_DRead already produced elsewhere
	//or since this thread joins the others, it can be stacked ...
//	E2_DRead *data = ALLOCATE0 (E2_DRead);
//	CHECKALLOCATEDWARN (data, return (GINT_TO_POINTER (E2DREAD_ENOMEM));)
//	data->aid = ID;

	//no BGL-racing because read-dir funcs do not do any GUI stuff
	gdk_threads_leave ();
	e2_window_show_status_message (_("Reading directory data"));
	gdk_threads_enter ();
	//make status message show, at session start at least
	//(must be inside BGL, dunno why)
	WAIT_FOR_EVENTS;
	gdk_threads_leave ();	//enable show of dialog, if that's needed

	//create monitor thread
	//redundant if read-thread has finished already, but then it gets killed
	//immediately, anyhow
	E2_DRead data = { ID, 0, NULL };
	pthread_attr_t attr;
	pthread_attr_init (&attr);
	pthread_attr_setdetachstate (&attr, PTHREAD_CREATE_DETACHED);
	if (pthread_create (&data.mid, &attr,
		(gpointer) _e2_fs_progress_monitor, &data) == 0)
			printd (DEBUG,"dir-monitor-thread (ID=%lu) started", data.mid);
	else
	{
		//FIXME message to user with BGL management
		printd (WARN,"read-dir-thread-create error!");
	}
	//block until read is finished or cancelled
	printd (DEBUG,"main thread blocking until read ends or times out");
	gpointer entries;
	pthread_join (ID, &entries);

	gdk_threads_enter ();
	//CHECKME BGL-racing with too-slow dialog in other thread?
	e2_window_clear_status_message ();	//no BGL management neeeded
	WAIT_FOR_EVENTS;
	//some race-minimisation here
	GtkWidget *wid = data.dialog;
	data.dialog = NULL;
	if (wid != NULL && GTK_IS_WIDGET (wid))
		gtk_widget_destroy (wid);
	ID = data.mid;
	data.mid = 0;
	if (ID > 0)
		pthread_cancel (ID);
	printd (DEBUG,"exiting function e2_fs_dir_foreach () with read result");
//	DEALLOCATE (E2_DRead, data);
	printd (DEBUG,"dir read data cleaned up in main thread");
	if (entries == PTHREAD_CANCELED)
		entries = GINT_TO_POINTER (E2DREAD_DNF);
	return entries;
}
/**
@brief get real path for @a local_path if it's a link

@param local_path store for pointer to freeable, absolute, localised, path string which may be a link name

@return TRUE if a newly-allocated replacement path is provided in @a local_path
*/
gboolean e2_fs_walk_link (gchar **local_path E2_ERR_ARG())
{
	struct stat sb;
	if (!e2_fs_lstat (*local_path, &sb E2_ERR_SAMEARG()) && S_ISLNK (sb.st_mode))
	{
#ifdef E2_VFS
		if (e2_fs_item_is_mounted (*local_path, E2_ERR_NAME))
		{
#endif

#ifdef __USE_GNU
			gchar *resolved_path = canonicalize_file_name (*local_path);
			if (resolved_path != NULL)
			{
				g_free (*local_path);
				*local_path = resolved_path;
				return TRUE;
			}
#else
			gchar *resolved_path = g_try_malloc (PATH_MAX);
			CHECKALLOCATEDWARN (resolved_path, );	//need BGL?
			if (resolved_path != NULL)
			{
				if (realpath (*local_path, resolved_path) != NULL)	//buffer o/flow risk
				{
					g_free (*local_path);
					*local_path = g_realloc (resolved_path, strlen (resolved_path) + 1); //making smaller, no error test
					return TRUE;
				}
				else
					g_free (resolved_path);
			}
#endif
#ifdef E2_VFS
		}
		else	//item is virtual
		{
//			g_error_free (*E2_ERR_NAME);	//just in case?
	# ifdef E2_VFSTMP
			//FIXME vfs if relevant, get information about virtual item, how ??
	# endif
			return FALSE;
		}
#endif
	}
	return FALSE;
}
/**
@brief stat item @a localpath, put results into @a buf

This looks through links, but does not fail for hanging links

@param localpath relative or absolute path of item to stat, localised string
@param buf pointer to buffer to hold the results

@return 0 (FALSE) if the stat succeeded, else -1 (TRUE)
*/
gint e2_fs_stat (const gchar *localpath, struct stat *buf E2_ERR_ARG())
{
#ifdef E2_VFS
# ifdef E2_VFSTMP
	//what about incomplete mount of fuse-dir
# endif
	GError *localerr = (E2_ERR_NAME == NULL) ? NULL : *E2_ERR_NAME ;
	if (e2_fs_item_is_mounted (localpath, E2_ERR_NAME))
	{
#endif
		if (stat (localpath, buf))
		{	//stat failed, but it could be just due to a hanging symlink
			struct stat statbuf;	//don't disturb the caller's buffer
			if (!e2_fs_lstat (localpath, &statbuf
#ifdef E2_VFS
				, &localerr
#endif
			)	//stat link succeeded
				//CHECKME more checks for hanging link?
				&& (S_ISLNK (statbuf.st_mode) &&
#ifdef E2_VFS
				localerr->code == ENOENT
#else
				E2_ERR_PIS (ENOENT)
#endif
			))
			{
#ifdef E2_VFSTMP
				g_error_free (localerr);	//clear whatever error
#endif
				*buf = statbuf;
				return 0;
			}
			return -1;
		}
		return 0;
#ifdef E2_VFS
	}
	else	//item is virtual
	{
# ifdef E2_VFSTMP
		//get information about virtual item, how ??
# endif
		return -1;
	}
#endif
}
/**
@brief tolerantly open file @a localpath and return its descriptor
File descriptors are only relevant for local files, so no vfs treatment
is needed
@param localpath absolute path of file to open, localised string
@param openflags bitflags to provide to open()
@param mode mode used by open() when creating a file

@return descriptor of opened file, -1 on error
*/
gint e2_fs_safeopen (const gchar *localpath, gint openflags, gint mode)
{
	gint res;
	do res = open (localpath, openflags, mode);
		while (res == -1 && errno == EINTR);
	return res;
}
/**
@brief open file @a filepath for writing
Error message expects BGL to be closed
@param utfpath path of file to open, utf8 string

@return pointer to data struct for opened file, or NULL if the open failed
*/
E2_FILE *e2_fs_open_file_to_write (gchar *utfpath E2_ERR_ARG())
{
	E2_FILE *f;
	gchar *localpath = F_FILENAME_TO_LOCALE (utfpath);
#ifdef E2_VFS
	if (e2_fs_item_is_mounted (localpath, E2_ERR_NAME))
	{
#endif
		f = e2_fs_open_stream (localpath, "w");

		if (f == NULL)
		{
			gchar *msg = g_strdup_printf (_("Cannot open '%s' for writing - %s"),
				utfpath, g_strerror (errno));
			e2_output_print_error (msg, TRUE);
		}
#ifdef E2_VFS
	}
	else	//non-native open
	{
# ifdef E2_VFSTMP
		//FIXME
		f = NULL;
# endif
	}
#endif
	F_FREE (localpath);
	return f;
}
/**
@brief write @a string to file @a utfpath
Error message expects BGL to be closed
@param utfpath file to be written, utf8 string, for error reporting
@param string NULL-terminated string to be written
@param stream handle for file

@return EOF if a write error occurs, otherwise a non-negative value
*/
gint e2_fs_file_put (gchar *utfpath, const gchar *string, E2_FILE *stream
	E2_ERR_ARG())
{
	gint retval;
#ifdef E2_VFS
	gchar *localpath = F_FILENAME_TO_LOCALE (utfpath);
	if (e2_fs_item_is_mounted ((const gchar *)localpath, E2_ERR_NAME))
	{
#endif
		retval = fputs (string, stream);	//local only
		if (retval == EOF)
		{
#ifdef E2_VFS
			GError *localerr = (E2_ERR_NAME == NULL) ? NULL : *E2_ERR_NAME;
			e2_fs_set_error_from_errno (&localerr);	//apply relevant error data
#endif
			gchar *msg = g_strdup_printf (_("Error writing file '%s' - %s"),
				utfpath,
#ifdef E2_VFS
				localerr->message
#else
				g_strerror (errno)
#endif
				);
			e2_output_print_error (msg, TRUE);
#ifdef E2_VFS
			if (E2_ERR_NAME == NULL)
				g_error_free (localerr);
#endif
		}
#ifdef E2_VFS
	}
	else	//non-native write
	{
# ifdef E2_VFSTMP
		//FIXME
		retval = EOF;
# endif
	}
#endif
	return retval;
}
/**
@brief read file contents into memory

@param utfpath path of file to be read, utf8 string
@param contents ptr to where the contents are to be stored

@return TRUE if @a filepath was read successfully
*/
gboolean e2_fs_get_file_contents (gchar *utfpath, gchar **contents E2_ERR_ARG())
{
	gboolean retval;
	gchar *local = F_FILENAME_TO_LOCALE (utfpath);
#ifdef E2_VFS
	if (e2_fs_item_is_mounted (local, E2_ERR_NAME))
	{
#endif
		retval = g_file_get_contents (local, contents, NULL, NULL);
#ifdef E2_VFS
	}
	else	//item is virtual
	{
//		g_error_free (*E2_ERR_NAME);	//just in case?
# ifdef E2_VFSTMP
		//get information about virtual item, how ??
# endif
		retval = FALSE;
	}
#endif
	F_FREE (local);
	return retval;
}
/**
@brief tolerantly close file descriptor
File descriptors are only relevant for local files, so no vfs treatment
is needed
@param file_desc file descriptor

@return 0 on success, -1 on error
*/
gint e2_fs_safeclose (gint file_desc)
{
	gint res;
	do res = (gint) close (file_desc);
		while (res == -1 && errno == EINTR);
	return res;
}
/**
@brief perform a native, blockwise, file copy
Error messages assume BGL is open
@param src localised string, absolute path of item to copy
@param src_sb pointer to valid struct stat for @a src
@param dest localised string, absolute path of copy destination

@return TRUE if the copy was completed successfully
*/
gboolean e2_fs_copy_file (const gchar *src, const struct stat *src_sb,
	const gchar *dest E2_ERR_ARG())
{
#ifdef E2_VFS
	if (e2_fs_item_is_mounted (src, E2_ERR_NAME) && e2_fs_item_is_mounted (dest, E2_ERR_NAME))
	{
#endif
//		struct stat src_sb;
		struct stat dest_sb;

		//CHECKME vfs e2_fs_open|close ever needed for closing while native copying ?

		//O_NOATIME is GNU-specific, != 'cp', and requires write-permission on item
		//O_NOFOLLOW causes spurious error
		gint src_desc = TEMP_FAILURE_RETRY (open (src, O_RDONLY));
		if (src_desc < 0)
		{
#ifdef E2_VFS
			GError *localerr = (E2_ERR_NAME == NULL) ? NULL : *E2_ERR_NAME ;
			e2_fs_set_error_from_errno (&localerr);
#endif
			return (e2_fs_error_local (_("Cannot open '%s' for reading"), (gchar *)src
#ifdef E2_VFS
				, localerr->message
#endif
				));
		}
		mode_t dest_mode = (src_sb->st_mode | S_IWUSR) & ALLPERMS;
		gint dest_desc =
		TEMP_FAILURE_RETRY (open (dest, O_WRONLY | O_CREAT | O_EXCL, dest_mode));
		if (dest_desc < 0)
		{
#ifdef E2_VFS
			GError *localerr = (E2_ERR_NAME == NULL) ? NULL : *E2_ERR_NAME ;
			e2_fs_set_error_from_errno (&localerr);
#endif
			e2_fs_error_local (_("Cannot create file %s"), (gchar *)dest
#ifdef E2_VFS
				, localerr->message
#endif
				);
			TEMP_FAILURE_RETRY (close (src_desc));
			return FALSE;
		}
#ifdef E2_VFS
		GError *localerr = (E2_ERR_NAME == NULL) ? NULL : *E2_ERR_NAME ;
#endif
		if (e2_fs_fstat (dest_desc, &dest_sb))
		{
#ifdef E2_VFS
			e2_fs_set_error_from_errno (&localerr);
#endif
			e2_fs_error_local (_("Cannot get information about %s"), (gchar *)dest
#ifdef E2_VFS
			, localerr->message
#endif
			);
#ifdef E2_VFS
			if (E2_ERR_NAME == NULL)
				g_error_free (localerr);
#endif
			TEMP_FAILURE_RETRY (close (src_desc));
			TEMP_FAILURE_RETRY (close (dest_desc));
			return FALSE;
		}
	//	use blocks the same size as dest block (like cp)
	//		blksize_t buf_size = dest_sb.st_blksize;  //or src ??
	//		gchar *buf = g_alloca (buf_size);
		//find a buffer size up to 1 MB (!= the 'cp' approach)
	//		blksize_t buf_size = 1048576;
		//find a buffer up to 16 times dest block
		//(compromise between accesses and multi-tasking latency)
		blksize_t buf_size = dest_sb.st_blksize * 16;
		size_t src_size = src_sb->st_size * 2;
		while (buf_size > src_size && buf_size > dest_sb.st_blksize)
			buf_size /= 2;
		gpointer buf;
#ifdef USE_GLIB2_10
		while ((buf = g_slice_alloc (buf_size)) == NULL)
#else
		while ((buf = g_try_malloc (buf_size)) == NULL)
#endif
		{
			if (buf_size < dest_sb.st_blksize)
			{
				gdk_threads_enter ();
				e2_utils_show_memory_message ();
				gdk_threads_leave ();
				TEMP_FAILURE_RETRY (close (src_desc));
				TEMP_FAILURE_RETRY (close (dest_desc));
				return FALSE;
			}
			buf_size /= 2;
		}
		//CHECKME suspend access-monitoring here ...
		gboolean retval = TRUE;
		while (1)
		{
			ssize_t n_read = TEMP_FAILURE_RETRY (read (src_desc, buf, buf_size));	//(potential cancellation point, LEAKS)
			if (n_read == 0)
				break;

			if (n_read < 0)
			{
#ifdef E2_VFS
				GError *localerr = (E2_ERR_NAME == NULL) ? NULL : *E2_ERR_NAME ;
				e2_fs_set_error_from_errno (&localerr);
#endif
				e2_fs_error_local (_("Error reading file %s"), (gchar *)src
#ifdef E2_VFS
					, localerr->message
#endif
					);
				retval = FALSE;
				break;
			}

			ssize_t n_write = 0;
			while (n_write < n_read)
			{
				n_write = TEMP_FAILURE_RETRY (write (dest_desc, buf, n_read));	//(potential cancellation point, LEAKS)
				if (n_write < 0)
				{
#ifdef E2_VFS
					e2_fs_set_error_from_errno (E2_ERR_NAME);
#endif
					e2_fs_error_local (_("Error writing file %s"), (gchar *)dest E2_ERR_MSGC());
					retval = FALSE;
					break;
				}
			}
			if (!retval)
				break;
//			pthread_testcancel ();	//swap threads, cancel if instructed (leaks if so)
		}
		TEMP_FAILURE_RETRY (close (src_desc));
		TEMP_FAILURE_RETRY (close (dest_desc));
		if (!retval)
			unlink (dest);
#ifdef USE_GLIB2_10
		g_slice_free1 (buf_size, buf);
#else
		g_free (buf);
#endif
		return retval;
#ifdef E2_VFS
	}
	else	//one or both of the src/dest is non-native
	{
		g_error_free (*E2_ERR_NAME);
# ifdef E2_VFSTMP
		//FIXME generalise this for vfs
# endif
		return FALSE;
	}
#endif
}
/**
@brief open pipe to read output from @a command
Error message expects BGL to be closed
@param command localised command string

@return pointer to opened pipe, or NULL if open failed
*/
E2_FILE *e2_fs_open_pipe (gchar *command)
{
	E2_FILE *pipe;
	if ((pipe = popen (command, "r")) == NULL)	//no vfs command-execution for piping
	{
		gchar *s, *utf;
		if ((s = e2_utils_find_whitespace (command)) != NULL)
			*s = '\0';
		utf = e2_utf8_from_locale (command);
		s = g_strdup_printf (_("Cannot open pipe for command '%s'"), utf);
		e2_output_print_error (s, TRUE);
		g_free (utf);
	}
	return pipe;
}
/**
@brief read piped command output into memory
A terminating 0 is always added
Error message expects BGL to be closed
@param command utf command string to be executed
@param output store for ptr to command output string

@return TRUE if @a command result was read successfully
*/
gboolean e2_fs_get_command_output (gchar *command, gchar **output)
{
	gchar *local = F_FILENAME_TO_LOCALE (command);
	E2_FILE *pipe = e2_fs_open_pipe (local);
	F_FREE (local);
	if (pipe == NULL)
		return FALSE;
	gchar *msg = g_strdup_printf (
			_("Command %s failed: not enough memory"), command);
	size_t total = 0;
	size_t bsize = 4096;	//block size
	gchar *buf = (gchar *) g_try_malloc ((gulong) bsize);
	if (buf == NULL)
	{
		e2_output_print_error (msg, TRUE);
		e2_fs_pipe_close (pipe);
		return FALSE;
	}
	gchar *store = buf;
	ssize_t bytes_read;
	while ((bytes_read = fread (store, 1, bsize, pipe)) > 0)
	{
		if (bytes_read == bsize)
		{
			total += bsize;
			buf = g_try_realloc (buf, total+bsize);
			if (buf == NULL)
			{
				e2_output_print_error (msg, TRUE);
				e2_fs_pipe_close (pipe);
				return FALSE;
			}
			store = buf + total;
		}
		else
		{
			*(store+bytes_read) = '\0';
			bytes_read++;
			total += bytes_read;
			buf = g_try_realloc (buf, total);
			if (buf == NULL)
			{
				e2_output_print_error (msg, TRUE);
				e2_fs_pipe_close (pipe);
				return FALSE;
			}
			break;
		}
	}

	g_free (msg);
	e2_fs_pipe_close (pipe);
	if (total == 0)
		g_free (buf);
	else
		*output = buf;
	return (total > 0);
}
/**
@brief read up to @a count -1 bytes of piped command output into memory
A terminating 0 is always added
@param command utf command string to be executed
@param contents ptr to where the contents are to be stored
@param count size of block to read + 1 (for added 0)

@return TRUE if @a command result was read successfully
*/
gboolean e2_fs_sniff_command_output (gchar *command, gchar **contents, gulong count)
{
	gchar *local = F_FILENAME_TO_LOCALE (command);
	E2_FILE *pipe = e2_fs_open_pipe (local);
	F_FREE (local);
	if (pipe == NULL)
		return FALSE;
	*contents = (gchar *) g_try_malloc (count);
	if (*contents == NULL)
	{
		//FIXME warn user
		e2_fs_pipe_close (pipe);
		return FALSE;
	}
	ssize_t bytes_read = fread (*contents, 1, (size_t)count-1, pipe);
	*(*contents + bytes_read) = '\0';
	e2_fs_pipe_close (pipe);
	return TRUE;
}
/* *
@brief blockwize read from stdin into memory
A terminating 0 is always added
Error message expects BGL to be closed
@param contents ptr to set to where the input is stored

@return TRUE if stdin was read successfully
*/
/*UNUSED
gboolean e2_fs_read_stdin (gchar **contents)
{
	fd_set fds;
	FD_ZERO (&fds);
	FD_SET (STDIN_FILENO, &fds);

	struct timeval wait;
	wait.tv_sec = 0;
	wait.tv_usec = 100000;

	if (select (STDIN_FILENO+1, &fds, NULL, NULL, &wait) != 1)
		return FALSE;

	size_t total = 0;
	size_t bsize = 1024;	//block size
	gchar *buf = (gchar *) g_try_malloc ((gulong) bsize);
	if (buf == NULL)
		return FALSE;
	gchar *store = buf;
	ssize_t bytes_read;
	while (feof (stdin) == 0)
	{
		bytes_read = fread (buf, 1, bsize, stdin);
		if (ferror (stdin) != 0)
		{
			g_free (buf);
			return FALSE;	//error msg ??
		}
		if (bytes_read == bsize)
		{
			total += bsize;
			buf = g_try_realloc (buf, total+bsize);
			if (buf == NULL)
			{
				e2_output_print_error (_("Failed stdin read: not enough memory"), FALSE);
				e2_fs_close_stream (stdin);
				return FALSE;
			}
			store = buf + total;
		}
		else
		{
			*(store+bytes_read) = '\0';
			bytes_read++;
			buf = g_try_realloc (buf, total + bytes_read);
			if (buf == NULL)
			{
				e2_fs_close_stream (stdin);
				return FALSE;
			}
			break;
		}
	}
	e2_fs_close_stream (stdin);
	*contents = buf;
	return TRUE;
} */
/* *
@brief write a sequence of bytes from memory to stdout

@param contents ptr to 0-terminated sequence of bytes to be written

@return TRUE if @a contents was sent successfully
*/
/*UNUSED
gboolean e2_fs_write_stdout (gchar *contents)
{
	fd_set fds;
	FD_ZERO (&fds);
	FD_SET (STDOUT_FILENO, &fds);

	struct timeval wait;
	wait.tv_sec = 0;
	wait.tv_usec = 100000;

	if (select (STDOUT_FILENO+1, NULL, &fds, NULL, &wait) != 1)
		return FALSE;

	return (e2_fs_file_put ("stdout", contents, stdout E2_ERR_NONE()) != EOF);
} */
/**
@brief convert file mode_t to a readable permissions string
NOTE this handles only single-byte strings, and some parts are
not translated
String is 10 bytes long

@param buf ptr to buffer where the result is to be stored
@param len integer length of @a buf
@param mode mode_t property to be parsed

@return
*/
void e2_fs_get_perm_string (gchar *buf, gint len, mode_t mode)
{
	gchar *perm_sets[8];
	perm_sets[0] = _("---");
	perm_sets[1] = _("--x");
	perm_sets[2] = _("-w-");
	perm_sets[3] = _("-wx");
	perm_sets[4] = _("r--");
	perm_sets[5] = _("r-x");
	perm_sets[6] = _("rw-");
	perm_sets[7] = _("rwx");

	gint u, g, o;
	u = (mode & S_IRWXU) >> 6;
	g = (mode & S_IRWXG) >> 3;
	o = (mode & S_IRWXO);

	g_snprintf (buf, len, "-%s%s%s", perm_sets[u], perm_sets[g], perm_sets[o]);

	//_I( FIXME the things below do not suppport translation !!
	if (S_ISLNK(mode))
		buf[0] = 'l';
	else if (S_ISDIR(mode))
		buf[0] = 'd';
	else if (S_ISBLK(mode))
		buf[0] = 'b';
	else if (S_ISCHR(mode))
		buf[0] = 'c';
	else if (S_ISFIFO(mode))
		buf[0] = 'f';
	else if (S_ISSOCK(mode))
		buf[0] = 's';

	if (mode & S_ISVTX)
		buf[9] = (buf[9] == '-') ? 'T' : 't';
	if (mode & S_ISGID)
		buf[6] = (buf[6] == '-') ? 'S' : 's';
	if (mode & S_ISUID)
		buf[3] = (buf[3] == '-') ? 'S' : 's';
}
/**
@brief update m/a/c times of config dir

This should be called after writing the config/cache files
to let any other instance know that it should re-read those
files (if that option is in force)
Error message expects BGL to be closed

@return
*/
void e2_fs_touch_config_dir (void)
{
#ifdef E2_FAM
	if (app.monitor_type != E2_MONITOR_DEFAULT)
		return;
#endif
	struct stat stat_buf;
	E2_ERR_DECLARE

	gchar *local = F_FILENAME_TO_LOCALE (e2_cl_options.config_dir);
	if (!e2_fs_utime (local, NULL E2_ERR_PTR()))
	{
		if (!e2_fs_stat (local, &stat_buf E2_ERR_PTR()))
			app.config_mtime = stat_buf.st_mtime;
		else
		{
			printd (WARN, "couldn't stat the config dir");
#ifdef E2_VFS
			e2_output_print_error (E2_ERR_NAME->message, FALSE);
#else
			e2_output_print_strerrno ();
#endif
			E2_ERR_CLEAR
		}
	}
	else
	{
		printd (WARN, "couldn't touch the config dir");
#ifdef E2_VFS
		e2_output_print_error (E2_ERR_NAME->message, FALSE);
#else
		e2_output_print_strerrno ();
#endif
		E2_ERR_CLEAR
	}
	F_FREE (local);
}
/**
@brief change time(s) of item @a utfpath to current system time

@param utfpath utf8 string with absolute path of item to touch

@return TRUE if the operation succeeded
*/
gboolean e2_fs_touchnow (gchar *utfpath E2_ERR_ARG())
{
#ifdef E2_VFSTMP
	//FIXME set the correct CWD
#else
	gchar *local = F_FILENAME_TO_LOCALE (curr_view->dir);
#endif
	gint result = e2_fs_utime (local, NULL E2_ERR_SAMEARG());
	F_FREE (local);
	return (result == 0);
}
/**
@brief display error message @a msg, with current system error description
Assumes BGL is open
@param msg utf-8 string message to display

@return FALSE (==error signal) always
*/
gboolean e2_fs_error (gchar *msg
#ifdef E2_VFS
	, gchar *reason
#endif
	)
{
	gchar *long_msg = g_strconcat (msg, " - ",
#ifdef E2_VFS
		reason
#else
		(gchar *) g_strerror (errno)
#endif
	, NULL);
	gdk_threads_enter ();
	e2_output_print_error (long_msg, TRUE);
	gdk_threads_leave ();
	return FALSE;
}
/**
@brief construct and display error message, with current system error description
Assumes BGL is open
@param format utf-8 format string (with a "%s") for the message to display
@param local localised itemname or path string to be incorporated into the message

@return FALSE (==error signal) always
*/
gboolean e2_fs_error_local (const gchar *format, gchar *local
#ifdef E2_VFS
		, gchar *reason
#endif
	)
{
	gchar *utf = F_DISPLAYNAME_FROM_LOCALE (local);
	//for output pane printing, no need to escape any pango-annoying component
	//of the itemname
	gchar *msg = g_strdup_printf (format, utf);
	e2_fs_error (msg
#ifdef E2_VFS
		, reason
#endif
		);
	F_FREE (utf);
	g_free (msg);
	return FALSE;
}
/**
@brief construct and display error message using @a format and @a local
Assumes BGL is open
@param format utf-8 string, format (with a "%s") for the message to display
@param local localised itemname or path string to be incorporated into the message

@return FALSE (==error signal) always
*/
gboolean e2_fs_error_simple (const gchar *format, gchar *local)
{
	gchar *utf = F_DISPLAYNAME_FROM_LOCALE (local);
	//no need to escape any pango-annoying component of the itemname
	gchar *msg = g_strdup_printf (format, utf);
	gdk_threads_enter ();
	e2_output_print_error (msg, TRUE);
	gdk_threads_leave ();
	F_FREE (utf);
	return FALSE;
}
