/* $Id: e2_option_bool.c 544 2007-07-20 12:48:20Z tpgww $

Copyright (C) 2004-2007 tooar <tooar@gmx.net>

This file is part of emelFM2.
emelFM2 is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

emelFM2 is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emelFM2; see the file GPL. If not, contact the Free Software
Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

/**
@file src/config/e2_option_bool.c
@brief boolean options handling

boolean options handling

*/

#include "emelfm2.h"
#include "e2_option.h"
#include "e2_option_bool.h"

  /*********************/
 /***** callbacks *****/
/*********************/

static void _e2_option_bool_toggle_d_expander_cb
	(GtkWidget *expander, GParamSpec *spec, E2_OptionSet *set)
{
	gboolean value = gtk_expander_get_expanded  (GTK_EXPANDER (expander));
	GtkWidget *show = g_object_get_data (G_OBJECT (expander), "e2-show-widget");
	if (show != NULL)
	{
		if (value)
		{
			if (!GTK_WIDGET_VISIBLE (show))
				gtk_widget_show (show);
		} else if (GTK_WIDGET_VISIBLE (show))
			gtk_widget_hide (show);

	}
	GtkWidget *controller = g_object_get_data (G_OBJECT (expander),
		"e2-controller-widget");
	if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (controller),
		"e2-controller-blocked")))
			return;
	e2_option_bool_set_direct (set, value);

}

static gboolean _e2_option_bool_toggle_d_expander_cb2
	(gpointer data, GtkWidget *expander)
{
	GtkWidget *controller = g_object_get_data (G_OBJECT (expander),
		"e2-controller-widget");
	if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (controller),
		"e2-controller-blocked")))
			return TRUE;
	gboolean value = GPOINTER_TO_INT (data);
	gboolean current = gtk_expander_get_expanded  (GTK_EXPANDER (expander));
	if (value != current)
		gtk_expander_set_expanded (GTK_EXPANDER (expander), value);
	return TRUE;
}

static void _e2_option_bool_value_changed_cb
	(GtkWidget *menu_item, E2_OptionSet *set)
{
	GtkWidget *controller = g_object_get_data (G_OBJECT (menu_item),
		"e2-controller-widget");
	if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (controller),
		"e2-controller-blocked")))
			return;
	e2_option_bool_set_direct (set, gtk_check_menu_item_get_active
		(GTK_CHECK_MENU_ITEM (menu_item)));
}

static void _e2_option_bool__value_changed_cb
	(GtkWidget *button, E2_OptionSet *set)
{
	GtkWidget *controller = g_object_get_data (G_OBJECT (button),
		"e2-controller-widget");
	if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (controller),
		"e2-controller-blocked")))
			return;
	e2_option_bool_set_direct (set, gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON (button)));
}

static gboolean _e2_option_bool_value_changed_cb2
	(gpointer data, GtkWidget *menu_item)
{
	GtkWidget *controller = g_object_get_data (G_OBJECT (menu_item),
		"e2-controller-widget");
	if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (controller),
		"e2-controller-blocked")))
			return TRUE;
	gboolean value = GPOINTER_TO_INT (data);
	gboolean current = gtk_check_menu_item_get_active
		(GTK_CHECK_MENU_ITEM (menu_item));
	if (value != current)
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menu_item), value);
	return TRUE;
}

static gboolean _e2_option_bool__value_changed_cb2
	(gpointer data, GtkWidget *button)
{
	GtkWidget *controller = g_object_get_data (G_OBJECT (button),
		"e2-controller-widget");
	if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (controller),
		"e2-controller-blocked")))
			return TRUE;
	gboolean value = GPOINTER_TO_INT (data);
	gboolean current = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON (button));
	if (value != current)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), value);
	return TRUE;
}

static gboolean _e2_option_bool_value_changed_cb3
	(gpointer data, GtkWidget *menu_item)
{
	gboolean value = GPOINTER_TO_INT (data);
	if ((GTK_WIDGET_SENSITIVE (menu_item)) && value)
		gtk_widget_set_sensitive (menu_item, FALSE);
	if ((!GTK_WIDGET_SENSITIVE (menu_item)) && !value)
		gtk_widget_set_sensitive (menu_item, TRUE);
	return TRUE;
}

static gboolean _e2_option_bool_value_changed_cb4
	(gpointer data, GtkWidget *menu_item)
{
	gboolean value = GPOINTER_TO_INT (data);
	if ((GTK_WIDGET_SENSITIVE (menu_item)) && value)
		gtk_widget_set_sensitive (menu_item, TRUE);
	if ((!GTK_WIDGET_SENSITIVE (menu_item)) && !value)
		gtk_widget_set_sensitive (menu_item, FALSE);
	return TRUE;
}

  /******************/
 /***** public *****/
/******************/

GtkWidget *e2_option_bool_add_menu_widget (GtkWidget *controller,
	GtkWidget *menu, E2_OptionSet *set, gpointer func, gpointer data)
{
	GtkWidget *menu_item = e2_menu_add_check (menu, set->desc,
		e2_option_bool_get_direct (set), func, data);
	g_object_set_data (G_OBJECT (menu_item), "e2-controller-widget", controller);
	//this will update the option value in case the check menu item changes
	g_signal_connect (G_OBJECT (menu_item), "toggled",
		G_CALLBACK (_e2_option_bool_value_changed_cb), set);
	//this will update the check menu item in case the option value changes
	e2_option_attach_value_changed_simple (set, menu_item,
		_e2_option_bool_value_changed_cb2, menu_item);
#ifdef USE_GTK2_12
	gtk_widget_set_tooltip_text (
#else
	e2_widget_set_tooltip (NULL,
#endif
		menu_item, set->tip);
	//handle dependencies
	if (set->depends != NULL)
	{
		if (set->depends[0] == '!')
		{
			E2_OptionSet *opt = e2_option_get (set->depends + 1);
			if ((opt != NULL) && (opt->type == E2_OPTION_TYPE_BOOL))
			{
				e2_option_attach_value_changed_simple (opt, menu_item,
					_e2_option_bool_value_changed_cb3, menu_item);
				gtk_widget_set_sensitive (menu_item,
					!e2_option_bool_get_direct (opt));
			}
		} else
		{
			E2_OptionSet *opt = e2_option_get (set->depends);
			if ((opt != NULL) && (opt->type == E2_OPTION_TYPE_BOOL))
			{
				e2_option_attach_value_changed_simple (opt, menu_item,
					_e2_option_bool_value_changed_cb4, menu_item);
				gtk_widget_set_sensitive (menu_item,
					e2_option_bool_get_direct (opt));
			}
		}
	}
	return menu_item;
}

GtkWidget *e2_option_bool_add_dialog_widget (GtkWidget *dialog, GtkWidget *box,
	GtkSizeGroup *size_group, E2_OptionSet *set)
{
	set->widget = e2_button_add_toggle (box, TRUE, set->ival, set->desc, NULL,
		FALSE, 0, _e2_option_bool__value_changed_cb, set);
	g_object_set_data (G_OBJECT (set->widget), "e2-controller-widget", dialog);
	e2_option_attach_value_changed_simple (set, set->widget,
		_e2_option_bool__value_changed_cb2, set->widget);
#ifdef USE_GTK2_12
	gtk_widget_set_tooltip_text (
#else
	e2_widget_set_tooltip (NULL,
#endif
		set->widget, set->tip);
	return set->widget;
}

GtkWidget *e2_option_bool_add_expander_widget (GtkWidget *controller,
	GtkWidget *box, gchar *text, E2_OptionSet *set, GtkWidget *child)
{
	GtkWidget *expander = gtk_expander_new (text);
	g_signal_connect (G_OBJECT (expander), "notify::expanded",
		G_CALLBACK (_e2_option_bool_toggle_d_expander_cb), set);
	e2_option_attach_value_changed_simple (set, expander,
		_e2_option_bool_toggle_d_expander_cb2, expander);
	gtk_container_set_border_width (GTK_CONTAINER (child), E2_PADDING_SMALL);

	g_object_set_data (G_OBJECT (expander), "e2-controller-widget", controller);
	gtk_box_pack_start (GTK_BOX (box), expander, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), child, FALSE, FALSE, 0);
	g_object_set_data (G_OBJECT (expander), "e2-show-widget", child);

	return expander;
}
/**
@brief register new boolean option

register new boolean option.

@param name    name of the option, generally a constant string but sometimes runtime-created
@param group   group the option belongs to, used in config dialog, a r-t string
@param desc    textual description of the option used in config dialog, a r-t _() string
@param tip     tooltip used when displaying the config dialog, a _() string
@param depends name of another option this one depends on
@param value   value for the option
@param flags   bitflags determining how the option data is to be handled

@return  E2_OptionSet of the new option
*/
E2_OptionSet *e2_option_bool_register (gchar *name, gchar *group, gchar *desc,
	gchar *tip, gchar *depends, gboolean value, E2_OptionFlags flags)
{
	E2_OptionSet *set = e2_option_register (E2_OPTION_TYPE_BOOL, name, group,
		desc, tip, depends, flags);
	//set ->i/sval
	set->hook_freezed = TRUE;
	e2_option_bool_set_direct (set, value);
	set->hook_freezed = FALSE;
	return set;
}
/**
@brief get current value of boolean option

@param option name of the option

@return current value (T/F) of the option
*/
gboolean e2_option_bool_get (gchar *option)
{
	E2_OptionSet *set = e2_option_get (option);
	if (set == NULL)
	{
		printd (WARN, "trying to get option '%s' which doesn't exist", option);
		return FALSE;
	}
	if (set->type == E2_OPTION_TYPE_BOOL)
		return e2_option_bool_get_direct (set);
	else
	{
		printd (WARN, "trying to get option '%s' as boolean", option);
		return FALSE;
	}
}
/**
@brief get current value of boolean option

@param set pointer to data struct for the option

@return current value (T/F) of the option
*/
gboolean e2_option_bool_get_direct (E2_OptionSet *set)
{
	return set->ival;
}
/**
@brief toggle value of boolean option

@param option name of the option

@return the new value (T/F) of the option
*/
gboolean e2_option_bool_toggle (gchar *option)
{
	E2_OptionSet *set = e2_option_get (option);
	if (set == NULL)
		return FALSE;
	if (set->type == E2_OPTION_TYPE_BOOL)
		return e2_option_bool_toggle_direct (set);
	else
	{
		printd (WARN, "trying to toggle bool option '%s', which isn't a bool",
			set->name);
		return FALSE;
	}
}
/**
@brief toggle value of boolean option

@param set pointer to data struct for the option

@return the new value (T/F) of the option
*/
gboolean e2_option_bool_toggle_direct (E2_OptionSet *set)
{
	if (e2_option_bool_get_direct (set))
	{
		e2_option_bool_set_direct (set, FALSE);
		return FALSE;
	} else
	{
		e2_option_bool_set_direct (set, TRUE);
		return TRUE;
	}
}
/**
@brief set current value of boolean option

@param option name of the option
@param value the state (T/F) to be set

@return the new value (T/F) of the option, FALSE if it doesn't exist
*/
gboolean e2_option_bool_set (gchar *option, gboolean value)
{
	E2_OptionSet *set = e2_option_get (option);
	if (set == NULL)
		return FALSE;
	if (set->type == E2_OPTION_TYPE_BOOL)
		e2_option_bool_set_direct (set, value);
	else
	{
		printd (WARN, "trying to bool set option '%s' which isn't a bool",
			option);
		return FALSE;
	}
	return value;
}
/**
@brief set current value of boolean option

@param set pointer to data struct for the option
@param value the state (T/F) to be set

@return the new value (T/F) of the option
*/
gboolean e2_option_bool_set_direct (E2_OptionSet *set, gboolean value)
{
	set->ival = value;
	set->sval = (value) ? "true" : "false";
	if (!set->hook_freezed)
		e2_hook_list_run (&set->hook_value_changed, GINT_TO_POINTER (value));
	return value;
}
