/*
 
    Ejecter - Safely, easily remove external peripherals
    Copyright 2008-2009, Federico Pelloni <federico.pelloni@gmail.com>
 
    
    This file is part of Ejecter.
 
    Ejecter is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Ejecter is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Ejecter.  If not, see <http://www.gnu.org/licenses/>.
   
*/ 

using Gtk, GLib, Gdu;


namespace Ejecter {

public class App: Gtk.Window {

    private Gtk.StatusIcon status_icon;
    private Gtk.Table table;
    private Gtk.ScrolledWindow scroll;
    private Gtk.EventBox container;
    private Gtk.Label title_label;
    private Ejecter.About about;
    private GLib.VolumeMonitor monitor;
    private GLib.HashTable<string, Ejecter.Device> devices;
    private GLib.SList<string> invalid_devices;
    private Ejecter.Menu menu;
    private Gdu.Pool pool;

    construct {
    
        // Set up main window
        this.decorated = false;
        this.skip_taskbar_hint = true;
        this.skip_pager_hint = true;
        // this.accept_focus = false;
        this.modal = true;
        this.set_type_hint(Gdk.WindowTypeHint.POPUP_MENU);
        this.resize(conf.default_width, conf.default_height);
        
        this.focus_out_event += (w, e) => { this.hide(); };
        
        
        // About window
        this.about = new Ejecter.About();
        
        // Status icon
        Gtk.IconTheme theme = Gtk.IconTheme.get_default();
        theme.prepend_search_path(Config.ICONS_DIR);
        this.status_icon = new Gtk.StatusIcon.from_icon_name("ejecter");
        this.status_icon.set_tooltip_text(_("Eject removable media"));
        this.status_icon.set_visible(false);
        this.status_icon.activate += this.handle_trayclick;
        
        // Main vbox
        Gtk.VBox box = new Gtk.VBox(false, 0);
        this.add(box);
        
        // Header label
        this.title_label = new Gtk.Label("");
        this.title_label.set_markup("<b>" + _("Eject removable media") + "</b>");
        this.title_label.set_alignment(0, (float) 0.5);
        this.title_label.set_padding(10, 10);
        box.pack_start(this.title_label, false, false, 0);
        
        // Right-click menu
        this.menu = new Ejecter.Menu();
        this.status_icon.popup_menu += (icon, button, time) => 
                                { this.menu.popup(null, null, null, button, time); };
        this.menu.about.activate += this.show_about;
        
        // Colorize header
        this.container = new Gtk.EventBox();
        Gtk.Style style = this.get_style();
        this.container.modify_bg(Gtk.StateType.NORMAL, style.@base[(int) Gtk.StateType.NORMAL]);
        box.pack_start(this.container, true, true, 0);
        
        // Scrollable
        this.scroll = new Gtk.ScrolledWindow(null, null);
        this.scroll.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.NEVER);
        this.container.add(this.scroll);
        
        // Viewport
        Gtk.Viewport viewport = new Gtk.Viewport(null, null);
        viewport.set_shadow_type(Gtk.ShadowType.NONE);
        viewport.modify_bg(Gtk.StateType.NORMAL, style.@base[(int) Gtk.StateType.NORMAL]);
        this.scroll.add(viewport);
        
        // Device table
        this.table = new Gtk.Table(1, 3, false);
        viewport.add(this.table);
        
        this.table.remove += (c, w) => { 
                                         this.check_icon();
                                         this.update_table_parent();
                                       };
        this.table.size_allocate += r => { this.reposition(); };
        this.show += t => { this.update_table_parent(); };
        
        // Device map
        this.devices = new GLib.HashTable<string , Ejecter.Device>(GLib.str_hash, GLib.str_equal);        
        this.invalid_devices = new GLib.SList<string>();
        
        // GIO monitor
        this.pool = new Gdu.Pool();
        this.monitor = GLib.VolumeMonitor.get();
        this.load_devices();
        
        // Watchers
        //this.monitor.drive_connected += (monitor, drive) => { this.manage_drive(drive); };
        this.monitor.volume_added += (monitor, volume) => { this.manage_volume(volume); };
        this.monitor.mount_added += (monitor, mount) => { this.manage_mount(mount); };
        
    }
    
    
    private void update_table_parent () {
    
        Gtk.Requisition req = Gtk.Requisition();
        this.table.size_request(out req);
        
        Gtk.Requisition title_req = Gtk.Requisition();
        this.title_label.size_request(out title_req);
        
        int max_height = (int) (Gdk.Screen.height() * 0.7) - title_req.height;
        int width = (int) GLib.Math.fmax(conf.default_width, GLib.Math.fmax(req.width, title_req.width));
        
        if (req.height > max_height) {
         
            this.resize(width, max_height);
            this.scroll.vscrollbar_policy = Gtk.PolicyType.AUTOMATIC;
            
        } else {
        
            this.resize(width, req.height + title_req.height);
            this.scroll.vscrollbar_policy = Gtk.PolicyType.NEVER;
        
        }
        
    }
    
    
    
    private void reposition () {
    
        if (this.status_icon.visible) {

            int x;
            int y;
            weak Gdk.Screen screen;
            Gdk.Rectangle area;
            Gtk.Orientation orientation;
        
            this.status_icon.get_geometry(out screen, out area, out orientation);
            int s_w = screen.get_width();
            int s_h = screen.get_height();
            
            int w_w;
            int w_h;
            this.get_size(out w_w, out w_h);
            
            
            if (orientation == Gtk.Orientation.HORIZONTAL) {
                
                if (area.y < 300)
                    y = area.y + area.height;
                else
                    y = area.y - w_h;
                    
                x = (area.x < s_w - w_w) ? area.x : s_w - w_w;
                
            } else {
                
                if (area.x < 300)
                    x = area.x + area.width;
                else
                    x = area.x - w_w;
                    
                y = (area.y < s_h - w_h) ? area.y : s_h - w_h;
                
            }
                
                
            this.move(x, y);
            
        }
        
    }
    
    
    private void handle_trayclick (Gtk.StatusIcon icon) {
        
        if (this.visible) {
        
            this.hide();
            
        } else {

            this.reposition();
            this.show();

        }
    
    }
    
    
    
    private void show_about () {
        
        if (!this.about.visible)
            this.about.run();
    
    }
    
    
    
    private void load_devices () {

        foreach (GLib.Volume v in (GLib.List<GLib.Volume>) this.monitor.get_volumes()) {
            
            debug("");
            
            GLib.Drive d = v.get_drive();
            
            this.manage_drive(d);
            this.manage_volume(v);
            
            GLib.Mount m = v.get_mount();
            if (m != null)
                this.manage_mount(m);

        }    

    }
    
    
    
    
    private Ejecter.Device? manage_drive (GLib.Drive? drive) {
    
        debug("");
        debug("NEW DEVICE");
        
        if (drive == null) { debug("Drive passed was null, skipping"); return null; };
    
        var id = drive.get_identifier(conf.device_identifier);
        if (id == null || id == "") { debug("No id"); return null; };
        if (this.invalid_devices.index(id) >= 0) { debug("Skipped drive (in invalid list)"); return null; };
        if (this.devices.lookup(id) != null) { debug("Skipped drive (already in list)"); return null; };
        debug("Drive id: " + id);
        
        Gdu.Device gdu_dev = this.pool.get_by_device_file(id);
        
        if (gdu_dev.is_system_internal()) {
            debug("Device is internal: skip");
            this.invalid_devices.append(id);
            return null;
        }
        
        Ejecter.Device d = new Ejecter.Device(drive, gdu_dev);
        this.devices.insert(id, d);
        d.attach(this.table);

        d.removed += d => { 
                    this.devices.remove(d.drive.get_identifier(conf.device_identifier)); 
                    this.check_icon();
                };

        this.update_table_parent();
        this.check_icon();

        return null;
        
    }
    
    
    private void manage_volume (GLib.Volume? volume) {
    
        debug("");                    
        debug("NEW VOLUME");
    
        if (volume == null) { debug("Volume was null, skipping"); return; };
    
        GLib.Drive drive = volume.get_drive();
        if (drive == null) { debug("Volume did not have a drive, skipping"); return; };
        
        string id = drive.get_identifier(conf.device_identifier);
        if (id == null || this.invalid_devices.index(id) >= 0) return;
        
        debug("Drive id: " + id);
        debug("Volume name: " + volume.get_name());
        
        Ejecter.Device dev = this.devices.lookup(id);
        
        if (dev == null) { 
            dev = this.manage_drive(drive);
            if (dev == null) { debug("No device - skipping"); return; };
        };
        
        dev.add_volume(volume);

        this.check_icon();
    
    }
    
    
    private void manage_mount (GLib.Mount? mount) {
    
        debug("");
        debug("NEW MOUNT");

        if (mount == null) { debug("Mount was null, skipping"); return; };

        GLib.Drive drive = mount.get_drive();
        
        if (drive == null) { debug("Mount did not have a drive, skipping"); return; };
        
        string id = drive.get_identifier(conf.device_identifier);
        if (id == null) return;
        
        debug("Mount id: " + id);
        debug("Mount name: " + mount.get_name());
        
        Ejecter.Device dev = this.devices.lookup(id);
        
        if (dev == null) return;
        
        dev.add_mount(mount);

        this.check_icon();
            
    }
    
    
    
    private void check_icon () {

        this.status_icon.set_visible((bool) this.devices.size());
        
        if (this.devices.size() <= 0 && this.visible) this.hide();
    
    }



    static int main (string[] args) {
    
   		Intl.bindtextdomain(Config.GETTEXT_PACKAGE, Config.LOCALE_DIR);
		Intl.textdomain(Config.GETTEXT_PACKAGE);

        Gtk.init(ref args);
        
        GLib.Environment.set_application_name(Config.PACKAGE_NAME);
        
        var ej = new Ejecter.App();
        ej.show_all();
        ej.hide();
        
        Gtk.main();
        return 0;
    
    }

}

}

