# -*- coding: utf-8; Mode: Python; indent-tabs-mode: nil; tab-width: 4 -*-

# Copyright (C) 2010-2012 Stephane Graber
# Author: Stephane Graber <stgraber@ubuntu.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import debconf
import os

from ubiquity.plugin import Plugin, PluginUI

NAME = 'edu-domain'
AFTER = 'edu-target'
WEIGHT = 10
OEM = False


class PageGtk(PluginUI):
    plugin_title = 'ubiquity/text/edu-domain_heading_label'

    def __init__(self, controller, *args, **kwargs):
        self.controller = controller
        try:
            from gi.repository import Gtk as gtk
            builder = gtk.Builder()
            builder.add_from_file(os.path.join(os.environ['UBIQUITY_GLADE'],
                'edu-domain.ui'))
            builder.connect_signals(self)
            self.controller.add_builder(builder)
            self.page = builder.get_object('edu-domain_window')

            # Create interface model
            self.interface_model = gtk.ListStore(str, str)

            # Load required objects
            self.description = builder.get_object('description')

            ## Workstation section
            self.rb_workstation = builder.get_object('rb_workstation')
            self.rb_workstation.connect('toggled', self.on_rb_changed, None)
            self.bx_workstation = builder.get_object('bx_workstation')
            self.eb_workstation = builder.get_object('eb_workstation')
            self.eb_workstation.connect('button-press-event',
                                            self.on_lbl_clicked,
                                            self.rb_workstation)
            self.lbl_workstation = builder.get_object('lbl_workstation')

            ## Terminal server section
            self.rb_termserver = builder.get_object('rb_termserver')
            self.rb_termserver.connect('toggled', self.on_rb_changed, None)
            self.sp_termserver = builder.get_object('bx_termserver')
            self.bx_termserver = builder.get_object('bx_termserver')
            self.eb_termserver = builder.get_object('eb_termserver')
            self.eb_termserver.connect('button-press-event',
                                            self.on_lbl_clicked,
                                            self.rb_termserver)
            self.lbl_termserver = builder.get_object('lbl_termserver')
            self.lbl_termserver_interface = builder.get_object(
                                            'lbl_termserver_interface')
            self.lbl_termserver_extint = builder.get_object(
                                            'lbl_termserver_extint')
            self.lbl_termserver_intint = builder.get_object(
                                            'lbl_termserver_intint')

            self.gr_termserver = builder.get_object('gr_termserver')
            self.cb_termserver_extint = builder.get_object(
                                            'cb_termserver_extint')
            self.cb_termserver_extint.set_model(self.interface_model)
            cell = gtk.CellRendererText()
            self.cb_termserver_extint.pack_start(cell, False)
            self.cb_termserver_extint.add_attribute(cell, 'text', 0)

            self.cb_termserver_intint = builder.get_object(
                                            'cb_termserver_intint')
            self.cb_termserver_intint.set_model(self.interface_model)
            cell = gtk.CellRendererText()
            self.cb_termserver_intint.pack_start(cell, False)
            self.cb_termserver_intint.add_attribute(cell, 'text', 0)

            ## Domain member section
            self.rb_member = builder.get_object('rb_member')
            self.rb_member.connect('toggled', self.on_rb_changed, None)
            self.sp_member = builder.get_object('sp_member')
            self.bx_member = builder.get_object('bx_member')
            self.eb_member = builder.get_object('eb_member')
            self.eb_member.connect('button-press-event',
                                            self.on_lbl_clicked,
                                            self.rb_member)
            self.lbl_member = builder.get_object('lbl_member')
            self.gr_member = builder.get_object('gr_member')
            self.lbl_member_domain = builder.get_object(
                                            'lbl_member_domain')
            self.lbl_member_username = builder.get_object(
                                            'lbl_member_username')
            self.lbl_member_password = builder.get_object(
                                            'lbl_member_password')

            ## Domain server section
            self.rb_server = builder.get_object('rb_server')
            self.rb_server.connect('toggled', self.on_rb_changed, None)
            self.sp_server = builder.get_object('sp_server')
            self.bx_server = builder.get_object('bx_server')
            self.eb_server = builder.get_object('eb_server')
            self.eb_server.connect('button-press-event',
                                            self.on_lbl_clicked,
                                            self.rb_server)
            self.gr_server1 = builder.get_object('gr_server1')
            self.gr_server2 = builder.get_object('gr_server2')
            self.lbl_server = builder.get_object('lbl_server')
            self.lbl_server_domain = builder.get_object(
                                            'lbl_server_domain')
            self.lbl_server_username = builder.get_object(
                                            'lbl_server_username')
            self.lbl_server_password = builder.get_object(
                                            'lbl_server_password')
            self.lbl_server_interface = builder.get_object(
                                            'lbl_server_interface')
            self.lbl_server_extint = builder.get_object(
                                            'lbl_server_extint')
            self.lbl_server_intint = builder.get_object(
                                            'lbl_server_intint')

            self.cb_server_extint = builder.get_object(
                                            'cb_server_extint')
            self.cb_server_extint.set_model(self.interface_model)
            cell = gtk.CellRendererText()
            self.cb_server_extint.pack_start(cell, False)
            self.cb_server_extint.add_attribute(cell, 'text', 0)

            self.cb_server_intint = builder.get_object(
                                            'cb_server_intint')
            self.cb_server_intint.set_model(self.interface_model)
            cell = gtk.CellRendererText()
            self.cb_server_intint.pack_start(cell, False)
            self.cb_server_intint.add_attribute(cell, 'text', 0)

        except Exception as e:
            self.debug('Could not create edu-domain page: %s', e)
            self.page = None
        self.plugin_widgets = self.page

    def on_lbl_clicked(self, widget, event, target):
        target.set_active(True)
        target.grab_focus()

    def on_rb_changed(self, widget, data):
        target = self.get_selected()

        to_disable = {"workstation": [],
                      "termserver": [self.gr_termserver],
                      "member": [self.gr_member],
                      "server": [self.gr_server1, self.gr_server2]}

        for key, widgets in to_disable.items():
            for widget in widgets:
                if key == target:
                    widget.set_sensitive(True)
                else:
                    widget.set_sensitive(False)

    def plugin_translate(self, lang):
        self.description.set_markup(self.controller.get_string(
                                'edu-domain_description_label', lang))

        self.lbl_workstation.set_markup(self.controller.get_string(
                                'edu-domain_workstation_label', lang))

        self.lbl_termserver.set_markup(self.controller.get_string(
                                'edu-domain_termserver_label', lang))
        self.lbl_termserver_interface.set_markup(self.controller.get_string(
                                'edu-domain_termserver_interface_label', lang))
        self.lbl_termserver_extint.set_markup(self.controller.get_string(
                                'edu-domain_common_extint_label', lang))
        self.lbl_termserver_intint.set_markup(self.controller.get_string(
                                'edu-domain_common_intint_label', lang))

        self.lbl_member.set_markup(self.controller.get_string(
                                'edu-domain_member_label', lang))
        self.lbl_member_domain.set_markup(self.controller.get_string(
                                'edu-domain_common_domain_label', lang))
        self.lbl_member_username.set_markup(self.controller.get_string(
                                'edu-domain_common_username_label', lang))
        self.lbl_member_password.set_markup(self.controller.get_string(
                                'edu-domain_common_password_label', lang))

        self.lbl_server.set_markup(self.controller.get_string(
                                'edu-domain_server_label', lang))
        self.lbl_server_domain.set_markup(self.controller.get_string(
                                'edu-domain_common_domain_label', lang))
        self.lbl_server_username.set_markup(self.controller.get_string(
                                'edu-domain_common_username_label', lang))
        self.lbl_server_password.set_markup(self.controller.get_string(
                                'edu-domain_common_password_label', lang))
        self.lbl_server_extint.set_markup(self.controller.get_string(
                                'edu-domain_common_extint_label', lang))
        self.lbl_server_intint.set_markup(self.controller.get_string(
                                'edu-domain_common_intint_label', lang))

    def get_selected(self):
        target = "workstation"
        if self.rb_termserver.get_active():
            target = "termserver"
        if self.rb_member.get_active():
            target = "member"
        elif self.rb_server.get_active():
            target = "server"

        return target


class Page(Plugin):
    def prepare(self):
        try:
            target = self.db.get('edubuntu-live/edu-target_target')
        except debconf.DebconfError:
            target = "workstation"

        # Long hardcoded list of what options to show
        if target == "termserver":
            self.ui.bx_workstation.set_visible(False)
            self.ui.bx_termserver.set_visible(True)
            self.ui.sp_termserver.set_visible(True)
            self.ui.rb_termserver.set_active(True)
            self.ui.bx_member.set_visible(True)
            self.ui.sp_member.set_visible(True)
            self.ui.bx_server.set_visible(False)
            self.ui.sp_server.set_visible(False)
        elif target == "server":
            self.ui.bx_workstation.set_visible(False)
            self.ui.bx_termserver.set_visible(False)
            self.ui.sp_termserver.set_visible(False)
            self.ui.bx_member.set_visible(True)
            self.ui.sp_member.set_visible(False)
            self.ui.bx_server.set_visible(True)
            self.ui.sp_server.set_visible(True)
            self.ui.rb_server.set_active(True)
        else:
            self.ui.bx_workstation.set_visible(True)
            self.ui.rb_workstation.set_active(True)
            self.ui.bx_termserver.set_visible(False)
            self.ui.sp_termserver.set_visible(False)
            self.ui.bx_member.set_visible(True)
            self.ui.sp_member.set_visible(True)
            self.ui.bx_server.set_visible(False)
            self.ui.sp_server.set_visible(False)

        # Restore selection
        try:
            select = self.db.get('edubuntu-live/edu-domain_select')
        except debconf.DebconfError:
            select = "workstation"

        if select == "workstation" and self.ui.bx_workstation.get_visible():
            self.ui.rb_workstation.set_active(True)
        elif select == "termserver" and self.ui.bx_termserver.get_visible():
            self.ui.rb_termserver.set_active(True)
        elif select == "member" and self.ui.bx_member.get_visible():
            self.ui.rb_member.set_active(True)
        elif select == "server" and self.ui.bx_server.get_visible():
            self.ui.rb_server.set_active(True)

        # Refresh interface lists
        self.ui.interface_model.clear()
        devices = self.list_devices()
        for device in devices:
            self.ui.interface_model.append(("%s (%s)" % (device,
                                            devices[device][0]),
                                            device))
        if not devices:
            self.ui.interface_model.append(("---", None))
        self.ui.cb_termserver_extint.set_active(0)
        self.ui.cb_termserver_intint.set_active(0)
        self.ui.cb_server_extint.set_active(0)
        self.ui.cb_server_intint.set_active(0)

    def list_devices(self):
        # Code based on edubuntu-netboot
        import subprocess
        devices = {}

        cmd = ['nmcli', '-e', 'yes', '-t', '-f', 'DEVICE,TYPE,STATE', 'dev']
        listdev = subprocess.Popen(cmd, stdout=subprocess.PIPE,
                            stderr=subprocess.PIPE, env={'LANG': 'C'})
        retval = listdev.wait()

        if retval != 0:
            return []
        else:
            for line in listdev.stdout.readlines():
                fields = line.decode('utf-8').strip().split(':')
                if len(fields) == 3:
                    if fields[1] != "802-3-ethernet" or \
                        fields[2] == "unavailable":
                        continue
                    devices[fields[0]] = (fields[1], fields[2])

        return devices

    def ok_handler(self):
        self.preseed('edubuntu-live/edu-domain_select', self.ui.get_selected())
        Plugin.ok_handler(self)
