#ifndef _ECS_VEC_INT_H_
#define _ECS_VEC_INT_H_

/*============================================================================
 *  Prototypes des fonctions
 *   associées à la structure `ecs_vec_int_t' décrivant un vecteur indexé entier
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilité
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_tab_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_vec_int.h"


/*============================================================================
 *                         Declaration de la structure
 *============================================================================*/

typedef struct _ecs_vec_int_t ecs_vec_int_t;


/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction allouant et initialisant une structure `ecs_vec_int_t'
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__cree
(                           /* <-- Pointeur sur une structure `ecs_vec_int_t' */
 void
);


/*----------------------------------------------------------------------------
 *  Fonction libérant une structure `ecs_vec_int_t' donnée en argument.
 *  Elle renvoie un pointeur NULL
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__detruit
(
 ecs_vec_int_t * this_vec_int
);


/*----------------------------------------------------------------------------
 *  Fonction qui alloue une structure `ecs_vec_int_t' ainsi que ses membres
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__alloue
(                      /* <-- Renvoie la référence à la structure créée       */
 size_t      pos_nbr,  /* --> Nombre de positions pour  tableau des positions */
 size_t      val_nbr   /* --> Nombre de valeurs   pour  tableau des valeurs   */
);


/*----------------------------------------------------------------------------
 *  Fonction qui redimensionne les membres d'une structure `ecs_vec_int_t'
 *----------------------------------------------------------------------------*/

void ecs_vec_int__redimensionne
(
 ecs_vec_int_t  *const this_vec_int,
 size_t                pos_nbr,  /* --> Nouvelle dim. du tableau des pos.     */
 size_t                val_nbr   /* --> Nouvelle dim. du tableau des val. x   */
);


/*----------------------------------------------------------------------------
 *  Fonction libérant une structure `ecs_vec_int_t'
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__libere
(                               /* <-- Renvoie un pointeur NULL               */
 ecs_vec_int_t  *this_vec_int   /* <-> Structure `ecs_vec_int_t' à liberer    */
);


/*----------------------------------------------------------------------------
 *  Fonction qui crée un vecteur `ecs_vec_int_t'
 *   et qui l'initialise avec les références données
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__initialise
(
 size_t      pos_nbr,
 ecs_size_t *pos_tab,
 ecs_int_t  *val_tab
);


/*----------------------------------------------------------------------------
 *  Fonction qui créé une structure `ecs_vec_int_t'
 *   à partir d'un tableau bi-dimensionnel `tab_elt' contenant
 *   pour chaque élément un ensemble de valeurs
 *  Si un élément `ielt' n'a pas de valeur associée, `tab_elt[ielt].nbr = 0'
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__transforme_bi_tab
(
 const ecs_tab_int_t *const tab_elt,
       size_t               nbr_elt,
       size_t               nbr_val
);


/*----------------------------------------------------------------------------
 *  Fonction retournant le nombre de positions
 *----------------------------------------------------------------------------*/

size_t ecs_vec_int__ret_pos_nbr
(
 const ecs_vec_int_t *const this_vec_int
);


/*----------------------------------------------------------------------------
 *  Fonction retournant le nombre de valeurs associées au tableau des positions
 *----------------------------------------------------------------------------*/

size_t ecs_vec_int__ret_val_nbr
(
 const ecs_vec_int_t *const this_vec_int
);


/*----------------------------------------------------------------------------
 *  Fonction retournant la référence du tableau des positions
 *----------------------------------------------------------------------------*/

ecs_size_t * ecs_vec_int__ret_pos_tab
(
 ecs_vec_int_t *this_vec_int
);


/*----------------------------------------------------------------------------
 *  Fonction retournant la référence du tableau des valeurs
 *----------------------------------------------------------------------------*/

ecs_int_t * ecs_vec_int__ret_val_tab
(
 ecs_vec_int_t *this_vec_int
);


/*----------------------------------------------------------------------------
 *  Fonction qui retourne le tableau des valeurs du vecteur indexé donné,
 *   dimensionné au nombre de valeurs distinctes du vecteur indexé,
 *   (valeurs qui correspondent à des références)
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_vec_int__ret_reference
(
 ecs_vec_int_t     *const this_vec_int
);


/*----------------------------------------------------------------------------
 *  Fonction qui incrémente les valeurs d'un vecteur indexé donné
 *   d'une constante donnée
 *----------------------------------------------------------------------------*/

void ecs_vec_int__incremente_val
(
       ecs_vec_int_t *const this_vec_int,
 const ecs_int_t            increment
);


/*----------------------------------------------------------------------------
 *  Fonction qui incrémente les valeurs d'un vecteur indexé donné
 *   d'une constante donnée
 *----------------------------------------------------------------------------*/

void ecs_vec_int__incremente_val_sgn
(
       ecs_vec_int_t *const this_vec_int,
 const ecs_int_t            increment
);


/*----------------------------------------------------------------------------
 *  Fonction réalisant la transformation d'un vecteur indexé
 *   en appliquant directement le vecteur de transformation donné
 *   sur ses valeurs
 *
 *  Le nombre de valeurs transformées peut être inférieur
 *  au nombre de valeurs avant transformation
 *----------------------------------------------------------------------------*/

void ecs_vec_int__transforme_val
(
       ecs_vec_int_t  *const this_vec_int,
       size_t                nbr_elt_new,
 const ecs_tab_int_t         vect_transf
);


/*----------------------------------------------------------------------------
 *  Fonction réalisant la transformation d'un vecteur indexé
 *   en appliquant directement le vecteur de transformation donné
 *   sur ses positions
 *
 *  Le nombre de valeurs transformées doit être égal
 *  au nombre de valeurs avant transformation
 *----------------------------------------------------------------------------*/

void ecs_vec_int__transforme_pos
(
       ecs_vec_int_t  *const this_vec_int,
       size_t                nbr_elt_ref,
 const ecs_tab_int_t         vect_transf
);


/*----------------------------------------------------------------------------
 *  Fonction réalisant la transformation d'un vecteur indexé
 *   en appliquant directement le vecteur de transformation donné
 *   sur les valeurs associées à ses éléments
 *----------------------------------------------------------------------------*/

void ecs_vec_int__renumerote
(
       ecs_vec_int_t  *const this_vec_int,
 const ecs_tab_int_t         vect_transf
);


/*----------------------------------------------------------------------------
 *  Fonction réalisant la transformation d'un vecteur indexé
 *   en appliquant directement le vecteur de transformation donné
 *   sur les valeurs associées à ses éléments
 *----------------------------------------------------------------------------*/

void ecs_vec_int__renumerote_et_sgn
(
       ecs_vec_int_t  *const this_vec_int,
 const ecs_tab_int_t         vect_transf,
 const ecs_tab_int_t         signe_elt
);


/*----------------------------------------------------------------------------
 *  Fonction qui remplace les références à des éléments
 *  en des références à d'autres éléments liés aux premiers
 *  par un vecteur indexé
 *----------------------------------------------------------------------------*/

void ecs_vec_int__remplace_ref
(
       ecs_vec_int_t  *const vec_rep,
 const ecs_vec_int_t  *const vec_def
);


/*----------------------------------------------------------------------------
 *  Fonction qui remplace les références à des éléments
 *  en des références à d'autres éléments liés aux premiers
 *  par un vecteur indexé ; les références à 0 sont ignorées
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__ret_remplace_ref
(
       ecs_vec_int_t  *const vec_rep,
 const ecs_vec_int_t  *const vec_def
);


/*----------------------------------------------------------------------------
 *  Fonction réalisant le transfert des valeurs d'un champ associées
 *   aux éléments, sur les sous-éléments
 *   participant à la définition des éléments
 *   dans une connectivité "élément -> sous-élément"
 *
 *  Un même sous-élément peut appartenir à plusieurs éléments
 *  Un élément est supposé n'avoir qu'une seule valeur
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__herite
(                                                /* <-- Valeurs transférées   */
 const ecs_vec_int_t  *const vec_val_elt,        /* --> Valeurs à transférer  */
 const ecs_vec_int_t  *const vec_elt_def_sselt,  /* --> Définition éléments   */
       size_t                nbr_sselt
);


/*----------------------------------------------------------------------------
 *  Fonction qui détermine un nouveau vecteur à partir d'un vecteur donné
 *   en extrayant de ce dernier les éléments sélectionnés
 *   par le tableau de booléens
 *
 *  Cette fonction renvoie le vecteur qui définit les anciens éléments
 * du vecteur de référence en fonction des nouveaux éléments du vecteur renvoyé
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__extrait
(
       ecs_vec_int_t   *const vec_new,
 const ecs_vec_int_t   *const vec_ref,
 const ecs_tab_bool_t         bool_elt_select
);


/*----------------------------------------------------------------------------
 *  Fonction qui construit la  table de connectivité inverse
 *  "sous-éléments -> éléments"
 *
 *  Si dans la connectivité "éléments -> sous-éléments",
 *  le sous-élément est référencé négativement dans la définition de l'élément,
 *  alors dans la connectivité inverse,
 *  l'élément est référencé négativement dans la définition du sous-élément
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__inverse
(
 const ecs_vec_int_t  *const vec_def_elt,
       size_t                nbr_sselt
);


/*----------------------------------------------------------------------------
 *  Fonction qui réalise l'ordination des valeurs d'un vecteur indexé
 *     en triant les valeurs pour chaque ensemble de valeurs entre 2 positions
 *  et en triant chaque ensemble de valeurs dans l'ordre lexicographique
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_vec_int__trie_val
(
 ecs_vec_int_t * *const this_vec_int
);


/*----------------------------------------------------------------------------
 *  Fonction qui compare les ensembles de valeurs comprises entre 2 positions
 *   et renvoie un tableau donnant pour chaque ensemble
 *      le numéro du 1er ensemble de chaque groupe d'ensembles identiques
 *
 *  Si entre 2 positions il n'y a pas de valeurs, le numéro est `-1'
 *
 *  Cette fonction prend tout son intérêt lorsque
 *   - les valeurs ont été triées dans chaque ensemble de valeurs
 *   - les ensembles de valeurs ont été tries dans l'ordre lexicographique
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_vec_int__compare_val_pos
(
 const ecs_vec_int_t  *const this_vec_int
);


/*----------------------------------------------------------------------------
 *  Fonction qui prolonge le tableau des valeurs d'une structure
 *  ecs_vec_int_t de manière à ce qu'à chaque position corresponde au
 *  moins une valeur.
 *
 *  Si la structure ecs_vec_int_t n'existe pas, on en crée une.
 *
 *  Cette fonction est utile notamment lors de la construction des familles,
 *  afin que chaque élément appartienne bien explicitement à une famille
 *  donnée, et non pas seulement implicitement à la famille 0 s'il ne porte
 *  aucune couleur ou groupe.
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__prolonge_val_pos
(
 ecs_vec_int_t *const this_vec_int,
 ecs_int_t            val_defaut,
 size_t               nbr_defaut
);


/*----------------------------------------------------------------------------
 *  Fonction qui renvoie un vecteur indexé donnant pour chaque élément
 *   le numéro de famille à laquelle il appartient
 *   à partir -    du vecteur indexé donnant pour chaque élément
 *               les numéros d'attribut le caractérisant
 *            - et du tableau donnant pour chaque élément :
 *                 - le 1er élément ayant la même liste d'attributs
 *                 - ou `-1' s'il n'a pas d'attribut
 *
 *  La fonction détermine aussi :
 *   - les définitions des familles en fonction des numéros de descripteur
 *   - les nombres de descripteurs composant chaque famille
 *   - le  nombre  de familles
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_vec_int__attribue_fam
(
 const ecs_vec_int_t    *vec_int_att_unifie,
 const ecs_tab_int_t     tab_idem,
       ecs_int_t      ***def_fam_descr,
       ecs_int_t       **nbr_descr_fam,
 const ecs_int_t         num_fam_deb,
       int              *nbr_fam
);


#endif /* _ECS_VEC_INT_H_ */
