
import os
import copy
import sys
import time
import logging
import shutil
from xml.dom.minidom import *

import gconf
import pynotify
import gtk
import gobject
import pynotify

from util.DesktopFiles import DesktopFiles
from util.Threads import threaded

from Client import Client
from Sink import Sink
from windows.Watcher import Watcher
from PulseAudio import PulseAudio

from ui.TrayIcon import EarCandyStatusIcon
from ui.EarCandyPrefs import EarCandayPrefs

from EarCandyDBus import EarCandyDBusClient

from earcandyconfig import getdatapath

log = logging.getLogger('EarCandy')
log.setLevel(logging.WARNING)


class EarCandy():

    def __init__(self):

        self.version = 0.9
        self.active = True
        
        self.display = {
                        "": "[ unknown ]", 
                        "phone" : "Phone (VoIP)", 
                        "video" : "Video Player", 
                        "music" : "Music Player", 
                        "event" : "Notification", 
                        "game" : "Game" }

        self.ignore = ["EsounD client (UNIX socket client)", 
                        "ear-candy", 
                        "Native client (UNIX socket client)", 
                        "PulseAudio Volume Control"]

        self.config_file = os.path.expanduser("~/.config/Ear Candy/settings.xml")     
        self.default_config_file = os.path.join(getdatapath(), 'defaults', 'settings.xml')

        self.pa = PulseAudio(self)
        self.window_watcher = Watcher()
        self.window_watcher.callback = self.on_active_window_change

        self.prefer_sink = None

        self.pref = None

        pynotify.init('earcandy')

    def notify(self, name, text):
        return        
        #n = pynotify.Notification(name, text, None)
        #n.show()

    def init(self):
        self.is_mute = False
        self.mute_phone = False

        self.fade_timer_speed = 0.01 
        self.mute_level = 20
        self.follow_new_outputs = True

        self.priority_stack = { "" : [], "phone" : [], "video" : [], "music" : [] , "game" : [] }   

        self.apply_volume_thread_running = False 
        self.select_client_thread_running = False

    def run(self):

        ecb = EarCandyDBusClient(self)
        if ecb.is_running():
            print "Ear candy already running..."
            ecb.show_window()
            sys.exit(0)

        ecb.start_service()

        self.init() 
        self.load()
        self.pa.connect()

        self.apply_volume_thread()
        self.select_client_thread()

        self.status_icon = EarCandyStatusIcon(self)
        self.start()

        self.window_watcher.check_all()
        
        
        #self.open_preferances()

    def exit(self):
        self.stop()
        self.pa.disconnect()
        gtk.main_quit()


    def stop(self):
        self.active = False
        while self.apply_volume_thread_running or self.select_client_thread_running:
            time.sleep(0.1)

    def start(self):
        self.active = True
        while not self.apply_volume_thread_running or not self.select_client_thread_running:
            time.sleep(0.1)

    def reset_all(self):
        # tricky we want to delete all client settings and reload our default xml file
        self.stop()
        self.pa.disconnect()
        self.init()
        self.load(False)
        self.pa.connect()
        self.start()
        return

    def reset(self):
        self.stop()
        self.pa.disconnect()
        self.init()
        self.load()
        self.pa.connect()
        self.start()
        return

    def save(self):
        path = os.path.dirname(self.config_file)
        if not os.path.exists( path ):
            os.makedirs(path)

        # New document
        doc = Document()
        ec = doc.createElement("earcandy")
        doc.appendChild(ec)
	
        rules = doc.createElement("rules")
        ec.appendChild(rules)
        skip = copy.copy(self.ignore)
        for client in self.pa.clients.values():
            if not client.name in skip and client.role:
                # Creates user element
                el = doc.createElement("rule")
                rules.appendChild(el)
                client.to_xml(el)
                skip.append(client.name)
        
        doc.documentElement.setAttribute("fade_timer_speed", str(self.fade_timer_speed))
        doc.documentElement.setAttribute("mute_level", str(self.mute_level))
        doc.documentElement.setAttribute("tray_visible", str(self.status_icon.get_visible()))
        doc.documentElement.setAttribute("prefer_sink", str(self.prefer_sink))
        doc.documentElement.setAttribute("follow_new_outputs", str(self.follow_new_outputs))
        doc.documentElement.setAttribute("version", str(self.version))
        doc.documentElement.setAttribute("mute_phone", str(self.mute_phone))

        # Record outputs
        outputs = doc.createElement("outputs")
        for sink in self.pa.sinks.values():
            # Creates user element
            el = doc.createElement("output")
            el.setAttribute("name", sink.name)
            el.setAttribute("priority", str(sink.priority))
            outputs.appendChild(el)
        ec.appendChild(outputs)

        # Plugin status
        """
        plugins = doc.createElement("plugins")
        for plugin in self.plugin_manager.plugins:
            # Creates user element
            el = doc.createElement("plugin")
            el.setAttribute("name", plugin.get_plugin_name())
            el.setAttribute("enabled", str(plugin.enabled))
            plugins.appendChild(el)
        ec.appendChild(plugins)"""

        fp = open(self.config_file,"w")
        doc.writexml(fp, "    ", "", "\n", "UTF-8")
        fp.close()

    def load(self, user_settings=True):

        settings_version = 0
        xml = None
        doc = None

        # Load the defaults
        f = open(self.default_config_file, "r")
        xml = f.read()
        f.close()
        default_doc = parseString(xml)
        default_version = float(default_doc.documentElement.getAttribute("version"))

        # Check for user settings
        if user_settings and os.path.exists( self.config_file ):
            # Load XML
            try:
                f = open(self.config_file, "r")
                xml = f.read()
                f.close()
                doc = parseString(xml)
                if doc.documentElement.hasAttribute("version"):
                    settings_version = float(doc.documentElement.getAttribute("version"))
            except:
                pass

        # Load defaults ?
        if not doc or default_version > settings_version:
            doc = default_doc
            
        # Load client rules
        for el in doc.getElementsByTagName("rule"):
            client = Client(self, "")
            client.from_xml(el)
            if client.role:
                log.info("load rule : %s " % client.description)
            self.pa.clients[client.name] = client
            #self.plugin_manager.check_client( client )


        # Load sink rules
        for el in doc.getElementsByTagName("output"):
            sink = Sink(el.getAttribute("name"), int(el.getAttribute("priority")) )
            self.pa.sinks[sink.name] = sink
            #self.plugin_manager.check_client( client )


        self.fade_timer_speed = float(doc.documentElement.getAttribute("fade_timer_speed"))
        #self.mute_level = float(doc.documentElement.getAttribute("mute_level"))
        #if doc.documentElement.hasAttribute("tray_visible"):
        #    self.status_icon.set_visible( doc.documentElement.getAttribute("tray_visible") == "True" )
        #if doc.documentElement.hasAttribute("prefer_sink"):
        #    self.prefer_sink = doc.documentElement.getAttribute("prefer_sink")
        #if doc.documentElement.hasAttribute("follow_new_outputs"):
        #    self.follow_new_outputs = doc.documentElement.getAttribute("follow_new_outputs") == "True"
        #if doc.documentElement.hasAttribute("mute_phone"):
        #    self.mute_phone = doc.documentElement.getAttribute("mute_phone") == "True"


    @threaded
    def apply_volume_thread(self):
        while True:
            #logging.debug(": apply_volume_thread " )
            if self.active:
                self.apply_volume_thread_running = True
                gobject.idle_add(self.__adjust_volumes)      
            else:
                self.apply_volume_thread_running = False
            time.sleep(self.fade_timer_speed)

    @threaded
    def select_client_thread(self):
        while True:
            if self.active:
                self.select_client_thread_running = True
                gobject.idle_add(self.set_active_clients)  
            else:
                self.select_client_thread_running = False
            time.sleep(0.5)

    def __adjust_volumes(self):
        # Always update based on active sinks
        for sink in self.pa.sink_inputs.values():
            if sink.set_volume():
                # set pa volume
                self.pa.pa_context_set_sink_input_volume(sink.index, sink.volume)  

    def __set_window_stack(self, active_client):
        for client in self.pa.clients.values():

            # Select the primary client and order previous clients
            for key in self.priority_stack.keys():
                if client.role == key:
                    # Add client to role
                    if not client in self.priority_stack[key]:
                        if active_client == client:
                            self.priority_stack[key].insert(0, client)
                        else:
                            self.priority_stack[key].append(client)

                    # reshuffle if active
                    elif active_client == client and not self.priority_stack[key][0] == client:
                        self.priority_stack[key].remove(client)
                        self.priority_stack[key].insert(0, client)

                # If role has changed remove old entry
                elif client in self.priority_stack[key]:
                    self.priority_stack[key].remove(client)

    def print_stacks(self):
        for key in self.priority_stack.keys():
            if key:
                text =""
                for client in self.priority_stack[key]:
                    text += client.description
                    text += " (" 
                    if client.is_active():
                        text += "*"
                    else:
                        text += str(len(client.sinks.values())) 
                    text += "), "
                if text: log.debug("stack " + key + " " + text)

    def set_active_clients(self):

        # Based on per sink stacks
        for sink in self.pa.sinks.values():

            # find the top ranking sink_input for this sink
            highest_sink_input = None
            highest_stack_media_role_index = -1   
            highest_stack_media_role_window_index = -1

            for sink_input in self.pa.sink_inputs.values():
                status = False

                if sink_input.is_active():

                    # check against media role   
                    if sink_input.role and sink_input.role in self.priority_stack.keys():
                    
                        index = self.priority_stack.keys().index(sink_input.role)
                        if index <= highest_stack_media_role_index or highest_stack_media_role_index == -1:

                            # check window stack for client
                            if sink_input.client in self.priority_stack[sink_input.role]:
                                window_index = self.priority_stack[sink_input.role].index(sink_input.client)
                                if window_index <= highest_stack_media_role_window_index or highest_stack_media_role_window_index == -1:
                                    
                                    # Set previous highest to inactive
                                    if highest_sink_input:
                                        highest_sink_input.set_status(False)

                                    # Set new sink to active
                                    highest_sink_input = sink_input
                                    highest_stack_media_role_index = index
                                    highest_stack_media_role_window_index = window_index
                                    status = True
                                    
                    else:
                        # unknown media role so leave it always on
                        status = True

                sink_input.set_status(status)
        #self.print_stacks()

    def on_active_window_change(self, application, state):  #, pid, window_name, x, y, icon, fullscreen
        if application:
            if state in ("active","open"):
                for client in self.pa.clients.values():
                    if self.match_client_to_application(client, application, state): 
                        log.info(": window to client match %s => %s" % (application.name, client.description) )
                        self.__set_window_stack(client)
                        break
                

    def match_client_to_application(self, client, application, index=0):
        if client.test_focus_window(application.pid, application.window_name, application.command, application.name): 
            client.fullscreen = application.fullscreen  
            if not client.icon : client.icon = application.icon
            if not client.icon_name : client.icon_name = application.icon_name
            if not client.description :client.description = application.description
            if not client.role: client.role = application.role
            client.application = application

            return True
        return False

    def set_auto_start(self, flag):
        filename = "earcandy.desktop"
        path = os.path.expanduser(os.getenv('XDG_CONFIG_HOME', '~/.config/autostart'))
        dest = os.path.join(path, filename)
        src = "/usr/share/applications/earcandy.desktop"

        if flag:
            # insure the autostart path exists
            if not os.path.exists(path):
                os.makedirs(path)

            if not os.path.exists( dest ):
                shutil.copyfile(src, dest)
        else:
            if os.path.exists( dest ):
                os.remove( dest )

    def is_auto_start(self):
        filename = "earcandy.desktop"
        dest = os.path.expanduser(os.getenv('XDG_CONFIG_HOME', '~/.config/autostart/' + filename))
        return os.path.exists(dest)

    def open_preferances(self):
        self.print_stacks()
        if not self.pref:
            self.pref = EarCandayPrefs(self)
        self.pref.run()

    def close_preferances(self):
        if self.pref:
            self.pref = None

    def get_current_sink_volume(self):
        self.pa.get_sink_info_by_name(self.pa.prefer_sink_index)


if __name__ == '__main__':
    logging.basicConfig(level=logging.INFO)
    import gtk
    # Turn on gtk threading
    gtk.gdk.threads_init()

    ec = EarCandy()
    ec.run()

    gtk.main()
