#include "ogg.h"
#include <vorbis/codec.h>

#ifdef HAVE_LIBVORBIS

#include <unistd.h>
#include <pthread.h>
#include <stdlib.h>
#include <fcntl.h>
#include <qthread.h>
#include <ctype.h>
#include "mainwnd.h"
#include "pump.h"

#define BLOCK_SKIP 4096

OggStreamSource::OggStreamSource(QWidget *dest) : StreamSource(dest)
{
	fp=NULL;
	file_pos=0L;
}

OggStreamSource::~OggStreamSource()
{
	if(fp)
	{
		if(fp)
			ov_clear(&ov);
	}
}

int OggStreamSource::attach(QString file)
{
	if(access(file, 0) < 0)
	{
		printf("File not found\n");
		return -1;
	}
	
	filename=file;

	fp=fopen(file, "r");
	if(!fp)
	{
		printf("Can't open file\n");
		return -1;
	}

	if(ov_open(fp, &ov, NULL, 0) != 0)
	{
		printf("Cannot open vorbis file\n");
		fclose(fp);
		return -1;
	}

	if(ov_streams(&ov) != 1)
	{
		printf("File has more than one stream\n");
		ov_clear(&ov);
		return -1;
	}

	vorbis_info *info=ov_info(&ov, -1);
	if(!info)
	{
		printf("No bitstream info found\n");
		ov_clear(&ov);
		return -1;
	}


	sample_rate=info->rate;
	if(info->channels != 2)
	{
		printf("Not a stereo file\n");
		ov_clear(&ov);
		return -1;
	}

	frame_pos=0L;
	file_pos=0L;
	max_frames=(unsigned long)ov_pcm_total(&ov, -1);

	m_artist="";
	m_title="";

	vorbis_comment *cmt=ov_comment(&ov, -1);
	if(cmt)
	{
		int i;
		for(i=0;i<cmt->comments;++i)
		{
			char *c=cmt->user_comments[i];
			char *eq=strchr(c, '=');
			if(!eq)
				continue;
			++eq;
			char b[eq-c];
			memcpy(b, c, eq-c);
			b[eq-c-1]='\0';
			int j;
			for(j=0;j<eq-c;j++)
				b[j]=toupper(b[j]);
			if(!strcmp(b, "ARTIST"))
				m_artist=eq;
			if(!strcmp(b, "TITLE"))
				m_title=eq;
		}
	}

	if(m_artist == "" && m_title == "")
	{
		char fnb[1024];
		strcpy(fnb, file);
		char *fn=strrchr(fnb, '/');
		if(!fn)
			fn=fnb;
		else
			++fn;
		char *dot=strrchr(fn, '.');
		if(dot)
			(*dot)='\0';
		char *sep=strstr(fn, " - ");
		if(sep)
		{
			(*sep)='\0';
			sep+=3;
		}

		if(sep)
		{
			m_artist=fn;
			m_title=sep;
		}
		else
		{
			m_artist="";
			m_title=fn;
		}
	}
	return 0;
}

int OggStreamSource::get_buffer(char *buf, int max)
{
	if(!fp || !playing)
		return 0;
	
	if(file_pos+max > max_frames)
		max=max_frames-file_pos;

	if(!max)
	{
		if(playing)
			QThread::postEvent(this, new QEvent(QEvent::User));
		playing=0;
		return 0;
	}

	if(play_samples)
	{
		if((unsigned long)max > play_samples)
			max=(int)play_samples;
	}

	long bytes_read=0;

	short tb[max*2];
	char *bufp=(char *)tb;

	while(1)
	{
		int bytes=ov_read(&ov, bufp, max*sizeof(short)*2-bytes_read, 0, 2, 1, &stream_num);
		if(!bytes)
		{
			max=bytes_read/(sizeof(short)*2);
			break;
		}
		if(bytes == OV_HOLE)
			continue;
		if(bytes == OV_EBADLINK)
			continue;
		bytes_read+=bytes;
		bufp+=bytes;
		if(bytes_read == max*sizeof(short)*2)
			break;
	}

	int i;
	for(i=0;i<max;i++)
	{
		*(((short *)buf)+i)=tb[i*2];
		*(((short *)buf)+max+i)=tb[i*2+1];
	}

	file_pos+=max;

	if(play_samples)
	{
		play_samples-=max;
		if(play_samples <= 0 || max == 0)
		{
			if(loop_play)
				play(last_start, last_samples, true);
			else
			{
				playing=0;
				QThread::postEvent(this, new QEvent(QEvent::User));
			}
		}
	}
	
	calcAgc(buf, max);
	return max;
}

int OggStreamSource::play(unsigned long start_frame, unsigned long samples, bool loop)
{
	if(!fp)
		return -1;
	
	if(!loop_play || !loop)
		stop();

	if(start_frame >= max_frames)
		return -1;

	loop_play=loop;
	last_start=start_frame;

	ov_pcm_seek(&ov, (ogg_int64_t) start_frame);

	file_pos=start_frame;
	frame_pos=start_frame;
	play_samples=samples;
	last_samples=play_samples;

	playing=1;
	app_window->get_pump()->work();
	return 0;
}

int OggStreamSource::stop(void)
{
	if(!player || !fp || !playing)
		return 0;

	StreamSource::stop();
	playing=0;
	return 0;
}

StreamSource *OggStreamSource::get_source(void)
{
	OggStreamSource *s=new OggStreamSource(app_window);
	s->attach(filename);
	return (StreamSource *)s;
}

void OggStreamSource::exit_loop(void)
{
	pthread_mutex_lock(&control_lock);
	play_samples=0;
	loop_play=false;
	pthread_mutex_unlock(&control_lock);
}

#endif /* HAVE_LIBVORBIS */
